/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 CodeFactory AB
 * Copyright (C) 2001 Mikael Hallendal <micke@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Mikael Hallendal <micke@codefactory.se>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define d(x)

#include <bonobo.h>
#include "util/type-utils.h"
#include "allocation-manager-client.h"

static void  allocation_mc_init        (AllocationManagerClient       *rmc);
static void  allocation_mc_class_init  (AllocationManagerClientClass  *klass);
static void  amc_destroy               (GtkObject                     *object);

static void  amc_listener              (ManagerClient                 *mc,
					gchar                         *subtype,
					CORBA_any                     *any);
static void  amc_project_set           (ManagerClient                 *amc);
static void  amc_project_unset         (ManagerClient                 *amc);

enum {
	ALLOCATION_ADDED,
	ALLOCATIONS_REMOVED,
	LAST_SIGNAL
};

static gint signals[LAST_SIGNAL] = { 0 };

struct _AllocationManagerClientPriv {
        GM_AllocationManager manager;

};

GNOME_CLASS_BOILERPLATE (AllocationManagerClient, allocation_mc,
                         ManagerClient,           manager_client);

static void
allocation_mc_init (AllocationManagerClient *amc)
{
        AllocationManagerClientPriv *priv;
        
	d(puts(__FUNCTION__));

        priv = g_new0 (AllocationManagerClientPriv, 1);
        
        amc->priv = priv;
}

static void
allocation_mc_class_init (AllocationManagerClientClass *klass)
{
	GtkObjectClass     *object_class;
	ManagerClientClass *mcc;
	
	object_class = (GtkObjectClass *) klass;
	mcc          = MANAGER_CLIENT_CLASS (klass);
	
	mcc->listener         = amc_listener;
	mcc->project_set      = amc_project_set;
	mcc->project_unset    = amc_project_unset;

	object_class->destroy = amc_destroy;

	d(puts(__FUNCTION__));

	signals[ALLOCATION_ADDED] = 
		gtk_signal_new ("allocation_added",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (AllocationManagerClientClass,
						   allocation_added),
				gtk_marshal_NONE__POINTER,
				GTK_TYPE_NONE,
				1, GTK_TYPE_POINTER);

	signals[ALLOCATIONS_REMOVED] = 
		gtk_signal_new ("allocations_removed",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (AllocationManagerClientClass,
						   allocations_removed),
				gtk_marshal_NONE__POINTER,
				GTK_TYPE_NONE,
				1, GTK_TYPE_POINTER);

	gtk_object_class_add_signals (object_class, signals, LAST_SIGNAL);
}

static void
amc_destroy (GtkObject *object)
{
        AllocationManagerClient     *amc;
        AllocationManagerClientPriv *priv;
	
        g_return_if_fail (object != NULL);
        g_return_if_fail (IS_ALLOCATION_MANAGER_CLIENT (object));
        
	d(puts (__FUNCTION__));

        amc  = ALLOCATION_MANAGER_CLIENT (object);
	priv = amc->priv;

	bonobo_object_release_unref (priv->manager, NULL);
	priv->manager = NULL;

	g_free (amc->priv);
        amc->priv = NULL;

	GNOME_CALL_PARENT_HANDLER (GTK_OBJECT_CLASS, destroy, (object));
}

static void 
amc_listener (ManagerClient *mc, gchar *subtype, CORBA_any *any)
{
	AllocationManagerClient     *amc;
	AllocationManagerClientPriv *priv;
        
	g_return_if_fail (mc != NULL);
	g_return_if_fail (IS_ALLOCATION_MANAGER_CLIENT (mc));
	
	d(puts(__FUNCTION__));

	amc  = ALLOCATION_MANAGER_CLIENT (mc);
	priv = amc->priv;
	
	/* Emit signals */

	if (!strcmp (subtype, "added")) {
                GM_Allocation   *allocation;

                allocation = (GM_Allocation *) any->_value;
                
                gtk_signal_emit (GTK_OBJECT (amc),
                                 signals[ALLOCATION_ADDED],
                                 allocation);
	}
	else if (!strcmp (subtype, "removed")) {
                GM_AllocationSeq   *allocations;
                GSList             *list;
                
                allocations = (GM_AllocationSeq *) any->_value;

		if (allocations) {
			list = corba_util_allocation_seq_to_list (allocations);
			
			gtk_signal_emit (GTK_OBJECT (amc),
					 signals[ALLOCATIONS_REMOVED],
					 list);
			
			g_slist_CORBA_free (list);
		} else {
			g_warning ("Got empty deassignment");
		}
	} else {
		d(g_print ("AllocationManagerClient: got unhandled"
			   "event: %s\n", subtype));
	}
}

static void
amc_project_set (ManagerClient *mc) 
{
	AllocationManagerClient     *amc;
	AllocationManagerClientPriv *priv;
	GM_Project                   project_co;
	CORBA_Environment            ev;
	
	g_return_if_fail (mc != NULL);
	g_return_if_fail (IS_ALLOCATION_MANAGER_CLIENT (mc));
	
	d(puts (__FUNCTION__));

	amc  = ALLOCATION_MANAGER_CLIENT (mc);
	priv = amc->priv;

	if (priv->manager) {
		bonobo_object_release_unref (priv->manager, NULL);
	}

	if (!mc->shell) {
		g_warning ("AllocationManagerClient doesn't have a shell");
		return;
	}

	CORBA_exception_init (&ev);

	project_co = 
		GNOME_MrProject_Shell_getProject (mc->shell, 
						  &ev);
	
	if (BONOBO_EX (&ev) || !project_co) {
		g_warning ("Shell doesn't have a project");
		CORBA_exception_free (&ev);
		return;
	}

	priv->manager = Bonobo_Unknown_queryInterface (project_co,
						       "IDL:GNOME/MrProject/AllocationManager:1.0",
						       &ev);
	
	if (BONOBO_EX (&ev)) {
		g_warning ("Couldn't get AllocationManager interface through project");
	}

	CORBA_Object_release (project_co, NULL);
	
	CORBA_exception_free (&ev);
}

static void
amc_project_unset (ManagerClient *mc)
{
	AllocationManagerClient     *amc;
	AllocationManagerClientPriv *priv;

	g_return_if_fail (mc != NULL);
	g_return_if_fail (IS_ALLOCATION_MANAGER_CLIENT (mc));

	d(puts (__FUNCTION__));

	amc  = ALLOCATION_MANAGER_CLIENT (mc);
	priv = amc->priv;

	if (priv->manager) {
		bonobo_object_release_unref (priv->manager, NULL);
	}

	priv->manager = CORBA_OBJECT_NIL;
}

ManagerClient *
allocation_mc_new (GM_Shell shell, gboolean set, CORBA_Environment *ev)
{
	AllocationManagerClient     *amc;
	AllocationManagerClientPriv *priv;
	
	d(puts (__FUNCTION__));

	amc  = gtk_type_new (ALLOCATION_MANAGER_CLIENT_TYPE);
	priv = amc->priv;
	
	manager_client_set_shell (MANAGER_CLIENT (amc),
				  shell,
				  "allocation",
				  ev);
	if (set) {
		amc_project_set (MANAGER_CLIENT(amc));
	}
 	
	return MANAGER_CLIENT(amc);
}

void
allocation_mc_allocate (AllocationManagerClient *amc,
			GM_Id                    task_id,
			GM_Id                    resource_id,
			short                    units,
			CORBA_Environment       *ev)
{
	AllocationManagerClientPriv *priv;

	g_return_if_fail (amc != NULL);
	g_return_if_fail (IS_ALLOCATION_MANAGER_CLIENT(amc));

	d(puts(__FUNCTION__));

	priv = amc->priv;

	GNOME_MrProject_AllocationManager_allocate (priv->manager,
						    task_id,
						    resource_id,
						    ev);

}

void
allocation_mc_deallocate (AllocationManagerClient *amc,
			  GM_Allocation           *allocation,
			  CORBA_Environment       *ev)
{
	AllocationManagerClientPriv *priv;
	
	g_return_if_fail (amc != NULL);
	g_return_if_fail (IS_ALLOCATION_MANAGER_CLIENT(amc));

	d(puts(__FUNCTION__));

	priv = amc->priv;
	
	GNOME_MrProject_AllocationManager_deallocate (priv->manager,
						      allocation,
						      ev);
	
}

void
allocation_mc_deallocate_ids (AllocationManagerClient *amc,
			      GM_Id                    task_id,
			      GM_Id                    resource_id,
			      CORBA_Environment       *ev)
{
	GM_Allocation *allocation;

	allocation             = GNOME_MrProject_Allocation__alloc ();
	allocation->taskId     = task_id;
	allocation->resourceId = resource_id;
	allocation->units      = 0;
	
	d(puts(__FUNCTION__));

	allocation_mc_deallocate (amc, allocation, ev);
	
	CORBA_free (allocation);
}

GSList *
allocation_mc_get_allocations_by_task (AllocationManagerClient *amc,
				       GM_Id                    task_id,
				       CORBA_Environment       *ev)
{
	AllocationManagerClientPriv *priv;
	GM_AllocationSeq            *allocations;
	GSList                      *list;

	g_return_val_if_fail (amc != NULL, NULL);
	g_return_val_if_fail (IS_ALLOCATION_MANAGER_CLIENT(amc), NULL);

	d(puts(__FUNCTION__));

	priv = amc->priv;
	
	allocations =  GNOME_MrProject_AllocationManager_getAllocationsByTask (priv->manager,
									       task_id,
									       ev);
	
	/* FIX: Check for exceptions */

	if (BONOBO_EX (ev) || !allocations) {
		g_warning ("Couldn't get initial allocations");
		return NULL;
	}
	

	list = corba_util_allocation_seq_to_list (allocations);

	CORBA_free (allocations);

	return list;
}

GSList *       
allocation_mc_get_allocations_by_resource (AllocationManagerClient *amc,
					   GM_Id                    res_id,
					   CORBA_Environment       *ev)
{
	AllocationManagerClientPriv *priv;
	GM_AllocationSeq            *allocations;
	GSList                      *list;
	
	g_return_val_if_fail (amc != NULL, NULL);
	g_return_val_if_fail (IS_ALLOCATION_MANAGER_CLIENT(amc), NULL);

	d(puts(__FUNCTION__));

	priv = amc->priv;

	allocations = GNOME_MrProject_AllocationManager_getAllocationsByResource (priv->manager,
										  res_id,
										  ev);
	
	/* FIX: Check for exceptions */

	list = corba_util_allocation_seq_to_list (allocations);

	CORBA_free (allocations);
	
	return list;
}

GSList *           
allocation_mc_get_all_allocations (AllocationManagerClient *amc,
				   CORBA_Environment       *ev)
{
	AllocationManagerClientPriv *priv;
	GM_AllocationSeq            *allocations;
	GSList                      *list;
	
	g_return_val_if_fail (amc != NULL, NULL);
	g_return_val_if_fail (IS_ALLOCATION_MANAGER_CLIENT(amc), NULL);

	d(puts(__FUNCTION__));

	priv = amc->priv;

	allocations = GNOME_MrProject_AllocationManager_getAllAllocations (priv->manager,
									ev);
	
	/* FIX: Check for exceptions */
	list = corba_util_allocation_seq_to_list (allocations);

	CORBA_free (allocations);
	
	return list;
}

void
allocation_mc_deallocate_all_resources (AllocationManagerClient *amc,
					GM_Id                    task_id,
					CORBA_Environment       *ev)
{
	AllocationManagerClientPriv *priv;
	
	g_return_if_fail (amc != NULL);
	g_return_if_fail (IS_ALLOCATION_MANAGER_CLIENT(amc));

	d(puts(__FUNCTION__));

	priv = amc->priv;

	GNOME_MrProject_AllocationManager_deallocateAllResources(priv->manager,
								 task_id,
								 ev);

}

void
allocation_mc_deallocate_all_tasks (AllocationManagerClient *amc,
				    GM_Id                    resource_id,
				    CORBA_Environment       *ev)
{
	AllocationManagerClientPriv *priv;
	
	g_return_if_fail (amc != NULL);
	g_return_if_fail (IS_ALLOCATION_MANAGER_CLIENT(amc));

	d(puts(__FUNCTION__));

	priv = amc->priv;

	GNOME_MrProject_AllocationManager_deallocateAllTasks (priv->manager,
							      resource_id,
							      ev);
	
}



