/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2000 CodeFactory AB
 * Copyright (C) 2000 Richard Hult <rhult@codefactory.se>
 * Copyright (C) 2001 Mikael Hallendal <micke@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gal/unicode/gunicode.h>
#include <dlfcn.h>

#include <signal.h>
#include <bonobo/bonobo-main.h>
#include <liboaf/liboaf.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include <libgnomevfs/gnome-vfs-init.h>
#include <gconf/gconf.h>
#include <glade/glade.h>

#include "libmrproject/GNOME_MrProject.h"
#include "project-window.h"
#include "util/eel-debug.h"
#include "util/corba-utils.h"
#include "widgets/mr-message-box.h"

#include "shell.h"

#define PROJECT_FACTORY_OAFIID "OAFIID:GNOME_MrProject_ProjectFactory"

GSList *open_projects;
static CORBA_Object factory_co;
static void (*segv_handler) (int) = NULL;

static guint
activate_idle (gpointer user_data)
{
	const char        **startup_files;
	CORBA_Environment   ev;
	gint                i;
	gchar              *uri;
	GtkWidget          *window;

	CORBA_exception_init (&ev);
	
	factory_co = oaf_activate_from_id (PROJECT_FACTORY_OAFIID,
					   0,    /* OAF_ActivationFlags */
					   NULL, /* OAF_ActivationID */
					   &ev);
	if (BONOBO_EX (&ev) || factory_co == CORBA_OBJECT_NIL) {
		gchar *msg = g_strdup_printf (
			"Could not activate the Project factory. This probably "
			"means that the prefix MrProject is installed into "
			"differs from GNOME's and OAF's prefix.\n\n"
			"You can solve this by setting the environment "
			"variable OAF_INFO_PATH to %sshare/oaf. Then "
			"terminate oafd and try again.", PREFIX);

		mr_message_box_show_message (
			GNOME_MESSAGE_BOX_ERROR,
			msg);

		gtk_main_quit ();
		exit (1);
	}
	
	startup_files = user_data;
	if (startup_files) {
		for (i = 0; startup_files[i]; ++i) {
			gchar *tmp, *filename;

			/* Make sure we have an absolute path. */
			if (startup_files[i][0] != '/') {
				tmp = g_get_current_dir ();
				filename = g_strconcat (tmp, "/", startup_files[i], NULL);
				g_free (tmp);
			} else {
				filename = g_strdup (startup_files[i]);
			}

			uri = gnome_vfs_get_uri_from_local_path (filename);
			g_free (filename);

			window = project_window_new (uri);
			g_free (uri);
		}
	} else {
		window = project_window_new (NULL);
	}
	
	return FALSE;
}

/* Do this slightly hacky thing to be nice to users. It needs to be handled
 * differently when there are more backends. Now we just pass a filename and
 * assume that the project can be saved there.
 */
static void
signal_handler (int signum)
{
	GSList            *list;
	gint               i;
	CORBA_Environment  ev;

	switch (signum) {
	case SIGSEGV:
		puts ("Handling SIGSEGV.");

		/* Restore the handler. */
		signal (SIGSEGV, SIG_DFL);

		CORBA_exception_init (&ev);

		mr_message_box_show_message (
			GNOME_MESSAGE_BOX_ERROR,
			"MrProject has crashed.\n\nAn attempt to save any open "
			"projects will be done. Files will be saved in your home "
			"directory, with names such as mrproject-saved<number>.mrproject.");

		/* Try to save the open projects. */
		for (i = 1, list = open_projects; list; i++, list = list->next) {
			gchar *home, *name;

			home = g_get_home_dir ();
			name = g_strdup_printf (
				"file:///%s/mrproject-saved%d.mrproject",
				home,
				i);
			GNOME_MrProject_Project_save (list->data, name, &ev);
			CORBA_exception_free (&ev);
		}
		break;

	default:
		break;
	}

	exit (1);
}

static void
log_handler (const gchar *log_domain, GLogLevelFlags log_level,
	     const gchar *message, gpointer user_data)
{
	switch (log_level) {
	case G_LOG_LEVEL_CRITICAL:
		mr_message_box_show_message (GNOME_MESSAGE_BOX_ERROR, message);
		break;
	case G_LOG_LEVEL_WARNING:
		mr_message_box_show_message (GNOME_MESSAGE_BOX_WARNING, message);
		break;
	case G_LOG_LEVEL_INFO:
	case G_LOG_LEVEL_MESSAGE:
	default:
		mr_message_box_show_message (GNOME_MESSAGE_BOX_INFO, message);
		break;
	}
}

int
main (int argc, char *argv[])
{
	CORBA_ORB           orb;
	const char        **startup_files;
	guint               handler_id;
	static gint         option_debug;
	static poptContext  pctx;
	static const struct poptOption mrproject_popt_options [] = {
		{
			"debug",
			'\0',
			POPT_ARG_INT,
			&option_debug,
			0,
			N_("Enables some debugging functions"),
			N_("LEVEL")
		},
		{
			NULL,
			'\0',
			0,
			NULL,
			0
		}
	};

	bindtextdomain (PACKAGE, GNOMELOCALEDIR);
	textdomain (PACKAGE);

        gnome_init_with_popt_table ("MrProject", VERSION, argc, 
				    argv, mrproject_popt_options, 0, &pctx);

	glade_gnome_init ();

	gtk_widget_push_visual (gdk_rgb_get_visual ());
	gtk_widget_push_colormap (gdk_rgb_get_cmap ());

	if (g_getenv ("MRPROJECT_DEBUG") != NULL) {
		eel_make_warnings_and_criticals_stop_in_debugger
			(G_LOG_DOMAIN, g_log_domain_glib,
			 "MrProject",
			 "Gantt",
			 "Resource",
			 "Network",
			 "Send",
			 "Python",
			 "e-table",
			 "Util",
			 NULL);
	}
	
	orb = oaf_init (argc, argv);
	gnome_vfs_init ();
	if (bonobo_init (orb, NULL, NULL) == FALSE) {
		g_error ("Could not initialize Bonobo.");
	}

	if (gconf_init (argc, argv, NULL) == FALSE) {
		g_error ("Could not initialize GConf.");
	}
	
	if (pctx) {
		startup_files = poptGetArgs (pctx);
	} else {
		startup_files = NULL;
	}

	open_projects = NULL;	

	/* Try to clean up when crashing. FIXME: should use sigaction. */
	if (g_getenv ("MRPROJECT_DEBUG") == NULL) {
		segv_handler = signal (SIGSEGV, signal_handler);
	}
	
	/* FIXME: This could happen when a component dies as well. Can
	 * we check for that and use bonobo_exception_add_handler_fn instead
	 * to give two different messages?
	 */
	/* Note to translators: the engine is a separate bonobo exe component. */
	bonobo_exception_add_handler_str ("IDL:CORBA/COMM_FAILURE:1.0", 
					  _("The connection to the engine was broken. "
					  "This probably means that the engine crashed."));

	/* This is not 100% correct (URI / file), but user friendly... */
	bonobo_exception_add_handler_str ("IDL:GNOME/MrProject/Project/URINotFound:1.0", 
					  _("File not found."));

	bonobo_exception_add_handler_str ("IDL:GNOME/MrProject/Project/MethodNotSupported:1.0", 
					  _("Could not find a file loader for this format."));

	g_idle_add ((GSourceFunc) activate_idle, startup_files);

	/* Redirect messages. FIXME: we should probably have a
	 * domain of our own and only use that one to display
	 * user errors/warnings/messages, and let the usual ones
	 * go to stderr. We are not supposed to have any warnings
	 * anyway so...
	 */
	handler_id = g_log_set_handler (G_LOG_DOMAIN, 
			                G_LOG_LEVEL_WARNING | 
					G_LOG_LEVEL_CRITICAL | 
			                G_LOG_LEVEL_MESSAGE | 
					G_LOG_LEVEL_INFO, 
			                log_handler, NULL);

/*	g_print ("before: %p\n", g_utf8_skip);
	{
		void *handle;
		gpointer sliff;
		gint i;
		
		for (i = 0; i < 10; i++) {
			g_print ("%d ", (int)g_utf8_skip[i]);
		}
		


 
		handle = dlopen ("libgal.so", RTLD_NOW);

		sliff = dlsym (handle, "g_utf8_skip");
		g_print ("dlopened: %p\n", sliff);
	}

	g_print ("after: %p\n", g_utf8_skip);
*/
	bonobo_main ();

	g_log_remove_handler (G_LOG_DOMAIN, handler_id);

	return 0;
}

