/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 CodeFactory AB
 * Copyright (C) 2001 Mikael Hallendal <micke@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Mikael Hallendal <micke@codefactory.se>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkmarshal.h>
#include <stdio.h> /* Remove when all puts are gone */
#include "alloc-table-model.h"
#include "util/corba-utils.h"
#include "util/id-map.h"
#include "util/type-utils.h"

static void     alloc_table_model_init       (AllocTableModel         *model);
static void     alloc_table_model_class_init (AllocTableModelClass    *klass);

static void     atm_destroy                  (GtkObject               *object);

static gint     atm_col_count                (ETableModel             *etm);
static gint     atm_row_count                (ETableModel             *etm);
static void *   atm_value_at                 (ETableModel             *etm,
					      gint                     col,
					      gint                     row);

static void     atm_set_value_at             (ETableModel             *etm,
					      gint                     col,
					      gint                     row,
					      const void              *value);

static gboolean atm_is_cell_editable         (ETableModel             *etm,
					      gint                     col,
					      gint                     row);

static void     atm_free_value               (ETableModel             *etm,
					      gint                     col,
					      void                    *value);
static gboolean atm_value_is_empty           (ETableModel             *etm,
					      gint                     col,
					      const void              *value);
static gchar *  atm_value_to_string          (ETableModel             *etm,
					      gint                     col,
					      const void              *value);

static gint     atm_get_row                  (AllocTableModel         *atm,
					      gint                     id);

struct _AllocTableModelPriv {
	GArray       *resources;
	gint          nr_of_cols;
	gint          nr_of_rows;

	IdMap        *allocations;
};

enum {
        RESOURCE_ALLOCATED,
        RESOURCE_DEALLOCATED,
        LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0 };
                         
GNOME_CLASS_BOILERPLATE (AllocTableModel, alloc_table_model,
			 ETableModel,     e_table_model);

#define d(x)

static void
alloc_table_model_init (AllocTableModel *model)
{
	AllocTableModelPriv *priv;
	
	priv              = g_new0 (AllocTableModelPriv, 1);
	model->priv       = priv;
	priv->resources   = g_array_new (FALSE, FALSE,
					  sizeof (GM_Resource *));
	priv->allocations = id_map_new (0);
	priv->nr_of_cols  = LAST_AT_COL;
	priv->nr_of_rows  =  0;
}

static void
alloc_table_model_class_init (AllocTableModelClass *klass)
{
	GtkObjectClass   *object_class;
	ETableModelClass *model_class;

	object_class = (GtkObjectClass *) klass;
	model_class  = E_TABLE_MODEL_CLASS (klass);
	parent_class = gtk_type_class (E_TABLE_MODEL_TYPE);
	
	/* -- GtkObject functions   -- */
	object_class->destroy         = atm_destroy;
	
	/* -- ETableModel functions -- */
	model_class->column_count     = atm_col_count;
 	model_class->row_count        = atm_row_count; 
	model_class->value_at         = atm_value_at;
	model_class->set_value_at     = atm_set_value_at;
 	model_class->is_cell_editable = atm_is_cell_editable; 
	model_class->free_value       = atm_free_value;
	model_class->value_is_empty   = atm_value_is_empty;
	model_class->value_to_string  = atm_value_to_string;

        signals[RESOURCE_ALLOCATED] =
		gtk_signal_new ("resource_allocated",
				GTK_RUN_LAST, 
				object_class->type,
				GTK_SIGNAL_OFFSET (AllocTableModelClass,
						   resource_allocated),
				gtk_marshal_NONE__INT,
				GTK_TYPE_NONE,
 				1, GTK_TYPE_INT);

	signals[RESOURCE_DEALLOCATED] =
		gtk_signal_new ("resource_deallocated",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (AllocTableModelClass,
						   resource_deallocated),
				gtk_marshal_NONE__INT,
				GTK_TYPE_NONE,
				1, GTK_TYPE_INT);
	
	gtk_object_class_add_signals (object_class, signals, LAST_SIGNAL);
}

static void
atm_destroy (GtkObject *object)
{
}

static gint
atm_col_count (ETableModel *etm)
{
	g_return_val_if_fail (etm != NULL, -1);
	g_return_val_if_fail (IS_ALLOC_TABLE_MODEL (etm), -1);
	
	return ALLOC_TABLE_MODEL (etm)->priv->nr_of_cols;
}

static gint
atm_row_count (ETableModel *etm)
{
	g_return_val_if_fail (etm != NULL, -1);
	g_return_val_if_fail (IS_ALLOC_TABLE_MODEL (etm), -1);
	
	return ALLOC_TABLE_MODEL (etm)->priv->nr_of_rows;
}

static void *
atm_value_at (ETableModel *etm, gint col, gint row)
{
	AllocTableModel     *atm;
	AllocTableModelPriv *priv;
	GM_Resource         *resource;
	gint                 allocated;
	
	g_return_val_if_fail (etm != NULL, NULL);
	g_return_val_if_fail (IS_ALLOC_TABLE_MODEL (etm), NULL);

	atm  = ALLOC_TABLE_MODEL (etm);
	priv = atm->priv;
	
	if (row >= priv->nr_of_rows) {
		g_warning ("get_value_at: Requested row to large: %d\n", row);
		return NULL;
	}
	
	resource = g_array_index (priv->resources,
				  GNOME_MrProject_Resource *,
				  row);

	if (!resource) {
		g_warning ("get_value_at: No ResourceRow at %d\n", row);
		return NULL;
	}

	switch (col) {
	case AT_COL_ID:
		return GINT_TO_POINTER (resource->resourceId);
		break;
	case AT_COL_ALLOCATED:
		allocated = 0;

		if (id_map_lookup (priv->allocations, resource->resourceId)) {
			allocated = 1;
		}
		return GINT_TO_POINTER (allocated);
		break;
	case AT_COL_NAME:
		return CORBA_string_dup (resource->name);
		break;
	case AT_COL_UNITS:
		return GINT_TO_POINTER  (resource->units);
		break;
	default:
		return NULL;
	}

	return NULL;
}

static void
atm_set_value_at (ETableModel *etm, gint col, gint row, const void *value)
{
	AllocTableModel     *atm;
	AllocTableModelPriv *priv;
	GM_Resource         *resource;
	gint                 allocated;
	
	g_return_if_fail (etm != NULL);
	g_return_if_fail (IS_ALLOC_TABLE_MODEL (etm));

	atm  = ALLOC_TABLE_MODEL (etm);
	priv = atm->priv;
	
	if (row >= priv->nr_of_rows) {
		g_warning ("get_value_at: Requested row to large: %d\n", row);
		return;
	}
	
	resource = g_array_index (priv->resources,
				  GNOME_MrProject_Resource *,
				  row);

	if (!resource) {
		g_warning ("get_value_at: No ResourceRow at %d\n", row);
		return;
	}

	switch (col) {
	case AT_COL_ALLOCATED:
		allocated = GPOINTER_TO_INT (value);

		if (allocated) {
			id_map_insert_id (priv->allocations, 
					  resource->resourceId,
					  GINT_TO_POINTER (100));

			gtk_signal_emit (GTK_OBJECT (atm),
					 signals[RESOURCE_ALLOCATED],
					 resource->resourceId);
		} else {
			id_map_remove (priv->allocations,
				       resource->resourceId);

			gtk_signal_emit (GTK_OBJECT (atm),
					 signals[RESOURCE_DEALLOCATED],
					 resource->resourceId);
		}
		break;
	default:
		/* Shouldn't get here, do nothing */
		break;
	}
}

static gboolean 
atm_is_cell_editable (ETableModel *etm, gint col, gint row)
{
	if (col == AT_COL_ALLOCATED) {
		return TRUE;
	}
	
	return FALSE;
}

static void     
atm_free_value (ETableModel *etm, gint col, void *value)
{
	d(puts (__FUNCTION__));
}

static gboolean 
atm_value_is_empty (ETableModel *etm, gint col, const void *value)
{
	d(puts (__FUNCTION__));
	
	return TRUE;
}

static gchar *  
atm_value_to_string (ETableModel *etm, gint col, const void *value)
{
	d(puts (__FUNCTION__));

	return g_strdup ("");
}

static gint
atm_get_row (AllocTableModel *atm, gint id)
{
	AllocTableModelPriv *priv;
	GM_Resource         *resource;
	gint                 i;
	
	g_return_val_if_fail (atm != NULL, -1);
	g_return_val_if_fail (IS_ALLOC_TABLE_MODEL (atm), -1);
	
	priv = atm->priv;
	
	for (i = 0; i < priv->nr_of_rows; ++i) {
		resource = g_array_index (priv->resources,
					  GNOME_MrProject_Resource *,
					  i);
		if (resource->resourceId == id) {
			return i;
		}
	}

	return -1;
}

ETableModel *
alloc_table_model_new ()
{
	AllocTableModel *atm;
	
	atm = gtk_type_new (ALLOC_TABLE_MODEL_TYPE);
	
	return E_TABLE_MODEL (atm);
}

void   
alloc_table_model_add_resource (AllocTableModel *atm, GM_Resource *res)
{
	AllocTableModelPriv *priv;
	GM_Resource         *resource;
	
	g_return_if_fail (atm != NULL);
	g_return_if_fail (IS_ALLOC_TABLE_MODEL (atm));
	
	priv = atm->priv;

	resource = corba_util_resource_duplicate(res);

 	g_array_append_val (priv->resources, resource); 

	priv->nr_of_rows++;

	e_table_model_row_inserted (E_TABLE_MODEL (atm), priv->nr_of_rows - 1);
  	e_table_model_changed (E_TABLE_MODEL (atm));  
}

void
alloc_table_model_update_resource (AllocTableModel *atm, GM_Resource *res)
{
	GM_Resource *resource;
	gint         row;
	
	g_return_if_fail (atm != NULL);
	g_return_if_fail (IS_ALLOC_TABLE_MODEL (atm));
	g_return_if_fail (res != NULL);

	d(puts (__FUNCTION__));
	
	row      = atm_get_row (atm, res->resourceId);
	resource = g_array_index (atm->priv->resources, GM_Resource *, row);
	
	if (!resource) {
		return;
	}

	if (corba_util_resource_update (resource, res)) {
		
/* 	e_table_model_row_changed (E_TABLE_MODEL (atm), atm->priv->nr_of_rows); */
		e_table_model_changed (E_TABLE_MODEL (atm));
	}
}

void
alloc_table_model_remove_resource (AllocTableModel *atm, GM_Id res_id)
{
	GM_Resource *resource;
	gint         row;
	
	g_return_if_fail (atm != NULL);
	g_return_if_fail (IS_ALLOC_TABLE_MODEL (atm));
	
	row      = atm_get_row (atm, res_id);
	resource = g_array_index (atm->priv->resources, GM_Resource *, row);
	
	if (!resource) {
		return;
	}

	atm->priv->resources = g_array_remove_index (atm->priv->resources, 
						     row);

	CORBA_free (resource);

	atm->priv->nr_of_rows--;

	e_table_model_row_deleted (E_TABLE_MODEL (atm), atm->priv->nr_of_rows);
	e_table_model_changed (E_TABLE_MODEL (atm));
}

void
alloc_table_model_set_allocation (AllocTableModel *atm,
				  GM_Id            res_id,
				  gboolean         allocate)
{
	AllocTableModelPriv *priv;
	gint                 row;
	
	g_return_if_fail (atm != NULL);
	g_return_if_fail (IS_ALLOC_TABLE_MODEL (atm));

	priv = atm->priv;
	row  = atm_get_row (atm, res_id);
	
	if (row == -1) {
		/* No resource we know of */
		return;
	}
	
	if (allocate) {
		id_map_insert_id (priv->allocations, res_id, 
				  GINT_TO_POINTER (1));
	} else {
		id_map_remove (priv->allocations, res_id);
	}

	e_table_model_changed (E_TABLE_MODEL (atm));
}

