/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2000-2001 CodeFactory AB
 * Copyright (C) 2000-2001 Richard Hult <rhult@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <math.h>
#include <libgnomeui/gnome-canvas-rect-ellipse.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-app.h>
#include <libgnomeui/gnome-app-helper.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-messagebox.h>
#include <libgnomeui/gnome-popup-menu.h>
#include <libgnomeui/gnome-uidefs.h>
#include <gal/widgets/e-canvas.h>
#include <gal/e-table/e-table.h>
#include <gal/e-table/e-tree-model.h>
#include <glade/glade.h>
#include "util/type-utils.h"
#include "util/time-utils.h"
#include "client/widgets/mr-hscrollbar.h"
#include "gantt-chart.h"
#include "gantt-item.h"
#include "gantt-header-item.h"
#include "gantt-model.h"
#include "gantt-scale.h"


typedef struct {
	GtkWidget *window;
	GtkWidget *custom_sb;
	GtkWidget *custom_om;
	GtkWidget *reset_btn;

	/* Because of the broken option menu we
	 * need to keep track of the currently
	 * selected value here. This is an index
	 * into the unit array.
	 */
	gint        custom_unit;
	
	/* Keep this around so we can reset to it. */
	GtkWidget *original_btn;
	gint       original_custom_unit; /* index in optionmenu */
	gfloat     original_custom_value;
	time_t     original_t1, original_t2;
} ZoomDialog;

/* Private members. */
struct _GanttChartPriv {
	ZoomDialog *zoom_dialog;
	GtkWidget  *timescale_dialog;
};

#define DEFAULT_SCALE 0.0005

static void gantt_chart_class_init (GanttChartClass *klass);
static void gantt_chart_init       (GanttChart      *chart);
static gdouble get_gantt_allocation_width (GanttChart *chart);

GNOME_CLASS_BOILERPLATE (GanttChart, gantt_chart, GtkTable, gtk_table);

static void
gantt_chart_destroy (GtkObject *object)
{
	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GANTT_CHART (object));
	g_return_if_fail (GTK_OBJECT_CONSTRUCTED (object));

	if (!GTK_OBJECT_DESTROYED (object)) {
		GanttChart *gantt = GANTT_CHART (object);

		g_free (gantt->priv);
		gantt->priv = NULL;
	}

	GNOME_CALL_PARENT_HANDLER (GTK_OBJECT_CLASS, destroy, (object));
}

static void
gantt_chart_class_init (GanttChartClass *klass)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;
	
	object_class = (GtkObjectClass *) klass;
	widget_class = (GtkWidgetClass *) klass;

	/* Gtk Object methods. */
	object_class->destroy = gantt_chart_destroy;
}

static void
gantt_chart_init (GanttChart *chart)
{
	GtkTable *table = GTK_TABLE (chart);

	chart->priv = g_new0 (GanttChartPriv, 1);

	table->homogeneous = FALSE;
	gtk_table_resize (table, 2, 3);
}

static void
header_canvas_realize (GtkWidget *widget, GanttChart *chart)
{
	/* We set the background pixmap to NULL so that X won't clear 
	 * exposed areas and thus be faster.
	 */
	gdk_window_set_back_pixmap (GTK_LAYOUT (widget)->bin_window,
				    NULL,
				    FALSE);
}

static void
gantt_canvas_realize (GtkWidget *widget, GanttChart *chart)
{
	GdkWindow      *window;
	GdkColormap    *colormap;
	GtkStyle       *style;

	/* We set the background pixmap to NULL so that X won't clear 
	 * exposed areas and thus be faster.
	 */
	gdk_window_set_back_pixmap (GTK_LAYOUT (widget)->bin_window,
				    NULL,
				    FALSE);

	window = widget->window;

	/* Set the background to white. */
	style = gtk_style_copy (widget->style);
	colormap = gtk_widget_get_colormap (widget);
	gdk_color_white (colormap, &style->bg[GTK_STATE_NORMAL]);
	gtk_widget_set_style (widget, style);
	gtk_style_unref (style);
}

static void
gantt_canvas_size_allocate (GtkWidget     *widget,
			    GtkAllocation *allocation,
			    GanttChart    *chart)
{
	gdouble x1, y1, x2, y2;
	gdouble width, height;

	gtk_object_get (GTK_OBJECT (chart->gantt_item),
			"x1", &x1,
			"y1", &y1,
			"x2", &x2,
			"y2", &y2,
			NULL);

	width = x2 - x1;
	height = y2 - y1;
	
	width = MAX ((int) width, allocation->width);
	height = MAX ((int) height, allocation->height);

	gnome_canvas_set_scroll_region (GNOME_CANVAS (widget),
					x1,
					y1,
					x1 + width - 1,
					y1 + height - 1);
}

static void
header_canvas_size_request (GtkWidget *widget, GtkRequisition *req, GanttChart *chart)
{
	gdouble header_height;

	gtk_object_get (GTK_OBJECT (chart->header_item),
			"height", &header_height,
			NULL);

	req->width = -1;
	req->height = header_height;
}

static void
header_canvas_size_allocate (GtkWidget     *widget,
			     GtkAllocation *alloc,
			     GanttChart    *chart)
{
	gantt_scale_set_min_viewport_width (chart->gantt_scale, alloc->width);
}

static void
hscroll_expand (GtkRange *range, MrScrollType type, GanttChart *chart)
{
	gtk_layout_freeze (GTK_LAYOUT (chart->gantt_canvas));
	gtk_layout_freeze (GTK_LAYOUT (chart->header_canvas));
	switch (type) {
	case MR_SCROLL_STEP_BACKWARD:
		gantt_scale_move_viewport (chart->gantt_scale, -60*60*24);
		chart->gantt_hadj->value = chart->gantt_hadj->lower;
		gtk_adjustment_value_changed (chart->gantt_hadj);
		break;

	case MR_SCROLL_STEP_FORWARD:
		gantt_scale_move_viewport (chart->gantt_scale, 60*60*24);
		chart->gantt_hadj->value = chart->gantt_hadj->upper -
			chart->gantt_hadj->page_size;
		gtk_adjustment_value_changed (chart->gantt_hadj);
		break;

	case MR_SCROLL_CONT_BACKWARD:
		chart->gantt_hadj->value = chart->gantt_hadj->lower;
		gtk_adjustment_value_changed (chart->gantt_hadj);
		gantt_scale_move_viewport (chart->gantt_scale, -60*60*24);
		break;
		
	case MR_SCROLL_CONT_FORWARD:
		chart->gantt_hadj->value = chart->gantt_hadj->upper -
			chart->gantt_hadj->page_size;
		gtk_adjustment_value_changed (chart->gantt_hadj);
		gantt_scale_move_viewport (chart->gantt_scale, 60*60*24);
		break;

	default:
		break;
	}

	gtk_layout_thaw (GTK_LAYOUT (chart->gantt_canvas));
	gtk_layout_thaw (GTK_LAYOUT (chart->header_canvas));
}

static void
timescale_close_clicked (GtkWidget *button, GanttChart *chart)
{
	gtk_widget_destroy (chart->priv->timescale_dialog);
	chart->priv->timescale_dialog = NULL;
}

typedef struct {
	gchar     *label1;
	gchar     *label2;
	guint      value;
} HackPopupItem;

static HackPopupItem scale_units[] = {
 	{ N_("Years"),    N_("years"),    GANTT_UNIT_YEAR },
	{ N_("Quarters"), N_("quarters"), GANTT_UNIT_QUARTER },
	{ N_("Months"),   N_("months"),   GANTT_UNIT_MONTH },
	{ N_("Weeks"),    N_("weeks"),    GANTT_UNIT_WEEK },
	{ N_("Days"),     N_("days"),     GANTT_UNIT_DAY },
	{ N_("Hours"),    N_("hours"),    GANTT_UNIT_HOUR },
	{ N_("Minutes"),  N_("minutes"),  GANTT_UNIT_MINUTE },
	{ NULL,           NULL,           GANTT_UNIT_NONE }
};

static void
timescale_major_changed (GtkWidget *menu_item, GanttChart *chart)
{
	gint      i;
	GanttUnit unit;
	
	i = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (menu_item), "nr"));
	unit = scale_units[i].value;
	gantt_scale_set_units (chart->gantt_scale, unit, GANTT_UNIT_NONE);
}

static void
timescale_minor_changed (GtkWidget *menu_item, GanttChart *chart)
{
	gint      i;
	GanttUnit unit;
	
	i = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (menu_item), "nr"));
	unit = scale_units[i].value;
	gantt_scale_set_units (chart->gantt_scale, GANTT_UNIT_NONE, unit);
}

static void
setup_hack_option_menu (GtkOptionMenu *option_menu,
			gboolean       label1,
			GtkSignalFunc  func,
			gpointer       user_data)
{
	GtkWidget *menu, *menu_item;
	gint       i;
	
	/* Workaround for stupid option menu that has screwed up size
	 * allocation.
	 */

	menu = gtk_option_menu_get_menu (option_menu);
	gtk_widget_destroy (menu);
	
	menu = gtk_menu_new ();
	
	for (i = 0; scale_units[i].label1; i++) {
		if (label1) {
			menu_item = gtk_menu_item_new_with_label (_(scale_units[i].label1));
		} else {
			menu_item = gtk_menu_item_new_with_label (_(scale_units[i].label2));
		}
		gtk_widget_show (menu_item);
		gtk_menu_append (GTK_MENU (menu), menu_item);
		
		gtk_object_set_data (GTK_OBJECT (menu_item),
				     "nr",
				     GINT_TO_POINTER (i));
		gtk_signal_connect (GTK_OBJECT (menu_item),
				    "activate",
				    func,
				    user_data);
	}		

	gtk_widget_show (menu);
	gtk_option_menu_set_menu (option_menu, menu);
}

static time_t
get_center_time (GanttChart *chart)
{
	GnomeCanvas *canvas;
	gint         cx;
	double       c2w[6], w2c[6];
	ArtPoint     pc, pw;
	time_t       t;

	canvas = GNOME_CANVAS (chart->gantt_canvas);
	
	/* This is in canvas coordinates. */
	gnome_canvas_get_scroll_offsets (canvas, &cx, NULL);

	cx += chart->gantt_canvas->allocation.width / 2;
	
	/* Convert to world coordinates. */
	gnome_canvas_w2c_affine (canvas, w2c);
	art_affine_invert (c2w, w2c);

	pc.x = cx;
	pc.y = 0;
	art_affine_point (&pw, &pc, c2w);
	
	t = gantt_scale_w2t (chart->gantt_scale, pw.x);
	return t;
}

static void
timescale_size_changed (GtkAdjustment *adjustment, GanttChart *chart)
{
	gdouble size;
	time_t  center;

	size = DEFAULT_SCALE * adjustment->value / 100.0;
	center = get_center_time (chart);
 	gantt_scale_set_scale_factor_ex (chart->gantt_scale, size, center);
}

static gboolean
timescale_key_event (GtkWidget   *widget,
		     GdkEventKey *event,
		     GtkWidget   *close_button)
{
	if (event->keyval == GDK_Escape) {
		gtk_button_clicked (GTK_BUTTON (close_button));
		return TRUE;
	} else {
		return FALSE;
	}
}

static void
timescale_cb (GtkMenuItem *item, GanttChart *chart)
{
	GladeXML      *glade;
        GtkWidget     *window, *button;
	GtkWidget     *major_menu, *minor_menu;
	GtkWidget     *size_sb;
	GtkAdjustment *adjustment;
	gint           i;

	if (chart->priv->timescale_dialog != NULL) {
		gdk_window_raise (chart->priv->timescale_dialog->window);
		return;
	}

        glade = glade_xml_new (MRPROJECT_DATADIR
			       "gantt-timescale-dialog.glade",
                               NULL);
	
	window = glade_xml_get_widget (glade, "toplevel");
	chart->priv->timescale_dialog = window;
	
	button = glade_xml_get_widget (glade, "button_close");
	major_menu = glade_xml_get_widget (glade, "om_major");
	minor_menu = glade_xml_get_widget (glade, "om_minor");
	size_sb = glade_xml_get_widget (glade, "sb_size");

	gtk_signal_connect (GTK_OBJECT (window), "key_press_event",
			    GTK_SIGNAL_FUNC (timescale_key_event), button);

	setup_hack_option_menu (GTK_OPTION_MENU (major_menu),
				TRUE, /* use label1 */
				timescale_major_changed,
				chart);
	setup_hack_option_menu (GTK_OPTION_MENU (minor_menu),
				TRUE, /* use label1 */
				timescale_minor_changed,
				chart);

	/* Reflect settings in the option menus. */
	for (i = 0; scale_units[i].label1; i++) {
		if (scale_units[i].value == chart->gantt_scale->major_unit) {
			gtk_option_menu_set_history (GTK_OPTION_MENU (major_menu), i);
		}
		if (scale_units[i].value == chart->gantt_scale->minor_unit) {
			gtk_option_menu_set_history (GTK_OPTION_MENU (minor_menu), i);
		}
	}

	/* FIXME: Set the spinbutton to the current value and use a unit spin button. */
	
	
	adjustment = gtk_spin_button_get_adjustment (GTK_SPIN_BUTTON (size_sb));
	gtk_signal_connect (GTK_OBJECT (adjustment),
			    "value-changed",
			    timescale_size_changed,
			    chart);
	
	gtk_signal_connect (GTK_OBJECT (button),
			    "clicked",
			    timescale_close_clicked,
			    chart);

	gtk_widget_show_all (window);
	gtk_object_unref (GTK_OBJECT (glade));
}

void
gantt_chart_customize_scale (GanttChart *chart)
{
	timescale_cb (NULL, chart);
}

static gint
zoom_key_event (GtkWidget *widget, GdkEventKey *event, GtkWidget *close_button)
{
	if (event->keyval == GDK_Escape) {
		gtk_button_clicked (GTK_BUTTON (close_button));
		return 1;
	} else {
		return 0;
	}
}

static void
zoom_close_clicked (GtkWidget *button, GanttChart *chart)
{
	ZoomDialog *dlg = chart->priv->zoom_dialog;

	gtk_widget_destroy (dlg->window);
	g_free (dlg);
	chart->priv->zoom_dialog = NULL;
}

static void
zoom_week_clicked (GtkWidget *button, GanttChart *chart)
{
	ZoomDialog *dlg = chart->priv->zoom_dialog;
	time_t      t, t1, t2;
	gdouble     width;
	
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button))) {
		gtk_widget_set_sensitive (dlg->reset_btn, TRUE);

		/* Get the date that is centered, and show half a week
		 * around it.
		 */

		t = gantt_chart_get_centered_time (chart);

		/* FIXME: this way... */

		/*t1 = time_week_begin (t, chart->gantt_scale->week_starts_on_monday);
		  t2 = time_week_end (t, chart->gantt_scale->week_starts_on_monday);
		  t_minor = gantt_scale_get_minor_tick (chart->gantt_scale);
		  t1 -= t_minor;
		  t2 += t_minor;*/

		/* ...or this? (ditto for the following functions) */
		
		t1 = t - 3.5*60*60*24;
		t2 = t + 3.5*60*60*24;
		
		width = get_gantt_allocation_width (chart);
		
		gantt_scale_zoom (chart->gantt_scale,
				  t1,
				  t2,
				  width);
	}
}

static void
zoom_month_clicked (GtkWidget *button, GanttChart *chart)
{
	ZoomDialog *dlg = chart->priv->zoom_dialog;
	time_t      t, t1, t2;
	gdouble     width;

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button))) {
		gtk_widget_set_sensitive (dlg->reset_btn, TRUE);

		/* Get the date that is centered, and show half a month
		 * around it.
		 */

		t = gantt_chart_get_centered_time (chart);

		/*t1 = time_month_begin (t);
		  t2 = time_month_end (t);
		  t_minor = gantt_scale_get_minor_tick (chart->gantt_scale);
		  t1 -= t_minor;
		  t2 += t_minor;
		*/

		t1 = t - 0.5*31*60*60*24;
		t2 = t + 0.5*31*60*60*24;
		
		width = get_gantt_allocation_width (chart);
		gantt_scale_zoom (chart->gantt_scale,
				  t1,
				  t2,
				  width);
	}
}

static void
zoom_quarter_clicked (GtkWidget *button, GanttChart *chart)
{
	ZoomDialog *dlg = chart->priv->zoom_dialog;
	time_t      t, t1, t2;
	gdouble     width;

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button))) {
		gtk_widget_set_sensitive (dlg->reset_btn, TRUE);

		/* Get the date that is centered, and show half a month
		 * around it.
		 */

		t = gantt_chart_get_centered_time (chart);
		t1 = t - 1.5*30.5*60*60*24;
		t2 = t + 1.5*30.5*60*60*24;

		width = get_gantt_allocation_width (chart);
		gantt_scale_zoom (chart->gantt_scale,
				  t1,
				  t2,
				  width);
	}
}

static void
zoom_selected_clicked (GtkWidget *button, GanttChart *chart)
{
	ZoomDialog *dlg = chart->priv->zoom_dialog;

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button))) {
		gtk_widget_set_sensitive (dlg->reset_btn, TRUE);
	}
}

static void
zoom_entire_clicked (GtkWidget *button, GanttChart *chart)
{
	ZoomDialog *dlg = chart->priv->zoom_dialog;
	
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button))) {
		gtk_widget_set_sensitive (dlg->reset_btn, TRUE);

		gantt_chart_zoom_entire (chart);
	}
}

static void
zoom_custom_changed (GanttChart *chart)
{
	ZoomDialog *dlg = chart->priv->zoom_dialog;
	GanttUnit   unit;
	gfloat      value;
	time_t      t, t1, t2, dt;
	gdouble     width; 
	
	unit = scale_units[dlg->custom_unit].value;
	
	gtk_widget_set_sensitive (dlg->reset_btn, TRUE);
		
	t = gantt_chart_get_centered_time (chart);

	value = gtk_spin_button_get_value_as_float (
		GTK_SPIN_BUTTON (dlg->custom_sb));

	dt = floor (value * gantt_scale_unit_get_period (unit) + 0.5);
	t1 = t - dt / 2;
	t2 = t + dt / 2;

	width = get_gantt_allocation_width (chart);
	
	gantt_scale_zoom (chart->gantt_scale,
			  t1,
			  t2,
			  width);
	
	gantt_chart_scroll_to_time (chart, t1);
}

static void
zoom_custom_clicked (GtkWidget *button, GanttChart *chart)
{
	ZoomDialog *dlg = chart->priv->zoom_dialog;

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button))) {
		gtk_widget_set_sensitive (dlg->custom_sb, TRUE);
		gtk_widget_set_sensitive (dlg->custom_om, TRUE);
		zoom_custom_changed (chart);
		gtk_widget_grab_focus (dlg->custom_sb);
	} else {
		gtk_widget_set_sensitive (dlg->custom_sb, FALSE);
		gtk_widget_set_sensitive (dlg->custom_om, FALSE);
	}
}

static void
zoom_custom_menu_changed (GtkWidget *menu_item, GanttChart *chart)
{
	ZoomDialog *dlg = chart->priv->zoom_dialog;
	gint        i;
	
	i = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (menu_item), "nr"));
	dlg->custom_unit = i;

	zoom_custom_changed (chart);
}

static void
zoom_custom_value_changed (GtkWidget *widget, GanttChart *chart)
{
	zoom_custom_changed (chart);
}

static void
zoom_reset_clicked (GtkWidget *button, GanttChart *chart)
{
	ZoomDialog *dlg = chart->priv->zoom_dialog;

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dlg->original_btn),
				     TRUE);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (dlg->custom_sb),
				   dlg->original_custom_value);

	gtk_option_menu_set_history (GTK_OPTION_MENU (dlg->custom_om),
				     dlg->original_custom_unit);

	dlg->custom_unit = dlg->original_custom_unit;
	
	gtk_widget_set_sensitive (button, FALSE);
}
	
static void
zoom_cb (GtkWidget *widget, GanttChart *chart)
{
	GladeXML      *glade;
	GtkWidget     *window, *button;
	GtkWidget     *om;
	GtkWidget     *rb_week, *rb_month, *rb_quarter;
	GtkWidget     *rb_selected, *rb_entire, *rb_custom;
	ZoomDialog    *zoom_dialog;
	time_t         t1, t2;
	gint           days;
	GtkAdjustment *adj;

	if (chart->priv->zoom_dialog != NULL) {
		gdk_window_raise (chart->priv->zoom_dialog->window->window);
		return;
	}
	
	glade = glade_xml_new (MRPROJECT_DATADIR
			       "gantt-zoom-dialog.glade",
			       NULL);
	if (!glade) {
		g_warning (_("Could not create zoom dialog."));
		return;
	}

	zoom_dialog = g_new0 (ZoomDialog, 1);

	window = glade_xml_get_widget (glade, "toplevel");
	button = glade_xml_get_widget (glade, "button_close");

	zoom_dialog->window = window;
	gtk_signal_connect (GTK_OBJECT (window), "key_press_event",
			    GTK_SIGNAL_FUNC (zoom_key_event), button);
	
	gtk_signal_connect (GTK_OBJECT (button),
			    "clicked",
			    zoom_close_clicked,
			    chart);

	zoom_dialog->reset_btn = glade_xml_get_widget (glade, "button_reset");
	gtk_signal_connect (GTK_OBJECT (zoom_dialog->reset_btn),
			    "clicked",
			    zoom_reset_clicked,
			    chart);

	rb_week = glade_xml_get_widget (glade, "rb_week");
	rb_month = glade_xml_get_widget (glade, "rb_month");
	rb_quarter = glade_xml_get_widget (glade, "rb_quarter");
	rb_selected = glade_xml_get_widget (glade, "rb_selected");
	rb_entire = glade_xml_get_widget (glade, "rb_entire");
	rb_custom = glade_xml_get_widget (glade, "rb_custom");
	
	gtk_widget_set_sensitive (rb_selected, FALSE);
	
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (rb_custom), TRUE);
	zoom_dialog->original_btn = rb_custom;
	
	gtk_signal_connect (GTK_OBJECT (rb_week),
			    "clicked",
			    zoom_week_clicked,
			    chart);
	gtk_signal_connect (GTK_OBJECT (rb_month),
			    "clicked",
			    zoom_month_clicked,
			    chart);
	gtk_signal_connect (GTK_OBJECT (rb_quarter),
			    "clicked",
			    zoom_quarter_clicked,
			    chart);
	gtk_signal_connect (GTK_OBJECT (rb_selected),
			    "clicked",
			    zoom_selected_clicked,
			    chart);
	gtk_signal_connect (GTK_OBJECT (rb_entire),
			    "clicked",
			    zoom_entire_clicked,
			    chart);
	gtk_signal_connect (GTK_OBJECT (rb_custom),
			    "clicked",
			    zoom_custom_clicked,
			    chart);

	om = glade_xml_get_widget (glade, "optionmenu_custom");
	zoom_dialog->custom_om = om;
	setup_hack_option_menu (GTK_OPTION_MENU (om),
				FALSE, /* use label2 */
				zoom_custom_menu_changed,
				chart);

	/* Get original values. */
	t1 = gantt_chart_get_first_visible_time (chart);
	t2 = gantt_chart_get_last_visible_time (chart);

	days = 30; /* fallback */
	time_diff (t2, t1, &days, NULL, NULL);
	
	zoom_dialog->original_custom_unit = 4; /* custom */
	zoom_dialog->custom_unit = 4;
	zoom_dialog->original_custom_value = days;
	zoom_dialog->original_t1 = t1;
	zoom_dialog->original_t2 = t2;
	gtk_option_menu_set_history (GTK_OPTION_MENU (zoom_dialog->custom_om),
				     zoom_dialog->original_custom_unit);

	chart->priv->zoom_dialog = zoom_dialog;
	zoom_dialog->custom_sb = glade_xml_get_widget (glade, "sb_custom");
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (zoom_dialog->custom_sb),
				   zoom_dialog->original_custom_value);
	adj = gtk_spin_button_get_adjustment (GTK_SPIN_BUTTON (zoom_dialog->custom_sb));
	gtk_signal_connect (GTK_OBJECT (adj),
			    "value-changed",
			    zoom_custom_value_changed,
			    chart);
	
	gtk_object_unref (GTK_OBJECT (glade));
}

void
gantt_chart_zoom_entire (GanttChart *chart)
{
	time_t      t1, t2, t_minor;
	gdouble     width;

	g_return_if_fail (chart != NULL);
	g_return_if_fail (IS_GANTT_CHART (chart));
		
	t1 = gantt_model_get_first_time (chart->gantt_model);
	t2 = gantt_model_get_last_time (chart->gantt_model);
	
	t_minor = gantt_scale_get_minor_tick (chart->gantt_scale);
	
	/* If we don't get first/last, leave it as it is. */
	if (t1 == -1 || t2 == -1) {
		return;
	}
	
	t1 -= t_minor;
	t2 += t_minor;
	
	/* Make sure we show at least 4 minor ticks. */
	if (t2 <= (t1 + 4 * t_minor)) {
		t2 = t1 + 4 * t_minor;
	}
	
	g_print ("%s -> %s\n", isodate_from_time_t (t1),
		 isodate_from_time_t (t2)); 
	
	width = get_gantt_allocation_width (chart);
	gantt_scale_zoom (chart->gantt_scale,
			  t1,
			  t2,
			  width);
	
	gantt_chart_scroll_to_time (chart, t1);
}

void
gantt_chart_zoom (GanttChart *chart)
{
	zoom_cb (NULL, chart);
}

static GnomeUIInfo header_popup_uiinfo[] = {

	{ GNOME_APP_UI_ITEM, N_("Customize Scale..."),
	  N_("Customize the Gantt Scale"), timescale_cb, NULL,
	  NULL, 0, 0, 0, 0 },

	{ GNOME_APP_UI_ITEM, N_("Zoom..."),
	  N_("Change the zoom factor of the Gantt chart"), zoom_cb, NULL,
	  NULL, 0, 0, 0, 0 },

	GNOMEUIINFO_END
};

static gboolean
header_canvas_event (GnomeCanvas *canvas, GdkEvent *event, GanttChart *chart)
{
	GtkWidget *popup_menu;

	switch (event->type) {
	case GDK_BUTTON_PRESS:
		if (event->button.button == 3) {
			popup_menu = gnome_popup_menu_new (header_popup_uiinfo);
			gnome_popup_menu_do_popup_modal (popup_menu,
							 NULL,
							 NULL,
							 (GdkEventButton *) event,
							 chart);
			gtk_widget_destroy (popup_menu);
			 
			return TRUE;
		}
		
		break;
	default:
		break;
	}
	
	return FALSE;
}

static void
gantt_chart_construct (GanttChart  *chart,
		       GanttModel  *gantt_model,
		       ETableModel *table_model)
{
	GanttChartPriv *priv;
	GtkTable       *table;
	GtkAdjustment  *vadj;
	time_t          t1, t2;

	GanttScale *gantt_scale;

	table = GTK_TABLE (chart);
	priv = chart->priv;

	chart->gantt_model = gantt_model;

	t1 = time (NULL);
	t2 = time_add_month (t1, 1);
	gantt_scale = gantt_scale_new (DEFAULT_SCALE, t1, t2);

	gantt_scale->major_unit = GANTT_UNIT_WEEK;
	gantt_scale->minor_unit = GANTT_UNIT_DAY;
	gantt_scale->week_starts_on_monday = TRUE;
	chart->gantt_scale = gantt_scale;
	
	chart->header_canvas = GTK_WIDGET (gnome_canvas_new ());
	chart->header_item = gnome_canvas_item_new (
		gnome_canvas_root (GNOME_CANVAS (chart->header_canvas)),
		gantt_header_item_get_type (),
		"gantt_model", gantt_model,
		"gantt_scale", gantt_scale,
		NULL);

	chart->gantt_canvas = gnome_canvas_new ();
	GNOME_CANVAS (chart->gantt_canvas)->close_enough = 5;

	gtk_object_set_data (GTK_OBJECT (chart->gantt_canvas),
			     "gantt_model", gantt_model);

	gtk_object_set_data (GTK_OBJECT (chart->gantt_canvas),
			     "gantt_scale", gantt_scale);

	chart->gantt_item = gnome_canvas_item_new (
		gnome_canvas_root (GNOME_CANVAS (chart->gantt_canvas)),
		gantt_item_get_type (),
		"gantt_model", gantt_model,
		"gantt_scale", gantt_scale,
		"table_model", table_model,
		NULL);

	gtk_signal_connect (GTK_OBJECT (chart->header_canvas),
			    "event",
			    GTK_SIGNAL_FUNC (header_canvas_event),
			    chart);

	gtk_signal_connect (GTK_OBJECT (chart->header_canvas),
			    "realize",
			    GTK_SIGNAL_FUNC (header_canvas_realize),
			    chart);

	gtk_signal_connect (GTK_OBJECT (chart->gantt_canvas),
			    "realize",
			    GTK_SIGNAL_FUNC (gantt_canvas_realize),
			    chart);
		
	gtk_signal_connect (GTK_OBJECT (chart->header_canvas),
			    "size_request",
			    GTK_SIGNAL_FUNC (header_canvas_size_request),
			    chart);
	
	gtk_signal_connect (GTK_OBJECT (chart->header_canvas),
			    "size_allocate",
			    GTK_SIGNAL_FUNC (header_canvas_size_allocate),
			    chart);

	gtk_signal_connect (GTK_OBJECT (chart->gantt_canvas),
			    "size_allocate",
			    GTK_SIGNAL_FUNC (gantt_canvas_size_allocate),
			    chart);

	/* Create scroll bars and connect the adjustments. */
	vadj = gtk_layout_get_vadjustment (GTK_LAYOUT (chart->gantt_canvas));
	vadj->step_increment = 20;
	chart->gantt_hadj = GTK_ADJUSTMENT (gtk_adjustment_new (0, 0, gantt_scale->x2 - gantt_scale->x1, 50, 50, 500));
	chart->hscroll = mr_hscrollbar_new (chart->gantt_hadj);
	chart->vscroll = gtk_vscrollbar_new (vadj);

	gtk_signal_connect (GTK_OBJECT (chart->hscroll), "expand",
			    GTK_SIGNAL_FUNC (hscroll_expand),
			    chart);

	gtk_layout_set_hadjustment (GTK_LAYOUT (chart->header_canvas), chart->gantt_hadj);
	gtk_layout_set_hadjustment (GTK_LAYOUT (chart->gantt_canvas), chart->gantt_hadj);

	gtk_table_attach (GTK_TABLE (table), chart->header_canvas,
			  0, 1, 0, 1,
			  GTK_EXPAND | GTK_FILL,
			  GTK_FILL,
			  0, 0);

	gtk_table_attach (GTK_TABLE (table), chart->hscroll,
			  0, 1, 2, 3,
			  GTK_FILL | GTK_SHRINK,
			  GTK_FILL | GTK_SHRINK,
			  0, 0);

	gtk_table_attach (GTK_TABLE (table), chart->vscroll,
			  1, 2, 1, 2,
			  GTK_FILL | GTK_SHRINK,
			  GTK_FILL | GTK_SHRINK,
			  0, 0);

	gtk_table_attach (GTK_TABLE (table), chart->gantt_canvas,
			  0, 1, 1, 2,
			  GTK_EXPAND | GTK_FILL,
			  GTK_EXPAND | GTK_FILL,
			  0, 0);
}

GtkWidget *
gantt_chart_new (GanttModel *gantt_model, ETableModel *table_model)
{
	GanttChart *chart;

	chart =  gtk_type_new (TYPE_GANTT_CHART);
	gantt_chart_construct (chart, gantt_model, table_model);
	
	return GTK_WIDGET (chart);
}

GtkAdjustment *
gantt_chart_get_vadjustment (GanttChart *chart)
{
	return gtk_layout_get_vadjustment (GTK_LAYOUT (chart->gantt_canvas));
}

void
gantt_chart_set_vadjustment (GanttChart *chart, GtkAdjustment *vadj)
{
	g_return_if_fail (chart != NULL);
	g_return_if_fail (IS_GANTT_CHART (chart));
	g_return_if_fail (vadj != NULL);

	gtk_layout_set_vadjustment (GTK_LAYOUT (chart->gantt_canvas), vadj);
}

time_t
gantt_chart_get_first_visible_time (GanttChart *chart)
{
	GnomeCanvas *canvas;
	gint         cx;
	double       c2w[6], w2c[6];
	ArtPoint     pc, pw;
	time_t       t;

	g_return_val_if_fail (chart != NULL, 0);
	g_return_val_if_fail (IS_GANTT_CHART (chart), 0);

	canvas = GNOME_CANVAS (chart->gantt_canvas);
	
	/* This is in canvas coordinates. */
	gnome_canvas_get_scroll_offsets (canvas, &cx, NULL);

	/* Convert to world coordinates. */
	gnome_canvas_w2c_affine (canvas, w2c);
	art_affine_invert (c2w, w2c);

	pc.x = cx;
	pc.y = 0;
	art_affine_point (&pw, &pc, c2w);
	
	t = gantt_scale_w2t (chart->gantt_scale, pw.x);
	return t;
}

time_t
gantt_chart_get_last_visible_time (GanttChart *chart)
{
	GnomeCanvas *canvas;
	gint         cx;
	double       c2w[6], w2c[6];
	ArtPoint     pc, pw;
	time_t       t;

	g_return_val_if_fail (chart != NULL, 0);
	g_return_val_if_fail (IS_GANTT_CHART (chart), 0);

	canvas = GNOME_CANVAS (chart->gantt_canvas);
	
	/* This is in canvas coordinates. */
	gnome_canvas_get_scroll_offsets (canvas, &cx, NULL);

	cx += GTK_WIDGET (canvas)->allocation.width;
	
	/* Convert to world coordinates. */
	gnome_canvas_w2c_affine (canvas, w2c);
	art_affine_invert (c2w, w2c);

	pc.x = cx;
	pc.y = 0;
	art_affine_point (&pw, &pc, c2w);
	
	t = gantt_scale_w2t (chart->gantt_scale, pw.x);
	return t;
}

/* world coordinates */
static gdouble
get_gantt_allocation_width (GanttChart *chart)
{
	gint     width;
	double   w2c[6], c2w[6], wx1, wx2;
	ArtPoint pw, pc;
	
	g_return_val_if_fail (chart != NULL, 0);
	g_return_val_if_fail (IS_GANTT_CHART (chart), 0);

	gnome_canvas_w2c_affine (GNOME_CANVAS (chart->gantt_canvas), w2c);
	art_affine_invert (c2w, w2c);

	pc.x = 0;
	pc.y = 0;
	art_affine_point (&pw, &pc, c2w);
	wx1 = floor (pw.x + 0.5);
	
	width = chart->gantt_canvas->allocation.width;
	pc.x = width;
	pc.y = 0;
	art_affine_point (&pw, &pc, c2w);
	wx2 = floor (pw.x + 0.5);

	return wx2 - wx1;
}

time_t
gantt_chart_get_centered_time (GanttChart *chart)
{
	time_t   t, t_width;
	
	g_return_val_if_fail (chart != NULL, 0);
	g_return_val_if_fail (IS_GANTT_CHART (chart), 0);

	t = gantt_chart_get_first_visible_time (chart);
	t_width = gantt_scale_w2t (chart->gantt_scale,
				   get_gantt_allocation_width (chart));

	return t + t_width / 2;
}

void
gantt_chart_scroll_to_time (GanttChart *chart, time_t t)
{
	gint     width, cx;
	double   w2c[6], c2w[6], wx, wx2;
	ArtPoint pw, pc;
	time_t   t_width, t_minor;
	
	g_return_if_fail (chart != NULL);
	g_return_if_fail (IS_GANTT_CHART (chart));

	gnome_canvas_w2c_affine (GNOME_CANVAS (chart->gantt_canvas), w2c);
	art_affine_invert (c2w, w2c);

	pc.x = 0;
	pc.y = 0;
	art_affine_point (&pw, &pc, c2w);
	wx = floor (pw.x + 0.5);
	
	width = chart->gantt_canvas->allocation.width;
	pc.x = width;
	pc.y = 0;
	art_affine_point (&pw, &pc, c2w);
	wx2 = floor (pw.x + 0.5);

	t_width = gantt_scale_w2t (chart->gantt_scale,
				   get_gantt_allocation_width (chart));
	t_minor = gantt_scale_get_minor_tick (chart->gantt_scale);

	/* Make sure that the time is included in the viewport. */
	gantt_scale_show_time (chart->gantt_scale, t + t_width + t_minor);
	gantt_scale_show_time (chart->gantt_scale, t - t_minor);

	/* This is in world coordinates. */
	wx = gantt_scale_t2w (chart->gantt_scale, t);

	/* Convert to canvas coordinates. */
	pw.x = wx;
	pw.y = 0;
	art_affine_point (&pc, &pw, w2c);

	cx = floor (pc.x + 0.5);
	cx = CLAMP (
		cx,
		chart->gantt_hadj->lower,
		chart->gantt_hadj->upper - chart->gantt_hadj->page_size);

	gtk_adjustment_set_value (chart->gantt_hadj, cx);
}

void
gantt_chart_select (GanttChart *chart, GSList *rows)
{
	GSList *l;

	g_return_if_fail (chart != NULL);
	g_return_if_fail (IS_GANTT_CHART (chart));

	gantt_item_unselect_all (GANTT_ITEM (chart->gantt_item));
	
	for (l = rows; l; l = l->next) {
		/*g_print ("chart select row %d\n", GPOINTER_TO_INT (l->data));*/
		gantt_item_select_row (GANTT_ITEM (chart->gantt_item),
				       GPOINTER_TO_INT (l->data));
	}
}


