/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 Thomas Nyberg <thomas@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Thomas Nyberg
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <glib.h>
#include <stdio.h>
#include "util/corba-utils.h"
#include "util/id-map.h"
#include "util/type-utils.h"
#include "resource-filter.h"

#define DEBUG 1
#include "util/debug.h"

static void resource_filter_init (ResourceFilter *filter);
static void resource_filter_class_init (ResourceFilterClass *class);
static void resource_filter_destroy (GtkObject *object);

static GtkObjectClass *parent_class;

typedef struct {
	GM_Id resource_id;
	GSList *tasks; /* list of taskId */

	gboolean visible; /* should this be displayed? */
} _ResourceInfo;

typedef struct {
	GM_Id task_id;
	GSList *resources; /* list of resourceId */
} _TaskInfo;

struct _ResourceFilterPriv {
	IdMap *resources; /* contains _ResourceInfo */
	IdMap *tasks; /* contains _TaskInfo */
};


GNOME_CLASS_BOILERPLATE (ResourceFilter, resource_filter,
			 GtkObject, gtk_object);

static void
resource_filter_init (ResourceFilter *filter)
{
	ResourceFilterPriv *priv;
	g_return_if_fail (filter != NULL);

	d(puts(__FUNCTION__));

	priv = (ResourceFilterPriv *)g_new0 (ResourceFilterPriv, 1);

	filter->priv = priv;

	priv->resources = id_map_new (0);
	priv->tasks = id_map_new (0);
}

static void
resource_filter_class_init (ResourceFilterClass *class)
{
	GtkObjectClass *object_class;

	object_class = GTK_OBJECT_CLASS (class);

	if (parent_class == NULL) {
		parent_class = gtk_type_class (GTK_TYPE_OBJECT);
	}

	object_class->destroy = resource_filter_destroy;
}

static void
resource_filter_destroy (GtkObject *object)
{
	ResourceFilter *filter;
	GSList *list, *node;
	_ResourceInfo *res_info;
	_TaskInfo *task_info;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_RESOURCE_FILTER (object));

	d(puts(__FUNCTION__));

	filter = RESOURCE_FILTER (object);
	g_return_if_fail (filter->priv != NULL);

	/* free the resource-list */
	list = id_map_get_objects (filter->priv->resources);
	for (node = list; node; node = node->next ) {
		g_assert (node->data != NULL);

		res_info = (_ResourceInfo *)node->data;
		g_slist_free (res_info->tasks);
		g_free (res_info);
	}
	g_slist_free (list);
	gtk_object_destroy (GTK_OBJECT (filter->priv->resources));
	
	/* free the task-list */
	list = id_map_get_objects (filter->priv->tasks);
	for (node = list; node; node = node->next) {
		g_assert (node->data != NULL);

		task_info = (_TaskInfo *)node->data;
		g_slist_free (task_info->resources);
		g_free (task_info);
	}
	g_slist_free (list);
	gtk_object_destroy (GTK_OBJECT (filter->priv->tasks));

	(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

ResourceFilter *
resource_filter_new (void)
{
	ResourceFilter *filter;

	filter = RESOURCE_FILTER (gtk_type_new (RESOURCE_FILTER_TYPE));

	g_assert (filter != NULL);

	return filter;
}

void
resource_filter_add_resource (ResourceFilter *filter, 
			      const GM_Resource *resource)
{
	_ResourceInfo *res_info;

	g_return_if_fail (filter != NULL);
	g_return_if_fail (IS_RESOURCE_FILTER (filter));
	g_return_if_fail (resource != NULL);

	g_assert (filter->priv != NULL);

	d(puts(__FUNCTION__));

	/* this is just to make sure we aren't making any mistakes */
	if (id_map_lookup (filter->priv->resources, 
			   resource->resourceId) != NULL) {
		g_warning ("Resource already exists");
		return;
	}

	res_info = (_ResourceInfo *)g_new0 (_ResourceInfo, 1);

	res_info->resource_id = resource->resourceId;
	res_info->tasks = NULL; /* just in case */
	res_info->visible = TRUE; /* default-action */
	id_map_insert_id (filter->priv->resources, 
			  resource->resourceId, 
			  res_info);
}

void 
resource_filter_add_task (ResourceFilter *filter, 
			  const GM_Task *task)
{
	_TaskInfo *task_info;

	g_return_if_fail (filter != NULL);
	g_return_if_fail (IS_RESOURCE_FILTER (filter));
	g_return_if_fail (task != NULL);

	d(puts(__FUNCTION__));

	/* this is just to be sure we aren't making any mistakes */
	if (id_map_lookup (filter->priv->tasks, 
			   task->taskId) != NULL) {
		g_warning ("Task already exists");
		return;
	}

	task_info = (_TaskInfo *)g_new0 (_TaskInfo, 1);
	task_info->task_id = task->taskId;
	task_info->resources = NULL; /* just in case */

	id_map_insert_id (filter->priv->tasks, task->taskId, task_info);
}

void
resource_filter_add_allocation (ResourceFilter *filter, 
				const GM_Allocation *allocation)
{
	_ResourceInfo *res_info;
	_TaskInfo *task_info;

	g_return_if_fail (filter != NULL);
	g_return_if_fail (IS_RESOURCE_FILTER (filter));
	g_return_if_fail (allocation != NULL);

	d(puts(__FUNCTION__));

	task_info = (_TaskInfo *)id_map_lookup (filter->priv->tasks,
						allocation->taskId);
	res_info = (_ResourceInfo *)id_map_lookup (filter->priv->resources,
						   allocation->resourceId);

	/* just in case */
	if (task_info == NULL || res_info == NULL) {
		g_warning ("Allocation issued with non-existing stuff");
		return;
	}

	/* now we add them to each others lists */
	task_info->resources = g_slist_prepend (task_info->resources,
						GINT_TO_POINTER (allocation->resourceId));
	res_info->tasks = g_slist_prepend (res_info->tasks,
					   GINT_TO_POINTER (allocation->taskId));

}

static void
resource_filter_remove_resource_from_task (ResourceFilter *filter,
					   GM_Id resource_id,
					   GM_Id task_id)
{
	_TaskInfo *task_info;

	g_assert (filter != NULL);
	g_assert (IS_RESOURCE_FILTER (filter));
	g_assert (filter->priv != NULL);

	task_info = (_TaskInfo *)id_map_lookup (filter->priv->tasks, task_id);
	g_assert (task_info != NULL);

	task_info->resources = g_slist_remove (task_info->resources, 
					       GINT_TO_POINTER (resource_id));
}

static void
resource_filter_remove_task_from_resource (ResourceFilter *filter,
					   GM_Id task_id,
					   GM_Id resource_id)
{
	_ResourceInfo *res_info;

	g_assert (filter != NULL);
	g_assert (IS_RESOURCE_FILTER (filter));
	g_assert (filter->priv != NULL);

	res_info = (_ResourceInfo *)id_map_lookup (filter->priv->resources, 
						   resource_id);
	g_assert (res_info != NULL);

	res_info->tasks = g_slist_remove (res_info->tasks,
					  GINT_TO_POINTER (task_id));
}

void 
resource_filter_remove_resource (ResourceFilter *filter, 
				 GM_Id id)
{
	_ResourceInfo *res_info;
	GSList *node;

	g_return_if_fail (filter != NULL);
	g_return_if_fail (IS_RESOURCE_FILTER (filter));

	d(puts(__FUNCTION__));

	res_info = (_ResourceInfo *)id_map_lookup (filter->priv->resources,
						   id);
	if (res_info == NULL) {
		g_warning ("Trying to remove nonexisting resource");
		return;
	}

	for (node = res_info->tasks; node; node = node->next) {
		g_assert (node->data != NULL);
		/* remove all tasks information about me */
		resource_filter_remove_resource_from_task (filter, id, 
							   GPOINTER_TO_INT (node->data));
	}

	id_map_remove (filter->priv->resources, id);
	g_slist_free (res_info->tasks);
	g_free (res_info);
}

void
resource_filter_remove_task (ResourceFilter *filter, 
			     GM_Id id)
{
	_TaskInfo *task_info;
	GSList *node;

	g_return_if_fail (filter != NULL);
	g_return_if_fail (IS_RESOURCE_FILTER (filter));

	d(puts(__FUNCTION__));

	task_info = (_TaskInfo *)id_map_lookup (filter->priv->tasks,
						id);
	if (task_info == NULL) {
		g_warning ("Trying to remove nonexisting task");
		return;
	}

	for (node = task_info->resources; node; node = node->next) {
		g_assert (node->data != NULL);

		resource_filter_remove_task_from_resource (filter, id,
							   GPOINTER_TO_INT (node->data));
	}

	id_map_remove (filter->priv->tasks, id);
	g_slist_free (task_info->resources);
	g_free (task_info);
}

void
resource_filter_remove_allocation (ResourceFilter *filter, 
				   const GM_Allocation *allocation)
{
	_ResourceInfo *res_info;
	_TaskInfo *task_info;

	g_return_if_fail (filter != NULL);
	g_return_if_fail (IS_RESOURCE_FILTER (filter));
	g_return_if_fail (allocation != NULL);

	d(puts(__FUNCTION__));

	res_info = (_ResourceInfo *)id_map_lookup (filter->priv->resources, 
						   allocation->resourceId);
	task_info = (_TaskInfo *)id_map_lookup (filter->priv->tasks,
						allocation->taskId);
	if (res_info == NULL || task_info == NULL) {
		g_warning ("Trying to remove nonexisting "
			   "allocation (resource: %d, task: %d)", 
			   allocation->resourceId, allocation->taskId);
		return;
	}

	res_info->tasks = g_slist_remove (res_info->tasks,
					  GINT_TO_POINTER (allocation->taskId));
	task_info->resources = g_slist_remove (task_info->resources,
					       GINT_TO_POINTER (allocation->resourceId));
}


gboolean
resource_filter_task_is_visible (ResourceFilter *filter, 
				 GM_Id id)
{
	_TaskInfo *task_info;
	_ResourceInfo *res_info;
	GSList *node;

	g_return_val_if_fail (filter != NULL, FALSE);
	g_return_val_if_fail (IS_RESOURCE_FILTER (filter), FALSE);
	g_assert (filter->priv != NULL);

	d(puts(__FUNCTION__));

	task_info = (_TaskInfo *)id_map_lookup (filter->priv->tasks,
						id);
	g_assert (task_info != NULL);

	/* traverse all resources this task has */
	for (node = task_info->resources; node; node = node->next) {
		g_assert (node->data != NULL);

		res_info = (_ResourceInfo *)
			id_map_lookup (filter->priv->resources,
				       GPOINTER_TO_INT (node->data));

		g_assert (res_info != NULL);
		/* if one is visible, it's good enough for us */
		if (res_info->visible) {
			return TRUE;
		}
	}

	/* if we got here, no visible resource was used by this task */
	return FALSE;
}

gboolean
resource_filter_resource_get_state (ResourceFilter *filter,
				    GM_Id id)
{
	_ResourceInfo *res_info;

	g_return_val_if_fail (filter != NULL, FALSE);
	g_return_val_if_fail (IS_RESOURCE_FILTER (filter), FALSE);

	d(puts(__FUNCTION__));

	res_info = (_ResourceInfo *)id_map_lookup (filter->priv->resources, id);

	g_assert (res_info != NULL);

	return res_info->visible;
}

void
resource_filter_resource_set_state (ResourceFilter *filter,
				    GM_Id id,
				    gboolean visible)
{
	_ResourceInfo *res_info;

	g_return_if_fail (filter != NULL);
	g_return_if_fail (IS_RESOURCE_FILTER (filter));
	g_assert (filter->priv != NULL);

	d(puts(__FUNCTION__));

	res_info = (_ResourceInfo *)id_map_lookup (filter->priv->resources, id);
	g_assert (res_info != NULL);

	res_info->visible = visible;
}

void
resource_filter_all_resources_set_state (ResourceFilter *filter,
					 gboolean        visible)
{
	GSList *list, *node;
	_ResourceInfo *res_info;

	g_return_if_fail (filter != NULL);
	g_return_if_fail (IS_RESOURCE_FILTER (filter));
	g_assert (filter->priv != NULL);

	d(puts(__FUNCTION__));

	list = id_map_get_objects (filter->priv->resources);
	
	for (node = list; node; node = node->next) {
		g_assert (node->data != NULL);

		res_info = (_ResourceInfo *)node->data;

		res_info->visible = visible;
	}
	g_slist_free (list);
}
