/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 Thomas Nyberg <thomas@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Thomas Nyberg
 */
/*
 * This file contains code taken from Evolution, and written by
 *  Damon Chaplin <damon@ximian.com>
 *  and
 *  Rodrigo Moya <rodrigo@ximian.com>
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <math.h>
#include "util/type-utils.h"
#include "util/corba-utils.h"
#include "util/id-map.h"
#include "util/time-utils.h"
#include "libmrproject/GNOME_MrProject.h"
#include "month-view.h"
#include "month-view-main-item.h"
#include "month-view-titles-item.h"
#include "month-view-task-item.h"
#include "month-view-summary-task-item.h"

#define DEBUG 0
#include "util/debug.h"

static void month_view_init (MonthView *month_view);
static void month_view_class_init (MonthViewClass *klass);
static void month_view_destroy (GtkObject *object);
static void month_view_realize (GtkWidget *widget);
static void month_view_unrealize (GtkWidget *widget);
static void month_view_size_allocate (GtkWidget *widget, GtkAllocation *allocation);
static void month_view_recalc_cell_sizes (MonthView *month_view);
static void month_view_style_set (GtkWidget *widget, GtkStyle *previous_style);
static void month_view_on_adjustment_changed (GtkAdjustment *adjustment,
					      MonthView *month_view) G_GNUC_UNUSED;
static void month_view_reflow (MonthView *month_view);
static void month_view_layout_tasks (MonthView *month_view);
static void month_view_create_spans (MonthView *month_view);
static void month_view_clear_visible_tasks (MonthView *month_view);

/* static void month_view_layout_tasks (MonthView *month_view); */

typedef struct _MonthViewTask MonthViewTask;
struct _MonthViewTask {
	GM_Task *task;
	GSList *spans; /* This is a list of all the MonthViewTaskItems */
	gint row; /* This is this tasks row in the list */
	GtkType type; /* type of task-item to use */
};

struct _MonthViewPriv {
	IdMap *map;
	GSList *tasks; /* this is a list containing all tasks that are visible this month */
	gint *rows_per_week;
	guint reflow_idle_id;
};

enum {
	FIRST_DAY_CHANGED,
	LAST_SIGNAL
};

static gint signals [LAST_SIGNAL] = { 0, };

GNOME_CLASS_BOILERPLATE (MonthView, month_view,
			 GtkTable,    gtk_table);

static void
month_view_init (MonthView *month_view)
{
	GtkAdjustment *adjustment;
	gint i;

	g_return_if_fail (month_view != NULL);
	g_return_if_fail (IS_MONTH_VIEW (month_view));


	d(puts(__FUNCTION__));

	month_view->weeks_shown = 6;
	month_view->rows = 6;
	month_view->columns = 2;
	month_view->compress_weekend = FALSE;
	month_view->week_start_day = 0;		/* Monday. */
	month_view->display_start_day = 0;	/* Monday. */
	month_view->row_offsets = NULL;
	month_view->row_heights = NULL;

	/* create the priv-struct */
	month_view->priv = g_new0 (MonthViewPriv, 1);
	month_view->priv->map = id_map_new (0);
	month_view->priv->rows_per_week = g_new0 (gint, month_view->weeks_shown);
	for (i = 0;i < month_view->weeks_shown; i++) {
		month_view->priv->rows_per_week[i] = 3;
	}

	g_date_clear (&month_view->base_date, 1);
	g_date_clear (&month_view->first_day_shown, 1);

	month_view->row_height = 10;
	month_view->rows_per_cell = 1;

	month_view->main_gc = NULL;

	/* Create the small font. */
	month_view->use_small_font = TRUE;

	/* Use the default one, otherwise we're in big trouble, l10n-wise /rh */
	month_view->small_font = GTK_WIDGET (month_view)->style->font;
	gdk_font_ref (month_view->small_font);

	if (!month_view->small_font) {
		g_warning ("Couldn't load font");
	}

	month_view->main_canvas = GNOME_CANVAS (gnome_canvas_new ());
	
	gtk_table_attach (GTK_TABLE (month_view), GTK_WIDGET (month_view->main_canvas),
			  1, 2, 1, 2,
			  GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 1, 1);
	gtk_widget_show (GTK_WIDGET (month_view->main_canvas)); 

	month_view->main_canvas_item =
		gnome_canvas_item_new (gnome_canvas_root (month_view->main_canvas),
				       month_view_main_item_get_type (),
				       "MonthViewMainItem::month_view", month_view,
				       NULL); 
	gnome_canvas_set_scroll_region (GNOME_CANVAS (month_view->main_canvas),
					0, 0, 0, 0);

	month_view->titles_canvas = GNOME_CANVAS (gnome_canvas_new ());
	gtk_table_attach (GTK_TABLE (month_view), GTK_WIDGET (month_view->titles_canvas),
			  1, 2, 0, 1, GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show (GTK_WIDGET (month_view->titles_canvas));
	month_view->titles_canvas_item =
		gnome_canvas_item_new (gnome_canvas_root (GNOME_CANVAS (month_view->titles_canvas)),
				       month_view_titles_item_get_type (),
				       "MonthViewTitlesItem::month_view", month_view,
				       NULL); 
	/* Set the height of the titles */
	gtk_widget_set_usize (GTK_WIDGET (month_view->titles_canvas), -1,
			      month_view->small_font->ascent + month_view->small_font->descent + 5);


#if 0
	/* Create the buttons to jump to each days. */
	pixbuf = gdk_pixbuf_new_from_xpm_data ((const char**) jump_xpm);

	for (i = 0; i < E_WEEK_VIEW_MAX_WEEKS * 7; i++) {
		week_view->jump_buttons[i] = gnome_canvas_item_new
			(canvas_group,
			 gnome_canvas_pixbuf_get_type (),
			 "GnomeCanvasPixbuf::pixbuf", pixbuf,
			 NULL);

		gtk_signal_connect (GTK_OBJECT (week_view->jump_buttons[i]),
				    "event",
				    GTK_SIGNAL_FUNC (e_week_view_on_jump_button_event),
				    week_view);
	}
	gdk_pixbuf_unref (pixbuf);
#endif
	/*
	 * Scrollbar.
	 */
	adjustment = gtk_layout_get_vadjustment (GTK_LAYOUT (month_view->main_canvas));
	month_view->vscrollbar = gtk_vscrollbar_new (GTK_ADJUSTMENT (adjustment));

	gtk_table_attach (GTK_TABLE (month_view), month_view->vscrollbar,
			  2, 3, 1, 2, 0, GTK_EXPAND | GTK_FILL, 0, 0);
	gtk_widget_show (month_view->vscrollbar);
#if 0
	return;

	adjustment = gtk_adjustment_new (0, -52, 52, 1, 1, 1);
	gtk_signal_connect (adjustment, "value_changed",
			    GTK_SIGNAL_FUNC (month_view_on_adjustment_changed),
			    month_view);

	month_view->vscrollbar = gtk_vscrollbar_new (GTK_ADJUSTMENT (adjustment));
	gtk_table_attach (GTK_TABLE (month_view), month_view->vscrollbar,
			  2, 3, 1, 2, 0, GTK_EXPAND | GTK_FILL, 0, 0);
	gtk_widget_show (month_view->vscrollbar);
#endif
}

static void
month_view_class_init (MonthViewClass *klass)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	parent_class = gtk_type_class (GTK_TYPE_TABLE);

	object_class = (GtkObjectClass *)klass;
	widget_class = (GtkWidgetClass *)klass;

	object_class->destroy = month_view_destroy;

	widget_class->realize = month_view_realize;
	widget_class->unrealize = month_view_unrealize;
	widget_class->size_allocate = month_view_size_allocate;
	widget_class->style_set = month_view_style_set;

	signals [FIRST_DAY_CHANGED] =
		gtk_signal_new ("first-day-changed",
				GTK_RUN_LAST,
				object_class->type,
				0,
				gtk_marshal_NONE__NONE,
				GTK_TYPE_NONE, 0);
	
	gtk_object_class_add_signals (object_class, signals, LAST_SIGNAL);
}

static void
month_view_destroy (GtkObject *object)
{
	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_MONTH_VIEW (object));
	
	d(puts(__FUNCTION__));
}

static void
month_view_realize (GtkWidget *widget)
{
	MonthView *month_view;
	GdkColormap *colormap;
	gboolean success[MONTH_VIEW_COLOR_LAST];
	gint nfailed;

	d(puts(__FUNCTION__));
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (IS_MONTH_VIEW (widget));

	month_view = MONTH_VIEW (widget);

	if (GTK_WIDGET_CLASS (parent_class)->realize) {
		(*GTK_WIDGET_CLASS (parent_class)->realize)(widget);
	}

	month_view->main_gc = gdk_gc_new (widget->window);

	colormap = gtk_widget_get_colormap (widget);

	/* Allocate the colors. */
	month_view->colors[MONTH_VIEW_COLOR_EVEN_MONTHS].red   = 0xeded;
	month_view->colors[MONTH_VIEW_COLOR_EVEN_MONTHS].green = 0xeded;
	month_view->colors[MONTH_VIEW_COLOR_EVEN_MONTHS].blue  = 0xeded;

	month_view->colors[MONTH_VIEW_COLOR_ODD_MONTHS].red   = 65535;
	month_view->colors[MONTH_VIEW_COLOR_ODD_MONTHS].green = 65535;
	month_view->colors[MONTH_VIEW_COLOR_ODD_MONTHS].blue  = 65535;

	gdk_color_parse ("light green", 
			 &month_view->colors[MONTH_VIEW_COLOR_EVENT_BACKGROUND]);

	gdk_color_parse ("light grey",
			 &month_view->colors[MONTH_VIEW_COLOR_SUMMARY_EVENT_BACKGROUND]);

	month_view->colors[MONTH_VIEW_COLOR_EVENT_BORDER].red   = 0;
	month_view->colors[MONTH_VIEW_COLOR_EVENT_BORDER].green = 0;
	month_view->colors[MONTH_VIEW_COLOR_EVENT_BORDER].blue  = 0;

	month_view->colors[MONTH_VIEW_COLOR_EVENT_TEXT].red   = 0;
	month_view->colors[MONTH_VIEW_COLOR_EVENT_TEXT].green = 0;
	month_view->colors[MONTH_VIEW_COLOR_EVENT_TEXT].blue  = 0;

	month_view->colors[MONTH_VIEW_COLOR_GRID].red   = 0 * 257;
	month_view->colors[MONTH_VIEW_COLOR_GRID].green = 0 * 257;
	month_view->colors[MONTH_VIEW_COLOR_GRID].blue  = 0 * 257;

	month_view->colors[MONTH_VIEW_COLOR_SELECTED].red   = 0 * 257;
	month_view->colors[MONTH_VIEW_COLOR_SELECTED].green = 0 * 257;
	month_view->colors[MONTH_VIEW_COLOR_SELECTED].blue  = 156 * 257;

	month_view->colors[MONTH_VIEW_COLOR_DATES].red   = 0 * 257;
	month_view->colors[MONTH_VIEW_COLOR_DATES].green = 0 * 257;
	month_view->colors[MONTH_VIEW_COLOR_DATES].blue  = 0 * 257;

	month_view->colors[MONTH_VIEW_COLOR_DATES_SELECTED].red   = 65535;
	month_view->colors[MONTH_VIEW_COLOR_DATES_SELECTED].green = 65535;
	month_view->colors[MONTH_VIEW_COLOR_DATES_SELECTED].blue  = 65535;

	nfailed = gdk_colormap_alloc_colors (colormap, month_view->colors,
					     MONTH_VIEW_COLOR_LAST, FALSE,
					     TRUE, success);
	if (nfailed) {
		g_warning ("Failed to allocate all colors");
	}
}

static void
month_view_unrealize (GtkWidget *widget)
{
	g_return_if_fail (widget != NULL);
	g_return_if_fail (IS_MONTH_VIEW (widget));

	d(puts(__FUNCTION__));
}

static void
month_view_style_set (GtkWidget *widget,
		      GtkStyle  *previous_style)
{
	MonthView *month_view;
	GdkFont *font;
	gint day, day_width, max_day_width, max_abbr_day_width;
	gint month, month_width, max_month_width, max_abbr_month_width;
	GDate date;
	gchar buffer[128];

	d(puts(__FUNCTION__));

	if (GTK_WIDGET_CLASS (parent_class)->style_set) {
		(*GTK_WIDGET_CLASS (parent_class)->style_set)(widget, previous_style);
	}

	month_view = MONTH_VIEW (widget);
	font = widget->style->font;

	/* Recalculate the height of each row based on the font size. */
	month_view->row_height = font->ascent + font->descent + 
		MONTH_VIEW_EVENT_BORDER_HEIGHT * 2 + 
		MONTH_VIEW_EVENT_TEXT_Y_PAD * 2;

	/* Set the height of the top canvas. */
	gtk_widget_set_usize (GTK_WIDGET (month_view->titles_canvas), -1,
			      font->ascent + font->descent + 5);

	/* Save the sizes of various strings in the font, so we can quickly
	   decide which date formats to use. */
	g_date_clear (&date, 1);
	g_date_set_dmy (&date, 27, 3, 2000);	/* Must be a Monday. */

	max_day_width = 0;
	max_abbr_day_width = 0;
	for (day = 0; day < 7; day++) {
		g_date_strftime (buffer, 128, "%A", &date);
		day_width = gdk_string_width (font, buffer);
		month_view->day_widths[day] = day_width;
		max_day_width = MAX (max_day_width, day_width);

		g_date_strftime (buffer, 128, "%a", &date);
		day_width = gdk_string_width (font, buffer);
		month_view->abbr_day_widths[day] = day_width;
		max_abbr_day_width = MAX (max_abbr_day_width, day_width);

		g_date_add_days (&date, 1);
	}

	max_month_width = 0;
	max_abbr_month_width = 0;
	for (month = 0; month < 12; month++) {
		g_date_set_month (&date, month + 1);

		g_date_strftime (buffer, 128, "%B", &date);
		month_width = gdk_string_width (font, buffer);
		month_view->month_widths[month] = month_width;
		max_month_width = MAX (max_month_width, month_width);

		g_date_strftime (buffer, 128, "%b", &date);
		month_width = gdk_string_width (font, buffer);
		month_view->abbr_month_widths[month] = month_width;
		max_abbr_month_width = MAX (max_abbr_month_width, month_width);
	}

	month_view->space_width = gdk_string_width (font, " ");
	month_view->colon_width = gdk_string_width (font, ":");
	month_view->slash_width = gdk_string_width (font, "/");
	month_view->digit_width = gdk_string_width (font, "5");
	if (month_view->small_font) {
		month_view->small_digit_width = gdk_string_width (month_view->small_font, "5");
	}
	month_view->max_day_width = max_day_width;
	month_view->max_abbr_day_width = max_abbr_day_width;
	month_view->max_month_width = max_month_width;
	month_view->max_abbr_month_width = max_abbr_month_width;
}

static void
month_view_size_allocate (GtkWidget *widget, 
			  GtkAllocation *allocation)
{
	MonthView *month_view;
	gdouble old_x2, old_y2, new_x2, new_y2;
	gint canvas_height;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (IS_MONTH_VIEW (widget));
	g_return_if_fail (allocation != NULL);

	month_view = MONTH_VIEW (widget);
	d(puts(__FUNCTION__));

	if (GTK_WIDGET_CLASS (parent_class)->size_allocate) {
		(*GTK_WIDGET_CLASS (parent_class)->size_allocate) (widget, allocation);
	}

	month_view_recalc_cell_sizes (month_view);

	/* Set the scroll region of the top canvas to its allocated size. */
	gnome_canvas_get_scroll_region (GNOME_CANVAS (month_view->titles_canvas),
					NULL, NULL, &old_x2, &old_y2);

	new_x2 = GTK_WIDGET (month_view->titles_canvas)->allocation.width - 1;
	new_y2 = GTK_WIDGET (month_view->titles_canvas)->allocation.height - 1;

	if (old_x2 != new_x2 || old_y2 != new_y2) {
		gnome_canvas_set_scroll_region (GNOME_CANVAS (month_view->titles_canvas),
						0, 0, new_x2, new_y2);
	}

	canvas_height = month_view->rows * (month_view->row_height + MONTH_VIEW_EVENT_Y_SPACING);
	d(g_print ("Canvasheight: %d\n", canvas_height));
	gnome_canvas_set_scroll_region (GNOME_CANVAS (month_view->main_canvas),
					0, 0, GTK_WIDGET (month_view->main_canvas)->allocation.width - 1, 
					canvas_height);
	/*	gnome_canvas_scroll_to (GNOME_CANVAS (month_view->main_canvas),
				0, 0);
	*/

	month_view_reflow (month_view);	
}

static void
month_view_recalc_cell_sizes (MonthView *month_view)
{
	gfloat canvas_width, canvas_height, offset;
	gint row, col;
	gint i;
	d(g_print ("%s\n", __FUNCTION__));
	d(g_print ("Rows: %d\n", month_view->rows));
	month_view->rows = 0;
	for (i = 0; i < month_view->weeks_shown; i++) {
		month_view->rows += month_view->priv->rows_per_week[i];
	}
	d(g_print ("Rows: %d\n", month_view->rows));
	if (month_view->rows < GTK_WIDGET (month_view->main_canvas)->allocation.height /
	    (month_view->row_height + MONTH_VIEW_EVENT_Y_SPACING)) {
		d(g_print ("Redistributing rows"));

		row = 1 + GTK_WIDGET (month_view->main_canvas)->allocation.height / 
			(month_view->row_height + MONTH_VIEW_EVENT_Y_SPACING) - month_view->rows;

		while (row > 0) {
			for (i = 0; i < month_view->weeks_shown; i++, row--) {
				month_view->priv->rows_per_week[i]++;
				month_view->rows++;
				if (row == 0) {
					break;
				}
			}
		}

	}

	if (month_view->row_heights) {
		g_free (month_view->row_heights);
	}
	if (month_view->row_offsets) {
		g_free (month_view->row_offsets);
	}
	month_view->row_heights = g_new0 (gint, month_view->rows);
	month_view->row_offsets = g_new0 (gint, month_view->rows + 1);

	month_view->columns = month_view->compress_weekend ? 6 : 7;

	/* Calculate the column sizes, using floating point so that pixels
	   get divided evenly. Note that we use one more element than the
	   number of columns, to make it easy to get the column widths.
	   We also add one to the width so that the right border of the last
	   column is off the edge of the displayed area. */
	canvas_width = GTK_WIDGET (month_view->main_canvas)->allocation.width + 1;
	canvas_width /= month_view->columns;
	offset = 0;
	for (col = 0; col <= month_view->columns; col++) {
		month_view->col_offsets[col] = floor (offset + 0.5);
		offset += canvas_width;
	}

	/* Calculate the cell widths based on the offsets. */
	for (col = 0; col < month_view->columns; col++) {
		month_view->col_widths[col] = month_view->col_offsets[col + 1]
			- month_view->col_offsets[col];
	}

	/* Now do the same for the row heights. */
	canvas_height = GTK_LAYOUT (month_view->main_canvas)->height + 1;
	offset = 0;
	for (row = 0; row <= month_view->rows; row++) {
		month_view->row_offsets[row] = floor (offset + 0.5);
		offset += month_view->row_height + MONTH_VIEW_EVENT_Y_SPACING;
	}

	/* Calculate the cell heights based on the offsets. */
	for (row = 0; row < month_view->rows; row++) {
		month_view->row_heights[row] = month_view->row_offsets[row + 1]
			- month_view->row_offsets[row];
	}
}

static void
month_view_on_adjustment_changed (GtkAdjustment *adjustment,
				  MonthView *month_view)
{
	GDate date;
	gint week_offset;
	guint32 old_first_day_julian, new_first_day_julian;

	d(puts(__FUNCTION__));

	/* If we don't have a valid date set yet, just return. */
	if (!g_date_valid (&month_view->first_day_shown))
		return;

	/* Determine the first date shown. */
	date = month_view->base_date;
	week_offset = floor (adjustment->value + 0.5);
	g_date_add_days (&date, week_offset * 7);

	d(g_print ("before: %d, after: %d\n",
		 g_date_day_of_year (&month_view->first_day_shown),
		 g_date_day_of_year (&date)));

	/* Convert the old & new first days shown to julian values. */
	old_first_day_julian = g_date_julian (&month_view->first_day_shown);
	new_first_day_julian = g_date_julian (&date);

	/* If we are already showing the date, just return. */
	if (old_first_day_julian == new_first_day_julian)
		return;

	/* Set the new first day shown. */
	month_view->first_day_shown = date;
	gtk_signal_emit (GTK_OBJECT (month_view), signals[FIRST_DAY_CHANGED], NULL);
	gtk_widget_queue_draw (GTK_WIDGET (month_view->main_canvas));
}

GtkWidget *
month_view_new (void)
{
	MonthView *month_view;

	d(puts(__FUNCTION__));

	month_view = gtk_type_new (MONTH_VIEW_TYPE);

	return GTK_WIDGET (month_view);
}


static void
month_view_layout_get_day_position	(MonthView      *month_view,
					 gint		 day,
					 gint		 weeks_shown,
					 gint		 display_start_day,
					 gboolean	 compress_weekend,
					 gint		*day_x,
					 gint		*day_y,
					 gint		*rows)
{
	gint week, day_of_week, col, weekend_col;
	gint week_row, i;

	*day_x = *day_y = *rows = 0;
	g_return_if_fail (day >= 0);

	g_return_if_fail (day < weeks_shown * 7);

	week = day / 7;
	week_row = 0;
	for (i = 0; i < week; i++) {
		week_row += month_view->priv->rows_per_week[i];
	}
	
	col = day % 7;
	day_of_week = (display_start_day + day) % 7;
	if (compress_weekend && day_of_week >= 5) {
		/* In the compressed view Saturday is above Sunday and
		   both have just one row as opposed to 2 for all the
		   other days. */
		if (day_of_week == 5) {
			*day_y = week_row;
			*rows = 1;
		} else {
			*day_y = week_row + 1;
			*rows = 1;
			col--;
		}
		/* Both Saturday and Sunday are in the same column. */
		*day_x = col;
	} else {
		/* If the weekend is compressed and the day is after
		   the weekend we have to move back a column. */
		if (compress_weekend) {
				/* Calculate where the weekend column is.
				   Note that 5 is Saturday. */
			weekend_col = (5 + 7 - display_start_day) % 7;
			if (col > weekend_col) {
				col--;
			}
		}
		
		*day_y = week_row;
		*rows = month_view->priv->rows_per_week[week];
		*day_x = col;
	}
}

void
month_view_get_day_position	(MonthView      *month_view,
				 gint		 day,
				 gint		*day_x,
				 gint		*day_y,
				 gint		*day_w,
				 gint		*day_h)
{
	gint cell_x, cell_y, cell_h;

	month_view_layout_get_day_position (month_view, day,
					    month_view->weeks_shown,
					    month_view->display_start_day,
					    month_view->compress_weekend,
					    &cell_x, &cell_y, &cell_h);
	
	*day_x = month_view->col_offsets[cell_x];
	*day_y = month_view->row_offsets[cell_y];

	*day_w = month_view->col_widths[cell_x];
	*day_h = 0;
	for (; cell_h > 0; cell_h--) {
		*day_h += month_view->row_heights[cell_y];	
	}
	/*	*day_h += month_view->row_heights[cell_y + 1]; */
	/*	if (cell_h == 2) {
		*day_h += month_view->row_heights[cell_y + 1];
		}*/
}

gboolean   
month_view_get_span_position (MonthView      *month_view,
			      gint            task_num,
			      gint            span_num,
			      gint           *span_x,
			      gint           *span_y,
			      gint           *span_w)
{
	MonthViewTask *mv_task;
	time_t start;
	time_t view_start, view_end;
	gint day_start, day_stop;
	gint week;
	gint week_row;
	gint i;

	g_return_val_if_fail (month_view != NULL, FALSE);
	g_return_val_if_fail (IS_MONTH_VIEW (month_view), FALSE);
	g_return_val_if_fail (span_x != NULL && span_y != NULL && span_w != NULL, FALSE);

	d(puts(__FUNCTION__));

	/* FIXME: is this the right fix? Fixes a crash when inserting
	 * a task and the calendar is not realized.
	 */
	if (!month_view->row_offsets) {
		return FALSE;
	}
	
	/* make sure we have valid data to work with */
	if ((mv_task = id_map_lookup (month_view->priv->map, (GM_Id)task_num)) == NULL) {
		return FALSE;
	}

	view_start = time_from_day (g_date_year (&month_view->first_day_shown),
				    g_date_month (&month_view->first_day_shown) - 1,
				    g_date_day (&month_view->first_day_shown));

	view_end = view_start + (7 * 24 * 60 * 60) * month_view->weeks_shown;

	start = mv_task->task->start + span_num * (7 * 24 * 60 * 60);

	if (start < view_start || start > view_end) {
		d(g_print ("Span %d not visible!\n", span_num));
		return FALSE; /* not visible now */
	}

	week = (start - view_start) / (7 * 24 * 60 * 60);
	week_row = 0;
	for (i = 0; i < week; i++) {
		week_row += month_view->priv->rows_per_week[i];
	}
	day_start = span_num > 0 ? 0 : (start - view_start) % (7 * 24 * 60 * 60);
	day_stop = MIN ((7 * 24 * 60 * 60),
			mv_task->task->end - view_start - week * (7 * 24 * 60 * 60));

	d(g_print ("Week: %d, day_start: %d, day_stop: %d\n", week, day_start, day_stop));
	*span_x = ((GTK_WIDGET (month_view->main_canvas)->allocation.width + 1) * day_start) / (7 * 24 * 60 * 60);
	*span_y = month_view->row_offsets[week_row] + month_view->row_height + 
		mv_task->row * (month_view->row_height + MONTH_VIEW_EVENT_Y_SPACING);

	*span_w = ((GTK_WIDGET (month_view->main_canvas)->allocation.width + 1) * day_stop) / 
		(7 * 24 * 60 * 60) - *span_x;

	d(g_print ("From %d, %d -> %d\n", *span_x, *span_y, *span_w));

	return TRUE;
}

const gchar *
month_view_get_task_title (MonthView *month_view,
			   gint       task_num)
{
	MonthViewTask *mv_task;

	mv_task = (MonthViewTask *)id_map_lookup (month_view->priv->map, (GM_Id)task_num);

	g_assert (mv_task != NULL);

	return mv_task->task->name == NULL ? "<no title>" : mv_task->task->name;
}

gint
month_view_get_time_string_width (MonthView *month_view)
{
        gint time_width;

        if (month_view->use_small_font && month_view->small_font)
                time_width = month_view->digit_width * 2
                        + month_view->small_digit_width * 2;
        else
                time_width = month_view->digit_width * 4
                        + month_view->colon_width;

        return time_width;
}


/* This sets the first day shown in the view. It will be rounded down to the
   nearest week. */
void
month_view_set_first_day_shown		(MonthView	*month_view,
					 GDate		*date)
{
	GDate base_date;
	gint weekday, day_offset;
	gboolean update_adjustment_value = FALSE;

	g_return_if_fail (month_view != NULL);
	g_return_if_fail (IS_MONTH_VIEW (month_view));

	/* Calculate the weekday of the given date, 0 = Mon. */
	weekday = g_date_weekday (date) - 1;

	/* Convert it to an offset from the start of the display. */
	day_offset = (weekday + 7 - month_view->display_start_day) % 7;

	/* Calculate the base date, i.e. the first day shown when the
	   scrollbar adjustment value is 0. */
	base_date = *date;
	g_date_subtract_days (&base_date, day_offset);

	/* See if we need to update the base date. */
	if (!g_date_valid (&month_view->base_date)
	    || g_date_compare (&month_view->base_date, &base_date)) {
		month_view->base_date = base_date;
		update_adjustment_value = TRUE;
	}

	/* See if we need to update the first day shown. */
	if (!g_date_valid (&month_view->first_day_shown)
	    || g_date_compare (&month_view->first_day_shown, &base_date)) {
		month_view->first_day_shown = base_date;
	}

	/* Reset the adjustment value to 0 if the base address has changed.
	   Note that we do this after updating first_day_shown so that our
	   signal handler will not try to reload the events. */
	if (update_adjustment_value)
		gtk_adjustment_set_value (GTK_RANGE (month_view->vscrollbar)->adjustment, 0);

	month_view_clear_visible_tasks (month_view);
	month_view_layout_tasks (month_view);
	month_view_create_spans (month_view);

	gtk_signal_emit (GTK_OBJECT (month_view), signals[FIRST_DAY_CHANGED], NULL);
	
	gtk_widget_queue_draw (GTK_WIDGET (month_view->main_canvas));
}

void
month_view_get_first_day_shown		(MonthView	*month_view,
					 GDate		*date)
{
	g_return_if_fail (month_view != NULL);
	g_return_if_fail (IS_MONTH_VIEW (month_view));

	*date = month_view->first_day_shown;
}

#if 0
static gint
month_view_compare_task_length (MonthViewTask *task1, 
				MonthViewTask *task2)
{
	gint l1, l2;

	g_return_val_if_fail (task1 != NULL, 0);
	g_return_val_if_fail (task2 != NULL, 0);

	l1 = ABS (task1->task->end - task1->task->start);
	l2 = ABS (task2->task->end - task2->task->start);

	if (l1 > l2) {
		return -1;
	}
	else if (l1 < l2) {
		return 1;
	}

	return 0;
}
#endif
#if 0
static gint
month_view_compare_task_start (MonthViewTask *task1, 
			       MonthViewTask *task2)
{
	if (task1->task->start > task2->task->start) {
		return 1;
	}
	else if (task1->task->start < task2->task->start) {
		return -1;
	}

	return 0;
}
#endif

static void
update_spans (gpointer value, gpointer user_data)
{
	MonthViewTask *mv_task;
	GSList *node;

	g_return_if_fail (value != NULL);

	mv_task = (MonthViewTask *)value;

	for (node = mv_task->spans; node; node = node->next) {
		g_assert (node->data != NULL);

		gnome_canvas_item_request_update (GNOME_CANVAS_ITEM (node->data));		
	}
}


static gboolean
month_view_reflow_idle (MonthView *month_view)
{
	d(puts (__FUNCTION__));

	g_assert (month_view != NULL);
	g_assert (IS_MONTH_VIEW (month_view));


	g_slist_foreach (month_view->priv->tasks, update_spans, month_view);
	
	gtk_widget_queue_draw (GTK_WIDGET (month_view->main_canvas));

	month_view->priv->reflow_idle_id = 0;
	
	return FALSE;
}

/* FIXME: This reflow should ideally have a lower priority than the
 * reflow idle handler in task-box.
 */
static void
month_view_reflow (MonthView *month_view)
{
	g_return_if_fail (month_view != NULL);
	g_return_if_fail (IS_MONTH_VIEW (month_view));
	
	if (!month_view->priv->reflow_idle_id) {
		month_view->priv->reflow_idle_id =
			g_idle_add ((GSourceFunc) month_view_reflow_idle, month_view);
	}
}	

static gint
month_view_task_sort (MonthViewTask *task1, 
		      MonthViewTask *task2)
{
	gint l1, l2;

	g_return_val_if_fail (task1 != NULL, 0);
	g_return_val_if_fail (task2 != NULL, 0);

	l1 = ABS (task1->task->end - task1->task->start);
	l2 = ABS (task2->task->end - task2->task->start);

	if (task1->task->type == task2->task->type) {
		if (l1 > l2) {
			return -1;
		}
		else if (l1 < l2) {
			return 1;
		}
		return 0;
	}

	if (task1->task->type == GNOME_MrProject_TASK_NORMAL) {
		return -1;
	}
	if (task2->task->type == GNOME_MrProject_TASK_NORMAL) {
		return 1;
	}
	return 0;
}
/*
 * the tast-list should be sorted thought that we have all the children of
 * root at the top. With the normal tasks coming before the summary-tasks.
 * Then this should be repeated for all summary-tasks acting as root.
 * There should probably be some recursive thing going down here.
 */

static GSList *
parent_hash2list (GHashTable *table, GM_Id parent)
{
	GSList *node;
	MonthViewTask *mv_task;
	GSList *result = NULL;

	typedef struct {
		GSList *children;
	} _HashEntry;
	_HashEntry *entry;

	d(g_print ("Working through parent: %d\n", parent));

	entry = g_hash_table_lookup (table, GINT_TO_POINTER (parent));
	if (entry == NULL) {
		return NULL;
	}
	entry->children = g_slist_sort (entry->children, 
					(GCompareFunc)month_view_task_sort);

	for (node = entry->children; node; node = node->next) {
		g_assert (node->data != NULL);

		mv_task = (MonthViewTask *)node->data;
		result = g_slist_append (result, mv_task);

		if (mv_task->task->type == GNOME_MrProject_TASK_SUMMARY) {
			GSList *tmp = parent_hash2list (table, 
							mv_task->task->taskId);
			result = g_slist_concat (result, tmp);
		}
	}
	return result;
}

/* used to destroy the parent-hashtable */
static gboolean
parent_hash_remove (gpointer key, gpointer value, gpointer user_data)
{
	struct _HashEntry {
		GSList *children;
	} *entry;

	g_return_val_if_fail (value != NULL, TRUE);

	entry = (struct _HashEntry *)value;
	if (entry->children != NULL) {
		g_slist_free (entry->children);
	}
	g_free (entry);

	return TRUE;
}

/* here we place the spans on the canvas */
static void
month_view_layout_tasks (MonthView *month_view)
{
	MonthViewTask *mv_task;
	GSList *node;
	GSList *task_list;
	time_t tmp_time;
	time_t view_start, view_end;
	gint *rows;
	gint start_day, end_day, day;
	gint *week_rows;
	gint max_row;
	gint canvas_height;
	GHashTable *parent_hash;
	GM_Id id;

	typedef struct {
		GSList *children;
	} _HashEntry;

	d(puts(__FUNCTION__));

	/* internal-function, we are cruel! */
	g_assert (month_view != NULL);
	g_assert (IS_MONTH_VIEW (month_view));

	view_start = time_from_day (g_date_year (&month_view->first_day_shown),
				    g_date_month (&month_view->first_day_shown) - 1,
				    g_date_day (&month_view->first_day_shown));

	view_end = view_start + (7 * 24 * 60 * 60) * month_view->weeks_shown;

	/* free the old list */
	if (month_view->priv->tasks != NULL) {
		g_slist_free (month_view->priv->tasks);
		month_view->priv->tasks = NULL;
	}

	/* create our hash-table */
	parent_hash = g_hash_table_new (g_direct_hash, g_direct_equal);
	task_list = id_map_get_objects (month_view->priv->map);

	/* find all tasks that should be shown */
	for (node = task_list; node; node = node->next) {
		g_assert (node->data != NULL);
		mv_task = (MonthViewTask *)node->data;

		/* make sure this task is within the boundaries of this month */
		if (!(mv_task->task->start > view_end ||
		      mv_task->task->end < view_start)) {
			_HashEntry *entry;

			id = mv_task->task->parentId;
			if (id_map_lookup (month_view->priv->map, id) == NULL) {
				/* parent not present, use root instead */
				id = 0;
			}
			/* retrieve the list of siblings */
			entry = g_hash_table_lookup (parent_hash, 
						     GINT_TO_POINTER (id));
			if (entry == NULL) { /* entry not found, create */
				entry = g_new0 (_HashEntry, 1);
				g_hash_table_insert (parent_hash, 
						     GINT_TO_POINTER (id), 
						     entry);
			}
			/* add this task to the list of siblings (children of our parent) */
			entry->children = g_slist_prepend (entry->children, 
							   mv_task);

		}
	}
	/* no, convert the hashtable into a list for processing */
	month_view->priv->tasks = parent_hash2list (parent_hash, 0);
	g_slist_free (task_list);
	/* now, destroy the hashtable */
	g_hash_table_foreach_remove (parent_hash, parent_hash_remove, NULL);
	g_hash_table_destroy (parent_hash);
	
	/* this is for pure debuging purposes */
	for (node = month_view->priv->tasks; node; node = node->next) {
		g_assert (node->data != NULL);
		d(g_print ("Id: %d, parent: %d\n", ((MonthViewTask *)node->data)->task->taskId,
			 ((MonthViewTask *)node->data)->task->parentId));
	}

	/* now, assign the row for each task to use */

	rows = g_new0 (gint, 7 * month_view->weeks_shown);
	week_rows = g_new0 (gint, month_view->weeks_shown);
	for (day = 0; day < month_view->weeks_shown; day++) {
		month_view->priv->rows_per_week[day] = 3; /* standard value */
	}
	month_view_recalc_cell_sizes (month_view);
	for (node = month_view->priv->tasks; node; node = node->next) {
		g_assert (node->data != NULL);
		mv_task = (MonthViewTask *)node->data;

		tmp_time = MAX (mv_task->task->start - view_start, 0);
		start_day = tmp_time / (24 * 60 * 60);

		tmp_time = MIN (mv_task->task->end, view_end) - view_start;
		end_day = tmp_time / (24 * 60 * 60);
		max_row = 0;
		for (day = start_day; day <= end_day; day++) {
			rows[day]++;
			if (rows[day] > month_view->priv->rows_per_week[day / 7] - 1) {
				month_view->priv->rows_per_week[day / 7] = rows[day] + 1;
			}
			max_row = MAX (max_row, rows[day] - 1);
		}
		mv_task->row = max_row;
	}
	g_free (rows);

	/* we need to fix the height-setting of the canvas, something really screws thisone up */
	canvas_height = month_view->rows * (month_view->row_height + MONTH_VIEW_EVENT_Y_SPACING);
	d(g_print ("Canvasheight: %d\n", canvas_height));
	gnome_canvas_scroll_to (GNOME_CANVAS (month_view->main_canvas), 0, 0);
	gnome_canvas_set_scroll_region (GNOME_CANVAS (month_view->main_canvas),
					0, 0, GTK_WIDGET (month_view->main_canvas)->allocation.width - 1, 
					canvas_height);

	/* update and redraw the canvas */
	gnome_canvas_item_request_update (GNOME_CANVAS_ITEM (month_view->main_canvas_item));
}


/* this function creates and/or removes spans as necessary */
static void
month_view_update_spans (MonthView *month_view, 
			 MonthViewTask *mv_task)
{
	GnomeCanvasItem *item;

	time_t t;
	gint left, right;
	gint span;
	gint num_spans;
	gint tmp;

	g_assert (month_view != NULL);
	g_assert (IS_MONTH_VIEW (month_view));
	g_assert (mv_task != NULL);

	d(puts(__FUNCTION__));

	t = mv_task->task->start;

	t = ABS(mv_task->task->end - 
		time_week_begin (mv_task->task->start, TRUE));

	span = g_slist_length (mv_task->spans);

	num_spans = (t / (7 * 24 * 60 * 60) + 1) - /* seconds in a week */
		span;
	d(g_print ("Num-spans to change: %d\n", num_spans));

	/*
	 * I don't think this will work if the start is altered :/
	 */
	if (num_spans < 0) { /* too many spans, remove some! */		
		d(puts("Removing spans..."));
		for (; num_spans < 0; num_spans++) {
			d(puts("Removing item"));
			item = GNOME_CANVAS_ITEM (mv_task->spans->data); /* first item */
			mv_task->spans = g_slist_remove (mv_task->spans, item);

			gtk_object_destroy (GTK_OBJECT (item));
		}
		g_assert (mv_task->spans != NULL);
		item = GNOME_CANVAS_ITEM (mv_task->spans->data);

		/* now, remove the arrow-right-flag */
		gtk_object_get (GTK_OBJECT (item), "MonthViewTaskItem::arrows", &tmp, NULL);

		gtk_object_set (GTK_OBJECT (item), "MonthViewTaskItem::arrows", 
				tmp & (~MONTH_VIEW_TASK_ITEM_ARROW_RIGHT), NULL);
	}
	else if (num_spans > 0) {
		d(puts("Adding spans"));

		right = 0;
		left = span > 0 ? MONTH_VIEW_TASK_ITEM_ARROW_LEFT : 0;
		if (left && mv_task->spans != NULL) {
			item = GNOME_CANVAS_ITEM (mv_task->spans->data);
			gtk_object_get (GTK_OBJECT (item), "MonthViewTaskItem::arrows", &tmp, NULL);
			gtk_object_set (GTK_OBJECT (item), "MonthViewTaskItem::arrows", 
					tmp | MONTH_VIEW_TASK_ITEM_ARROW_RIGHT, NULL);
		}
		d(g_print ("Length: %ld\n", t));
		for (; num_spans > 0; num_spans--, span++) {
			d(g_print ("Adding span: %d\n", span));

			if (num_spans > 1) {
				right = MONTH_VIEW_TASK_ITEM_ARROW_RIGHT;
			}
			
			item = gnome_canvas_item_new (gnome_canvas_root (GNOME_CANVAS (month_view->main_canvas)),
						      mv_task->type,
						      "MonthViewTaskItem::month_view", month_view,
						      "MonthViewTaskItem::arrows", left | right,
						      "MonthViewTaskItem::task_num", mv_task->task->taskId,
						      "MonthViewTaskItem::span_num", span,
						      NULL);
			
			mv_task->spans = g_slist_prepend (mv_task->spans, item);

			left = right ? MONTH_VIEW_TASK_ITEM_ARROW_LEFT : 0;
			right = 0;
		} 
	}
}

static void
month_view_assign_task_item_type (MonthViewTask *mv_task)
{
	switch (mv_task->task->type) {
	case GNOME_MrProject_TASK_SUMMARY:
		mv_task->type = month_view_summary_task_item_get_type ();
		break;
	case GNOME_MrProject_TASK_NORMAL:
		mv_task->type = month_view_task_item_get_type ();
		break;
	default:
		g_warning ("Unsupported task-type: %d\n", mv_task->task->type);
		mv_task->type = month_view_task_item_get_type ();
		break;
	}
}

void
month_view_insert_task (MonthView     *month_view, 
			GM_Task       *task)
{
	MonthViewTask *mv_task;

	g_return_if_fail (month_view != NULL);
	g_return_if_fail (IS_MONTH_VIEW (month_view));
	g_return_if_fail (task != NULL);

	d(puts(__FUNCTION__));


	if (id_map_lookup (month_view->priv->map, task->taskId) != NULL) {
		/* task already existing */
		return;
	}

	mv_task = g_new0 (MonthViewTask, 1);
	mv_task->task = corba_util_task_duplicate (task);
	mv_task->spans = NULL;
	mv_task->row = 0;
	month_view_assign_task_item_type (mv_task);

	id_map_insert_id (month_view->priv->map, mv_task->task->taskId, mv_task);
	
	month_view_update_spans (month_view, mv_task);
	month_view_layout_tasks (month_view);
	month_view_reflow (month_view);
}

static void
month_view_remove_task_spans (MonthViewTask *mv_task)
{
	GSList *span;

	g_assert (mv_task != NULL);

	/* now remove all spans from this node */
	for (span = mv_task->spans; span; span = span->next) {
		g_assert (IS_MONTH_VIEW_TASK_ITEM (span->data));
		
		gtk_object_destroy (GTK_OBJECT (span->data));
	}
	/* free the memory */
	g_slist_free (mv_task->spans);
	mv_task->spans = NULL;
}

void
month_view_update_task (MonthView *month_view,
			GM_Task *task)
{
	MonthViewTask *mv_task;
	gboolean recreate_spans;

	g_return_if_fail (month_view != NULL);
	g_return_if_fail (IS_MONTH_VIEW (month_view));
	g_return_if_fail (task != NULL);

	d(g_print ("%s, Id: %d\n", __FUNCTION__, task->taskId));
	d(puts(__FUNCTION__));

	mv_task = id_map_lookup (month_view->priv->map, task->taskId);

	/* make sure we find a task to play with */
	if (mv_task == NULL) {
		return;
	}

	recreate_spans = (mv_task->task->type != task->type);

	corba_util_task_update (mv_task->task, task, TASK_CHANGE_ALL);

	if (recreate_spans) {
		d(puts("Changing task-type"));
		month_view_remove_task_spans (mv_task);
		month_view_assign_task_item_type (mv_task);
	}

	/* update every task */
	month_view_update_spans (month_view, mv_task);
	month_view_layout_tasks (month_view);
	month_view_reflow (month_view);
}

void 
month_view_reparent_task (MonthView      *month_view,
			  GM_Id          task_id,
			  GM_Id          new_parent_id)
{
	MonthViewTask *mv_task;

	d(puts(__FUNCTION__));

	g_return_if_fail (month_view != NULL);
	g_return_if_fail (IS_MONTH_VIEW (month_view));

	mv_task = id_map_lookup (month_view->priv->map, task_id);

	if (mv_task == NULL) {
		return;
	}

	mv_task->task->parentId = new_parent_id;

	month_view_reflow (month_view);
}


void
month_view_remove_task (MonthView *month_view, 
			GSList *tasks)
{
	GSList *node;
	GM_Id id;
	MonthViewTask *mv_task;
	
	d(puts(__FUNCTION__));

	g_return_if_fail (month_view != NULL);
	g_return_if_fail (IS_MONTH_VIEW (month_view));
	g_return_if_fail (tasks != NULL);

	/* traverse all the nodes */
	for (node = tasks; node; node = node->next) {
		g_assert (node->data != NULL);

 		id = GPOINTER_TO_INT (node->data);

		mv_task = (MonthViewTask *)id_map_lookup (month_view->priv->map, id);

		if (mv_task) {
			id_map_remove (month_view->priv->map, id);

			month_view_remove_task_spans (mv_task);

			CORBA_free (mv_task->task);
			g_free (mv_task);
		}
	}
	month_view_layout_tasks (month_view);
	month_view_reflow (month_view);
}

static void
month_view_clear_visible_tasks (MonthView *month_view)
{
	GSList *node;
	MonthViewTask *mv_task;

	g_return_if_fail (month_view != NULL);
	g_return_if_fail (IS_MONTH_VIEW (month_view));

	for (node = month_view->priv->tasks; node; node = node->next) {
		g_assert (node->data != NULL);

		mv_task = (MonthViewTask *)node->data;

		month_view_remove_task_spans (mv_task);
	}
	g_slist_free (month_view->priv->tasks);
	month_view->priv->tasks = NULL;
}

static void
month_view_create_spans (MonthView *month_view)
{
	GSList *node;

	g_return_if_fail (month_view != NULL);
	g_return_if_fail (IS_MONTH_VIEW (month_view));

	for (node = month_view->priv->tasks; node; node = node->next) {
		g_assert (node->data != NULL);

		month_view_update_spans (month_view, (MonthViewTask *)node->data);
	}
	month_view_reflow (month_view);
}

static void
month_view_clear_tasks (gpointer key, gpointer value, gpointer user_data)
{
	MonthViewTask *mv_task;

	g_return_if_fail (value != NULL);

	mv_task = (MonthViewTask *)value;

	month_view_remove_task_spans (mv_task);

	CORBA_free (mv_task->task);
	g_free (mv_task);
}

void 
month_view_clear (MonthView *month_view)
{
	g_return_if_fail (month_view != NULL);
	g_return_if_fail (IS_MONTH_VIEW (month_view));


	id_map_foreach (month_view->priv->map, month_view_clear_tasks, NULL);
	gtk_object_destroy (GTK_OBJECT (month_view->priv->map));
	month_view->priv->map = id_map_new (0);

	if (month_view->priv->tasks != NULL) {
		g_slist_free (month_view->priv->tasks);
		month_view->priv->tasks = NULL;
	}

	month_view_reflow (month_view);
}

GM_Task *
month_view_get_task (MonthView *month_view, gint task_num)
{
	MonthViewTask *mv_task;
	
	mv_task = (MonthViewTask *) id_map_lookup (month_view->priv->map, 
						   (GM_Id) task_num);
	
	return mv_task->task;
}
