/************************************************************
 *   mpgtx an mpeg toolbox                                      *
 *   by Laurent Alacoque <laureck@users.sourceforge.net>     *   
 *   (c) 2001                                                *
 *   You may copy, modify and redistribute this              *
 *   source file under the terms of the GNU Public License   *
 ************************************************************/

#include "mpegOut.hh"
#include "common.hh"
#include <stdio.h>
#include <stdlib.h>

mpegOut::mpegOut(char* filename, bool status) {
	BTRACK;
	print_status = status;
	//	fprintf(stderr, "constructing mpegOut %s\n",filename);
	if (!access(filename, F_OK)) {
		// file already exist
		//fprintf(stderr,"File %s already exist\n",filename);
	}
	//open file for writing/truncate
	MpegOut = fopen(filename, "wb");

	if (MpegOut == 0) {
		// Gosh ! unable to open
		fprintf(stderr, "Unable to open %s for writing\n", filename);
		perror("reason: ");
	}

	HasAudio = HasVideo = false;
	FileType = mpeg_EMPTY;	
}


mpegOut::mpegOut(FILE* filehandle, bool status) {
	BTRACK;
	print_status=status;
	//	fprintf(stderr,"constructing mpegOut with filehandle\n");
	MpegOut = filehandle;	
	HasAudio = HasVideo = false;
	FileType = mpeg_EMPTY;	
}


mpegOut::~mpegOut() {
	BTRACK;
	//	printf("destructing mpegOut \n");
	if (MpegOut != 0) fclose (MpegOut);
}


void mpegSystemOut::WriteHeader(mpeg* Mpeg){
	BTRACK;
	//Have to write system headers and video sequence
	if (!MpegOut) return;
	if (!Mpeg) return;
	mpeg_version = Mpeg->Version();

	if(Mpeg->System->video_system_header)
		fwrite(Mpeg->System->video_system_header,
			1,	Mpeg->System->video_system_header_length,	MpegOut);
	if (Mpeg->System->audio_system_header)
		fwrite(Mpeg->System->audio_system_header,
			1,	Mpeg->System->audio_system_header_length,	MpegOut);

	// okay system headers are written...
	// let's write the PACK + video packet header
	// we keep the remaining in mem because we need to 
	// correct the packet length before outputing it (stdout is not
	// seekable:/
	off_t PACKlength = 0;

	if ((Mpeg->System->first_video_packet[4] & 0xF0) == 0x20) {
		//standard mpeg1 pack
		PACKlength = 12;
	} else {
		if ((Mpeg->System->first_video_packet[4] & 0xC0) == 0x40) {
			//new mpeg2 pack : 14 bytes + stuffing
			PACKlength = 14 + (Mpeg->System->first_video_packet[13] & 0x07);
		} else {
			//wazup?
			// May 28 2001: maybe there's no pack at all...
			byte packetcode = Mpeg->System->first_video_packet[3] & 0xF0;
			if (packetcode == 0xE0 || packetcode == 0xD0 ) {
				// this is a video packet that dosnt have a pack before
				PACKlength = 0;
			} else {

				if(Mpeg->System->first_video_packet[3] != 0xba) {
					//neither a PACK nor a video packet
					fprintf(stderr, "I'm lost, ciao\n");
					exit(1);
				}
				fprintf(stderr, "Weird PACK encountered don't seem to be mpeg 1 nor mpeg 2.\n");
				fprintf(stderr, "I most certainly crash in a minute but I love risk.\n");

				// try to find the end of the pack "manually" i.e. till the first video packet
				bool endofpackfound = false;
				PACKlength = 12;
				for (; PACKlength < Mpeg->System->first_video_packet_length; PACKlength++)
				{
					if (	Mpeg->System->first_video_packet[PACKlength] == 0x0 &&
							Mpeg->System->first_video_packet[PACKlength + 1] == 0x0 &&
							Mpeg->System->first_video_packet[PACKlength + 2] == 0x1 &&
							((Mpeg->System->first_video_packet[PACKlength + 3] & 0xF0) == 0xE0||
							 (Mpeg->System->first_video_packet[PACKlength + 3] & 0xF0) == 0xD0)
						)
					{
						endofpackfound=true;
						break;
						PACKlength--;	// TODO: Laurent what's this? break exits..
					} //if
				} //for

				if ((!endofpackfound) ||
						(Mpeg->System->first_video_packet_length-PACKlength-4-2 <=0))
				{
					fprintf(stderr,"Now I'm really lost, ciao\n");
					exit(1);
				}
			}
			// end May 28
		}
	}

	fwrite(Mpeg->System->first_video_packet,
			1, PACKlength + 4, MpegOut);

	// now allocate required mem
	partial_packet_length = Mpeg->System->first_video_packet_length-PACKlength - 4 - 2;
	//after the size field
	partial_packet_length += Mpeg->Video->video_header_size;
	partial_packet = new byte[partial_packet_length];
	//	printf("allocating %ld bytes\n",partial_packet_length);

	memcpy(partial_packet,
			(Mpeg->System->first_video_packet) + PACKlength + 4 + 2,
			Mpeg->System->first_video_packet_length - PACKlength - 4 - 2);

	memcpy(&partial_packet[(Mpeg->System->first_video_packet_length) - PACKlength - 4 - 2],
			Mpeg->Video->video_header,
			Mpeg->Video->video_header_size);

	HasAudio = HasVideo = true;
	FileType = mpeg_SYSTEM;
	currentTS = Mpeg->System->initial_TS;
	fflush(MpegOut);
}


void mpegVideoOut::WriteHeader(mpeg* Mpeg) {
	BTRACK;
	//Have to write video sequence
	if (!MpegOut) return;
	fwrite(Mpeg->Video->video_header,
		1, Mpeg->Video->video_header_size, MpegOut);		
}


int mpegVideoOut::WriteChunk(
		mpeg* Mpeg,
		off_t from, bool from_included,
		off_t to, bool to_included)
{
	BTRACK;
	if (!MpegOut) return 0;

	off_t real_from, real_to;

	if ((from > to) || (from < 0) || (to < 0)) {	
		//pffff
		return 0;
	}	

	if (from_included) {
		real_from = Mpeg->bdFindNextMarker(from, 0xB8);
		if (real_from == -1) {
			//couldn't find a GOP before from!
			real_from = Mpeg->FindNextMarker(from, 0xB8);
			if (real_from == -1) {
				//gosh couldn't find GOP at all!
				return 0;
			}	
		}
	} else {
		real_from = Mpeg->FindNextMarker(from, 0xB8);
		if (real_from == -1) {
			real_from = Mpeg->bdFindNextMarker(from, 0xB8);
			if(real_from == -1) {
				//couldn't find any GOP!
				return 0;
			}
		}
	}

	//okay we have a real from
	if (to_included) {
		real_to = Mpeg->FindNextMarker(to, 0xB8);
		if (real_to == -1) real_to = Mpeg->FileSize;
	} else {
		real_to = Mpeg->bdFindNextMarker(to, 0xB8);
		if (real_to == -1) {
			//couldn't find any GOP!
			return 0;
		}
	}

	if (real_from >= real_to) return 0;

	//now let's Copy from GOP boundary to GOP boundary
	if (print_status) fprintf(stderr, "        ");
	Copy(Mpeg->MpegFile, real_from,real_to);
	return 1;
}

int mpegAudioOut::WriteChunk(
		mpeg* Mpeg,
		off_t from, bool from_included,
		off_t to,bool to_included)
{
	BTRACK;

	if (!MpegOut) return 0;
	off_t real_from,real_to;

	// 23 March 2001, real_to an real_from are now seeked
	// because there are plenty of variable bitrates mp3

	/*	real_from=from -(from% Mpeg->Audio->frame_length);
		if ((real_from!=from)&&(!from_included))
		real_from+=Mpeg->Audio->frame_length;

		real_to= to -(from%Mpeg->Audio->frame_length);
		if ((real_to!=to)&&(to_included)) 
		real_to += Mpeg->Audio->frame_length;
		if (real_to > Mpeg->Size()) real_to=Mpeg->Size();		
	 */	
	if (from_included)
		real_from = Mpeg->bdFindMatchingAudio(from);
	else 
		real_from = Mpeg->FindMatchingAudio(from);

	if (to_included)
		real_to=Mpeg->FindMatchingAudio(to);
	else 
		real_to=Mpeg->bdFindMatchingAudio(to);


	if (real_to==-1 || real_from==-1 || real_from >= real_to)
		return false;

	if(print_status) fprintf(stderr,"        ");

	Copy(Mpeg->MpegFile, real_from, real_to);
	return 1;
}


int mpegSystemOut::WriteChunk(
		mpeg* Mpeg,
		off_t from, bool from_included,
		off_t to, bool to_included)
{
	BTRACK;
	if (!MpegOut) return 0;

	off_t real_from, real_to;

	//given the from and to offsets, find GOP boundaries
	if ((from > to) || (from < 0) || (to < 0)) {
		//pffff
		return 0;
	}	

	if (from_included) {
		real_from = Mpeg->bdFindNextMarker(from, 0xB8);
		if (real_from == -1) {
			//couldn't find a GOP before from!
			real_from = Mpeg->FindNextMarker(from, 0xB8);
			if (real_from == -1) {
				//gosh couldn't find GOP at all!
				return 0;
			}
		}
	} else {
		real_from= Mpeg->FindNextMarker(from, 0xB8);
		if (real_from == -1) {
			//couldn't find any GOP!
			return 0;
		}
	}

	//okay we have a real from
	if (to_included) {
		real_to = Mpeg->FindNextMarker(to, 0xB8);
		if (real_to == -1) real_to = Mpeg->FileSize;
	} else {
		real_to = Mpeg->bdFindNextMarker(to, 0xB8);
		if (real_to == -1) {
			//couldn't find any GOP!
			return 0;
		}
	}

	if (real_from >= real_to) return 0;

	//	printf("[%ld - %ld ] becomes [%ld - %ld]\n",from,to, real_from,real_to);
	//	while(getchar()!='\n');

	//okay the file is there but artifacts too..
	//we have to correct the packet length
	//we already know the data size we have,
	//let's compute what we just added : from GOP to end of packet

	off_t EOP = real_from - 1;
	marker mark;
	while (true) {
		EOP = Mpeg->FindNextMarker(EOP + 1, &mark);
		if (((mark & 0xF0) != 0x0) && (mark != 0xB8)) {
			// this is neither a slice, nor a pic nor a gop : 
			// we are then at the end of the packet
			break;
		}
		if (EOP == -1) break;
	}
	off_t GOP2EOP = EOP - real_from;


	//if FindNextMarker returned -1 there was no pack after->take filesize
	if (GOP2EOP < 0) GOP2EOP = Mpeg->FileSize-real_from;
	//okay the packet size must now be ::partial_packet_length+GOP2EOP
	//let's write that 
	off_t real_packet_length = partial_packet_length+GOP2EOP;
	byte low = real_packet_length & 0xFF;
	byte high = (real_packet_length >> 8) & 0xFF;
	//fseeko(MpegOut, size_field_offset, SEEK_SET);
	fwrite(&high, 1, 1, MpegOut);
	fwrite(&low, 1, 1, MpegOut);
	//fseeko(MpegOut, 0, SEEK_END);
	fflush(MpegOut);	
	//write what we kept in mem and free memory
	fwrite(partial_packet, 1, partial_packet_length, MpegOut);
	fflush(MpegOut);
	//	printf("deleting partial_packet in WriteChunk\n");
	delete[] partial_packet;

	//now recreate buffer with everything from start of packet
	//to GOP end (real_to) and copy it in partial_packet
	off_t last_packet_offset= Mpeg->bdFindNextMarker(real_to, 0xE0);
	partial_packet_length = real_to-last_packet_offset - 6;
	partial_packet = new byte[partial_packet_length];
	//	printf("allocating %ld bytes\n",partial_packet_length);

	FSEEK(Mpeg->MpegFile, last_packet_offset + 6, SEEK_SET);

	fread(partial_packet, 1, partial_packet_length, Mpeg->MpegFile);

	//now let's Copy from start of packet boundary to beg of last packet
	if (print_status) fprintf(stderr,"        ");
	Copy(Mpeg->MpegFile, real_from,last_packet_offset + 4);

	// Copy updated the current time stamp.
	// we must now correct the partial packet timestamp.
	byte* OldBuffer = buffer;
	buffer = partial_packet;
	CorrectTS(partial_packet_length);
	buffer = OldBuffer;

	return 1;
}

void mpegSystemOut::Copy(FILE* file, off_t from, off_t to) {
	BTRACK;
#ifdef ENABLE_OPTIMIZATION
	double Copy_time_start=Clock_Now();
#endif

	if (!MpegOut) return ;
	//	long DBGsize;
	off_t size = to - from;
	off_t read_size = 0;
	float percent;
	first_TS_correction = true;
	//fprintf(stderr,"Starting copy @%lx\n",ftell(MpegOut));
	size_t size2read = (size > COPYBUFFERSIZE)?COPYBUFFERSIZE:size;
	buffer = new byte[size2read];

	while(read_size < size) {
		size2read = ((size - read_size) > COPYBUFFERSIZE)?COPYBUFFERSIZE:size-read_size;

		FSEEK(file, from + read_size, SEEK_SET);

		read_size += fread(buffer, 1, size2read, file);
		if (first_TS_correction) {
			// hehe we have to set the broken link flag on the first GOP!
			// why? because those players will try to play the leading B frames
			// wich are predicted from former frames (and they're not any, cause
			// we just happen to cut them, remember?)
			// broken_link flag is just for that!
			// we must be on a GOP start here
			// broken link is 27 bit after GOP start sequence
			// it is @ 4 + 3 bytes + 3 bits
			buffer[7] |= 0x20;
			// pfiouuu that one line was hard to find!

			// now that broken link is set we must unset the closed gop bit
			buffer[7] &= 0xBF;


			//TODO: some players ignore this broken link flag...
			// (I hate them)
			// another dirty way to do that is to find all B (and P?)
			// frames preceding the first I frame and change their type
			// (offset of frame type : 00 00 01 00 +10bits /3bits
			// to something like 000b wich is not a valid frame type,
			// they may be discarded by the player

			// I guess the best way to make it robust is to parse the whole
			// GOP, delete the preceding B frames and change the temporal ref
			// of every other frames...
			// This is a though job because a single frame can span over multiple
			// video packets...

			// final versions of this soft might cut mpg files on frame
			// boundaries... we'll take care of that at this very moment.
		}
		CorrectTS(size2read);
		fwrite(buffer, 1, size2read, MpegOut);
		//printf("Written %ld bytes on %ld wanted\n",DBGsize,size2read);
		if (print_status) {
			START_CLOCK

			percent = (read_size*1.0) / (size/100.0);
			percent = (percent>100.0)?100.0:percent;
			fprintf(stderr,"\b\b\b\b\b\b\b\b%5.2f%%  ", percent);
			fflush(stdout);

			STOP_CLOCK(PRINTF);

		}
	}
	delete buffer;
	
#ifdef ENABLE_OPTIMIZATION
	AddTime(Copy_time_start,COPY);
#endif
}

void mpegAudioOut::Finish() {
	// write id3 tag if present
}

void mpegSystemOut::Finish(){
	BTRACK;

	if (!MpegOut) return;
	if (partial_packet) {
		byte low = partial_packet_length & 0xFF;
		byte high = (partial_packet_length >> 8) & 0xFF;
		fwrite(&high, 1, 1, MpegOut);
		fwrite(&low, 1, 1, MpegOut);
		fwrite(partial_packet, 1, partial_packet_length, MpegOut);
		//		printf("deleting partial packet in finish\n");
		delete[] partial_packet;
	}
	fclose(MpegOut);
}


void mpegVideoOut::Copy(FILE* file, off_t from, off_t to){
	BTRACK;
#ifdef ENABLE_OPTIMIZATION
	double Copy_time_start=Clock_Now();
#endif
	if (!MpegOut) return ;
	off_t size = to - from;
	off_t read_size = 0;
	first_TS_correction = true;
	float percent;
	//	fprintf(stderr,"Starting copy @%lx\n",ftell(MpegOut));
	size_t size2read = (size > COPYBUFFERSIZE)?COPYBUFFERSIZE:size;
	buffer = new byte[size2read];

	while (read_size < size) {
		size2read = ((size - read_size) > COPYBUFFERSIZE)?COPYBUFFERSIZE:size-read_size;

		FSEEK(file, from + read_size, SEEK_SET);

		read_size += fread(buffer, 1, size2read, file);
		if (first_TS_correction) {
			buffer[7] |= 0x20;
			first_TS_correction = false;
		}
		fwrite(buffer, 1, size2read, MpegOut);
		if (print_status) {
			//lperc=(read_size*100)/(size/100);
			START_CLOCK;
			percent = (read_size*1.0) / (size/100.0);
			fprintf(stderr, "\b\b\b\b\b\b\b\b%5.2f%%  ", percent);
			fflush(stderr);
			STOP_CLOCK(PRINTF);
		}
	}
	delete buffer;
#ifdef ENABLE_OPTIMIZATION
		AddTime(Copy_time_start,COPY);
#endif
}

void mpegAudioOut::Copy(FILE* file, off_t from, off_t to) {
	BTRACK;
#ifdef ENABLE_OPTIMIZATION
	double Copy_time_start=Clock_Now();
#endif	

	if (!MpegOut) return;
	off_t size = to - from;
	off_t read_size = 0;
	float percent;
	//	fprintf(stderr,"Starting copy @%lx\n",ftell(MpegOut));
	size_t size2read = (size > COPYBUFFERSIZE)?COPYBUFFERSIZE:size;
	buffer = new byte[size2read];

	while (read_size < size) {
		size2read = ((size - read_size) > COPYBUFFERSIZE)?COPYBUFFERSIZE:size-read_size;


		FSEEK(file, from + read_size, SEEK_SET);

		read_size += fread(buffer, 1, size2read, file);
		fwrite(buffer, 1, size2read, MpegOut);
		if (print_status) {
			START_CLOCK;
			//lperc=(read_size*100)/(size/100);
			percent = (read_size*1.0) / (size/100);
			percent = (percent>100.0)?100.0:percent;
			fprintf(stderr, "\b\b\b\b\b\b\b\b%5.2f%%  ", percent);
			fflush(stderr);
			STOP_CLOCK(PRINTF);
		}
	}
	delete buffer;
#ifdef ENABLE_OPTIMIZATION
	AddTime(Copy_time_start,COPY);
#endif
}


void foo(){
	printf("ici");
}

#ifndef _WIN32
inline
#endif

void mpegOutWithVideo::CorrectTS(int bufferlength) {
	BTRACK;
	// for each pack we encounter in that buffer, 
	// get the timestamp

	double ts;
	double ts2;

	//added April 24 2001
	if	(first_TS_correction) 
		ts_correction = 0.0;
	//end added
	int nbofTS;
	int i;
	int saved_i;
	//9 is for 4 bytes PACKstartcode + 5 bytes PACKTS

	for ( i=0; i < bufferlength - 9; i++){	
		//	printf("i vaut %d\n",i);
		if ((buffer[i]== 0x00) &&
				(buffer[i + 1] == 0x00) &&
				(buffer[i + 2] == 0x01))
		{
			switch(buffer[i + 3]) {
				case 0xBA:
					//we have a PACK
					//DBG
					//	printf("PACK ");
					if (mpeg_version == 2)
						memReadTS(i + 4, &ts, true);
					else memReadTS(i + 4, &ts);
					if (first_TS_correction) {
						first_TS_correction = false;
						ts_correction=currentTS-ts+0.015;
					}		
#ifdef _DEBUG_
					fprintf(stderr,"TS: %f     before : %f (delta %f) [pack]\n",ts+ts_correction,ts,ts_correction);
#endif			
					currentTS=ts + ts_correction;
					//	printf("pack current TS is now %f\n",currentTS);
					currentTS = (currentTS < 0)?0:currentTS;
					if (mpeg_version == 2) memWriteTS(i + 4, currentTS, true);
					else memWriteTS(i + 4, currentTS);
#if 0
#ifdef _DEBUG_			
					// debug
					fprintf(stderr,"PACK ");
					memReadTS(i+4,&ts,true);

#endif
#endif

					break;
				case 0xC0 :
				case 0xE0 :
					//skip packet start code & length field
					saved_i = i + 6;
					i += 6;
					nbofTS = memReadPktTS(&i, &ts, &ts2, bufferlength);
					if (first_TS_correction) {
						//ooops ts_correction not initialized... cheat.
						if (nbofTS >= 1) ts_correction=currentTS - ts + 0.015;
					}


					if (nbofTS == 1) {
						//only one pts to change
						ts += ts_correction;

						ts = (ts < 0)?0:ts; //I know this is dirty
						//	printf("pts current TS is now %f\n",ts);
						memWriteTS(i - 5, ts);
//						printf("1 TS written at %d\n", (i - 5) - saved_i);
//						if (i - 5 - saved_i == 0) fprintf(stderr, "ZERO FOUND\n");
					}
					if (nbofTS == 2) {
						//two ts to change
						ts  += ts_correction;
						ts2 += ts_correction;
						ts   = (ts  < 0)?0:ts;
						ts2  = (ts2 < 0)?0:ts2;
						//				printf("pts current TS is now %f\n",ts);
						//				printf("dts current TS is now %f\n",ts2);
						/*				memWriteTS(i-5,ts);
										memWriteTS(i,ts2);
						 */
						memWriteTS(i - 10, ts);
						memWriteTS(i - 5, ts2);
	//					printf("2 TS written at %d and %d\n",
//								(i - 10) - saved_i, (i - 5) - saved_i);
					}
					break;

			} //switch
		} //if
	} //for
}


void mpegOutWithVideo::memReadTS(int offset, double* ts, bool mpeg2pack){
	BTRACK;
	byte highbit;
	unsigned long low4Bytes;
	double TS;
#ifdef _DEBUG_
	printf("Reading TS at offset [" _OFF_x "]\n", offset + FTELL(MpegOut));
#endif //debug
	if (!mpeg2pack) {

		highbit= (buffer[offset]>>3)&0x01;	

		low4Bytes = ((buffer[offset] >> 1) & 0x03) << 30;
		low4Bytes |= buffer[offset + 1] << 22;
		low4Bytes |= (buffer[offset + 2] >> 1) <<15;
		low4Bytes |= buffer[offset + 3] << 7;
		low4Bytes |= buffer[offset + 4] >> 1;

#define FLOAT_0x10000 (double)((unsigned long)1 << 16)
#define STD_SYSTEM_CLOCK_FREQ (unsigned long)90000

		TS =  (double)(highbit * FLOAT_0x10000 * FLOAT_0x10000);
		TS += (double)(low4Bytes);
		TS /= (double)(STD_SYSTEM_CLOCK_FREQ);

		*ts = TS;
	} else {
		//mpeg2 pack header

		unsigned long sys_clock_ref;	
		highbit= (buffer[offset] & 0x20) >> 5;	

		low4Bytes = ((buffer[offset] & 0x18) >> 3) << 30;
		low4Bytes |= (buffer[offset] & 0x03) << 28;
		low4Bytes |= buffer[offset + 1] << 20;

		low4Bytes |= ((buffer[offset + 2] & 0xF8)) << 12;

		low4Bytes |= (buffer[offset + 2] & 0x03) << 13;

		low4Bytes |= buffer[offset + 3] << 5;
		low4Bytes |= (buffer[offset + 4]) >> 3;

		sys_clock_ref=(buffer[offset + 4] & 0x3) << 7;
		sys_clock_ref|=(buffer[offset + 5] >> 1);

		TS =  (double)(highbit * FLOAT_0x10000 * FLOAT_0x10000);
		TS += (double)(low4Bytes);
		if (sys_clock_ref == 0)	TS /= (double)(STD_SYSTEM_CLOCK_FREQ);
		else {
			// this is what I understood... CHECK
			// if not zero, then we have a 27 MHz accuracy with a max of 27 MHz
			// so clockfreq might well be 27MHz / sys_clock_ref
			TS /= (double)(27000000 / sys_clock_ref);
		}
		//debug

#ifdef _DEBUG_
		//	printf("TS2 : base : 0x%d,%032lx ref : %016lx\n",highbit,low4Bytes,sys_clock_ref);
		//	printf ("TS2 : %f\n",TS);
		//	fprintf(stderr,"\nTS mpeg 2 : %02x, %02x %02x %02x %02x, %lx [%lx]\n",
		//		highbit,
		//		byte((low4Bytes>>24)&0xFF),
		//		byte((low4Bytes>>16)&0xFF),
		//		byte((low4Bytes>>8)&0xFF),
		//		byte((low4Bytes)&0xFF),
		//		sys_clock_ref,
		//		(unsigned long) (offset)
		//		);
		fprintf(stderr,"%02x %02x %02x %02x %02x %02x %02x %02x %02x %02x   highBit=%d  low4Bytes=%ld sysclockref=%ld\n",
			buffer[offset-4],
			buffer[offset-3],
			buffer[offset-2],
			buffer[offset-1],
			buffer[offset-0],
			buffer[offset+1],
			buffer[offset+2],
			buffer[offset+3],
			buffer[offset+4],
			buffer[offset+5],
			highbit,
			low4Bytes,
			sys_clock_ref
			 );
#endif

		*ts=TS;
	}
	//DBG
	//	printf("------ TS = %f\n",TS);
}


void mpegOutWithVideo::memWriteTS (int offset, double ts, bool mpeg2pack){
	BTRACK;

#ifdef _DEBUG_
	printf("Writing TS at offset [" _OFF_x "]\n\n\n",(off_t)offset+FTELL(MpegOut));
	fprintf(stderr, " ts is %f\n",ts);
	if (mpeg2pack) fprintf (stderr," mpeg2pack\n");
	fprintf(stderr,"\nbefore write : %02x %02x %02x %02x %02x %02x\n",
			buffer[offset],
			buffer[offset+1],
			buffer[offset+2],
			buffer[offset+3],
			buffer[offset+4],
			buffer[offset+5]);
#endif

	byte uppercode = buffer[offset] & 0xF0;
	byte hiBit;
	long lowInt;
	double TS=ts;

	TS *= STD_SYSTEM_CLOCK_FREQ;
	if (TS > FLOAT_0x10000 * FLOAT_0x10000) {
		hiBit = 1;
		TS -= FLOAT_0x10000 * FLOAT_0x10000;
	} else {
		hiBit = 0;
	}

	lowInt = (long) TS;
#if 0
#ifdef _DEBUG_
	fprintf(stderr,"\nhiBit,lowInt: %02x, %02x %02x %02x %02x \n",
			hiBit,
			byte((lowInt>>24)&0xFF),
			byte((lowInt>>16)&0xFF),
			byte((lowInt>>8)&0xFF),
			byte((lowInt)&0xFF));
#endif
#endif

	if (!mpeg2pack){
		uppercode = uppercode |
			(hiBit << 3) |
			((lowInt & 0xC0000000) >> 29) |
			0x01;

		lowInt = 0x00010001 |
			((lowInt & 0x3FFF8000) << 2) |
			((lowInt & 0x00007FFF) << 1);

		buffer[offset] = uppercode;
		buffer[offset+1] = (byte)(lowInt>>24);
		buffer[offset+2] = (byte)((lowInt>>16)&0xFF);
		buffer[offset+3] = (byte)((lowInt>>8)&0xFF);
		buffer[offset+4] = (byte)(lowInt&0xFF);	
	} else {
		byte work;
		work = 0x40; //marker bit
		work |= hiBit << 5; //hiBit
		work |= lowInt >> 27 ; // two MSBs
		work |= 0x04; //marker bit
		work &= 0xFC; // clear the two LSB
		work |= (lowInt >> 28) & 0x03; // two more MSBs
		buffer[offset] = work; // marker
		buffer[offset+1] = (byte)((lowInt>>20)&0xFF);
		buffer[offset+2] = (byte)((((lowInt>>12)&0xF8))|((lowInt>>13)&0x03)|0x04);
		buffer[offset+3] = (byte)((lowInt>>5)&0xFF);
		buffer[offset+4] = (byte)(((lowInt<<3)&0xFF)|0x04);
		buffer[offset+5] = 0x01;//sys clock ref extension =0

#ifdef _DEBUG_
		fprintf(stderr,"\nafter write : %02x %02x %02x %02x %02x %02x\n",
				buffer[offset],
				buffer[offset+1],
				buffer[offset+2],
				buffer[offset+3],
				buffer[offset+4],
				buffer[offset+5]);
#endif

	}
}






int  mpegOutWithVideo::memReadPktTS(int* off, double* pts, double* dts, int bufferlength) {
	BTRACK;
	//just after the packet start code. and the length field
	byte mybyte;
	off_t offset=*off;
	//get rid of stuffing bytes
	mybyte=buffer[offset];
	while((mybyte & 0x80) && (offset <= bufferlength - 5)) {
		//this has to be a stuffing byte
		mybyte = buffer[++offset];
	}

	//here mybyte is the first valid byte
	if (mpeg_version == 2){
		//skip the first byte
		++offset;
		mybyte = buffer[offset];
		mybyte >>= 6;
		switch (mybyte) {
			case 0: return 0;
			case 1: 
#ifdef _DEBUG_		
					  fprintf(stderr,"Invalid PTS DTS flag at [" _OFF_x "]\n", offset);
#endif
					  return 0;
			case 2: //only PTS;
					  offset++;
					  memReadTS(offset,pts);
					  offset += 5;
					  *off = offset;
					  return 1;
			case 3: //both PTS and DTS
					  offset++;
					  memReadTS(offset, pts);
					  offset += 5;
					  memReadTS(offset, dts);
					  offset += 5;
					  *off = offset;
					  return 2;
		} //switch
		return 0;
	} else {
		if ((mybyte >> 6) == 0x01) {
			//std buffer scale stuff... let's skip it
			offset += 2;
			mybyte = buffer[offset];
		}

		if ((mybyte >> 4) == 0x02) {
			// dts Time stamp

			if (offset > bufferlength - 5) {
				// we have a valid header but no room to read the entire TS
				return -1;
			}

			//DBG
			//			printf("PTS "); 

			memReadTS(offset, pts);
			offset += 5;
			*off = offset;
			return 1;
		}
		else if ((mybyte >> 4) == 0x03) {
			//both pts and dts time stamps
			if (offset > bufferlength - 5) {
				// we have a valid header but no room to read the entire TS
				return -1;
			}
			*off = offset;
			//DBG
			// printf("PTS "); 

			memReadTS(offset, pts);
			offset += 5;
			if (offset > bufferlength - 5) {
				// we have a valid header but no room to read the entire TS
				*off = offset;
				return 1; //1 for one valid TS found
			}
			//DBG
			//	printf("DTS "); 

			memReadTS(offset, dts);
			offset += 5;
			*off = offset;
			return 2;
		}
	}
	return 0;
}



mpegOut* mpegOutFactory::NewMpegFrom(mpeg* MpegIn, char* filename, bool status){
	BTRACK;
	mpegOut* object;
	switch(MpegIn->MpegType) {
		case mpeg_AUDIO:
			object = new mpegAudioOut(filename,status);
			break;
		case mpeg_VIDEO:
			object = new mpegVideoOut(filename,status);
			break;
		case mpeg_SYSTEM:
			object = new mpegSystemOut(filename,status);
			break;
		default:
			object = 0;
			break;
	}

	if (object) {
		if (object->OutFile())
			object->WriteHeader(MpegIn);
		else {
			delete object;
			object = 0;
		}
	}
	return object;
}




mpegOut* mpegOutFactory::NewMpegFrom(mpeg* MpegIn, FILE* filehandle, bool status){
	BTRACK;
	mpegOut* object;
	switch (MpegIn->MpegType) {
		case mpeg_AUDIO:
			object = new mpegAudioOut(filehandle,status);
			break;
		case mpeg_VIDEO:
			object = new mpegVideoOut(filehandle,status);
			break;
		case mpeg_SYSTEM:
			object = new mpegSystemOut(filehandle,status);
			break;
		default:
			object = 0;
			break;
	}
	if (object) object->WriteHeader(MpegIn);
	return object;
}







/*
	class demuxer{
	public:
	demuxer(mpeg* _Mpeg,char* videofile, char* audiofile);
	WriteChunk(off_t from, off_t to, bool from_included, bool to_included);
	protected:
	demuxer(){};	
	FILE* AudioOut;
	FILE* VideoOut;
	mpeg* Mpeg;
	};
 */





demuxer::demuxer(mpeg* _Mpeg, char* _basename, bool _confirm)
:n_audio(0),n_video(0),n_programs(0),Mpeg(_Mpeg),confirm(_confirm),buffer(0)
{
	BTRACK;
	basename = new char[strlen(_basename) + 1];
	strcpy(basename,_basename);
	// convention for Audiofile and Videofile :
	// if Audiofile[i]==0 : file is to be opened with basename
	// if Audiofile[i]==-1  stream is to be ignored.
	// this way piping a stream to stdout is eq to Audiofile[i]=stdout 
	
	for(int i = 0; i < 16; i++){
		AudioFile[i] = VideoFile[i] = 0;
	}

}







demuxer::~demuxer() {
	BTRACK;
	if (buffer) delete[]buffer;
	for (int i = 0; i < 16; i++) {
		if (AudioFile[i]) fclose(AudioFile[i]);
		if (VideoFile[i]) fclose(VideoFile[i]);
	}
}








int demuxer::Process(){
	BTRACK;
	if (Mpeg == 0) return 0;
	if (Mpeg->MpegType == mpeg_SYSTEM) return ProcessProgramStream();
	if (Mpeg->MpegType == mpeg_TRANSPORT) return ProcessTransportStream();

	//else :)

	fprintf(stderr,"mpeg file %s can not be demultiplexed"
			" (neither System nor Program stream)\n", Mpeg->FileName);
	return 0;
}




int demuxer::ProcessTransportStream() {
	BTRACK;
	if (!Mpeg->Transport) return 0;
	int pr;
	int stream_num;
	off_t offset = 0;
	off_t FileSize = Mpeg->Size();
	PID pid;
	EStream *currentStream;
	off_t start, end;
	n_programs = Mpeg->Transport->n_progs;
	int feedback = 0;
	
	while (true) {
		pid = Mpeg->NextTrPacket(&offset,&start,&end);

		if (start == -1) {
			fprintf(stderr," Demultiplexing : [Programs %02d audio %02d  video %02d] [100.00%%]\n",
				n_programs,n_audio,n_video);
			return n_audio + n_video;
		}

		if (start == end) continue;

		for (pr = 0; pr < Mpeg->Transport->n_progs; pr++)
		{
			currentStream = Mpeg->Transport->programs[pr].TStreams;
			stream_num = 0;
			while (currentStream !=0){
				stream_num++;
				if (currentStream->pid == pid) {
					// this is the good one
					if (currentStream->demuxFileOk) {
						if (currentStream->demuxFile == 0x0) {
							// file is not created 
							char* tempname = new char[300];
							sprintf(tempname, "%s-%d-%d.",
									basename, Mpeg->Transport->programs[pr].prog_num, stream_num);
							switch (currentStream->type) {
								case 1: strcat(tempname, "m1v"); n_video++; break;
								case 2: strcat(tempname, "m2v"); n_video++; break;
								case 3: strcat(tempname, "mp3"); n_audio++; break;
								case 4: strcat(tempname, "mp3"); n_audio++; break;
								default: strcat(tempname, "unk"); break;
							}
							currentStream->demuxFile = openfile(tempname);
							delete[] tempname;
						}
						// if currentStream->demuxFile still 0 : problem
						if (currentStream->demuxFile == 0x0) {
							fprintf(stderr, "skipping Program %d stream %d\n",
									Mpeg->Transport->programs[pr].prog_num, stream_num);
							currentStream->demuxFileOk = false;
						} else {
							//		Copy(currentStream->demuxFile,start,end);
							DemuxTrPkt(currentStream->demuxFile, start, end);
							if ((feedback++ == 10000)) {
								feedback = 0;
								fprintf(stderr,
									" Demultiplexing : [Programs %02d audio %02d  video %02d] [%6.2f%%]\r",
									n_programs, n_audio, n_video, offset*100.0/FileSize);
							}
						} //else
					} //if
				} //if
				currentStream = currentStream->next;
			} //while
		} //for

#ifdef _DEBUG_
//		printf("PID %d has payload [%llx %llx]\n", pid, start, end);
#endif
	} //while (true)
	return 0;
}




int demuxer::DemuxTrPkt(FILE* out, off_t start, off_t end){
	BTRACK;
	// we have a transport packet, let's remove the program packet headers
	marker mark;
	off_t packetheader = Mpeg->FindNextMarker(start, &mark);
	off_t packetend = Mpeg->SkipPacketHeader(packetheader);
	mark &= 0xF0;

	if (mark==0xC0 || mark==0xE0 || mark==0xD0) {
		if (packetheader < 0 || packetheader > end) {
			// no packet header in the boundaries
			Copy(out, start, end);
			return 0;
		}
		if (packetend > end) {
			//serious problem
			return -1;
		}
		//here we have to issue two Copy
		Copy(out, start, packetheader);
		Copy(out, packetend, end);
		return 1;
	} else {
		Copy(out, start, end);
		return 1;
	}
}

int demuxer::ProcessProgramStream(){
	BTRACK;
	marker mark;
	off_t from, to;
	off_t FileSize = Mpeg->Size();
	off_t bytestoskip = 1;
	FILE* outfile;
	char* filename;
	int streamnumber;
	off_t offset;

	for (offset = 0; offset < Mpeg->Size(); offset += bytestoskip) {
		offset = Mpeg->FindNextMarker(offset,&mark);
		if (offset == -1) {
			// either we finished or unrecoverable error
			fprintf(stderr," Demultiplexing : [audio %02d  video %02d] [100.00%%]\n",
					n_audio,n_video);
			return n_audio+n_video;			
		}

		fprintf(stderr," Demultiplexing : [audio %02d  video %02d] [%6.2f%%]\r",
				n_audio,n_video, offset*100.0/FileSize);

		fflush(stderr);
		if (((mark&0xF0) == 0xC0) || ((mark&0xF0) == 0xD0)) {
			//new in mpeg 2 : audio streams are : 110x xxxx
			//audio packet
			streamnumber = mark&0x0F;
			if (AudioFile[streamnumber]) {
				//file is already opened or stream has to be ignored
			} else {
				//file has to be opened
				filename  = new char[300];
				n_audio++;
				sprintf(filename,"%s-%d.mp%d", basename, streamnumber, Mpeg->Audio->layer);
				AudioFile[streamnumber] = openfile(filename);					
				delete[] filename;
			}
			outfile = AudioFile[streamnumber];
		}
		else
			if ((mark&0xF0)==0xE0){
				//video packet
				streamnumber=mark&0x0F;
				if (VideoFile[streamnumber]){
					//file is already opened
				}
				else{
					//file has to be opened
					filename=new char[300];
					n_video++;			
					sprintf(filename,"%s-%d.m%dv",basename,streamnumber,Mpeg->mpeg_version);
					VideoFile[streamnumber]=openfile(filename);					
					delete[] filename;
				}
				outfile=VideoFile[streamnumber];

			} else {
				//neither Audio nor Video
				continue;
			}

		//whatever it is it outfile mustn't be 0 nor 1
		// if outfile is still 0 there was a problem opening it
		// if outfile is -1 we were asked to skip it
		if (outfile==0 || outfile==(FILE*)(0x1)) {
//			fprintf(stderr,"skipped stream %x\n",mark);
			continue;
		}

		// now we have to find out where it begins and where it ends
		int length = ((Mpeg->Byte(offset+4)<<8) | Mpeg->Byte(offset+5));
		from = Mpeg->SkipPacketHeader(offset);
		to = offset + length + 6;
		if (to <= from) {
			fprintf(stderr, "Erroneous packet size, skipping\n");
			offset++;
			continue;
		}
		START_CLOCK
		Copy(outfile, from, to);
		STOP_CLOCK(COPY)
		offset = to - 1;

		if (to >= FileSize) offset = FileSize;		
	}
	START_CLOCK
	fprintf(stderr," Demultiplexing : [audio %02d  video %02d] [100.00%%]\n",
			n_audio,n_video);
	STOP_CLOCK(PRINTF)
	return 1;
}


off_t demuxer::Copy(FILE* into, off_t from, off_t to) {
	BTRACK;
	off_t size = to -from;
	off_t read_size = 0;
	//	float percent;
	//	fprintf(stderr,"Starting copy @%lx\n",ftell(MpegOut));
	size_t size2read=(size>COPYBUFFERSIZE)?COPYBUFFERSIZE:size;
	if(!buffer) buffer= new byte[COPYBUFFERSIZE];

	while (read_size < size) {

		size2read =
			((size-read_size) > COPYBUFFERSIZE)?COPYBUFFERSIZE:size-read_size;

		FSEEK(Mpeg->MpegFile, from + read_size, SEEK_SET);

		read_size += fread(buffer, 1, size2read, Mpeg->MpegFile);
		fwrite(buffer, 1, size2read, into);
		//	if(print_status){
		//lperc=(read_size*100)/(size/100);
		//		percent= (read_size*1.0)/(size/100);
		//		fprintf(stderr,"\b\b\b\b\b\b\b\b%5.2f%%  ",percent);
		//		fflush(stderr);
		//	}
	}
	//	delete buffer;
	return to;
}


FILE* demuxer::openfile(char* filename) {
	BTRACK;
	FILE* temp;
	if (!confirm){
		temp = fopen(filename, "wb");
		if (temp==0){
			// problem
			fprintf(stderr,"\n");
			perror(filename);
		}
		return temp;
	} else {
		//must check if file exist
		temp = fopen(filename, "rb");
		if (temp != 0) {
			//file is readable, it exists
			fprintf(stderr,"file %s exists, overwrite ? [N/y/a]:     \b\b\b\b",filename);
			fflush(stderr);
			char answer = getchar();
			switch (answer) {
				case 'a':
				case 'A': confirm = false;
				case 'y':
				case 'Y':
					 temp = fopen(filename, "wb");
					 if (temp == 0) {

						 // problem
						 fprintf(stderr, "\n");
						 perror(filename);
					 }
					 return temp;
					 break;			
				default: return 0;
			} //switch
		} else {
			//file does not exist
			temp = fopen(filename, "wb");
			if (temp == 0) {
				// problem
				fprintf(stderr, "\n");
				perror(filename);
			}
			return temp;
		}
	}
}
