﻿using System.Collections;
using System.Collections.Generic;
using System.Web.Util;

namespace System.Web.UI.WebControls {
    public abstract class AutoFieldsGenerator : IAutoFieldGenerator, IStateManager {
        private List<AutoGeneratedFieldProperties> _autoGeneratedFieldProperties;
        private bool _tracking;

        internal bool InDataBinding {
            get;
            set;
        }

        internal object DataItem {
            get;
            set;
        }

        private ICollection CreateAutoGeneratedFieldsFromFieldProperties() {
            List<AutoGeneratedField> autoFields = new List<AutoGeneratedField>();

            foreach (AutoGeneratedFieldProperties fieldProperties in AutoGeneratedFieldProperties) {
                autoFields.Add(CreateAutoGeneratedFieldFromFieldProperties(fieldProperties));
            }

            return autoFields;
        }

        //internal for unit testing.
        protected internal List<AutoGeneratedFieldProperties> AutoGeneratedFieldProperties {
            get {
                if (_autoGeneratedFieldProperties == null) {
                    _autoGeneratedFieldProperties = new List<AutoGeneratedFieldProperties>();
                }
                return _autoGeneratedFieldProperties;
            }
        }

        /// <devdoc>
        /// Create a single autogenerated field.  This function can be overridden to create a different AutoGeneratedField.
        /// </devdoc>
        protected virtual AutoGeneratedField CreateAutoGeneratedFieldFromFieldProperties(AutoGeneratedFieldProperties fieldProperties) {
            AutoGeneratedField field = new AutoGeneratedField(fieldProperties.DataField);
            string name = fieldProperties.Name;
            ((IStateManager)field).TrackViewState();

            field.HeaderText = name;
            field.SortExpression = name;
            field.ReadOnly = fieldProperties.IsReadOnly;
            field.DataType = fieldProperties.Type;

            return field;
        }

        protected bool IsTrackingViewState {
            get {
                return _tracking;
            }
        }

        protected virtual void TrackViewState() {
            _tracking = true;
            foreach (object fieldProps in AutoGeneratedFieldProperties) {
                ((IStateManager)fieldProps).TrackViewState();
            }
        }

        protected virtual void LoadViewState(object savedState) {
            if (savedState != null) {
                object[] autoGenFieldStateArray = (object[])savedState;
                int fieldCount = autoGenFieldStateArray.Length;
                AutoGeneratedFieldProperties.Clear();

                for (int i = 0; i < fieldCount; i++) {
                    AutoGeneratedFieldProperties fieldProps = new AutoGeneratedFieldProperties();
                    ((IStateManager)fieldProps).TrackViewState();
                    ((IStateManager)fieldProps).LoadViewState(autoGenFieldStateArray[i]);
                    AutoGeneratedFieldProperties.Add(fieldProps);
                }
            }
        }

        protected virtual object SaveViewState() {
            // pack up the auto gen'd field properties
            int autoGenFieldPropsCount = AutoGeneratedFieldProperties.Count;
            object[] autoGenFieldState = new object[autoGenFieldPropsCount];

            for (int i = 0; i < autoGenFieldPropsCount; i++) {
                autoGenFieldState[i] = ((IStateManager)AutoGeneratedFieldProperties[i]).SaveViewState();
            }
            return autoGenFieldState;
        }

        /// <summary>
        /// Method that creates the AutoGeneratedFields. This would also typically populate the AutoGeneratedFieldProperties collection
        /// which are tracked by view state.
        /// </summary>
        /// <returns></returns>
        public abstract List<AutoGeneratedField> CreateAutoGeneratedFields(object dataItem, Control control);

        /// <summary>
        /// A flag to indicate whether Enum fields should be generated by this FieldGenerator.
        /// If this flag is not set, the FieldGenerator uses <see cref='System.Web.UI.WebControls.Control.RenderingCompatibility'/> of
        /// <see cref='System.Web.UI.WebControls.Control'/> to determine the default behavior which is 
        /// true when RenderingCompatibility is greater than or equal to 4.5 and false in older versions.
        /// </summary>
        public bool? AutoGenerateEnumFields {
            get;
            set;
        }

        #region Implementation of IAutoFieldGenerator

        public virtual ICollection GenerateFields(Control control) {
            if (InDataBinding) {
                //Reset this collection as this would be populated by CreateAutoGenerateFields.
                _autoGeneratedFieldProperties = null;
                return CreateAutoGeneratedFields(DataItem, control);
            }
            else {
                return CreateAutoGeneratedFieldsFromFieldProperties();
            }
        }

        #endregion

        #region Implementation of IStateManager

        bool IStateManager.IsTrackingViewState {
            get {
                return IsTrackingViewState;
            }
        }

        void IStateManager.LoadViewState(object savedState) {
            LoadViewState(savedState);
        }

        object IStateManager.SaveViewState() {
            return SaveViewState();
        }

        void IStateManager.TrackViewState() {
            TrackViewState();
        }

        #endregion
    }
}
