/*
 *  mod_bt - Making Things Better For Seeders
 *  Copyright 2006 Tyler MacDonald <tyler@yi.org>
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */


#include <libbtutil/util.h>
#include <libbtutil/types/bt_metainfo.h>

#include <stdlib.h>
#include <stdio.h>
#include <strings.h>

#include <apr.h>
#include <apr_pools.h>

static inline uint64_t last_piece_size(bt_metainfo* t) {
    return t->total_size - ((t->piece_count-1) * t->piece_size);
}

#define no_piece_remainder(t) \
    (t->piece_size == last_piece_size(t))

static inline uint64_t piece_remainder(bt_metainfo* t) {
    if(no_piece_remainder(t)) {
        return 0;
    } else {
        return last_piece_size(t);
    }
}

static inline unsigned int piece_count(bt_metainfo* t) {
    if(no_piece_remainder(t)) {
        return t->piece_count;
    } else {
        return t->piece_count - 1;
    }
}

static const char* banner = "\
bt_showmetainfo "VERSION" - decode BitTorrent metainfo files\n\
\n\
";

static const char* metainfo_header = "\
metainfo file.: %s\n\
info hash.....: %s\n\
";

static const char* metainfo_footer = "announce url..: %s\n\n";

static const char* single_metainfo = "\
file name.....: %s\n\
file size.....: %llu (%u * %u + %llu)\n\
";

static const char* multi_metainfo_header = "\
directory name: %s\n\
files.........:\n\
";

static const char* multi_metainfo_file = "\
   %s (%lu)\n\
";

static const char* multi_metainfo_footer = "\
archive size..: %llu (%u * %u + %llu)\n\
";

static inline void show_single_metainfo(bt_metainfo* t) {
    printf(
        single_metainfo,
        t->name, t->total_size, piece_count(t), t->piece_size,
        piece_remainder(t)
    );
}

static inline void show_archive_metainfo(bt_metainfo* t) {
    int i;
    
    printf(multi_metainfo_header, t->name, t->file_count);
    
    for(i=0; i<t->file_count; i++)
        printf(
            multi_metainfo_file,
            t->files[i].name + strlen(t->name) + 1,
            t->files[i].length
        );
    
    printf(
        multi_metainfo_footer, 
        t->total_size, piece_count(t), t->piece_size, piece_remainder(t)
    );
}

static inline void show_metainfo(apr_pool_t* p, bt_metainfo* torrent) {
    char *file;
    
    if(((file = rindex(torrent->torrent, '/') + 1)) == (void*)1)
        file = torrent->torrent;
    
    printf(metainfo_header, file, bt_str_infohash(p, (char*)torrent->hash));
    
    if(torrent->file_count == 1)
        show_single_metainfo(torrent);
    else
        show_archive_metainfo(torrent);
    
    printf(
        metainfo_footer,
        bt_args2url(
            p, torrent->tracker_address, torrent->tracker_port,
            torrent->tracker_announce
        )
    );
}


int main(int argc, char **argv) {
    apr_pool_t*         pool = NULL;
    bt_metainfo*        torrent = NULL;
    int i;

    printf(banner);
    
    if(argc < 2) {
        printf(
            "usage: %s file1.torrent file2.torrent file3.torrent ...\n",
            argv[0]
        );
        exit(19);
    }
    
    if(apr_app_initialize(NULL, NULL, NULL) != APR_SUCCESS) {
        fprintf(stderr, "apr_app_initialize() failed!\n");
        fflush(stderr);
        exit(20);
    }
 
    atexit(apr_terminate);
 
    if(apr_pool_initialize() != APR_SUCCESS) {
        fprintf(stderr, "apr_pool_initialize() failed!\n");
        fflush(stderr);
        exit(2);
    }
 
    if(apr_pool_create(&pool, NULL) != APR_SUCCESS) {
        fprintf(stderr, "apr_pool_create() failed!\n");
        fflush(stderr);
        exit(3);
    }

    for(i=1;i<argc;i++) {
        if((torrent = bt_metainfo_parse(pool, argv[i]))) {
            show_metainfo(pool, torrent);
        } else {
            fprintf(stderr, "Failed to parse torrent\n");
            fflush(stderr);
            exit(10);
        }
    }
    
    exit(0);
}
