# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Dialog to set tolerance.

# load libs
import wx

# load modules
from nucleus import mwx


class dlgSetTolerance(wx.Dialog):
    """ Set tolerance for matching and comparing. """

    # ----
    def __init__(self, parent, tolerance, errortype):
        wx.Dialog.__init__(self, parent, -1, "Set Tolerance", style=wx.DEFAULT_DIALOG_STYLE)

        self.tolerance = tolerance
        self.errortype = errortype

        # pack main frame
        mainBox = wx.BoxSizer(wx.VERTICAL)
        if wx.Platform == '__WXMAC__':
            mainBox.Add(self.makeParamsBox(), 0, wx.ALIGN_CENTER|wx.TOP|wx.LEFT|wx.RIGHT, 20)
            mainBox.Add(self.makeButtonBox(), 0, wx.ALIGN_CENTER|wx.ALL, 10)
        else:
            mainBox.Add(self.makeParamsBox(), 0, wx.ALIGN_CENTER|wx.ALL, 5)
            mainBox.Add(self.makeButtonBox(), 0, wx.ALIGN_CENTER|wx.ALL, 5)

        # fit layout
        mainBox.Fit(self)
        self.SetSizer(mainBox)
        self.Centre()
    # ----


    # ----
    def makeParamsBox(self):
        """ Make box for parameters. """

        # make items
        tolerance_label = wx.StaticText(self, -1, "Tolerance value: ")
        self.tolerance_value = wx.TextCtrl(self, -1, str(self.tolerance), size=(80, -1), validator=mwx.txtValidator('float'))

        errortype_choices = ['Da', 'ppm', '%']
        errortype_label = wx.StaticText(self, -1, "Tolerance units: ")
        self.errortype_combo = wx.ComboBox(self, -1, size=(80, -1), choices=errortype_choices, style=wx.CB_READONLY)

        # pack items
        grid = mwx.GridBagSizer()
        grid.Add(tolerance_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.tolerance_value, (0, 1))
        grid.Add(errortype_label, (1, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.errortype_combo, (1, 1))

        # set defaults
        self.errortype_combo.Select(errortype_choices.index(self.errortype))

        return grid
    # ----


    # ----
    def makeButtonBox(self):
        """ Make box for main buttons. """

        # make items
        OK_button = wx.Button(self, wx.ID_OK, "OK")
        Cancel_button = wx.Button(self, wx.ID_CANCEL, "Cancel")

        # pack items
        buttonBox = wx.BoxSizer(wx.HORIZONTAL)
        buttonBox.Add(OK_button, 0, wx.ALL, 5)
        buttonBox.Add(Cancel_button, 0, wx.ALL, 5)

        # set events
        wx.EVT_BUTTON(self, wx.ID_OK, self.onDlgOK)

        # set defaults
        OK_button.SetDefault()

        return buttonBox
    # ----


    # ----
    def onDlgOK(self, evt):
        """ Get parameters. """

        # get tolerance
        tolerance = self.tolerance_value.GetValue()
        tolerance = tolerance.replace(',', '.')
        try:
            self.tolerance = float(tolerance)
        except ValueError:
            return

        # get errortype
        self.errortype = self.errortype_combo.GetValue()

        # end dialog
        self.EndModal(wx.ID_OK)
    # ----


    # ----
    def getData(self):
        """ Get dialog data. """
        return self.tolerance, self.errortype
    # ----
