# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Main document data format.

# import modules
from description import description
from spectrum import spectrum
from peaklist import peaklist
from sequence import sequence
from monitor import monitor


class docData:
    """ Basic document definition. """

    # ----
    def __init__(self):
        self.description = description()
        self.spectrum = spectrum()
        self.peaklist = peaklist()
        self.sequence = sequence()
        self.monitor = monitor()
    # ----


    # ----
    def setMassParam(self, param, value):
        """ Set masstype, errortype, tolerance. """
        self.monitor.setMassParam(param, value)
    # ----


    # ----
    def setPath(self, path):
        """ Set document file path. """
        self.monitor.setPath(path)
    # ----


    # ----
    def setDocType(self, docType):
        """ Set document type. """
        self.monitor.setDocType(docType)
    # ----


    # ----
    def setScanID(self, scanID):
        """ Set spectrum scan ID. """
        self.monitor.setScanID(scanID)
    # ----


    # ----
    def setChangeStatus(self, status):
        """ Set document change status. """
        self.monitor.setChangeStatus(status)
    # ----


    # ----
    def setProcessingStatus(self, status):
        """ Set spectrum processing status. """
        self.monitor.setProcessingStatus(status)
    # ----


    # ----
    def setDataStatus(self, status, module=None):
        """ Set data status for current or selected module. """
        self.monitor.setDataStatus(status, module)
    # ----


    # ----
    def setMatchStatus(self, status, module=None):
        """ Set matching status for current or selected module. """
        self.monitor.setMatchStatus(status, module)
    # ----


    # ----
    def setDescription(self, parameter, value):
        """ Set description parameter. """

        self.description.setParameter(parameter, value)
        self.monitor.setChangeStatus(True)
    # ----


    # ----
    def setSpectrum(self, spectrum):
        """ Set new points to spectrum. """

        # set new spectrum data
        if not self.monitor.getProcessingStatus():
            self.spectrum.setSpectrum(spectrum, store=True)
        else:
            self.spectrum.setSpectrum(spectrum)

        # update data status
        self.monitor.setChangeStatus(True)
        if spectrum:
            self.monitor.setDataStatus(True, 'mSpec')
        else:
            self.monitor.setDataStatus(False, 'mSpec')
    # ----


    # ----
    def setPeaklist(self, peaks, undo=True):
        """ Set new peaks to peaklist. """

        # set new peaklist data
        if not self.monitor.getProcessingStatus():
            self.peaklist.setPeaklist(peaks, undo, store=True)
        else:
            self.peaklist.setPeaklist(peaks, undo)

        # update data status
        self.monitor.setChangeStatus(True)
        self.monitor.resetMatchStatus()
        if peaks:
            self.monitor.setDataStatus(True, 'mPeak')
        else:
            self.monitor.setDataStatus(False, 'mPeak')
    # ----


    # ----
    def setSequence(self, sequence=[]):
        """ Set new data to sequence. """

        self.sequence.setSequence(sequence)
        self.monitor.setChangeStatus(True)
    # ----


    # ----
    def setSequenceTitle(self, title=''):
        """ Set new sequence title. """

        self.sequence.setTitle(title)
        self.monitor.setChangeStatus(True)
    # ----


    # ----
    def setParsedSequence(self, sequence=[]):
        """ Set new data to parsed sequence. """

        self.sequence.setParsedSequence(sequence)
        if sequence:
            self.monitor.setDataStatus(True, 'mSeq')
        else:
            self.monitor.setDataStatus(False, 'mSeq')
    # ----


    # ----
    def setModifications(self, modifications=[]):
        """ Set new data to sequence modifications. """

        self.sequence.setModifications(modifications)
        self.monitor.setChangeStatus(True)
    # ----


    # ----
    def getMassParam(self, param=None):
        """ Get masstype, errortype, tolerance. """
        return self.monitor.getMassParam(param)
    # ----


    # ----
    def getMassParams(self, param=None):
        """ Get masstype, errortype, tolerance. """
        return self.monitor.setMassParams(param)
    # ----


    # ----
    def getPath(self):
        """ Get document file path. """
        return self.monitor.getPath()
    # ----


    # ----
    def getDocType(self):
        """ Get document type. """
        return self.monitor.getDocType()
    # ----


    # ----
    def getScanID(self):
        """ Get spectrum scan ID. """
        return self.monitor.getScanID()
    # ----


    # ----
    def getChangeStatus(self):
        """ Get document change status. """
        return self.monitor.getChangeStatus()
    # ----


    # ----
    def getProcessingStatus(self):
        """ Get spectrum processing status. """
        return self.monitor.getProcessingStatus()
    # ----


    # ----
    def getPeaklistUndoStatus(self):
        """ Get status of peaklist undo/redo. """
        return self.peaklist.getUndoStatus()
    # ----


    # ----
    def getDataStatus(self, module):
        """ Get data status for current or selected module. """
        return self.monitor.getDataStatus(module)
    # ----


    # ----
    def getMatchStatus(self, module):
        """ Get matching status for current or selected module. """
        return self.monitor.getMatchStatus(module)
    # ----


    # ----
    def getDescription(self, parameter):
        """ Get description parameter. """
        return self.description.getParameter(parameter)
    # ----


    # ----
    def getSpectrum(self):
        """ Get spectrum points. """
        return self.spectrum.getSpectrum()
    # ----


    # ----
    def getSpectrumXRange(self):
        """ Get spectrum mass range. """
        return self.spectrum.getXRange()
    # ----


    # ----
    def getSpectrumLength(self):
        """ Get number of points in the spectrum. """
        return self.spectrum.getLength()
    # ----


    # ----
    def getPeaks(self, indexes=None):
        """ Get selected peaks or whole peaklist data. """
        return self.peaklist.getPeaks(indexes)
    # ----


    # ----
    def getPeaklistLength(self):
        """ Get number of peaks in peaklist. """
        return self.peaklist.getLength()
    # ----


    # ----
    def getSequenceTitle(self):
        """ Get sequence title. """
        return self.sequence.getTitle()
    # ----


    # ----
    def getSequence(self, index1=None, index2=None):
        """ Get sequence slice or whole sequence. """
        return self.sequence.getSequence(index1, index2)
    # ----


    # ----
    def getSequenceString(self, index1=None, index2=None):
        """ Get sequence slice or whole sequence. """
        return self.sequence.getSequenceString(index1, index2)
    # ----


    # ----
    def getParsedSequence(self, index1=None, index2=None):
        """ Get parsed sequence slice or whole parsed sequence. """
        return self.sequence.getParsedSequence(index1, index2)
    # ----


    # ----
    def getSequenceLength(self):
        """ Get number of amino acids in the sequence. """
        return self.sequence.getLength()
    # ----


    # ----
    def getModifications(self, aminoacid=None):
        """ Get modifications for selected amino acid or whole sequence. """
        return self.sequence.getModifications(aminoacid)
    # ----


    # ----
    def getModificationsCount(self, index1=None, index2=None):
        """ Get modifications for selected amino acid or whole sequence. """
        return self.sequence.getModificationsCount(index1, index2)
    # ----


    # ----
    def addPeak(self, peak):
        """ Add new peak to peaklist. """

        # set new peaklist data
        if not self.monitor.getProcessingStatus():
            self.peaklist.addPeak(peak, store=True)
        else:
            self.peaklist.addPeak(peak)

        # update data status
        self.monitor.setChangeStatus(True)
        self.monitor.resetMatchStatus()
        self.monitor.setDataStatus(True, 'mPeak')
    # ----


    # ----
    def editPeak(self, index, peak):
        """ Update selected peak. """

        # set new peaklist data
        if not self.monitor.getProcessingStatus():
            self.peaklist.editPeak(index, peak, store=True)
        else:
            self.peaklist.editPeak(index, peak)

        # update data status
        self.monitor.setChangeStatus(True)
        self.monitor.resetMatchStatus()
    # ----


    # ----
    def deletePeaks(self, indexes=None):
        """ Delete all or selected peaks. """

        # set new peaklist data
        if not self.monitor.getProcessingStatus():
            self.peaklist.deletePeaks(indexes, store=True)
        else:
            self.peaklist.deletePeaks(indexes)

        # update data status
        self.monitor.setChangeStatus(True)
        self.monitor.resetMatchStatus()
        self.monitor.setDataStatus(bool(self.peaklist.getLength()), 'mPeak')
    # ----


    # ----
    def annotatePeaklist(self, annotations):
        """ Annotate peaklist. """

        # set new peaklist data
        if not self.monitor.getProcessingStatus():
            self.peaklist.annotate(annotations, store=True)
        else:
            self.peaklist.annotate(annotations)

        # update data status
        self.monitor.setChangeStatus(True)
    # ----


    # ----
    def deleteAnnotations(self, indexes=None):
        """ Delete annotations in all or selected peaks. """

        # set new peaklist data
        if not self.monitor.getProcessingStatus():
            self.peaklist.deleteAnnotations(indexes, store=True)
        else:
            self.peaklist.deleteAnnotations(indexes)

        # update data status
        self.monitor.setChangeStatus(True)
    # ----


    # ----
    def revertSpectrum(self):
        """ Revert spectrum to init data. """

        self.spectrum.revert()
        self.monitor.setChangeStatus(True)
        self.monitor.setProcessingStatus(False)
        self.monitor.setDataStatus(bool(self.spectrum.getLength()), 'mSpec')
    # ----


    # ----
    def storeSpectrum(self):
        """ Store current spectrum data as main for undo. """
        self.spectrum.store()
    # ----


    # ----
    def revertPeaklist(self):
        """ Revert spectrum to original data. """

        self.peaklist.revert()
        self.monitor.setChangeStatus(True)
        self.monitor.setProcessingStatus(False)
        self.monitor.resetMatchStatus()
        self.monitor.setDataStatus(bool(self.peaklist.getLength()), 'mPeak')
    # ----


    # ----
    def storePeaklist(self):
        """ Store current peaklist data as main for undo. """
        self.peaklist.store()
    # ----


    # ----
    def undoPeaklist(self):
        """ Step back in peaklist history. """

        # set new peaklist data
        if not self.monitor.getProcessingStatus():
            self.peaklist.undo(store=True)
        else:
            self.peaklist.undo()

        # update data status
        self.monitor.setChangeStatus(True)
        self.monitor.resetMatchStatus()
        self.monitor.setDataStatus(bool(self.peaklist.getLength()), 'mPeak')
    # ----


    # ----
    def redoPeaklist(self):
        """ Step forward in peaklist history. """

        # set new peaklist data
        if not self.monitor.getProcessingStatus():
            self.peaklist.redo(store=True)
        else:
            self.peaklist.redo()

        # update data status
        self.monitor.setChangeStatus(True)
        self.monitor.resetMatchStatus()
        self.monitor.setDataStatus(bool(self.peaklist.getLength()), 'mPeak')
    # ----
