// smartmem.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/

#include "application.h"
#include "localdefs.h"
#include "smartmem.h"
#ifdef NeXT
#include <m68k/math.h>
#else
#include <values.h>
#endif


unsigned SmartMemory::totalBytes_Allocated = 0;
unsigned SmartMemory::max_TotalAllocation = unsigned(2.0e+08);	// 200 Mb.
unsigned SmartMemory::max_SingleAllocation = unsigned(1.0e+08);	// 100 Mb.

void
SmartMemory::setMaxTotalAllocation(unsigned maxTot) {
	max_TotalAllocation = maxTot;
}

void
SmartMemory::setMaxSingleAllocation(unsigned maxSingle) {
	max_SingleAllocation = maxSingle;
}

void *
SmartMemory::allocate(unsigned* size) {
	void* mem = 0;
	if(isLegalAllocation(*size)) {
		if((mem = calloc(*size, 1)) != 0)
			totalBytes_Allocated += *size;
		else {
			Application::error("Memory allocation failure:");
			*size = 0;
		}
	}
	else *size = 0;
	return mem;
}

void
SmartMemory::free(void* ptr, unsigned size) {
	if(ptr) {
		::cfree(ptr);
		totalBytes_Allocated -= size;
	}
}

void*
SmartMemory::changeAllocation(void* ptr, unsigned oldSize, unsigned* size) {
	void* oldMem = ptr;
	int delta = 0;
	if(*size < oldSize)					// deallocating
		delta = int(*size) - int(oldSize);
	else {								// allocating
		delta = int(*size - oldSize);
		if(delta < 0) delta = MAXINT;	// check for int overflow
	}
	if(delta < 0 || isLegalAllocation(delta)) {
		if((ptr = realloc(ptr, *size)) != 0) {
			totalBytes_Allocated += delta;
			if(delta > 0)
				zero(addr(ptr) + oldSize, delta);	// zero out extended space
		}
		else {
			Application::error("Memory reallocation failure:");
			ptr = oldMem;
			*size = oldSize;
		}
	}
	else *size = oldSize;
	return ptr;
}

void*
SmartMemory::zero(void* ptr, unsigned size, int zero) {
	if(ptr)
		memset(ptr, zero, size);
	return ptr;
}

void*
SmartMemory::copy(void* from, void* to, unsigned size) {
	if(from != nil && to != nil)
		bcopy(from, to, size);
	return to;
}

int
SmartMemory::isLegalAllocation(unsigned size) {
	int status = false;
	char msg[120];
	const float million = 1.0e+06;
	if(size > maxSingleAllocation()) {
		sprintf(msg, "Request for %.2fmB exceeds current limit (%.2fmB).",
			size/million, maxSingleAllocation()/million);
		status = Application::confirm("Memory limit alert:",
			msg, "Attempt anyway?");
	}
	else if(totalBytes_Allocated + size > maxTotalAllocation()) {
		sprintf(msg, "Request for %.2fmB goes beyond current total allocation limit (%.2fmB).",
			size/million, maxTotalAllocation()/million);
		status = Application::confirm("Memory limit alert:",
			msg, "Attempt anyway?");
	}
	else status = true;	
	return status;
}
