// dataview.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// The base class for the objects which control the visual display of Data
// objects.  Its members are:  a pointer to the Data instance, a pointer to 
// the associated Controller, an array of Graph subclass instances, and various
// pointers to internal graphic elements.  The Controller invokes DataView
// methods to change the display of the Data, and the DataView passes these
// messages on to its Graph subclass instances, which do the actual "work".

#ifndef DATAVIEW_H
#ifdef __GNUG__
#pragma interface
#endif
#define DATAVIEW_H

#include <InterViews/scene.h>
#include "range.h"

class HScale;
class VBox;
class Panner;
class Event;
class Perspective;
class Data;
class Controller;
class Graph;
class EditStatusDelegate;
class Block;

// Assisting class passed as an argument to the DataView constructor.

class ViewInfo {
public:
	ViewInfo();
	ViewInfo(RangeUnit units);
	ViewInfo(RangeUnit, const Range&, const Range&);
	RangeUnit rangeUnits;
	Range frameRange;
	Range channelRange;
};

class DataView : public MonoScene {
public:
	DataView(Controller *, ViewInfo &);
	virtual ~DataView();
	
	redefined void Adjust(Perspective &);
	redefined void Handle(Event &);
	redefined void Update();
	redefined void Reconfig();

	// methods invoked by menu items
	virtual void horizontalZoom();
	virtual void horizontalUnZoom();
	virtual void verticalZoom();
	virtual void verticalUnZoom();
	virtual void setVisibleFrameRange();
	virtual void resetVerticalRange();
	virtual void setChannelRange() {}
	virtual void shiftChannelViewUp() {}
	virtual void shiftChannelViewDown() {}
	virtual void setInsertPoint();
	virtual void setEditRegion();
	virtual void toggleGraphMode() {}

	virtual Range getHorizScaleRange()=0;
	virtual Range getVisibleFrameRange()=0;
	virtual void resetHorizontalScale() {}
	virtual void setVisibleFrameRange(const Range &)=0;
	virtual void setInsertPoint(int, const Range&, boolean scroll=true)=0;
	virtual void setEditRegion(const Range&, const Range&, boolean scroll=true)=0;
	virtual void unselect();
	virtual Range getChannelRange()=0;
	virtual void setChannelRange(const Range &)=0;
	virtual Range getDisplayedHorizRange();

	virtual void setHorizScaleLabel(const char *);
	virtual void showVerticalScales(boolean show);
	virtual boolean horizScaleInFrames();
	virtual RangeUnit horizRangeUnits() { return currentRangeUnits; }
	virtual boolean frameScaleInFrames()=0;
	virtual double horizontalScalingFactor();
	virtual boolean keyCommand(unsigned long cmd);
protected:
	redefined void Resize();

	virtual void setHorizScaleTo(RangeUnit);
	virtual void setHorizRangeUnits(RangeUnit unit) { currentRangeUnits = unit; }
	Perspective& constrainView(Perspective &);
	virtual int graphsVisible() { return graphsShown; }
	virtual void incrementGraphsVisible() { graphsShown++; }
	virtual void decrementGraphsVisible() { graphsShown--; }
	virtual void updateGraphs();
	virtual void doChange();
	virtual void graphEvent(Interactor *, Event &);
	virtual void unselectOthers(Interactor *);
	virtual void addGraph(Graph *);
	virtual void removeGraph(boolean update);
	boolean doForGraphs(Block &);
	Graph* topGraph() { return graphs[0]; }	// for use by subclasses only
protected:
	Data *dataptr;
	Controller *controller;
	VBox *interior;
	VBox *graphBox;
	HScale *hscale;
	Perspective* shown;
private:
	friend Controller;
	friend class ViewChanger;

	void Init(ViewInfo &);
	virtual void initPerspective()=0;
	virtual void checkPerspective()=0;
	virtual Perspective* getVerticalPerspective() { return GetPerspective(); }
	virtual Perspective* getHorizontalPerspective() { return GetPerspective(); }
	Interactor *makeEditStatusPanel();
	EditStatusDelegate* getEditDelegate() { return editDelegate; }
	virtual const char* plotWidthAttribute()=0;
	virtual void constrainHorizontalView(Perspective &);
	virtual void constrainVerticalView(Perspective &) {}
	virtual int horizScaleLeftSpacer();
	virtual int horizScaleRightSpacer() { return 0; }
	void expandGraphArray();
	
	EditStatusDelegate* editDelegate;
	Coord cursorLoc;
	Graph **graphs;
	Interactor* viewScroller;
	int graphsShown;
	int maxGraphs;
	boolean perspectiveSet;		// used to determine whether to initialize
	RangeUnit currentRangeUnits;
};

class StatusPanel;

// This class functions as a relay for messages invoked by the DataEditor
// object regarding the numeric display of information about the data and its
// editing status.  These messages are passed on to the StatusPanel members
// which display the numbers (these are visually contained within the View
// window).

class EditStatusDelegate : public Resource {
public:
	virtual ~EditStatusDelegate();
	void update(const Range& current);
	void update() { update(editRange); }
private:
	friend DataView;
	EditStatusDelegate(DataView* view, StatusPanel* start, StatusPanel* end,
		StatusPanel* dur);
	StatusPanel* myStart;
	StatusPanel* myEnd;
	StatusPanel* myDur;
	DataView* myView;
	Range editRange;
};
	
#endif
