/*
------------------------------------------------------------------------------
MetaCam - Extract EXIF information from digital camera files, with
support for Vendor specific blocks.
Copyright (C) 2000 Daniel Stephens (daniel@cheeseplant.org)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
------------------------------------------------------------------------------
*/

#include <iostream.h>
#include <math.h>
#include <stdio.h>

#include "metacam.h"
#include "dpyfuncs.h"

static const char *rcsid="$Id: dpyfuncs.cc,v 1.7 2001/12/25 19:49:36 daniel Exp $";

void
dpyISO(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    os << name << ": " << v[1] << " (" << v[0] << ")" << endl;
}

void
dpyString(ostream &os, const char *name, const IFDEntry &e)
{
    vector<string> v = e.Get_STRINGS();
    os << name << ": " << v[0] << endl;
}

void Display_Rational(ostream &os, const tiffRATIONAL &r)
{
    if (r.num == 0) {
	os << "0";
    } else if (r.den == 1) {
	os << r.num;
    } else {
	os << r;
    }
}

void Display_Rational(ostream &os, const tiffSRATIONAL &r)
{
    if (r.num == 0) {
	os << "0";
    } else if (r.den == 1) {
	os << r.num;
    } else {
	os << r;
    }
}

void
dpyLens(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffRATIONAL> v = e.Get_RATIONALS();
    v[0] . Normalize();
    v[1] . Normalize();
    v[2] . Normalize();
    v[3] . Normalize();
    os << name << ": ";
    if (v[0] == v[1]) {
	Display_Rational(os,v[0]);
	os << "mm ";
    } else {
	Display_Rational(os,v[0]);
	os << "-";
	Display_Rational(os,v[1]);
	os << "mm ";
    }

    if (v[2] == v[3]) {
	os << "f" << (double)v[2];
    } else {
	os << "f" << (double)v[2];
	os << "-";
	os << "f" << (double)v[3];
    }

    os << endl;
}

void
dpyZoom(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffRATIONAL> v = e.Get_RATIONALS();
    v[0] . Normalize();
    os << name << ": ";
//    Display_Rational(os,v[0]);
    double d = v[0];
    d *= 100.0;
    d = floor(d);
    d /= 100.0;
    
    os << d << "mm" << endl;
}

void
dpyExpAdjust(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffSRATIONAL> v = e.Get_SRATIONALS();
    v[0] . Normalize();
    os << name << ": ";
    double d = v[0];
    d *= 1000.0;
    d = floor(d);
    d /= 1000.0;
    if (d > 0.0) {os << "+";}
    os << d << " EV" << endl;
}

void
dpyShutter(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffRATIONAL> v = e.Get_RATIONALS();
    v[0] . Normalize();
    os << name << ": ";
    Display_Rational(os,v[0]);
    os << " Sec." << endl;
}

void
dpyAperture(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffRATIONAL> v = e.Get_RATIONALS();
    os << name << ": f" << (double)v[0] << endl;
}

void
dpyPixels(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    os << name << ": " << v[0] << " pixels" << endl;
}

void
dpySigned(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffSIGNED> v = e.Get_SVALUES();
    if (v[0] > 0) {
	os << name << ": +" << v[0] << endl;
    } else {
	os << name << ": " << v[0] << endl;
    }
}

void
dpyUnsigned(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    os << name << ": " << v[0] << endl;
}

void
dpyResolution(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffRATIONAL> v = e.Get_RATIONALS();
    v[0] . Normalize();
    os << name << ": ";
    Display_Rational(os,v[0]);
    os << " Pixels/" << resolution_unit << endl;
}

void
dpyNULL(ostream &os, const char *name, const IFDEntry &e)
{}

void
dpyYes(ostream &os, const char *name, const IFDEntry &e)
{
    os << name << ": Yes" << endl;
}

void
dpyResolutionType(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    resolution_unit = "???";
    switch (v[0]) {
    case 1:
	resolution_unit = "???";
	break;
    case 2:
	resolution_unit = "Inch";
	break;
    case 3:
	resolution_unit = "Centimeter";
	break;
    }
}

void
dpyBitsPerSample(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    os << name << ": (";
    vector<tiffUNSIGNED>::iterator iter;
    bool first=true;
    for (iter = v.begin(); iter != v.end(); ++iter) {
	if (first) {
	    first=false;
	} else {
	    os << ",";
	}
	os << (*iter);
    }
    os << ")" << endl;
}

void
dpyPhotometric(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    os << name << ": ";
    switch (v[0]) {
    case 0: os << "White Is Zero" << endl; return;
    case 1: os << "Black Is Zero" << endl; return;
    case 2: os << "RGB" << endl; return;
    case 3: os << "RGB Palette" << endl; return;
    case 4: os << "Transparency Mask" << endl; return;
    case 5: os << "CMYK" << endl; return;
    case 6: os << "YCbCr" << endl; return;
    case 8: os << "CIELab" << endl; return;
    }
    os << "Unknown (" << v[0] << ")" << endl;
}

void
dpyCompression(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    os << name << ": ";
    switch (v[0]) {
    case 1: os << "Uncompressed" << endl; return;
    case 2: os << "CCITT 1D" << endl; return;
    case 3: os << "Group 3 FAX" << endl; return;
    case 4: os << "Group 4 Fax" << endl; return;
    case 5: os << "LZW" << endl; return;
    case 6: os << "JPEG" << endl; return;
    case 32773: os << "PackBits" << endl; return;
    }
    os << "Unknown (" << v[0] << ")" << endl;
}

void
dpyYCbCrPositioning(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    os << name << ": ";
    switch (v[0]) {
    case 1: os << "Center of Array" << endl; return;
    case 2: os << "Datum Point" << endl; return;
    }
    os << "Unknown (" << v[0] << ")" << endl;
}

void
dpyExposureProgram(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    os << name << ": ";
    switch (v[0]) {
    case 1: os << "Manual Control" << endl; return;
    case 2: os << "Program Normal" << endl; return;
    case 3: os << "Aperture Priority" << endl; return;
    case 4: os << "Shutter Priority" << endl; return;
    case 5: os << "Program Creative" << endl; return;
    case 6: os << "Program Action" << endl; return;
    case 7: os << "Portrait Mode" << endl; return;
    case 8: os << "Landscape Mode" << endl; return;
    }
    os << "Unknown (" << v[0] << ")" << endl;
}

void
dpyMeteringMode(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    os << name << ": ";
    switch (v[0]) {
    case 1: os << "Average" << endl; return;
    case 2: os << "Center Weighted Average" << endl; return;
    case 3: os << "Spot" << endl; return;
    case 4: os << "Multi-Spot" << endl; return;
    case 5: os << "Multi-Segment" << endl; return;
    }
    os << "Unknown (" << v[0] << ")" << endl;
}

void
dpyLightSource(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    os << name << ": ";
    switch (v[0]) {
    case 0: os << "Automatic" << endl; return;
    case 1: os << "Daylight" << endl; return;
    case 2: os << "Fluorescent" << endl; return;
    case 3: os << "Tungsten" << endl; return;
    case 10: os << "Flash" << endl; return;
    case 20: os << "Overcast" << endl; return; 
	/* Added 20 - Overcast for Olympus D-510Z
	   Thanks to Geoff Kuenning (geoff@cs.hmc.edu) for patch */
    }
    os << "Unknown (" << v[0] << ")" << endl;
}

void
dpyYesNo(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    os << name << ": ";
    switch (v[0]) {
    case 0: os << "No" << endl; return;
    case 1: os << "Yes" << endl; return;
    }
    os << "Unknown (" << v[0] << ")" << endl;
}

void
dpySensingMethod(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    os << name << ": ";
    switch (v[0]) {
    case 2: os << "Single Chip Color Area Sensor" << endl; return;
    }
    os << "Unknown (" << v[0] << ")" << endl;
}

void
dpyExifVersion(ostream &os, const char *name, const IFDEntry &e)
{
    vector<unsigned char> v = e.Get_OPAQUE();
    os << name << ": ";
    vector<unsigned char>::iterator iter;
    for (iter=v.begin(); iter!=v.end(); ++iter) {
	unsigned char c = (*iter);
	if ((c < 32) || (c>126)) {c = ' ';}
	os << c;
    }
    os << endl;
}

void
dpyExifAperture(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffRATIONAL> v = e.Get_RATIONALS();
    os << name << ": ";
    double a = v[0];
    double f = pow(M_SQRT2, a);
    f = f * 10.0;
    f = floor(f);
    f = f / 10.0;
    os << "f" << f << endl;
}

void
dpyExifShutter(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffSRATIONAL> v = e.Get_SRATIONALS();
    os << name << ": ";
    double a = v[0];
    if (a > 0.0) {
	double f = pow(2.0, a);
	os << "1/" << f << " Sec." << endl;
    } else {
	double f = pow(2.0, -a);
	os <<  f << " Sec." << endl;
    }
}

void
dpyRationalAsDouble(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffRATIONAL> v = e.Get_RATIONALS();
    os << name << ": ";
    double a = v[0];
    os << a << endl;
}

void
dpyOlymSpecialMode(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    os << name << ": ";
    switch (v[0]) {
    case 0: os << "Normal"; break;
    case 1: os << "Unknown"; break;
    case 2: os << "Fast"; break;
    case 3: os << "Panorama"; break;
    default:
	os << "Unknown (" << v[0] << ")"; break;
    }
    os << "; Seq " << v[1];
    if (v[0] == 3) {
	switch(v[2]) {
	case 1: os << " Left -> Right"; break;
	case 2: os << " Right -> Left"; break;
	case 3: os << " Bottom -> Top"; break;
	case 4: os << " Top -> Bottom"; break;
	default:
	    os << " Unknown (" << v[2] << ")"; break;
	}
    }
    os << endl;
}

void
dpyOlymJPEGQuality(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    os << name << ": ";
    switch (v[0]) {
    case 1: os << "SQ" << endl; return;
    case 2: os << "HQ" << endl; return;
    case 3: os << "SHQ" << endl; return;
    }
    os << "Unknown (" << v[0] << ")"; return;
}

void
dpyOlymZoom(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffRATIONAL> v = e.Get_RATIONALS();
    os << name << ": ";
    double a = v[0];
    if (a == 0.0) {
	os << "Normal" << endl;
	return;
    }
    a = a * 100.0;
    a = floor(a);
    a = a / 100.0;
    os << "x" << a << endl;
}


static void fmt(ostream &os, const char *fmtblock, const char *prefix, const char *value)
{
    int plen = strlen(prefix), flen = strlen(fmtblock);

    os << fmtblock + flen - (flen - plen) << prefix << ": " << value << endl;
}


void
dpyCanonBlock1(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();

    try {
	int n = v[0] / 2;

	if (v[1])
	    fmt(os, name, "Macro Mode", v[1] == 1 ? "Macro" : "No Macro");

	if (v[2]) {
	    char tl[32];
	    snprintf(tl, sizeof tl, "%.1g", (double)v[2] / 10.0);
	    fmt(os, name, "Self Timer", tl);
	}

	if (v[4]) {
	    static char *flashmodes[] = {
		"Auto", "On", "Red-Eye Reduction", "Slow Synchro",
		"Auto + Red-Eye Reduction", "On + Red-Eye Reduction"
	    };
	    if (v[4] <= 6)
		fmt(os, name, "Flash Mode", flashmodes[v[4]]);
	    else if (v[4] == 16)
		fmt(os, name, "Flash Mode", "External");
	} else
	    fmt(os, name, "Flash Mode", "No Flash");

	fmt(os, name, "Drive Mode", v[5] ? "Continuous" : (v[2] ? "Timer" : "Normal"));

	if (n >= 32 && v[32]) {
	    fmt(os, name, "Focus Mode", "Continuous");
	} else if (v[7] <= 6) {
	    static char *focusModes[] = {
		"One-Shot", "AI Servo", "AI Focus", "Manual Focus", 
		"Single Focus", "Continuous Focus", "Manual Focus"
	    };
	    fmt(os, name, "Focus Mode", focusModes[v[7]]);
	}

	if (v[10] <= 2)	{
	    static char *imageSizes[] = {
		"Large", "Medium", "Small"
	    };
	    fmt(os, name, "Image Size", imageSizes[v[10]]);
	}

	{
	    static char *adjustment[] = { "Low", "Normal", "High", "unknown" };

	    fmt(os, name, "Contrast", adjustment[((short)v[13] + 1) & 0x3]);
	    fmt(os, name, "Saturation", adjustment[((short)v[14] + 1) & 0x3]);
	    fmt(os, name, "Sharpness", adjustment[((short)v[15] + 1) & 0x3]);
	}

	if (v[16] >= 15 && v[16] <= 19) {
	    static char *speeds[] = { "auto", "50", "100", "200", "400" };

	    fmt(os, name, "ISO Speed", speeds[v[16] - 15]);
	}

	if (v[17] >= 3 && v[17] <= 5) {
	    static char *meterModes[] = {
		"Evaluative", "Partial", "Center-weighted"
	    };
	    fmt(os, name, "Metering Mode", meterModes[v[17] - 3]);
	}

	if (v[18] >= 0x3000 && v[18] <= 0x3004) {
	    static char *afPoints[] = {
		"Manual Focus", "Auto-Select", "Right", "Center", "Left"
	    };
	    fmt(os, name, "Autofocus Point", afPoints[v[18] - 0x3000]);
	}

	if (v[20] <= 5) {
	    static char *exposureModes[] = {
		"Preprogrammed", "Program", "Tv-priority", "Av-priority",
		"Manual", "A-DEP"
	    };
	    fmt(os, name, "Exposure Mode", exposureModes[v[20]]);
	}
	if (v[20] == 0 && v[11] <= 11) {
	    static char *shootModes[] = {
		"Full Auto", "Manual", "Landscape", "Fast Shutter", "Slow Shutter",
		"Night", "B&W", "Sepia", "Portrait", "Sports", "Macro / Close-Up",
		"Pan Focus"
	    };
	    fmt(os, name, "Preprogrammed Mode", shootModes[v[11]]);
	}

	// printf("%d, %d, %d\n", v[23], v[24], v[25]); // DEBUG
	if (v[25] && v[23] && v[24]) {
	    double maxFocalLen = v[23];
	    double minFocalLen = v[24];
	    double focalFactor = (double)v[25];
	    char zoom[32];
	    if (maxFocalLen == minFocalLen) {
		snprintf(zoom, sizeof zoom, "%.1gmm", maxFocalLen / focalFactor);
	    } else {
		snprintf(zoom, sizeof zoom, "%g-%gmm", minFocalLen / focalFactor,
			 maxFocalLen / focalFactor);
	    }
	    fmt(os, name, "Focal Length", zoom);
	}

	if (v[29]) {
#define V29BIT(N) (v[29] & (1 << (N)))
	    string flashDetails;

	    if (V29BIT(14))
		flashDetails += ", External E-TTL";
	    if (V29BIT(13))
		flashDetails += ", Internal";
	    if (V29BIT(11))
		flashDetails += ", FP Sync";
	    if (V29BIT(4))
		flashDetails += ", FP Sync Enabled";

	    if (flashDetails != "")
		fmt(os, name, "Flash Details", flashDetails.data() + 2);
	}

    } catch (...) {
	return;
    }
}

extern void dpyCanonBlock4(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();

    try {	
//	int n = v[0] / 2;

	if (v[7] <= 6) {
	    static char *whiteBalances[] = {
		"Auto", "Sunny", "Cloudy", "Tungsten", "Flourescent", "Flash", "Custom"
	    };
	    fmt(os, name, "White Balance", whiteBalances[v[7]]);
	}

	if (v[9] > 1) {
	    char n[32];

	    snprintf(n, sizeof n, "%u", (unsigned int)v[9]);
	    fmt(os, name, "Burst Sequence Number", n);
	}

	{
	    struct { unsigned short value; char *bias; } flashBias[] = {
		{0xffc0, "-2 EV"}, {0xffcc, "-1.67 EV"}, {0xffd0, "-1.50 EV"},
		{0xffd4, "-1.33 EV"}, {0xffe0, "-1 EV"}, {0xffec, "-0.67 EV"},
		{0xfff0, "-0.50 EV"}, {0xfff4, "-0.33 EV"}, {0x0000, "0 EV"},
		{0x000c, "0.33 EV"}, {0x0010, "0.50 EV"}, {0x0014, "0.67 EV"},
		{0x0020, "1 EV"}, {0x002c,  "1.33 EV"}, {0x0030, "1.50 EV"},
		{0x0034, "1.67 EV"}, {0x0040, "2 EV"}, {0, NULL}
	    };
	    for (int fb = 0; flashBias[fb].bias != NULL; fb++) {
		if (v[15] == flashBias[fb].value) {
		    fmt(os, name, "Flash Exposure Bias", flashBias[fb].bias);
		    break;
		}
	    }
	}
	
    } catch (...) {
	return;
    }
}


extern void dpyCanonImageNumber(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();

    try {
	unsigned long n = v[0];

	char img[32];
	snprintf(img, sizeof img, "%03lu-%04lu", n / 10000, n % 10000);

	os << name << ": " << img << endl;

    } catch (...) {
	return;
    }
}




extern void dpyCanonSerialNumber(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();

    try {
	unsigned long n = v[0];

	char sn[32];
	snprintf(sn, sizeof sn, "%04lX%05lu", n >> 16, n & 0xffff);
	
	fmt(os, name, "Camera Serial Number", sn);
    } catch (...) {
	return;
    }
}

extern void dpyCasioRecMode(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    os << name << ": ";
    switch (v[0]) {
    case 1: os << "Single Shutter" << endl; return;
    case 2: os << "Panorama" << endl; return;
    case 3: os << "Night Scene" << endl; return;
    case 4: os << "Portrait" << endl; return;
    case 5: os << "Landscape" << endl; return;
    }
    os << "Unknown (" << v[0] << ")"; return;
}

extern void dpyCasioQuality(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    os << name << ": ";
    switch (v[0]) {
    case 1: os << "Economy" << endl; return;
    case 2: os << "Normal" << endl; return;
    case 3: os << "Fine" << endl; return;
    }
    os << "Unknown (" << v[0] << ")"; return;
}

extern void dpyCasioFocusMode(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    os << name << ": ";
    switch (v[0]) {
    case 2: os << "Macro" << endl; return;
    case 3: os << "Auto Focus" << endl; return;
    case 4: os << "Manual Focus" << endl; return;
    case 5: os << "Infinity" << endl; return;
    }
    os << "Unknown (" << v[0] << ")"; return;
}

extern void dpyCasioFlashMode(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    os << name << ": ";
    switch (v[0]) {
    case 1: os << "Auto" << endl; return;
    case 2: os << "On" << endl; return;
    case 3: os << "Off" << endl; return;
    case 4: os << "Red Eye Reduction" << endl; return;
    }
    os << "Unknown (" << v[0] << ")"; return;
}

extern void dpyCasioFlashInten(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    os << name << ": ";
    switch (v[0]) {
    case 11: os << "Weak" << endl; return;
    case 13: os << "Normal" << endl; return;
    case 15: os << "Strong" << endl; return;
    }
    os << "Unknown (" << v[0] << ")"; return;
}

extern void dpyCasioDistance(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    
    const char* units = "mm";
    double dist = double(v[0]);

    if (dist > 1000.0)
    {
	    dist /= 1000.0;
	    units = "m";
    }
    else if (dist > 10)
    {
	    dist /= 10;
	    units = "cm";
    }

    os << name << ": " << dist << units << endl;
}

extern void dpyCasioWhiteBalance(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    os << name << ": ";
    switch (v[0]) {
    case 1: os << "Auto" << endl; return;
    case 2: os << "Tungsten" << endl; return;
    case 3: os << "Daylight" << endl; return;
    case 4: os << "Flurescent" << endl; return;
    case 5: os << "Shade" << endl; return;
    case 129: os << "Manual" << endl; return;
    }
    os << "Unknown (" << v[0] << ")"; return;
}

extern void dpyCasioDigitalZoom(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    os << name << ": ";
    switch (v[0]) {
    case 0x10000: os << "Off" << endl; return;
    case 0x10001: os << "2X" << endl; return;
    }
    os << "Unknown (" << v[0] << ")"; return;
}

extern void dpyCasioSharpness(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    os << name << ": ";
    switch (v[0]) {
    case 0: os << "Normal" << endl; return;
    case 1: os << "Soft" << endl; return;
    case 2: os << "Hard" << endl; return;
    }
    os << "Unknown (" << v[0] << ")"; return;
}

extern void dpyCasioContrast(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    os << name << ": ";
    switch (v[0]) {
    case 0: os << "Normal" << endl; return;
    case 1: os << "Low" << endl; return;
    case 2: os << "High" << endl; return;
    }
    os << "Unknown (" << v[0] << ")"; return;
}

extern void dpyCasioSaturation(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    os << name << ": ";
    switch (v[0]) {
    case 0: os << "Normal" << endl; return;
    case 1: os << "Low" << endl; return;
    case 2: os << "High" << endl; return;
    }
    os << "Unknown (" << v[0] << ")"; return;
}

extern void dpyCasioSensitivity(ostream &os, const char *name, const IFDEntry &e)
{
    vector<tiffUNSIGNED> v = e.Get_UVALUES();
    os << name << ": ";
    switch (v[0]) {
	    /* QV3000 Only: */
    case 64: os << "Normal" << endl; return;
    case 125: os << "+1.0" << endl; return;
    case 250: os << "+2.0" << endl; return;
    case 244: os << "+3.0" << endl; return;
	    /* QV2000/QV8000 Only: */
    case 80: os << "Normal" << endl; return;
    case 100: os << "High" << endl; return;
    }
    os << "Unknown (" << v[0] << ")"; return;
}

