/* memsize.c - MemTest-86  Version 2.9
 *
 * Released under version 2 of the Gnu Public License.
 * By Chris Brady, cbrady@sgi.com
 */

#include <string.h>
#include "test.h"
#include "defs.h"
#include "config.h"

extern ulong p1, p2;
extern volatile ulong *p;
extern struct vars *v;

ulong m_lim = 0;

/*
 * Find out how much memory there is.
 */
void mem_size()
{
	v->reserved = 0;
	v->test_mem = 0;

	switch (v->memsz_mode) {
	case SZ_MODE_BIOS:
		/* Get the memory size from the BIOS */
		memsize_820(0);
		break;
	case SZ_MODE_BIOS_RES:
		/* Get the memory size from the BIOS, include reserved mem */
		memsize_820(1);
		break;
	case SZ_MODE_PROBE:
		/* Probe to find memory */
		memsize_probe();
		cprint(LINE_INFO, COL_MMAP, " Probed ");
		break;
	}
	v->lim_lower = 0;
	v->lim_upper = (ulong)v->rmap[v->msegs-1].end;
	v->rmap[0].start = (ulong *)(START_ADR);

	adj_mem();
	dprint(LINE_INFO, 19, v->reserved/1024, 7, 0);
}

void memsize_820(int res)
{
	int i, n, nr;
	ulong end;
	struct e820entry nm[E820MAX];

	/* Clean up, adjust and copy the BIOS-supplied E820-map. */
	/* If the res arg is true reclassify reserved memory as E820_RAM */
	nr = sanitize_e820_map(v->e820, nm, v->e820_nr, res);

	/* If there is not a good 820 map use the BIOS 801/88 info */
	if (nr < 1 || nr > E820MAX) {
		memsize_801();
		return;
	}

	/* Build the memory map for testing */
	n = 0;
	for (i=0; i<nr; i++) {

		/* Don't try to use any reserved memory that starts at the */
		/* top of the address space. */
		if (nm[i].type == E820_RAM && nm[i].addr < 0xff000000) {

			/* Don't ever use memory between 640 and 1024k */
			if (nm[i].addr > RES_START && nm[n].addr < RES_END) {
				if (nm[i].addr + nm[n].size < RES_END) {
					continue;
				}
				nm[n].size -= (RES_END - nm[i].addr);
				nm[n].addr = RES_END;
			}
			end = nm[i].size + nm[i].addr;
			if (end == 0) {
				end = 0xfffffffc; }
			if (end > RES_START && end < RES_END) {
				nm[n].size -= (end - RES_START);
				end = nm[i].size + nm[i].addr;
			}
			v->rmap[n].start = (ulong *)nm[i].addr;
			v->rmap[n].end = (ulong *)end;
			v->test_mem += (end - nm[i].addr);
			n++;
		} else {
			/* If this is the top of memory then it's not */
			/* real ram so don't confuse things by counting it. */
			if (nm[i].addr < 0xff000000) {
				v->reserved += nm[i].size;
			}
		}
	}
	v->msegs = n;
	if (res) {
		cprint(LINE_INFO, COL_MMAP, "e820-All");
	} else {
		cprint(LINE_INFO, COL_MMAP, "e820-Std");
	}
}
	
void memsize_801()
{
	ulong mem_size;

	/* compare results from 88 and 801 methods and take the greater */
	/* These sizes are for extended memory in 1k units. */

	if (v->alt_mem_k < v->ext_mem_k) {
		mem_size = v->ext_mem_k;
		cprint(LINE_INFO, COL_MMAP, "e88-Std ");
	} else {
		mem_size = v->alt_mem_k;
		cprint(LINE_INFO, COL_MMAP, "e801-Std");
	}
	/* First we map in the first 640k */
	v->rmap[0].start = 0;
	v->rmap[0].end = (ulong *)RES_START;
	v->test_mem = RES_START;

	/* Now the extended memory */
	v->rmap[1].start = (ulong *)RES_END;
	v->rmap[1].end = (ulong *)((mem_size+1024)*1024);
	v->test_mem += mem_size * 1024;
	v->msegs = 2;
}

/*
 * Sanitize the BIOS e820 map.
 *
 * Some e820 responses include overlapping entries.  The following 
 * replaces the original e820 map with a new one, removing overlaps.
 *
 */
int sanitize_e820_map(struct e820entry *orig_map, struct e820entry *new_bios,
	short old_nr, short res)
{
	struct change_member {
		struct e820entry *pbios; /* pointer to original bios entry */
		unsigned long long addr; /* address for this change point */
	};
	struct change_member change_point_list[2*E820MAX];
	struct change_member *change_point[2*E820MAX];
	struct e820entry *overlap_list[E820MAX];
	struct e820entry biosmap[E820MAX];
	struct change_member *change_tmp;
	ulong current_type, last_type;
	unsigned long long last_addr;
	int chgidx, still_changing;
	int overlap_entries;
	int new_bios_entry;
	int i;

	/*
		Visually we're performing the following (1,2,3,4 = memory types)...
		Sample memory map (w/overlaps):
		   ____22__________________
		   ______________________4_
		   ____1111________________
		   _44_____________________
		   11111111________________
		   ____________________33__
		   ___________44___________
		   __________33333_________
		   ______________22________
		   ___________________2222_
		   _________111111111______
		   _____________________11_
		   _________________4______

		Sanitized equivalent (no overlap):
		   1_______________________
		   _44_____________________
		   ___1____________________
		   ____22__________________
		   ______11________________
		   _________1______________
		   __________3_____________
		   ___________44___________
		   _____________33_________
		   _______________2________
		   ________________1_______
		   _________________4______
		   ___________________2____
		   ____________________33__
		   ______________________4_
	*/
	/* First make a copy of the map */
	for (i=0; i<old_nr; i++) {
		biosmap[i].addr = orig_map[i].addr;
		biosmap[i].size = orig_map[i].size;
		biosmap[i].type = orig_map[i].type;
	}

	/* bail out if we find any unreasonable addresses in bios map */
	for (i=0; i<old_nr; i++) {
		if (biosmap[i].addr + biosmap[i].size < biosmap[i].addr)
			return 0;
		/* If we want to test the reserved memory include everything */
		if (res) {
			biosmap[i].type = E820_RAM;
		} else {
			/* It should always be safe to test ACPI ram */
			if ( biosmap[i].type == E820_ACPI) {
				biosmap[i].type = E820_RAM;
			}
		}
	}

	/* create pointers for initial change-point information (for sorting) */
	for (i=0; i < 2*old_nr; i++)
		change_point[i] = &change_point_list[i];

	/* record all known change-points (starting and ending addresses) */
	chgidx = 0;
	for (i=0; i < old_nr; i++)	{
		change_point[chgidx]->addr = biosmap[i].addr;
		change_point[chgidx++]->pbios = &biosmap[i];
		change_point[chgidx]->addr = biosmap[i].addr + biosmap[i].size;
		change_point[chgidx++]->pbios = &biosmap[i];
	}

	/* sort change-point list by memory addresses (low -> high) */
	still_changing = 1;
	while (still_changing)	{
		still_changing = 0;
		for (i=1; i < 2*old_nr; i++)  {
			/* if <current_addr> > <last_addr>, swap */
			/* or, if current=<start_addr> & last=<end_addr>, swap */
			if ((change_point[i]->addr < change_point[i-1]->addr) ||
				((change_point[i]->addr == change_point[i-1]->addr) &&
				 (change_point[i]->addr == change_point[i]->pbios->addr) &&
				 (change_point[i-1]->addr != change_point[i-1]->pbios->addr))
			   )
			{
				change_tmp = change_point[i];
				change_point[i] = change_point[i-1];
				change_point[i-1] = change_tmp;
				still_changing=1;
			}
		}
	}

	/* create a new bios memory map, removing overlaps */
	overlap_entries=0;	 /* number of entries in the overlap table */
	new_bios_entry=0;	 /* index for creating new bios map entries */
	last_type = 0;		 /* start with undefined memory type */
	last_addr = 0;		 /* start with 0 as last starting address */
	/* loop through change-points, determining affect on the new bios map */
	for (chgidx=0; chgidx < 2*old_nr; chgidx++)
	{
		/* keep track of all overlapping bios entries */
		if (change_point[chgidx]->addr == change_point[chgidx]->pbios->addr)
		{
			/* add map entry to overlap list (> 1 entry implies an overlap) */
			overlap_list[overlap_entries++]=change_point[chgidx]->pbios;
		}
		else
		{
			/* remove entry from list (order independent, so swap with last) */
			for (i=0; i<overlap_entries; i++)
			{
				if (overlap_list[i] == change_point[chgidx]->pbios)
					overlap_list[i] = overlap_list[overlap_entries-1];
			}
			overlap_entries--;
		}
		/* if there are overlapping entries, decide which "type" to use */
		/* (larger value takes precedence -- 1=usable, 2,3,4,4+=unusable) */
		current_type = 0;
		for (i=0; i<overlap_entries; i++)
			if (overlap_list[i]->type > current_type)
				current_type = overlap_list[i]->type;
		/* continue building up new bios map based on this information */
		if (current_type != last_type)	{
			if (last_type != 0)	 {
				new_bios[new_bios_entry].size =
					change_point[chgidx]->addr - last_addr;
				/* move forward only if the new size was non-zero */
				if (new_bios[new_bios_entry].size != 0)
					if (++new_bios_entry >= E820MAX)
						break; 	/* no more space left for new bios entries */
			}
			if (current_type != 0)	{
				new_bios[new_bios_entry].addr = change_point[chgidx]->addr;
				new_bios[new_bios_entry].type = current_type;
				last_addr=change_point[chgidx]->addr;
			}
			last_type = current_type;
		}
	}
	return(new_bios_entry);
}

void memsize_probe()
{
	int i, n;

	/* Since all address bits may not be decoded, the search for memory
	 * must be limited.  The max address is found by checking for
	 * memory wrap from 1MB to 4GB.  */
	v->rmap[0].start = (ulong *)0x1234569;
	p1 = (ulong)&v->rmap[0].start;
	m_lim = 0xfffffffc; 
	for (p2 = 0x100000; p2; p2 <<= 1) {  
		p = (ulong *)(p1 + p2);
		if (*p == 0x1234569) {
			m_lim = --p2;
			break;
		}
	}

	/* Turn on cache */
	set_cache(1);

	/* Find all segments of RAM */

	p = (ulong *)(START_ADR);
	i = 0;
	v->rmap[i].start = 0;

	/* Limit search for memory to m_lim and make sure we don't 
	 * overflow the 32 bit size of p.  */
	while ((ulong)p < m_lim && (ulong)p >= START_ADR) {
		/*
		 * Skip over reserved memory
		 */
		if ((ulong)p < RES_END &&
				(ulong)p >= RES_START) {
			v->rmap[i].end = (ulong *)RES_START;
			v->rmap[i].v = 1;
			v->test_mem += (v->rmap[i].end - v->rmap[i].start);
			p = (ulong *)RES_END;
			i++;
			v->rmap[i].start = 0;
			goto fstart;
		}

		if (check_ram() == 0) {
			/* ROM or nothing at this address, record end addrs */
			v->rmap[i].end = p;
			v->rmap[i].v = 1;
			v->test_mem += (v->rmap[i].end - v->rmap[i].start);
			i++;
			v->rmap[i].start = 0;
fstart:

			/* We get here when there is a gap in memory.
			 * Loop until we find more ram, the gap is more
			 * than 32768k or we hit m_lim */
			n = 32768;
			while ((ulong)p < m_lim &&
					(ulong)p >= START_ADR) {

				/* Skip over video memory */
				if ((ulong)p < RES_END &&
					(ulong)p >= RES_START) {
					p = (ulong *)RES_END;
				}
				if (check_ram() == 1) {
					/* More RAM, record start addrs */
					v->rmap[i].start = p;
					break;
				}

				/* If the gap is 32768k or more then there
				 * is probably no more memory so bail out */
				if (--n <= 0) {
					p = (ulong *)m_lim;
					break;
				}
				p += 0x400;
			}
		}
		p += 0x400;
	}

	/* If there is ram right up to the memory limit this will record
	 * the last address.  */
	if (v->rmap[i].start) {
		v->rmap[i].end = (ulong *)m_lim;
		v->rmap[i].v = 1;
		v->test_mem += (v->rmap[i].end - v->rmap[i].start);
		i++;
	}
/*
v->rmap[i].start = 0xd0000000;
v->rmap[i].end = 0xd0001000;
v->test_mem += (v->rmap[i].end - v->rmap[i].start);
i++;
*/
	v->msegs = i;
	v->test_mem *= 4;
}

/* check_ram - Determine if this address points to memory by checking
 * for a wrap pattern and then reading and then writing the complement.
 * We then check that at least one bit changed in each byte before
 * believing that it really is memory.  */

int check_ram() {
        int s;

        p1 = *p;

        /* write the complement */
        *p = ~p1;
        p2 = *p;
        s = 0;

        /* Now make sure a bit changed in each byte */
        if ((0xff & p1) != (0xff & p2)) {
                s++;
        }
        if ((0xff00 & p1) != (0xff00 & p2)) {
                s++;
        }
        if ((0xff0000 & p1) != (0xff0000 & p2)) {
                s++;
        }
        if ((0xff000000 & p1) != (0xff000000 & p2)) {
                s++;
        }
        if (s == 4) {
                /* RAM at this address */
                return 1;
        }

        return 0;
}
