/* Mednafen - Multi-system Emulator
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "pce.h"
#include "hes.h"
#include "../player.h"

static uint8 mpr_start[8];
static uint8 IBP[0x10];
static uint8 *rom = NULL;
static uint8 CurrentSong;
static bool8 bootstrap;

uint8 ReadIBP(unsigned int A)
{
 // Lovely EVIL speed hack.
 if(!PCE_InDebug)
  if((A&0xF) == 0x5)
  {
   int cycrun = HuCPU.count - HuCPU.tcount;
   if(cycrun > 0 && HuCPU.timer_div > 0)
   {
    if(HuCPU.timer_div < cycrun)
     cycrun = HuCPU.timer_div;
    HuC6280_StealCycles(cycrun);
   }
  }

 return(IBP[A & 0xF]);
}

static uint16 De16(uint8 *buf)
{
 return(buf[0] | (buf[1] << 8));
}

static uint32 De32(uint8 *buf)
{
 return(buf[0] | (buf[1]<<8) | (buf[2]<<16) | (buf[3] << 24));
}

static DECLFR(HESROMRead)
{
 if(bootstrap)
 {
  if(A == 0x1FFE)
   return(0x00);
  else if(A == 0x1FFF)
  {
   if(!PCE_InDebug)
   {
    HuCPU.A = CurrentSong;
    HuCPU.PC = 0x1C00;

    HuCPUFastMap[0] = rom;

    for(int x=0;x<8;x++)
    {
     HuCPU.MPR[x] = mpr_start[x];
    }
    HuCPU.MPR[8] = HuCPU.MPR[0];
   
    HuC6280_FlushMPRCache();

    bootstrap = 0;
   }
   return(0x1C);
  }
 }
 return(rom[A]);
}

int PCE_HESLoad(uint8 *buf, uint32 size)
{
 uint32 LoadAddr, LoadSize;
 uint16 InitAddr;
 uint8 *tmp;
 uint8 StartingSong;
 int TotalSongs;

 InitAddr = De16(&buf[0x6]);

 tmp = &buf[0x10];

 rom = (uint8 *)malloc(1024 * 1024);
 memset(rom, 0, 1024 * 1024);

 while(tmp < (buf + size - 0x10))
 {
  LoadSize = De32(&tmp[0x4]);
  LoadAddr = De32(&tmp[0x8]);
  tmp += 0x10;
  if(tmp >= (buf + size + LoadSize)) break;

  if((LoadAddr + LoadSize) > 0x100000) break;
  memcpy(rom + LoadAddr,tmp,LoadSize);
  tmp += LoadSize;
 }


 //system_init(hes);
 //system_reset(hes);

 CurrentSong = StartingSong = buf[5];
 TotalSongs = 256;

 IBP[0x00] = 0x20;
 IBP[0x01] = InitAddr;
 IBP[0x02] = InitAddr >> 8;
 IBP[0x03] = 0x4C;
 IBP[0x04] = 0x03;
 IBP[0x05] = 0x1C;

 MDFNMP_Init(TotalSongs, NULL, NULL, NULL, NULL); //UTF8 **snames);

 for(int x = 0; x < 8; x++)
  mpr_start[x] = buf[0x8 + x];

 for(int x = 0; x < 0x80; x++)
 {
  HuCPUFastMap[x] = rom;
  PCERead[x] = HESROMRead;
 }
 return(1);
}

void HES_Reset(void)
{
 HuCPUFastMap[0] = NULL;
 HuC6280_FlushMPRCache();
 bootstrap = 1;
}


void HES_Draw(uint32 *pXBuf, int16 *SoundBuf, int32 SoundBufSize)
{
 extern uint8 pce_jp_data[5];
 static uint8 last = 0;
 bool needreload = 0;

 if((pce_jp_data[0] & 0x20) && !(last & 0x20))
 {
  CurrentSong++;
  needreload = 1;
 }

 if((pce_jp_data[0] & 0x80) && !(last & 0x80))
 {
  CurrentSong--;
  needreload = 1;
 }

 if((pce_jp_data[0] & 0x8) && !(last & 0x8))
  needreload = 1;

 if((pce_jp_data[0] & 0x10) && !(last & 0x10))
 {
  CurrentSong += 10;
  needreload = 1;
 }

 if((pce_jp_data[0] & 0x40) && !(last & 0x40))
 {
  CurrentSong -= 10;
  needreload = 1;
 }

 last = pce_jp_data[0];

 if(needreload)
  PCE_Power();

 MDFNMP_Draw(pXBuf, CurrentSong, SoundBuf, SoundBufSize);
}

void HES_Close(void)
{
 if(rom)
 {
  free(rom);
  rom = NULL;
 }
}
