/*!
  @file           IFRUtil_Configuration.cpp
  @author         D039759
  @ingroup        IFR_Common
  @brief          Configuration settings.
  @see

\if EMIT_LICENCE



    ========== licence begin  GPL
    Copyright (c) 2001-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end







\endif
*/

#include "Interfaces/Runtime/Util/IFRUtil_Configuration.h"
#include "SAPDBCommon/SAPDB_Names.h"
#include "RunTime/RTE_IniFileHandling.h"
#include "SAPDBCommon/MemoryManagement/SAPDBMem_Alloca.h"

#include <stdio.h>

#ifndef WIN32
#include <unistd.h>
#define GETCWD getcwd
#else
#include <direct.h>
#define GETCWD _getcwd
#endif


#define SQLDBC_CONFIGFILENAME  SAPDB_RUNTIMES_INI_FILE
#define SQLDBC_CONFIGSECTION   "SQLDBC"

#define SQLDBC_TRACEFILENAME_KEY     "TraceFileName"
#define SQLDBC_TRACEFLAGS_KEY        "TraceFlags"
#define SQLDBC_TRACESHAREDMEM_KEY    "TraceSharedMemory"

//----------------------------------------------------------------------
static
IFR_Retcode internalGetTraceFileName(char *filename, IFR_size_t length)
{
    RTE_IniFileErrtext ignoredErrorText;
    RTE_IniFileResult  configResult;

    RTE_GetUserConfigString(SQLDBC_CONFIGFILENAME,
                            SQLDBC_CONFIGSECTION,
                            SQLDBC_TRACEFILENAME_KEY,
                            filename,
                            (SAPDB_Int4)length,
                            ignoredErrorText,
                            configResult);

    switch(configResult) {
    case SAPDB_INIFILE_RESULT_OK:
        return IFR_OK;
    case SAPDB_INIFILE_RESULT_TRUNCATED:
        return IFR_DATA_TRUNC;
    }
    return IFR_NOT_OK;
}

//----------------------------------------------------------------------
IFR_Retcode
IFRUtil_Configuration::getTraceFileName(char *filename, IFR_size_t length)
{
    if(length) *filename='\0';

    RTE_IniFileErrtext ignoredErrorText;
    RTE_IniFileResult  configResult;

    if(internalGetTraceFileName(filename, length) == IFR_OK) {
        return IFR_OK;
    }

    RTE_PutUserConfigString(SQLDBC_CONFIGFILENAME,
                            SQLDBC_CONFIGSECTION,
                            SQLDBC_TRACEFILENAME_KEY,
                            "sqldbctrace-%p.prt",
                            ignoredErrorText,
                            configResult);

    if(configResult == SAPDB_INIFILE_RESULT_OK) {
        return internalGetTraceFileName(filename, length);
    } else {
        return IFR_NOT_OK;
    }
}

//----------------------------------------------------------------------
IFR_Retcode
IFRUtil_Configuration::setTraceFileName(char *filename)
{
    if(filename==0 || strlen(filename) >= 1024) {
        return IFR_NOT_OK;
    }

    RTE_IniFileErrtext ignoredErrorText;
    RTE_IniFileResult  configResult;

    RTE_PutUserConfigString(SQLDBC_CONFIGFILENAME,
                            SQLDBC_CONFIGSECTION,
                            SQLDBC_TRACEFILENAME_KEY,
                            filename,
                            ignoredErrorText,
                            configResult);

    if(configResult == SAPDB_INIFILE_RESULT_OK) {
        return IFR_OK;
    } else {
        return IFR_NOT_OK;
    }
}

//----------------------------------------------------------------------
static IFR_Bool
fileNameIsAbsolute(char *filename)
{
#ifdef WIN32
    if(*filename && isalpha(*filename) && filename[1]==':') {
        return true;
    }
    if(filename[0]=='\\' && filename[1]=='\\') {
        return true;
    }
    return false;
#else
    if(filename[0]=='/') {
        return true;
    }
    return false;
#endif
}

//----------------------------------------------------------------------
static IFR_Bool
fileNameIsRelative(char *filename)
{
    if(*filename && *filename=='.') {
        return filename[1]=='/' || filename[1]=='\\';
    } else {
        return false;
    }
}

//----------------------------------------------------------------------
IFR_Retcode
IFRUtil_Configuration::getFullFileName(char *tmpfilename, 
                                       char *filename, 
                                       IFR_size_t length)
{
    if(fileNameIsAbsolute(tmpfilename)) {
        strcpy(filename, tmpfilename);
        return IFR_OK;
    } else if(fileNameIsRelative(tmpfilename)) {
        char fileLocation[512];
        GETCWD(fileLocation, 512);
        if(strlen(fileLocation) + strlen(tmpfilename) + 1 > length) {
            return IFR_NOT_OK;
        }
        strcpy(filename, fileLocation);
        char endc = filename[strlen(filename) - 1];
#ifdef WIN32
        if(endc == '/' || endc == '\\') {
#else
        if(endc == '/') {
#endif
            strcat(filename, tmpfilename + 2);
        } else {
#ifdef WIN32
            strcat(filename, "\\");
#else
            strcat(filename, "/");
#endif
            strcat(filename, tmpfilename + 2);
        }
        return IFR_OK;
    } else {
        RTE_Path           fileLocation;
        RTE_IniFileErrtext errtext;

        if(RTE_GetUserSpecificConfigPath(fileLocation, true, errtext)) {
            if(strlen(fileLocation) + strlen(tmpfilename) + 1 > length) {
                return IFR_NOT_OK;
            }
            strcpy(filename, fileLocation);
            strcat(filename, tmpfilename);
            return IFR_OK;
        } else {
            return IFR_NOT_OK;
        }
    }
    
}
//----------------------------------------------------------------------
IFR_Retcode
IFRUtil_Configuration::getFullTraceFileName(char *filename, IFR_size_t length)
{
    char *tmpfilename = (char *)alloca(length);
    IFR_Retcode rc=getTraceFileName(tmpfilename, length);
    if(rc != IFR_OK) {
        return IFR_NOT_OK;
    }
    if(fileNameIsAbsolute(tmpfilename)) {
        strcpy(filename, tmpfilename);
        return IFR_OK;
    } else if(fileNameIsRelative(tmpfilename)) {
        char fileLocation[512];
        GETCWD(fileLocation, 512);
        if(strlen(fileLocation) + strlen(tmpfilename) + 1 > length) {
            return IFR_NOT_OK;
        }
        strcpy(filename, fileLocation);
        char endc = filename[strlen(filename) - 1];
#ifdef WIN32
        if(endc == '/' || endc == '\\') {
#else
        if(endc == '/') {
#endif
            strcat(filename, tmpfilename + 2);
        } else {
#ifdef WIN32
            strcat(filename, "\\");
#else
            strcat(filename, "/");
#endif
            strcat(filename, tmpfilename + 2);
        }
        return IFR_OK;
    } else {
        RTE_Path           fileLocation;
        RTE_IniFileErrtext errtext;

        if(RTE_GetUserSpecificConfigPath(fileLocation, true, errtext)) {
            if(strlen(fileLocation) + strlen(tmpfilename) + 1 > length) {
                return IFR_NOT_OK;
            }
            strcpy(filename, fileLocation);
            strcat(filename, tmpfilename);
            return IFR_OK;
        } else {
            return IFR_NOT_OK;
        }
    }
}

//----------------------------------------------------------------------
static IFR_Retcode internalGetSharedMemoryName(char *filename,
                                               IFR_size_t length)
{
    RTE_IniFileErrtext ignoredErrorText;
    RTE_IniFileResult  configResult;

    RTE_GetUserConfigString(SQLDBC_CONFIGFILENAME,
                            SQLDBC_CONFIGSECTION,
                            SQLDBC_TRACESHAREDMEM_KEY,
                            filename,
                            (SAPDB_Int4)length,
                            ignoredErrorText,
                            configResult);

    switch(configResult) {
    case SAPDB_INIFILE_RESULT_OK:
        return IFR_OK;
    case SAPDB_INIFILE_RESULT_TRUNCATED:
        return IFR_DATA_TRUNC;
    }
    return IFR_NOT_OK;
}



//----------------------------------------------------------------------
IFR_Retcode
IFRUtil_Configuration::getSharedMemoryName(char *filename, IFR_size_t length)
{
    if(length) *filename='\0';

    RTE_IniFileErrtext ignoredErrorText;
    RTE_IniFileResult  configResult;

    if(internalGetSharedMemoryName(filename, length) == IFR_OK) {
        return IFR_OK;
    }

    RTE_PutUserConfigString(SQLDBC_CONFIGFILENAME,
                            SQLDBC_CONFIGSECTION,
                            SQLDBC_TRACESHAREDMEM_KEY,
                            "sqldbctrace.shm",
                            ignoredErrorText,
                            configResult);

    if(configResult == SAPDB_INIFILE_RESULT_OK) {
        return internalGetSharedMemoryName(filename, length);
    } else {
        return IFR_NOT_OK;
    }
}

//----------------------------------------------------------------------
IFR_Retcode
IFRUtil_Configuration::getFullSharedMemoryName(char *sharedmemoryname, IFR_size_t length)
{
    char *tmpfilename = (char *)alloca(length);
    IFR_Retcode rc=getSharedMemoryName(tmpfilename, length);
    if(rc != IFR_OK) {
        return IFR_NOT_OK;
    }
    if(fileNameIsAbsolute(tmpfilename)) {
        strcpy(sharedmemoryname, tmpfilename);
        return IFR_OK;
    } else {
        RTE_Path           fileLocation;
        RTE_IniFileErrtext errtext;

        if(RTE_GetUserSpecificConfigPath(fileLocation, true, errtext)) {
            if(strlen(fileLocation) + strlen(tmpfilename) + 1 > length) {
                return IFR_NOT_OK;
            }
            strcpy(sharedmemoryname, fileLocation);
            strcat(sharedmemoryname, tmpfilename);
            return IFR_OK;
        } else {
            return IFR_NOT_OK;
        }
    }
}


//----------------------------------------------------------------------
static IFR_Retcode internalGetTraceFlags(char *flags,
                                         IFR_size_t length)
{
    RTE_IniFileErrtext ignoredErrorText;
    RTE_IniFileResult  configResult;

    RTE_GetUserConfigString(SQLDBC_CONFIGFILENAME,
                            SQLDBC_CONFIGSECTION,
                            SQLDBC_TRACEFLAGS_KEY,
                            flags,
                            (SAPDB_Int4)length,
                            ignoredErrorText,
                            configResult);

    switch(configResult) {
    case SAPDB_INIFILE_RESULT_OK:
        return IFR_OK;
    case SAPDB_INIFILE_RESULT_TRUNCATED:
        return IFR_DATA_TRUNC;
    }
    return IFR_NOT_OK;
}


//----------------------------------------------------------------------
IFR_Retcode
IFRUtil_Configuration::getTraceFlags(char *flags, IFR_size_t length)
{
    if(length) *flags='\0';

    RTE_IniFileErrtext ignoredErrorText;
    RTE_IniFileResult  configResult;

    if(internalGetTraceFlags(flags, length) == IFR_OK) {
        return IFR_OK;
    }

    RTE_PutUserConfigString(SQLDBC_CONFIGFILENAME,
                            SQLDBC_CONFIGSECTION,
                            SQLDBC_TRACEFLAGS_KEY,
                            "",
                            ignoredErrorText,
                            configResult);

    if(configResult == SAPDB_INIFILE_RESULT_OK) {
        return internalGetTraceFlags(flags, length);
    } else {
        return IFR_NOT_OK;
    }
}

//----------------------------------------------------------------------
IFR_Retcode
IFRUtil_Configuration::setTraceFlags(char *flags)
{
    RTE_IniFileErrtext ignoredErrorText;
    RTE_IniFileResult  configResult;

    RTE_PutUserConfigString(SQLDBC_CONFIGFILENAME,
                            SQLDBC_CONFIGSECTION,
                            SQLDBC_TRACEFLAGS_KEY,
                            flags,
                            ignoredErrorText,
                            configResult);

    if(configResult == SAPDB_INIFILE_RESULT_OK) {
        return IFR_OK;
    } else {
        return IFR_NOT_OK;
    }
}

