/*!
    @file     SharedSQL_IPlan.hpp
    @ingroup  SharedSQL
    @author   DirkT
    @brief    AVLTree to store the execution plan for SQLCommands
    @see            

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
\endif
*/

#ifndef SHAREDSQL_IPLAN_HPP
#define SHAREDSQL_IPLAN_HPP

/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "SQLManager/SharedSQL/SharedSQL_Types.hpp"

#include "SAPDBCommon/SAPDB_Types.hpp"

/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

/*---------------------------------------------------------------------------*/
/*!
    @interface      SharedSQL_IPlan
    @brief          Interface to SharedSQL_Plan
 */
class SharedSQL_IPlan
{
public:
    /*!
        @brief  Drop the execution plan
        @param  DelMasterParseID [in] - delete also the master parseid
    */
    virtual void Drop( bool DelMasterParseID ) = 0;

    /*!
        @brief  Returns wheather or not there is a plan
        @return SAPDB_Bool
    */
    virtual SAPDB_Bool IsEmpty( void ) const = 0;

    /*!
        @brief  Returns the size of the execution plan
        @return SAPDB_Int4
    */
    virtual SAPDB_Int4 GetSize( void ) const = 0;

    /*!
        @brief  Returns some plan element identified by the key
        @param  Key [in]
        @return const void*
    */
    virtual const void* Find( void* Key ) const = 0;

    /*!
        @brief  Insert the execution plan
        @param  PID  [in] - ParseID used during preparation
        @param  P    [in] - Pointer to the execution plan
        @param  Size [in] - Size of the execution plan
        @return ...
    */
    virtual SAPDB_Bool Insert( SharedSQL_ParseID PID, void *const P, SAPDB_ULong Size ) = 0;

    /*!
        @brief  Returns the master parse id. e.g. the parse id used to prepare the command.
        @return SharedSQL_ParseID&
    */
    virtual SharedSQL_ParseID& GetMasterParseID( void ) = 0;

    /*!
        @brief  Lock the execution plan
        @param  exclusive [in]
    */
    virtual void EnterLock(bool exclusive) = 0;

    /*!
        @brief  Try to lock the execution plan
        @param  exclusive [in]
        @return TRUE if successful
    */
    virtual bool TryEnterLock(bool exclusive) = 0;

    /*!
        @brief  Leave the lock
        @param  exclusive [in]
    */ 
    virtual void LeaveLock(bool exclusive) = 0;

    /*!
        @brief  Switch the lock from EXCLUSIVE TO NONEXCLUSIVE
    */
    virtual void SwitchLockToNonExclusive( void ) = 0;

};

/* ------------------------------------------------------------------------- */

#endif  /* SHAREDSQL_IPLAN_HPP */

