#   MailScanner - SMTP E-Mail Virus Scanner
#   Copyright (C) 2001  Julian Field
#
#   $Id: lock.pl,v 1.6 2002/01/15 00:05:08 nwp Exp $
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#   The author, Julian Field, can be contacted by email at
#      Jules@JulianField.net
#   or by paper mail at
#      Julian Field
#      Dept of Electronics & Computer Science
#      University of Southampton
#      Southampton
#      SO17 1BJ
#      United Kingdom
#

# Provide functions to deal with opening + locking spool files

package Lock;

use strict;
use Fcntl qw(:DEFAULT :flock);

# Open and lock a file.
#
# Pass in a filehandle, a filespec (including ">", "<", or
# whatever on the front), and (optionally) the type of lock
# you want - "r" or "s" for shared/read lock, or pretty much
# anything else (but "w" or "x" really) for exclusive/write
# lock.
#
# Lock type used (flock or fcntl/lockf/posix) depends on
# config. If you're using posix locks, then don't try asking
# for a write-lock on a file opened for reading - it'll fail
# with EBADF (Bad file descriptor).
#
sub openlock {
    my ($fh, $fn, $rw) = @_;
    
    my ($locktype,$struct_flock);

    $locktype = ($Config::LockType)?
      $Config::LockType : $MTA::LockType;

    defined $rw or $rw = ((substr($fn,0,1) eq '>')?"w":"r");
    $rw =~ /^[rs]/i or $rw = 'w';

    unless (open($fh, $fn)) {
	Log::InfoLog("Could not open file $fn: %s", $!);
	return 0;
    }

    if ($locktype =~ /posix/i) {
	# HORRIBLY HARDWIRED
	# would like to "use File::lockf" but that would make
	# installation harder.
	#
	# I guess the pack() is not too bad so long as most parms
	# are zero ;)
	#
	# I've seen sslls, ssllll and all sorts used there...
	# ...not too sure what the best most portable way is :(
	#
	# from linux 2.2 /usr/include/asm/fcntl.h:
	#	struct flock {
	#        short l_type;
	#        short l_whence;
	#        off_t l_start;
	#        off_t l_len;
	#        pid_t l_pid;
	#	};
	#
	# size of off_t appears to depend on whether we've got large file support etc. ugh.
	#
	# from "man fcntl" and /usr/include/sys/fcntl.h on OBSD 2.7:
	#     struct flock {
	#             off_t   l_start;        /* starting offset */
	#             off_t   l_len;          /* len = 0 means until end of file */
	#             pid_t   l_pid;          /* lock owner */
	#             short   l_type;         /* lock type: read/write, etc. */
	#             short   l_whence;       /* type of l_start */
	#     };
	#
	# from solaris 2.7 /usr/include/sys/fcntl.h:
	#	/* regular version, for both small and large file compilation environment */
	#	typedef struct flock {
	#		short   l_type;
	#		short   l_whence;
	#		off_t   l_start;
	#		off_t   l_len;          /* len == 0 means until end of file */
	#		int     l_sysid;
	#		pid_t   l_pid;
	#		long    l_pad[4];               /* reserve area */
	#	} flock_t;
	#
	# and:
	#	/* transitional large file interface version */
	#
	#	#if     defined(_LARGEFILE64_SOURCE)
	#
	#	typedef struct flock64 {
	#		short   l_type;
	#		short   l_whence;
	#		off64_t l_start;
	#		off64_t l_len;          /* len == 0 means until end of file */
	#		int     l_sysid;
	#		pid_t   l_pid;
	#		long    l_pad[4];               /* reserve area */
	#	} flock64_t;
	#
	# and:
	#	/* SVr3 flock type; needed for rfs across the wire compatibility */
	#	typedef struct o_flock {
	#		int16_t l_type;
	#		int16_t l_whence;
	#		int32_t l_start;
	#		int32_t l_len;          /* len == 0 means until end of file */
	#		int16_t l_sysid;
	#		int16_t l_pid;
	#	} o_flock_t;
	#
	# so even thought that one's not used in solaris any more, I guess there'll
	# be systems "out there" that use it.
	#
	#
	# $^O returns either "linux", "openbsd", "solaris" on my systems. Have to
	# have a look at perl sources to find more.

#TODO - MAKE THIS ALL PORTABLE. May have to use File::Lockf
	
#	Log::DebugLog("Using fcntl() to lock $fn");
	$struct_flock =  pack('ssx32',($rw eq 'w' ? F_WRLCK : F_RDLCK),0);
	fcntl($fh, F_SETLK, $struct_flock) and return 1;
    }
    elsif ($locktype =~ /flock/i) {
#	Log::DebugLog("Using flock() to lock $fn");
	flock($fh, ($rw eq 'w' ? LOCK_EX : LOCK_SH) + LOCK_NB) and return 1;
    }
    else {
	Log::DebugLog("Not locking spool file $fn");
	return 1;
    }

    close ($fh);

    Log::DebugLog("Failed to lock $fn: %s", $!);

    return 0;
}


sub unlockclose {
    my ($fh) = @_;

    my $locktype;

    $locktype = ($Config::LockType)?
      $Config::LockType : $MTA::LockType;

    if ($locktype =~ /posix/i) {
	fcntl($fh, &F_SETLK, pack('sslls',&F_UNLCK,0,0,0,0));
    }
    elsif ($locktype =~ /flock/i) {
	flock($fh, LOCK_UN);
    }
# else {
#   default - do nothing, as we didn't lock it in the first place
# }

    close ($fh);
    return 1;
}

1;
