/*  ti_link - link program for TI calculators
 *  Copyright (C) 1999, 2000  Romain Lievin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * This file manage the registry file named 'tiffep.reg'. This file
 * contains the list of plugins & applications.
 * It contains a set of functions for managing the file, querying,
 * modfying, ... the registry data base.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

#include "hfiles.h"
#include "tiffep.h"

extern gchar *reg_path;
TiffepRegistry *registry = NULL;

/*
  Internal use !
  Destroy a registry entry and free first allocated memory
  - entry [in]: the registry entry to destroy
*/
void destroy_registry_entry(TiffepRegistryEntry *entry)
{
  g_string_free(entry->name, TRUE);
  g_string_free(entry->description, TRUE);
  g_string_free(entry->path, TRUE);
  g_string_free(entry->type, TRUE);
  g_string_free(entry->calc, TRUE);
  g_string_free(entry->var, TRUE);
  g_string_free(entry->status, TRUE);
  g_free(entry);
}

/*
  Internal use !
  GList callback to apply to each element 
  - data [in]: a data of the linked list
  - user_data [in]: user data, unused here
*/
void destroy_registry_entry2(gpointer data, gpointer user_data)
{
  TiffepRegistryEntry *entry = (TiffepRegistryEntry *)data;
  
  destroy_registry_entry(entry);
}

/*
  Destroy the entire registry (a linked list).
*/
void destroy_registry()
{
  g_list_foreach(registry, destroy_registry_entry2, NULL);
  g_list_free(registry);
  registry = NULL;
}

/*
  Display an error with line number.
  - line [in]: a line number
*/
static void stop (int line)
{
  g_error(gettext("TiffepRegistry file error at line %i.\n"), line);
}

/* Find a sub-string */
static char *find_str(char *s, const char *t)
{
  char *p;

  p = strstr(s, t);
  //fprintf(stdout, "find_str:<%s|%s>\n", s, p);
  if(p==NULL) return NULL;
  else return p+strlen(t);
}

/*
  Check whether the registry file exists.
  - int [out]: 0 if it does not exist.
*/
#ifndef F_OK
# define F_OK 0
#endif

TIEXPORT
int registry_is_file_exist(void)
{
  gchar *reg_file;
  gint ret = 0;

  reg_file = reg_path;

  ret = !access(reg_path, F_OK);
  //g_free(reg_path);

  return (ret);
}

/*
  Read the registry file for extracting informations and place them into
  the registry variable.
  - int [out]: 0 if success.
*/
TIEXPORT
int registry_read_file(void)
{
  FILE *txt;
  char buffer[256];
  char *p;
  int l=0;
  gchar *reg_file;
  TiffepRegistryEntry *entry;

  destroy_registry();
  reg_file = reg_path;
  fprintf(stderr, "TiffepRegistry filename: %s\n", reg_file);
  
  txt = fopen(reg_file, "rt");
  if(txt == NULL)
    {
      g_error(gettext("TiffepRegistry file not found, use default values.\n"));
      
      return 0;
    }
  //g_free(reg_file);

  while(!feof(txt))
    {
      /* Load a line */
      fgets(buffer, 256, txt);
      l++;
      /* Check end of reg file */
      if(strstr(buffer, "REG_END") != NULL) 
	{
	  fclose(txt);
	  return 0;
	}
      buffer[strlen(buffer)-1]='\0';
      /* Skip comments or emty lines */
      if(buffer[0]=='#' || !strlen(buffer)) continue;
      //printf("<%s>\n", buffer);
      
      /* Parse an entry and store it in the linked list */
      entry = (TiffepRegistryEntry *)g_malloc(sizeof(TiffepRegistryEntry));
      if( (p=find_str(buffer, "Name: ")) )
	{
	  entry->name = g_string_new(p);
	}
      else
	{
	  stop(l);
	  fclose(txt);
	  return ERR_PARSE_FILE;
	}

      fgets(buffer, 256, txt);
      buffer[strlen(buffer)-1]='\0';
      if( (p=find_str(buffer, "Description: ")) )
	{
	  entry->description = g_string_new(p);
	}
      else
	{
	  stop(l);
	  fclose(txt);
	  return ERR_PARSE_FILE;
	}

      fgets(buffer, 256, txt);
      buffer[strlen(buffer)-1]='\0';
      if( (p=find_str(buffer, "Path: ")) )
	{
	  entry->path = g_string_new(p);
	}
      else
	{
	  stop(l);
	  fclose(txt);
	  return ERR_PARSE_FILE;
	}

      fgets(buffer, 256, txt);
      buffer[strlen(buffer)-1]='\0';
      if( (p=find_str(buffer, "Type: ")) )
	{
	  entry->type = g_string_new(p);
	}
      else
	{
	  stop(l);
	  fclose(txt);
	  return ERR_PARSE_FILE;
	}

      fgets(buffer, 256, txt);
      buffer[strlen(buffer)-1]='\0';
      if( (p=find_str(buffer, "Calculator: ")) )
	{
	  entry->calc = g_string_new(p);
	}
      else
	{
	  stop(l);
	  fclose(txt);
	  return ERR_PARSE_FILE;
	}

      fgets(buffer, 256, txt);
      buffer[strlen(buffer)-1]='\0';
      if( (p=find_str(buffer, "Variable: ")) )
	{
	  entry->var = g_string_new(p);
	}
      else
	{
	  stop(l);
	  fclose(txt);
	  return ERR_PARSE_FILE;
	}

      fgets(buffer, 256, txt);
      buffer[strlen(buffer)-1]='\0';
      if( (p=find_str(buffer, "Status: ")) )
	{
	  entry->status = g_string_new(p);
	}
      else
	{
	  stop(l);
	  fclose(txt);
	  return ERR_PARSE_FILE;
	}

      fgets(buffer, 256, txt);
      buffer[strlen(buffer)-1]='\0';
      registry = g_list_append(registry, entry);

    }
  fclose(txt);

  return 0;
}

/*
  Write the registry variable into the registry file.
  - int [out]: 0 if success.
*/
int registry_write_file(void)
{
  FILE *txt;
  gchar *reg_file;
  gint i = 1;
  TiffepRegistryEntry *entry;

  reg_file = reg_path;
  txt=fopen(reg_file, "wt");
  if(txt==NULL)
    {
      fprintf(stderr, gettext("Unable to create the config file.\n"));
    }
  //g_free(reg_file);

  fprintf(txt, "# Config file for TiFFEP\n");
  fprintf(txt, "# Copyright (C) 2000 Romain Lievin <rlievin@mail.com>\n");
  fprintf(txt, "# Warning: any comments that you add to this file WILL be overwritten\n");
  fprintf(txt, "\n");

  for(i=1; i<g_list_length(registry); i++)
    {
      fprintf(txt, "# Application %i\n", i);
      entry = (TiffepRegistryEntry *)g_list_nth(registry, i);
      fprintf(txt, "Name: %s\n", entry->name->str);
      fprintf(txt, "Description: %s\n", entry->description->str);
      fprintf(txt, "Path: %s\n", entry->path->str);
      fprintf(txt, "Type: %s\n", entry->type->str);
      fprintf(txt, "Calculator: %s\n", entry->calc->str);
      fprintf(txt, "Variable: %s\n", entry->type->str);
      fprintf(txt, "Status: %s\n", entry->status->str);
      fprintf(txt, "\n");
    }

  fprintf(txt, "REG_END\n");

  fclose(txt);

  return 0;
}

/*
 * Add a registry entry in the file
 * - entry [in]: the entry to add
 * - int [out]: 0 if success
 */
TIEXPORT
int registry_add_entry(TiffepRegistryEntry *entry)
{
  TRY(registry_read_file());
  registry = g_list_append(registry, entry);

  return 0;
}

/*
 * Remove a registry entry in the file by its name
 * - name [in]: the entry to remove
 * - int [out]: 0 if success
 */
TIEXPORT
int registry_remove_entry(gchar *name)
{
  gint i;
  TiffepRegistryEntry *entry;

  TRY(registry_read_file());
  for(i=0; i<g_list_length(registry); i++)
    {
      entry = (TiffepRegistryEntry *)g_list_nth(registry, i);
      if(!strcmp(entry->name->str, name))
	{
	  destroy_registry_entry(entry);
	  registry = g_list_remove(registry, entry);
	}
    }
  TRY(registry_write_file());

  return 0;
}

/*
 * Search a registry entry by its name
 * - name [in]: the entry name to find
 * - main_registry [in]: the registry variable
 * - sub_registry [out]: a sub list which contain the sorted entries
 * - int [out]: 0 if success
 */
TIEXPORT
int registry_search_entry_by_name(gchar *name, TiffepRegistry *main_registry, 
				  TiffepRegistry **sub_registry)
{
  TiffepRegistryEntry *entry;
  TiffepRegistry *ptr = g_list_first(main_registry);
  gint found = ERR_NOT_FOUND;

  *sub_registry = NULL;
  //TRY(registry_read_file());

  while(ptr != NULL)
    {
      entry = (TiffepRegistryEntry *)ptr->data;
      if(strstr(entry->name->str, name) != NULL)
	{
	  if(!strcmp(entry->status->str, "enabled"))
	    {
	      *sub_registry = g_list_append(*sub_registry, entry);
	      found = 0;
	    }
	}
      
      ptr = g_list_next(ptr);
    }
  
  return found;
}

/*
 * Search a registry entry by its type
 * - name [in]: the entry type to find
 * - main_registry [in]: the registry variable
 * - sub_registry [out]: a sub list which contain the sorted entries
 * - int [out]: 0 if success
 */

TIEXPORT
int registry_search_entry_by_type(gchar *type, TiffepRegistry *main_registry, 
				  TiffepRegistry **sub_registry)
{
  TiffepRegistryEntry *entry;
  TiffepRegistry *ptr = g_list_first(main_registry);
  gint found = ERR_NOT_FOUND;

  *sub_registry = NULL;

  while(ptr != NULL)
    {
      entry = (TiffepRegistryEntry *)ptr->data;
      if(strstr(entry->type->str, type) != NULL)
	{
	  if(!strcmp(entry->status->str, "enabled"))
	    {
	      *sub_registry = g_list_append(*sub_registry, entry);
	      found = 0;
	    }
	}
      
      ptr = g_list_next(ptr);
    }
  
  return found;
}

/*
 * Search a registry entry by its calc type
 * - name [in]: the entry calc to find
 * - main_registry [in]: the registry variable
 * - sub_registry [out]: a sub list which contain the sorted entries
 * - int [out]: 0 if success
 */

TIEXPORT
int registry_search_entry_by_calc(gchar *calc, TiffepRegistry *main_registry, 
				  TiffepRegistry **sub_registry)
{
  TiffepRegistryEntry *entry;
  TiffepRegistry *ptr = g_list_first(main_registry);
  gint found = ERR_NOT_FOUND;

  *sub_registry = NULL;

  while(ptr != NULL)
    {
      entry = (TiffepRegistryEntry *)ptr->data;
      //fprintf(stdout, "<%s>\n", entry->calc->str);
      if(strstr(entry->calc->str, calc) != NULL)
	{
	  if(!strcmp(entry->status->str, "enabled"))
	    {
	      *sub_registry = g_list_append(*sub_registry, entry);
	      found = 0;
	    }
	}
      
      ptr = g_list_next(ptr);
    }
  
  return found;
}

/*
 * Search a registry entry by its calc type
 * - name [in]: the entry calc to find
 * - main_registry [in]: the registry variable
 * - sub_registry [out]: a sub list which contain the sorted entries
 * - int [out]: 0 if success
 */

TIEXPORT
int registry_search_entry_by_var(gchar *var, TiffepRegistry *main_registry, 
				  TiffepRegistry **sub_registry)
{
  TiffepRegistryEntry *entry;
  TiffepRegistry *ptr = g_list_first(main_registry);
  gint found = ERR_NOT_FOUND;

  *sub_registry = NULL;

  while(ptr != NULL)
    {
      entry = (TiffepRegistryEntry *)ptr->data;
      //fprintf(stdout, "<%s %s>\n", entry->var->str, var);
      if(strstr(entry->var->str, var) != NULL)
	{
	  if(!strcmp(entry->status->str, "enabled"))
	    {
	      *sub_registry = g_list_append(*sub_registry, entry);
	      found = 0;
	    }
	}
      
      ptr = g_list_next(ptr);
    }
  
  return found;
}

/*
 * Returns the registry pointer
 * - registry [out]: a pointer on the current registry
 * - int [out]: 0 if success
 */

TIEXPORT
int registry_get_pointer(GList **registry_pointer)
{
  *registry_pointer = NULL;
  TRY(registry_read_file());
  *registry_pointer = registry;

  return 0;
}

/*
 * Enable a registry entry
 * - entry [in]: the registry entry to enable
 * int [out]: 0 if success
 */
TIEXPORT
int registry_enable_entry(TiffepRegistryEntry *entry_to_enable)
{
  entry_to_enable->status = g_string_assign(entry_to_enable->status,
					    "enabled");
  TRY(registry_write_file());

  return 0;
}

/*
 * Disable a registry entry
 * - entry [in]: the registry entry to disable
 * - int [out]: 0 if success
 */
TIEXPORT
int registry_disable_entry(TiffepRegistryEntry *entry_to_disable)
{
  entry_to_disable->status = g_string_assign(entry_to_disable->status,
					    "disabled");
  TRY(registry_write_file());

  return 0;
}

/*
 * Chech whether a registry entry is enabled or disabled
 * - entry [in]: the registry entry to check
 * - int [out]: 0 if success
 */
TIEXPORT 
int registry_entry_is_enabled(TiffepRegistryEntry *entry)
{
  if(!strcmp(entry->status->str, "enabled"))
    return TRUE;
  else
    return FALSE;
}



