/*
 *  Copyright (c) 2018 The WebRTC project authors. All Rights Reserved.
 *
 *  Use of this source code is governed by a BSD-style license
 *  that can be found in the LICENSE file in the root of the source
 *  tree. An additional intellectual property rights grant can be found
 *  in the file PATENTS.  All contributing project authors may
 *  be found in the AUTHORS file in the root of the source tree.
 */

#include "modules/audio_processing/agc2/signal_classifier.h"

#include <array>
#include <functional>
#include <limits>

#include "modules/audio_processing/agc2/agc2_testing_common.h"
#include "modules/audio_processing/logging/apm_data_dumper.h"
#include "rtc_base/gunit.h"
#include "rtc_base/random.h"

namespace webrtc {

namespace {
Random rand_gen(42);
ApmDataDumper data_dumper(0);
constexpr int kNumIterations = 100;

// Runs the signal classifier on audio generated by 'sample_generator'
// for kNumIterations. Returns the number of frames classified as noise.
int RunClassifier(std::function<float()> sample_generator, int rate) {
  SignalClassifier classifier(&data_dumper);
  std::array<float, 480> signal;
  classifier.Initialize(rate);
  const size_t samples_per_channel = rtc::CheckedDivExact(rate, 100);
  int number_of_noise_frames = 0;
  for (int i = 0; i < kNumIterations; ++i) {
    for (size_t j = 0; j < samples_per_channel; ++j) {
      signal[j] = sample_generator();
    }
    number_of_noise_frames +=
        classifier.Analyze({&signal[0], samples_per_channel}) ==
        SignalClassifier::SignalType::kStationary;
  }
  return number_of_noise_frames;
}

float WhiteNoiseGenerator() {
  return static_cast<float>(rand_gen.Rand(std::numeric_limits<int16_t>::min(),
                                          std::numeric_limits<int16_t>::max()));
}
}  // namespace

// White random noise is stationary, but does not trigger the detector
// every frame due to the randomness.
TEST(AutomaticGainController2SignalClassifier, WhiteNoise) {
  for (const auto rate : {8000, 16000, 32000, 48000}) {
    const int number_of_noise_frames = RunClassifier(WhiteNoiseGenerator, rate);
    EXPECT_GT(number_of_noise_frames, kNumIterations / 2);
  }
}

// Sine curves are (very) stationary. They trigger the detector all
// the time. Except for a few initial frames.
TEST(AutomaticGainController2SignalClassifier, SineTone) {
  for (const auto rate : {8000, 16000, 32000, 48000}) {
    test::SineGenerator gen(600.f, rate);
    const int number_of_noise_frames = RunClassifier(gen, rate);
    EXPECT_GE(number_of_noise_frames, kNumIterations - 5);
  }
}

// Pulses are transient if they are far enough apart. They shouldn't
// trigger the noise detector.
TEST(AutomaticGainController2SignalClassifier, PulseTone) {
  for (const auto rate : {8000, 16000, 32000, 48000}) {
    test::PulseGenerator gen(30.f, rate);
    const int number_of_noise_frames = RunClassifier(gen, rate);
    EXPECT_EQ(number_of_noise_frames, 0);
  }
}
}  // namespace webrtc
