/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

/*! \file TeView.h
    This file provides TerraLib's definition of a View
*/
#ifndef  __TERRALIB_INTERNAL_VIEW_H
#define  __TERRALIB_INTERNAL_VIEW_H

#include "TeLayer.h"
#include "TeTheme.h"
#include <algorithm>

using namespace std;

//! A class to deal with views in TerraLib
/*!
	A view in TerraLib has its own projections and may belong to a particular user
	\sa TeTheme TeViewNode
*/
class TeView {

public:

	//! Constructor
    TeView( const string& name="", const string& user="", int id=0)
		: id_(id), name_(name), user_(user), proj_(0), is_visible_(true), connectedId_(0)
	{}

	//! Destructor
	~TeView () {}

	//! Returns view unique id
	int	id () { return id_; }

	//! Sets view unique id
	void id (int id)
	{
		id_ = id;	// update view id
		for (unsigned int th=0;th<viewTree_.size();th++)	//update its themes
			viewTree_.retrieve(th)->view (id_);
	}

	//! Returns the view name
	string	name () { return name_; }

	//! Sets the view name
	void name (const string& s) { name_ = s; }

	//! Returns the view user name
	string	user () { return user_; }

	//! Sets the view user name
	void user (const string& i) { user_ = i; }

	//! Returns the view projection
	TeProjection* projection () { return proj_; }

	//! Sets the view projection
	void projection (TeProjection* p) { proj_ = p; }

	//! Returns TRUE if view is visible
	bool isVisible () { return is_visible_; }

	//! Sets whether view is visible
	void isVisible (bool v) { is_visible_ = v; }
	
	//! Returns the view tree where view belongs
	TeViewTree* root () { return &viewTree_; }

	//! Adds a view node to the view tree
	void add( TeViewNode* node )
	{
		node->view (id_);
		viewTree_.add ( node );
	}

	//! Removes a view node through its name from the view tree
	TeViewNode* remove(string name)
	{
		return viewTree_.remove(name);
	}

	//! Removes a view node through its identifier from the view tree
	TeViewNode* remove (int id)
	{
		return viewTree_.removeID(id);
	}

	//! Moves a node up
	void moveUp (TeViewNode* node)
	{
		viewTree_.moveUp(node);
	}

	//! Moves a node down
	void moveDown(TeViewNode* node)
	{
		viewTree_.moveDown(node);
	}

	//! Returns a node identified by its position
	TeViewNode* get(int i)
	{
		return viewTree_.retrieve(i);
	}

	//! Returns a node identified by its name and version
	TeTheme* get(string themeName)
	{
		TeTheme *ret = 0;
		for (unsigned int th = 0;th < viewTree_.size();th++)
		{
			TeViewNode *node = viewTree_.retrieve(th);
			if (node->type() == TeTHEME)
			{
				TeTheme *tmp = (TeTheme*)node;
				if (tmp->name() == themeName)
				{
					ret = tmp;
					break;
				}
			}
		}
		return ret;
	}
	
	//! Sort the themes in the view
	void sort() 
		{ viewTree_.sort(); }

	//! Swap the order of two themes
	void swap( unsigned int i, unsigned int j)
	{
		viewTree_.swap(i,j);
	}

	//! Returns the number of themes in the view
	unsigned int size()
		{ return viewTree_.size() ; }

	//! Returns the vector of themes of the view sorted
	//! according to their priorities
	vector<TeViewNode*>& themes()
		{ return viewTree_.nodes();}

	void connectedId(int id) {connectedId_ = id;}

	int connectedId() {return connectedId_;}


private:
	int				id_;	// view id
	string			name_;	// view name
	string			user_;	// user id
	TeProjection*   proj_;
	bool			is_visible_;

	TeViewTree		viewTree_;
	int				connectedId_; // connected view
};

//! A map from identifiers to pointers to views
typedef map<int,TeView*> TeViewMap;

#endif



