/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.
You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#include "TeTheme.h"
#include "TeUtils.h"
#include "TeDatabase.h"
#include "TeGroupingAlgorithms.h"

extern int  yyparse(string& sqlOut);
extern int  initParse(const string& strIn, TeDatabase* db);

TeTheme& 
TeTheme::operator= (TeTheme& other)
{
	if ( this != &other )
	{
		myParent_ = other.parent(); 
		id_ = other.id(); 
		name_= other.name(); 
		view_ = other.view(); 
		type_ = other.type();
		priority_ = other.priority();
		
		layerId_ = other.layerId();
		layer_ = other.layer();      //the same layer pointer
		generateAttributeRest_ = other.attributeRest();
		generateTemporalRest_ = other.temporalRest();
		generateSpatialRest_ = other.spatialRest(); 
		spatialRelation_ = other.spatialRelation();
		hasSpatialRes_ =  other.hasSpatialRest();
		boxRest_ = other.boxRestriction();
		geomRest_ = other.geomRestriction(); 
		geomRepRest_ = other.geomRepRestriction();	
		minScale_ = other.minScale();
		maxScale_ = other.maxScale();
		collectionTable_ = other.collectionTable();
		collectionAuxTable_ = other.collectionAuxTable();
		visibleRep_ = other.visibleRep();
		enableVisibility_ = other.visibility();

		if(grouping_)
			delete grouping_;
		grouping_ = 0;
		if(other.grouping())
		{
			grouping_ = new TeGrouping();
			(*grouping_) = (*other.grouping());
		}

		legend_ = other.legend();
		outOfCollectionLegend_ = other.outOfCollectionLegend();
		withoutDataConnectionLegend_ = other.withoutDataConnectionLegend();	
		defaultLegend_ = other.defaultLegend ();
		pointingLegend_ = other.pointingLegend();
		queryLegend_ = other.queryLegend();
		queryAndPointingLegend_ = other.queryAndPointingLegend(); 
		
		attTableVector_ = other.attrTables();
		sqlFrom_ =  other.sqlFrom();		
		sqlJoin_ = other.sqlGridJoin();
		sqlGridFrom_ = other.sqlGridFrom();
		sqlGridJoin_ = other.sqlGridJoin();
		aliasVector_ = other.aliasVector();
		sqlAttList_ = other.sqlAttList();
		sqlNumAttList_ = other.sqlNumAttList();		
	
		if(rasterVisual_)
			delete rasterVisual_;
		rasterVisual_ = 0;
		if(other.rasterVisual())
		{
			rasterVisual_ = new TeRasterTransform();
			(*rasterVisual_) = (*other.rasterVisual());
		}
	}
	return *this;
}

void 
TeTheme::setSpatialRest(TeBox& box, TeGeomRep rep, TeSpatialRelation relation)
{
	hasSpatialRes_ = true;
	boxRest_ = box;
	spatialRelation_ = relation;
	
	if(rep==TeGEOMETRYNONE)
		geomRepRest_ = layer()->vectRepres()[0]->geomRep_;
	else
		geomRepRest_ = rep;
	
	geomRest_ = 0;
}

void 
TeTheme::setSpatialRest(TeGeometry* geom, TeGeomRep rep, TeSpatialRelation relation)
{
	hasSpatialRes_ = true;
	geomRest_ = geom;
	spatialRelation_ = relation;
	
	if(rep==TeGEOMETRYNONE)
		geomRepRest_ = layer()->vectRepres()[0]->geomRep_;
	else
		geomRepRest_ = rep;
	
	boxRest_ = TeBox();
}

void 
TeTheme::grouping(TeGrouping* g) 
{
	if(grouping_==g)
		return;

	if(grouping_) 
		delete (grouping_);
	
	grouping_ = g; 
} 

bool 
TeTheme::save()  
{
	TeDatabase* db = layer()->database();
	if(!db)
		return false;

	//insert theme in database 
	if(id()==0)
	{
		if(!db->insertTheme(this))
		{
			db->deleteTheme(this->id());
			return false;
		}
	}

	//update te_theme_tables table
	if(!db->updateThemeTable(this))
	{
		db->deleteTheme(this->id());
		return false;
	}
	
	//collection table 
	if(collectionTable().empty())
		collectionTable("te_collection_"+ Te2String(id())); 

	if(!db->createCollectionTable(collectionTable_))
	{
		db->deleteTheme(this->id());
		return false;
	}

	//collection aux table
	collectionAuxTable(collectionTable() + "_aux");
	addThemeTable(collectionAuxTable());

	if(!createCollectionAuxTable())
	{
		db->deleteTheme(this->id());
		return false;
	}

	return true;
}

bool
TeTheme::buildCollection()
{
	if(id()==0)
		return false;
	
	if(!populateCollection())
		return false;

	if(!populateCollectionAux())
		return false;

	return true;
}

bool 
TeTheme::generateLabelPositions()
{ 
	TeDatabase* db = layer()->database();
	if(!db)
		return false;

	return (db->generateLabelPositions(this)); 
}

void 
TeTheme::resetGrouping ()
{
	if(grouping_)
	{
		if (grouping_->groupMode_ == TeRasterSlicing && rasterVisual_)
			this->removeRasterVisual();
		
		delete(grouping_);
		grouping_ = 0;
	}

	legend_.clear();
	return;
}

bool 
TeTheme::buildGrouping(TeGrouping* g, vector<TeSlice>& slices)
{
	if(slices.size() > 200)
		return false;
	grouping(g);
	legend_.clear(); 
	
	for(unsigned int j=0; j<slices.size(); j++)
	{
		TeLegendEntry legend(slices[j]);
		legend.group(j);
		legend.theme(id());
		legend_.push_back(legend);
	}	
	return true;
}

bool  
TeTheme::buildGrouping(TeGrouping* g, TeSelectedObjects selectedObjects, vector<double>* dValuesVec)   
{ 
	TeDatabase* db = layer_->database();
	if(!db || !g)
		return false;

	unsigned int i;
	vector<TeSlice> slices;

	if (g->groupMode_ == TeRasterSlicing)
	{
		int b = atoi(g->groupAttribute_.name_.c_str());
		if (!layer_->raster() ||  
			b < 0 ||
			b > layer_->raster()->params().nBands() ||
			g->groupNumSlices_ <= 0 ||
			g->groupNumSlices_ > 200 )
			return false;

		if (g->groupMaxVal_ == TeMINFLOAT)
			g->groupMaxVal_ = layer_->raster()->params().vmax_[b];

		if (g->groupMinVal_ == TeMAXFLOAT)
			g->groupMinVal_ = layer_->raster()->params().vmin_[b];

		grouping(g);
		TeGroupByEqualStep(g->groupMinVal_, g->groupMaxVal_,g->groupNumSlices_, slices, g->groupPrecision_);
	}
	else
	{
		if(g->groupAttribute_.name_.empty())
			return false;
		grouping(g);

		//verify what the objects will be consired
		string	input;
		if(selectedObjects == TeSelectedByPointing)
		{
			input = " WHERE (grid_status = 1 OR grid_status = 3";
			input += " OR (grid_status is null AND (c_object_status = 1 OR c_object_status = 3)))";
		}
		else if(selectedObjects == TeNotSelectedByPointing)
		{
			input = " WHERE (grid_status = 0 OR grid_status = 2";
			input += " OR (grid_status is null AND (c_object_status = 0 OR c_object_status = 2)))";
		}
		else if(selectedObjects == TeSelectedByQuery)
		{
			input = " WHERE (grid_status = 2 OR grid_status = 3";
			input += " OR (grid_status is null AND (c_object_status = 2 OR c_object_status = 3)))";
		}
		else if(selectedObjects == TeNotSelectedByQuery)
		{
			input = " WHERE (grid_status = 0 OR grid_status = 1";
			input += " OR (grid_status is null AND (c_object_status = 0 OR c_object_status = 1)))";
		}
		else if(selectedObjects == TeGrouped)
		{
			input = " WHERE c_legend_id <> 0";
		}
		else if(selectedObjects == TeNotGrouped)
		{
			input = " WHERE c_legend_id = 0";
		}

		TeDatabasePortal* portal = db->getPortal();
		string query;
		bool normal = false;
		string aggrFunc = "";
		if(grouping_->groupFunction_.empty())
			aggrFunc = " MIN";
		else
			aggrFunc = grouping_->groupFunction_;

		if(grouping_->groupNormAttribute_.empty())
		{
//			if(grouping_->groupAttribute_.type_== TeSTRING)
//				query = " SELECT "+ grouping_->groupAttribute_.name_;  
//			else
				query = " SELECT "+ aggrFunc +"("+ grouping_->groupAttribute_.name_ +")";  
		}
		else
		{
			query = " SELECT "+ aggrFunc +"("+ grouping_->groupAttribute_.name_ +") / "+ aggrFunc +"("+ grouping_->groupNormAttribute_ + ")";
			normal = true;
		}
		query += sqlGridFrom(); 
		
		if(selectedObjects != TeAll)
			query += input;

//		if(grouping_->groupAttribute_.type_!= TeSTRING)
			query += " GROUP BY " + collectionTable() + ".c_object_id";
		
		if(!portal->query(query) || !portal->fetchRow())
		{
			delete portal;
			return false;
		}

		//mount vector in memory
		vector<double> dValues;  //inputvect
		vector<string> sValues;	//svec

		double mean, sum; 
		mean = sum = 0.;
		int	nullValues = 0;
		do  
		{
			string val = portal->getData(0);
			string valNorm = Te2String(atof(val.c_str()), grouping_->groupPrecision_);

			if (!val.empty())
			{
				if(grouping_->groupMode_== TeUniqueValue)
				{
					if(normal)
						sValues.push_back(valNorm);
					else
						sValues.push_back(val);
				}
				else
				{
					dValues.push_back(atof(valNorm.c_str()));
					sum += atof(valNorm.c_str());
				}

			}
			else
				nullValues++;
		}while(portal->fetchRow());
		delete portal;
		
		if(dValues.empty() && sValues.empty())
			return false;

		if(grouping_->groupMode_== TeEqualSteps)
			TeGroupByEqualStep(dValues.begin(), dValues.end(), grouping_->groupNumSlices_, slices, grouping_->groupPrecision_);
		else if(grouping_->groupMode_== TeQuantil)
			TeGroupByQuantil(dValues.begin(), dValues.end(), grouping_->groupNumSlices_, slices, grouping_->groupPrecision_);
		else if(grouping_->groupMode_== TeStdDeviation)
		{
			string m = Te2String(mean);
			TeGroupByStdDev(dValues.begin(), dValues.end(), grouping_->groupStdDev_, slices, m, grouping_->groupPrecision_);
		}
		else if(grouping_->groupMode_== TeUniqueValue)
		{
			if(grouping_->groupFunction_ == "COUNT")
				TeGroupByUniqueValue(sValues, TeINT, slices, grouping_->groupPrecision_);
			else
				TeGroupByUniqueValue(sValues, grouping_->groupAttribute_.type_, slices, grouping_->groupPrecision_);
		}

		if(grouping_->groupNullAttr_ && nullValues > 0)
		{
			TeSlice ps;
			ps.count_ = nullValues;
			ps.from_ = "Missing Data";
			slices.push_back(ps);
			grouping_->groupNumSlices_ = slices.size() - 1;
		}
		else
			grouping_->groupNumSlices_ = slices.size();

		if (dValuesVec)
		{
			for (i = 0; i < dValues.size(); ++i)
				dValuesVec->push_back(dValues[i]);
		}
	}

	legend_.clear(); 
	if(slices.size() > 200)
		return false;
	
	for(i=0; i<slices.size(); ++i)
	{
		TeLegendEntry legend(slices[i]);
		legend.group(i);
		legend.theme(id());
		legend_.push_back(legend);
	}
	
	return true;
}

bool
TeTheme::setGroupingVisual(int n, TeVisual& visual, TeGeomRep rep)
{
	if(	(n > grouping_->groupNumSlices_)	||
		((int)legend_.size() < n)				|| 
		(legend_.empty()) )
		return false;

	legend_[(n-1)].setVisual (visual, rep); 

	return true;
}

bool
TeTheme::setGroupingVisual(int n, TeGeomRepVisualMap& vismap)
{
	if(	(n > grouping_->groupNumSlices_)	||
		((int)legend_.size() < n)		|| 
		(legend_.empty()) )
		return false;

	TeGeomRepVisualMap::iterator it = vismap.begin();
	while (it != vismap.end())
	{
		legend_[(n-1)].setVisual(it->second,it->first);
		++it;
	}
	return true;
}

TeSliceVector 
TeTheme::getSlices()
{
	TeSliceVector sliceVec;
	for(unsigned int x=0; x<legend_.size(); ++x)
	{
		TeSlice slice = legend_[x].slice();
		sliceVec.push_back (slice);
	}
	return sliceVec;
}

bool 
TeTheme::getAttTables(TeAttrTableVector& attrs, TeAttrTableType attType)
{
	TeAttrTableVector::iterator it = attTableVector_.begin();
	while (it != attTableVector_.end())
	{
		if ((attType == TeAllAttrTypes) || ((*it).tableType() == attType))
			attrs.push_back((*it));
		++it;
	}
	return (!attrs.empty());
}

bool 
TeTheme::getTemporalTable(TeTable& table)
{
	TeAttrTableVector::iterator it = attTableVector_.begin();
	while (it != attTableVector_.end())
	{
		if (((*it).tableType() == TeAttrEvent) || ((*it).tableType() == TeFixedGeomDynAttr))
		{
			table = (*it); 
			return true;
		}
		++it;
	}
	return false;
}

bool
TeTheme::saveGrouping(TeSelectedObjects selectedObjects)
{ 
	TeDatabase* db = layer_->database();
	if(!db || !grouping_)
		return false;

	//verify if the collection table exists
	//if there is a grouping definition
	if (grouping_->groupMode_ != TeRasterSlicing)
	{
		if (!db->tableExist(collectionTable_) ||
		   (grouping_->groupAttribute_.name_.empty()))	
				return false;
	}
	if(legend_.empty())
		buildGrouping(grouping());

	//save grouping in te_gouping table, the legends in te_legend table and
	//the visual of each legend in the te_visual table
	if(!db->updateTheme(this))
		return false;

	if(grouping_->groupMode_ != TeRasterSlicing)
	{	//save legend in collection table
		if(!saveLegendInCollection(selectedObjects))
			return false;
	}
	return true;
}  

bool 
TeTheme::addThemeTable(TeTable& inputTable)  
{
	bool result = true;
	TeAttrTableType type = inputTable.tableType();

	if(inputTable.name() == collectionAuxTable_)
	{
		loadTablesJoin();
		return false;
	}
	
	if (type == TeAttrMedia)
		return false; 

	if(type != TeAttrStatic)
	{
		bool hasTemporal = false;
		bool hasExtern = false; 

		TeAttrTableVector::iterator it = attTableVector_.begin();
		while(it!=attTableVector_.end())
		{
			//temporal
			if( (it->tableType()==TeAttrEvent) || 
				(it->tableType()==TeFixedGeomDynAttr) ||
				(it->tableType()==TeDynGeomDynAttr))
				hasTemporal = true;

			//extern
			if((it->tableType())==TeAttrExternal)
				hasExtern = true;

			++it;
		}

		if( ((type==TeAttrEvent) || 
			(type==TeFixedGeomDynAttr) ||
			(type==TeDynGeomDynAttr)) && (hasTemporal || hasExtern))
			result = false;
		else if ((type==TeAttrExternal) && hasTemporal)
			result = false;
	}

	if(!result)
		return false;

	attTableVector_.push_back(inputTable);
	loadAliasVector();
	loadAttrLists();
	loadTablesJoin(); 
	return true;
}

void 
TeTheme::addThemeTable(string tableName)
{
	TeTable table(tableName);
	addThemeTable(table);
}

bool
TeTheme::setAttTables(TeAttrTableVector& attrs)
{	
	attTableVector_.clear();	
	int countTemporal = 0;
	int countExtern = 0;
	bool result = true;

	TeAttrTableVector::iterator it = attrs.begin();
	while(it!=attrs.end())
	{
		//temporal
		if( (it->tableType()==TeAttrEvent) || 
			(it->tableType()==TeFixedGeomDynAttr) ||
			(it->tableType()==TeDynGeomDynAttr))
			++countTemporal;

		//extern
		if(it->tableType()==TeAttrExternal)
			++countExtern;

		if( (it->tableType()==TeAttrMedia)     ||
			(countTemporal>0 && countExtern>0) ||
			(countTemporal>1) ||
			(it->name() == collectionAuxTable_) )
		{
			result = false;
		}
		else
		{
			attTableVector_.push_back(*it);
			loadAliasVector();
			loadAttrLists();
			loadTablesJoin(); 
		}
			
		++it;
	}

	return result;
}

bool 
TeTheme::removeThemeTable(unsigned int index)  
{
	if (index > (attTableVector_.size() - 1))
		return false;

	TeAttrTableVector::iterator it;
	it = attTableVector_.begin() + index;

	attTableVector_.erase(it);
	loadAliasVector();
	loadAttrLists();
	loadTablesJoin();

	return true; 
}

bool 
TeTheme::isThemeTable(int tableId)  
{
	bool isThemeTable = false;
	for (unsigned int i = 0; i < attTableVector_.size(); ++i)
	{
		if (attTableVector_[i].id() == tableId)
		{
			isThemeTable = true;
			break;
		}
	}
	return isThemeTable;
}

bool 
TeTheme::isThemeTable(string tableName)  
{
	bool isThemeTable = false;
	for (unsigned int i = 0; i < attTableVector_.size(); ++i)
	{
		if (attTableVector_[i].name() == tableName)
		{
			isThemeTable = true;
			break;
		}
	}
	return isThemeTable;
}

string
TeTheme::getTableName(const string& attrName)
{
	string tableName;
	size_t pos = attrName.find(".");

	if (pos != string::npos)
		return tableName = attrName.substr(0, pos-1);

	for (unsigned int i = 0; i < attTableVector_.size(); ++i)
	{
		TeAttributeList& attrList = attTableVector_[i].attributeList();
		for (unsigned j = 0; j < attrList.size(); ++j)
		{
			if (attrList[j].rep_.name_ == attrName)
			{
				tableName = attTableVector_[i].name();
				return tableName;
			}
		}	
	}

	return tableName;
}


bool 
TeTheme::loadThemeTables() 
{
	clearAttTableVector();
	TeDatabase* db = this->layer()->database(); 
	if(!db)
		return false;
	
	return (db->loadThemeTable(this));
}


bool 
TeTheme::deleteGrouping()  
{
	TeDatabase* db = layer_->database();
	if(!db)
		return false;

	resetGrouping();
	
	//delete te_legend table
	if(!db->deleteLegend (this->id()))
		return false;
	
	//delete te_grouping table
	string sql = "DELETE FROM te_grouping WHERE theme_id= "+ Te2String(this->id());
	db->execute (sql);

	return true;
}

bool 
TeTheme::createCollectionAuxTable()   
{
	unsigned int i, j;
	bool status;
	
	TeDatabase* db = layer_->database();

	if(!db->tableExist(collectionTable_))
		return false;

	if(db->tableExist(collectionAuxTable_))
	{
		status = db->execute("DROP TABLE " + collectionAuxTable_);
		if(!status)
			return false; 
	}

	TeAttributeList attList;

	TeAttribute at;
	at.rep_.type_ = TeSTRING;
	at.rep_.numChar_ = 50;
	at.rep_.name_ = "object_id";
	attList.push_back(at);

	j = 0;
	for (i = 0; i < attTableVector_.size(); ++i)
	{
		if (attTableVector_[i].tableType() == TeAttrExternal ||
			attTableVector_[i].tableType() == TeFixedGeomDynAttr)
		{
			at.rep_.name_ = "aux" + Te2String(j++);
			attList.push_back(at);
		}
	}

	at.rep_.isPrimaryKey_ = true;
//	at.rep_.numChar_ = 200;
	at.rep_.numChar_ = 0;
	at.rep_.name_ = "unique_id";
	at.rep_.isAutoNumber_ = true;
	at.rep_.type_ = TeINT;
	attList.push_back(at);
	
//	at.rep_.type_ = TeINT;
	at.rep_.name_ = "grid_status";
	at.rep_.isPrimaryKey_ = false;
	at.rep_.isAutoNumber_ = false;
	at.rep_.numChar_ = 0;
	attList.push_back(at);	
	
	status = db->createTable(collectionAuxTable_,attList);
	if(!status)
		return false;

	//create index to object_id
	db->createIndex(collectionAuxTable_, collectionAuxTable_+"_index1", "object_id"); 
	return true;
} 

bool 
TeTheme::populateCollectionAux()  
{	
	TeDatabase* db = layer_->database();
	if((!db) || (collectionTable_.empty()) || (collectionAuxTable_.empty()))
		return false;
	
	string whereClause; 

	whereClause = " WHERE 1=1 ";
	
	// Populate the collection auxiliary table
	string ins = "INSERT INTO " + collectionAuxTable_ + " (object_id";
    unsigned int i, j;
	j = 0;
	for (i = 0; i < attTableVector_.size(); ++i)
	{
		if (attTableVector_[i].tableType() == TeAttrExternal ||
			attTableVector_[i].tableType() == TeFixedGeomDynAttr)
			ins += ", aux" + Te2String(j++);
	}

//	ins += ", unique_id, grid_status) SELECT c_object_id";
	ins += ", grid_status) SELECT c_object_id";
	for (i = 0; i < attTableVector_.size(); ++i)
	{
		if (attTableVector_[i].tableType() == TeAttrExternal ||
			attTableVector_[i].tableType() == TeFixedGeomDynAttr)
			ins += "," + aliasVector()[i] + "." + attTableVector_[i].uniqueName();
	}

	ins += ",c_object_status";

	//attribute restriction
	if(!generateAttributeRest_.empty())
		whereClause += " AND "+ generateAttributeRest_; 

	//temporal restriction
	if(!generateTemporalRest_.empty())
	{
		string sqlTemp;
		initParse(generateTemporalRest_, db); 
				
		if(!yyparse(sqlTemp))  //0: accept  1: reject
			whereClause += " AND "+ sqlTemp;
		else
			return false;
	}

	Keys objs;

	//spatial restriction
	if(hasSpatialRes_)
	{
		//TeRepresentation* rep = layer_->vectRepres()[0];
		string geomTable = layer_->tableName(geomRepRest_); 
		
		if(boxRest_.isValid())
		{
			TeBox box = boxRest_;
			whereClause += " AND "+ db->getSQLBoxWhere(box, geomRepRest_);
		}
		else if (geomRest_)
		{
			
			TePrecision::instance().setPrecision(TeGetPrecision(layer()->projection()));

			if(db->spatialRelation(geomTable, geomRepRest_, geomRest_, objs, spatialRelation_))
			{
				string obs;
				for(unsigned int i=0; i<objs.size(); i++)
				{
					if(i!=0)
						obs += ",";
					obs += "'"+ objs[i] +"'";
				}
					
				whereClause += " AND "+ geomTable +".object_id IN ("+ obs +")";
			}
		}
	}

	bool usaTemporal = false;
	for (i = 0; i < attTableVector_.size(); ++i)
	{
		if (attTableVector_[i].tableType() == TeAttrExternal ||
			attTableVector_[i].tableType() == TeFixedGeomDynAttr ||
			attTableVector_[i].tableType() == TeDynGeomDynAttr)
		{
			whereClause += " AND ";
			whereClause += aliasVector()[i] + "." + attTableVector_[i].uniqueName();
			whereClause += " IS NOT NULL ";
		}
		if (attTableVector_[i].tableType() == TeFixedGeomDynAttr ||
			attTableVector_[i].tableType() == TeDynGeomDynAttr)
			usaTemporal = true;
	}
	
	string result = ins + sqlFrom() + whereClause + " ";
	bool ret = db->execute(result);
	if(ret == false)
		return false;

	TeAttribute at;
	at.rep_.name_ = "unique_id";
	at.rep_.type_ = TeSTRING;
	at.rep_.numChar_ = 100;
	at.rep_.isAutoNumber_ = false;
	at.rep_.isPrimaryKey_ = true;
	ret = db->alterTable(collectionAuxTable_, at.rep_);

	if(usaTemporal) // filter collection table
	{
		string s = "DELETE FROM " + collectionTable_ + " WHERE c_object_id NOT IN";
		s += " (SELECT DISTINCT object_id FROM " + collectionAuxTable_ + ")";
		return(db->execute(s));
	}
	return true;
}


bool 
TeTheme::locatePolygon (TeCoord2D &pt, TePolygon &polygon, const double& tol)  
{
	if (!layer()->database() || collectionTable().empty())
		return false;

	string geomTable = layer()->tableName(TePOLYGONS);
	string sqlFrom = " "+ geomTable +" INNER JOIN " + collectionTable();
	sqlFrom += " ON "+ collectionTable() +".c_object_id = "+ geomTable +".object_id "; 

	if(!layer()->database()->locatePolygon(sqlFrom, pt, polygon, tol))
		return false;

	return true;
}


bool 
TeTheme::locatePolygonSet (TeCoord2D &pt, double tol, TePolygonSet &polygons)  
{
	if (!layer()->database() || collectionTable().empty())
		return false;

	string geomTable = layer()->tableName(TePOLYGONS);
	string sqlFrom = " "+ geomTable +" INNER JOIN " + collectionTable();
	sqlFrom += " ON "+ collectionTable() +".c_object_id = "+ geomTable +".object_id "; 

	if(!layer()->database()->locatePolygonSet(sqlFrom, pt, tol, polygons))
		return false;

	return true;
}

bool 
TeTheme::locateLine (TeCoord2D &pt, TeLine2D &line, const double& tol)  
{
	if (!layer()->database() || collectionTable().empty())
		return false;

	string geomTable = layer()->tableName(TeLINES);
	string sqlFrom = " "+ geomTable +" INNER JOIN " + collectionTable();
	sqlFrom += " ON "+ collectionTable() +".c_object_id = "+ geomTable +".object_id "; 

	if(!layer()->database()->locateLine(sqlFrom, pt, line, tol))
		return false;

	return true;
}

bool 
TeTheme::locatePoint (TeCoord2D &pt, TePoint &point, const double& tol)  
{
	if (!layer()->database() || collectionTable().empty())
		return false;

	string geomTable = layer()->tableName(TePOINTS);
	string sqlFrom = " "+ geomTable +" INNER JOIN " + collectionTable();
	sqlFrom += " ON "+ collectionTable() +".c_object_id = "+ geomTable +".object_id "; 

	if(!layer()->database()->locatePoint(sqlFrom, pt, point, tol))
		return false;

	return true;
}

bool 
TeTheme::locateCell (TeCoord2D &pt, TeCell &cell, const double&  tol)  
{
	if (!layer()->database() || collectionTable().empty())
		return false;

	string geomTable = layer()->tableName(TeCELLS);
	string sqlFrom = " "+ geomTable +" INNER JOIN " + collectionTable();
	sqlFrom += " ON "+ collectionTable() +".c_object_id = "+ geomTable +".object_id "; 

	if(!layer()->database()->locateCell(sqlFrom, pt, cell, tol))
		return false;

	return true;
}

//------------------------------ protected methods
void 
TeTheme::loadAliasVector()  
{
	unsigned int i, count;
	TeTable table;
	multimap<string, int> tableMMap;

	aliasVector_.clear();

	for (i = 0; i < attTableVector_.size(); ++i)
	{
		table = attTableVector_[i];

		if (table.tableType() != TeAttrExternal)
			aliasVector_.push_back(table.name());
		else
		{
			count = tableMMap.count(table.name());
			if (count == 0)
				aliasVector_.push_back(table.name());
			else
				aliasVector_.push_back(table.name() + "_" + Te2String(count));

			tableMMap.insert(multimap<string,int>::value_type(table.name(), ++count));
		}
	}
}

string 
TeTheme::sqlGridFrom(const string& geomTable) 
{ 
	if(geomTable.empty())
		return sqlGridFrom_; 
	
	string result;
	loadTablesJoin(geomTable);
	result = sqlGridFrom_;
	loadTablesJoin();
	return result;
}

string 
TeTheme::sqlWhereRestrictions(TeRepresentation* rep)
{
	Keys objs;
	string whereClause= " 1 = 1 ";
	TeDatabase* db = layer()->database();
	
	// load the first representation 
	if(!rep)
		rep = (layer()->vectRepres())[0];

	// temporal restrictions are applied only to temporal tables
	if(hasTemporalRest())
	{
		string result = "";
		
		initParse(temporalRest(), db); 
				
		if(!yyparse(result))  //0: accept  1: reject
			whereClause += " AND "+ result;
		else 
			return "";
	}
			
	// we should test if the attribute restriction is valid
	if (hasAttrRest())
		whereClause += " AND "+ attributeRest();

	// spatial restriction with other geometry representation
	if (hasSpatialRest() && rep)
	{
		if(boxRestriction().isValid())
		{
            TeBox b =  boxRestriction();
            TeGeomRep gRep = rep->geomRep_;
//			whereClause += " AND "+ db->getSQLBoxWhere(boxRestriction(), rep->geomRep_);
			whereClause += " AND "+ db->getSQLBoxWhere(b, gRep);
		}
		else if(geomRestriction())
		{
			string geomTableRest = layer()->tableName(geomRepRestriction());
			TePrecision::instance().setPrecision(TeGetPrecision(layer()->projection()));

			if(db->spatialRelation(geomTableRest, geomRepRestriction(), geomRestriction(),  
							   objs, spatialRelation()))
			{
				string obs;
				for(unsigned int i=0; i<objs.size(); i++)
				{
					if(i!=0)
						obs += ",";
					obs += "'"+ objs[i] +"'";
				}
				
				whereClause += " AND "+ rep->tableName_ +".object_id IN ("+ obs +")";
			}
		}
	}
	return whereClause;
}



void 
TeTheme::loadTablesJoin(const string& geomTable)  //sqlGridFrom and sqlGridJoin
{
	unsigned int i, count;
	multimap<string, int> tableMMap;
    TeTable table;
    bool hasExternalTable = false;

    // Set the new sqlGridFrom_ clause and the new sqlGridJoin_ string
    sqlGridFrom_.clear();
    sqlGridJoin_.clear();

    TeAttrTableVector tableVec;
    vector<string>    aliasVec;

    if(!geomTable.empty())
    {
        TeTable table;
        table.name(geomTable);
        table.setLinkName("object_id");
        table.setUniqueName("object_id");
        table.setTableType(TeAttrStatic);
        tableVec.push_back(table);

        for(i=0; i<attTableVector_.size(); ++i)
            tableVec.push_back(attTableVector_[i]);

        aliasVec.push_back(geomTable);
        for(i=0; i<aliasVector_.size(); ++i)
            aliasVec.push_back(aliasVector_[i]);
    }
    else
    {
        tableVec = attTableVector_;
        aliasVec = aliasVector_;
    }

    //verify if there is external table
    for (i = 0; i < tableVec.size(); ++i)
    {
        if (tableVec[i].tableType() == TeAttrExternal)
        {
            hasExternalTable = true;
            break;
        }
    }
   
    if(!collectionAuxTable_.empty())
    {
        sqlGridFrom_ = " FROM ";

        for (i = 0; i <= tableVec.size(); ++i)
            sqlGridFrom_ += "(";

        sqlGridFrom_ += collectionAuxTable_;
        sqlGridJoin_ = "SELECT ";

        if(hasExternalTable)
            sqlGridFrom_ += " RIGHT JOIN "+ collectionTable_;
        else
            sqlGridFrom_ += " LEFT JOIN "+ collectionTable_;

        sqlGridFrom_ += " ON "+ collectionAuxTable_ +".object_id = "+ collectionTable_ +".c_object_id )";

        int numAux = 0;
        for (i = 0; i < tableVec.size(); ++i)
        {
            table = tableVec[i];
            if ((table.tableType()==TeAttrStatic) || (table.tableType()==TeAttrMedia) || (table.tableType()==TeAttrEvent))
            {
                sqlGridFrom_ += " LEFT JOIN " + aliasVec[i];
                sqlGridFrom_ += " ON " + collectionTable_ +".c_object_id = " + aliasVec[i] + "." + table.linkName() +")";
            }
            else if (table.tableType() == TeAttrExternal)
            {
				count = tableMMap.count(table.name());
				if (count == 0)
					sqlGridFrom_ += " LEFT JOIN " + aliasVec[i];		
				else
					sqlGridFrom_ += " LEFT JOIN " + table.name() + " AS " + aliasVec[i];

				tableMMap.insert(multimap<string,int>::value_type(table.name(), ++count));
                sqlGridFrom_ += " ON " + collectionAuxTable_ + ".aux" + Te2String(numAux++) +" = "+ aliasVec[i] +"."+ table.uniqueName() +")";
            }
            else
            {
                sqlGridFrom_ += " LEFT JOIN " + aliasVec[i];
                sqlGridFrom_ += " ON " + collectionAuxTable_ + ".aux" + Te2String(numAux++) +" = "+ aliasVec[i] +"."+ table.uniqueName() +")";
            }
           
            sqlGridJoin_ += aliasVec[i] + ".*,";
        }

        sqlGridJoin_ += collectionTable_ + ".*, " +  collectionAuxTable_ + ".* " + sqlGridFrom_;
    }
    loadThemeTablesJoin();
}


void TeTheme::loadThemeTablesJoin()  ////sqlJoin and sqlFrom
{
	unsigned int i, count;
	multimap<string, int> tableMMap;
	TeTable table;

	// Set the new from clause and the new join string
	sqlFrom_.clear();
	sqlJoin_.clear();

	sqlFrom_ = " FROM ";
	sqlJoin_ = "SELECT ";

	if (collectionTable_.empty() == false)
	{
		for (i = 0; i < attTableVector_.size(); ++i)
			sqlFrom_ += "(";
		sqlFrom_ += collectionTable_;

		for (i = 0; i < attTableVector_.size(); ++i)
		{
			table = attTableVector_[i];
			if (table.tableType() != TeAttrExternal)
			{
				sqlFrom_ += " LEFT JOIN " + aliasVector_[i];
				sqlFrom_ += " ON " + collectionTable_ + ".c_object_id = " + aliasVector_[i] + "." + table.linkName() + ")";
			}
			else
			{
				count = tableMMap.count(table.name());
				if (count == 0)
					sqlFrom_ += " LEFT JOIN " + aliasVector_[i];		
				else
					sqlFrom_ += " LEFT JOIN " + table.name() + " AS " + aliasVector_[i];

				tableMMap.insert(multimap<string,int>::value_type(table.name(), ++count));
				
				sqlFrom_ += " ON " + table.relatedTableName() + "." + table.relatedAttribute() + " = ";
				sqlFrom_ +=  aliasVector_[i] + "." + table.linkName() + ")";
			}

			sqlJoin_ += aliasVector_[i] + ".*,";
		}

		sqlJoin_ += collectionTable_ + ".*" + sqlFrom_;
	}
	else
	{
		if (attTableVector_.size() == 1)
		{
			table = attTableVector_[0];
			sqlFrom_ += table.name();
			sqlJoin_ = "SELECT " + table.name() + ".*" + sqlFrom_;
		}
		else
		{
			for (i = 0; i < attTableVector_.size() - 1; ++i)
				sqlFrom_ += "(";

			TeTable firstTable = attTableVector_[0];
			sqlFrom_ += firstTable.name();
			sqlJoin_ += firstTable.name() + ".*,";
			for (i = 1; i < attTableVector_.size(); ++i)
			{
				table = attTableVector_[i];
				sqlFrom_ += " LEFT JOIN " + aliasVector_[i];
				sqlFrom_ += " ON " + firstTable.name() + "." + firstTable.linkName() + " = " + aliasVector_[i] + "." + table.linkName() + ")";

				if (i == attTableVector_.size() - 1)
					sqlJoin_ += aliasVector_[i] + ".*";
				else
					sqlJoin_ += aliasVector_[i] + ".*,";
			}

			sqlJoin_ += sqlFrom_;
		}
	}
}

void 
TeTheme::loadAttrLists()  
{
	// Set the new list of attributes of all the theme tables and its new numerical list,
	unsigned int i, j, count;
	TeTable table;
	multimap<string, int> attrMMap;
	string attrName;

	sqlAttList_.clear();
	sqlNumAttList_.clear();

	if (layer() == 0)
		return;
	TeDatabase *db = layer()->database();

	for (i = 0; i < attTableVector_.size(); ++i)
	{
		table = attTableVector_[i];

		// Set the map of attribute names
		TeAttributeList attrList;
		db->getAttributeList(table.name(), attrList);
		for (j = 0; j < attrList.size(); ++j)
		{
			attrName = attrList[j].rep_.name_; 
			count = attrMMap.count(attrName);
			attrMMap.insert(multimap<string,int>::value_type(attrName, ++count));
		}
	}

	// Set the list of attribute names that contains all the attribute names
	// of the theme tables
	for (i = 0; i < attTableVector_.size(); ++i)
	{
		TeAttributeList attrList;
		table = attTableVector_[i];
		db->getAttributeList(table.name(), attrList);
		for (j = 0; j < attrList.size(); ++j)
		{
			attrName = attrList[j].rep_.name_; 
			count = attrMMap.count(attrName);
			if (count == 1)
				attrList[j].rep_.name_ = attrName;
			else
				attrList[j].rep_.name_ = aliasVector_[i] + "." + attrName;

			sqlAttList_.push_back(attrList[j]);
		}		
	}

	// Set the list of attribute names that contains all the numeric attribute names
	// of the theme tables
	for(i = 0; i < sqlAttList_.size(); ++i)
	{
		if(sqlAttList_[i].rep_.type_ == TeREAL || sqlAttList_[i].rep_.type_ == TeINT)
			sqlNumAttList_.push_back(sqlAttList_[i]);
	}
}

bool
TeTheme::saveLegendInCollection(TeSelectedObjects selectedObjects)   //colocar o count por grupo
{
	unsigned int i;
	if(!grouping_)
		return false; 

	TeAttrDataType		type = grouping_->groupAttribute_.type_;
	TeLegendEntryVector legVec = legend_;
	string			groupingAttr = grouping_->groupAttribute_.name_;
	
	string input;
	if(selectedObjects == TeSelectedByPointing)
	{
		input = " WHERE (grid_status = 1 OR grid_status = 3";
		input += " OR (grid_status IS NULL AND (c_object_status = 1 OR c_object_status = 3)))";
	}
	else if(selectedObjects == TeNotSelectedByPointing)
	{
		input = " WHERE (grid_status = 0 OR grid_status = 2";
		input += " OR (grid_status is null AND (c_object_status = 0 OR c_object_status = 2)))";
	}
	else if(selectedObjects == TeSelectedByQuery)
	{
		input = " WHERE (grid_status = 2 OR grid_status = 3";
		input += " OR (grid_status is null AND (c_object_status = 2 OR c_object_status = 3)))";
	}
	else if(selectedObjects == TeNotSelectedByQuery)
	{
		input = " WHERE (grid_status = 0 OR grid_status = 1";
		input += " OR (grid_status is null AND (c_object_status = 0 OR c_object_status = 1)))";
	}
	else if(selectedObjects == TeGrouped)
		input = " WHERE c_legend_id <> 0";
	else if(selectedObjects == TeNotGrouped)
		input = " WHERE c_legend_id = 0";
	

	string func;
	if(grouping_->groupFunction_.empty())
		func = " MIN";
	else
		func = grouping_->groupFunction_;

	if(grouping_->groupFunction_ == "COUNT")
		type = TeINT;

	string query = "SELECT MIN(" + collectionTable_ + ".c_object_id)";
	if(grouping_->groupNormAttribute_.empty())
		query += ", "+ func +"(" + groupingAttr + ")" + sqlGridFrom();
	else
		query += ", "+ func +"(" + groupingAttr + ") / "+ func +"(" + grouping_->groupNormAttribute_ + ")" + sqlGridFrom();
	
	if(selectedObjects != TeAll)
		query += input;

	query += " GROUP BY " + collectionTable_ + ".c_object_id";

	map<int, vector<string> > legMap;
	TeDatabase *db = layer()->database();
	TeDatabasePortal* portal = db->getPortal();
	if(portal->query(query) == false)
	{
		delete portal;
		return false;
	}

	vector<string> idVec;
	vector<string> nullIdVec;
	vector<string> valVec;
	while(portal->fetchRow())
	{
		string val = portal->getData(1);
		string oid = portal->getData(0);
		if (val.empty() == false)
		{
			idVec.push_back(oid);
			valVec.push_back(val);
		}
		else
			nullIdVec.push_back(oid);
	}
	if (grouping_->groupMode_ == TeUniqueValue)
	{
		unsigned int j = 0;
		while( j < idVec.size())
		{
			string val = valVec[j];
			string oid = idVec[j];
			if(type == TeREAL)
			{
				double a = atof(val.c_str());
				val = Te2String(a, grouping_->groupPrecision_);
			}
			else if(type == TeINT)
			{
				int a = atoi(val.c_str());
				val = Te2String(a);
			}
				
			unsigned int siz = legend_.size();
			if (nullIdVec.size())
				--siz;
			for(i=0; i < siz; i++)
			{
				TeLegendEntry& leg = legend_[i];
				if(val == leg.from())
				{
					legMap[leg.id()].push_back(oid);
					break;
				}
			}
			j++;
		}
	}
	else
	{
		unsigned int j = 0;
		while(j < idVec.size())
		{
			string val = valVec[j];
			string oid = idVec[j];
			if(type == TeREAL)
			{
				double a = atof(val.c_str());
				val = Te2String(a, grouping_->groupPrecision_);
			}
			
			unsigned int siz = legend_.size();
			if (nullIdVec.size())
				--siz;
			for(i=0; i < siz; i++)
			{
				TeLegendEntry& leg = legend_[i];
				int f = leg.from().find("mean", string::npos);
				if(f >= 0)
					continue;
				double dval = atof(val.c_str());
				double dfrom = atof(leg.from().c_str());
				double dto = atof(leg.to().c_str());
				if(i < legend_.size()-1)
				{
					if(dval >= dfrom && dval < dto)
					{
						legMap[leg.id()].push_back(oid);
						break;
					}
				}
				else
				{
					if(dval >= dfrom && dval <= dto)
					{
						legMap[leg.id()].push_back(oid);
						break;
					}
				}
			}
			j++;
		}
	}
	delete portal;

	int legId = legend_[legend_.size()-1].id();
	for(i = 0; i < nullIdVec.size(); ++i)
	{
		string oid = nullIdVec[i];
		legMap[legId].push_back(oid);
	}

	vector<string> svec;
	map<int, vector<string> > :: iterator it = legMap.begin();
	while(it != legMap.end())
	{
    	// --- Generate In Clauses ----
		unsigned int i, j, k, size, chunkSize = 200, nChunks;
		string inClause;
		size = it->second.size();
		if (size % chunkSize)
			nChunks = size / chunkSize + 1;
		else
			nChunks = size / chunkSize;

		j = 0;		
		for (k = 0; k < nChunks; ++k)
		{
			i = 0;
			inClause = "(";
			while (j < size && i < chunkSize)
			{
				inClause += "'" + db->escapeSequence(it->second[j]) + "',";
				i++;
				j++;
			}
			inClause[inClause.size() - 1] = ')';
			svec.push_back(inClause);
		}

		//--- generateInClause

		for(i=0; i<svec.size(); i++)
		{
			string up = "UPDATE " + collectionTable_ + " SET c_legend_id = " + Te2String(it->first);
			up += " WHERE c_object_id IN " + svec[i];

			if(db->execute(up) == false)
				continue;
		}
		it++;
		svec.clear();
	}
	if(legend_.size())
		visibleRep_ = visibleRep_ | 0x40000000;
	else
		visibleRep_ = visibleRep_ | 0xbfffffff;
	string upVis = "UPDATE te_theme SET visible_rep=" + Te2String(visibleRep_) + " WHERE theme_id=" + Te2String(id_);
	return (db->execute(upVis));
}

bool 
TeTheme::populateCollection()  
{
	TeDatabase* db = layer_->database();
	if(!db || collectionTable_.empty())
		return false;

	if(attTableVector_.empty())
		attTableVector_ = layer_->attrTables();
	
//	TeRepresentation* rep = layer_->vectRepres()[0];
	TeRepresPointerVector& represVec = layer_->vectRepres();
	for (unsigned int i = 0; i < represVec.size(); ++i)
	{
		TeRepresentation* rep = represVec[i];
		if (rep->geomRep_ == TeTEXT)
			continue;

		string geomTable = layer_->tableName(rep->geomRep_); 

		string sqlSelect, sqlFrom, sqlWhere;
		
		sqlSelect = " SELECT DISTINCT "+ geomTable +".object_id ";  
		sqlFrom = tableJoin(attTableVector_, geomTable, "object_id");
		
//		sqlWhere = " WHERE 1=1 ";
		sqlWhere = " WHERE " + geomTable + ".object_id NOT IN (SELECT c_object_id FROM " + collectionTable_ + ")";
		
		//attribute restriction
		if(!generateAttributeRest_.empty())
			sqlWhere += " AND "+ generateAttributeRest_; 

		//temporal restriction
		if(!generateTemporalRest_.empty())
		{
			string sqlTemp;
			initParse(generateTemporalRest_, db); 
					
			if(!yyparse(sqlTemp))  //0: accept  1: reject
				sqlWhere += " AND "+ sqlTemp;
			else
				return false;
		}

		Keys objs;

		//spatial restriction
		if(hasSpatialRes_)
		{
			if(boxRest_.isValid())
			{
				TeBox box = boxRest_;
				sqlWhere += " AND "+ db->getSQLBoxWhere(box, geomRepRest_);
			}
			else if (geomRest_)
			{
				TePrecision::instance().setPrecision(TeGetPrecision(layer()->projection()));

				if(db->spatialRelation(geomTable, geomRepRest_, geomRest_, objs, spatialRelation_))
				{
					string obs;
					for(unsigned int i=0; i<objs.size(); i++)
					{
						if(i!=0)
							obs += ",";
						obs += "'"+ objs[i] +"'";
					}
						
					sqlWhere += " AND "+ geomTable +".object_id IN ("+ obs +")";
				}
			}
		}

		//populate the collection table
		string popule;
		popule = " INSERT INTO "+ collectionTable_ +" (c_object_id) ";
		popule += sqlSelect +" FROM "+ sqlFrom + sqlWhere; 	
		
		if (!db->execute(popule))
			return false;
	}

//	int defaultLegend = defaultLegend_.id();
	string popule = "UPDATE " + collectionTable_;
	popule += " SET c_legend_id=0, c_legend_own=0, c_object_status=0 ";
	
	if (!db->execute(popule))
		return false;

	return true;
}


void TeTheme::createRasterVisual(TeRaster* rst)
{
	if (rasterVisual_)
		delete rasterVisual_;

	if (!rst)
		rst = layer_->raster();

	if (!rst)
		return;

	rasterVisual_ = new TeRasterTransform();
										
	if (rst->params().photometric_[0] == TeRASTERPALETTE || // raster palette -> uses its palette
	    rst->params().photometric_[0] == TeRASTERTHEMATIC) 
	{
		rasterVisual_->setTransfFunction(&TeRasterTransform::Pallete2ThreeBand);
		rasterVisual_->setLutSize(rst->params().lutr_.size());
		return;
	}

	if (visibleRep_ & 0x40000000 && grouping_ &&		// sliced raster -> generate the
		grouping_->groupMode_ == TeRasterSlicing)	// appropriate palette
	{
		int band = atoi(grouping_->groupAttribute_.name_.c_str());
		rasterVisual_->setSrcBand(band);
		if (rst->params().dataType_[band] != TeUNSIGNEDCHAR)
			rasterVisual_->generateLUT(legend_, 1024, defaultLegend_.visual(TePOLYGONS).color());
		else
			rasterVisual_->generateLUT(legend_, 256, defaultLegend_.visual(TePOLYGONS).color());
		rasterVisual_->setTransfFunction(&TeRasterTransform::LUT2ThreeBand);
		return;
	}

	if (rst->params().dataType_[0] != TeUNSIGNEDCHAR)	// non unsigned char -> generate linear transformation
		rasterVisual_->setLinearTransfParameters(rst->params().vmin_[0],rst->params().vmax_[0], 0, 255);

	if (rst->params().nBands() == 1)				
		rasterVisual_->setTransfFunction(&TeRasterTransform::Mono2ThreeBand);
	else
		rasterVisual_->setTransfFunction(&TeRasterTransform::Band2Band);
}


