/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

/*! \file TeSTEvent.h
	This file contains structures and definitions to deal with spatial temporal events. 
	An spatial temporal event is represented by a point, a time and a set of attributes. 
*/

#ifndef  __TERRALIB_INTERNAL_STEVENT_H
#define  __TERRALIB_INTERNAL_STEVENT_H

#include "TeGeometry.h"
#include "TeAttribute.h" 
#include "TeTimeInterval.h"

class TeTheme;

//! A class to represent spatial temporal events
class TeSTEvent
{
protected:
	vector<string>	unique_id_;		// instance identification for each attribute table
	string			object_id_;		// event identification
	TePoint			point_;			// point where the event happened  
	long			time_;			// time when the event happened   

	vector<double>			attrValue_;		// attribute values 
	vector<TeAttributeRep>	attrRep_;		// attribute representations (name, type, size) 
	vector<int>				attrOut_;		// indexes of the attributes that were generated

	TeTheme*		theme_;			// theme associated to this event
		
public:
	
	//! Empty constructor
	TeSTEvent()
	{ }

	//! Constructor
	TeSTEvent(const string& objId, const TePoint& point, const long& time);  

	//! Constructor
	TeSTEvent(const string& objId, const TePoint& point, const vector<double>& attrValue, 
		const vector<TeAttributeRep>& attrRep, const long& time);
	
	//! Copy constructor
	TeSTEvent(const TeSTEvent& other); 
						
	//! Assignment operator
	TeSTEvent& operator= (const TeSTEvent& other);
		
	//! Operator ==
	bool operator== (const TeSTEvent& other) const;
	
	//! Destructor
	~TeSTEvent() {} 

	//! Returns the event identification
	string objectId (void)
	{ return object_id_; }

	//! Sets the event identification
	void objectId (const string& objId)
	{ object_id_ = objId; }

	//! Returns the instance identification
	vector<string>& uniqueId () 
	{	return unique_id_;	}

	//! Returns the instance identification
	string uniqueId (int index_) 
	{	return unique_id_[index_];	}
	
	//! Sets the instance identifications
	void uniqueId (const vector<string>& ids) 
	{	unique_id_ = ids;	}
	
	//! Get point geometry
	void getGeometry(TePoint& result)
	{	result = point_; }

	//! Set point geometry 
	void setGeometry(const TePoint& result)
	{	point_ = result; }

	//! Get time
	long eventTime()
	{	return time_; }

	//! Set time
	void eventTime(const long& t)
	{ time_ = t; }

	//! Get the time as a structure TeTimeInterval
	TeTimeInterval timeInterval(); 
	
	//! Get theme pointer
	TeTheme* theme();

	//! Set theme pointer
	void theme(TeTheme* t);

	//! Get attribute values
	vector<double>& attrValues()
	{ return attrValue_; }

	//! Set attribute values
	void attrValues (const vector<double>& attrVal)
	{ attrValue_ = attrVal; }

	//! Get attribute representations (name, type, size)
	vector<TeAttributeRep>& getAttributeRep()
	{ return attrRep_; }

	//! Get the attribute representation of the index-th attribute
	TeAttributeRep getAttributeRep(const int& index)
	{ return  attrRep_[index]; }

	//! Set attribute representations (name, type, size)
	void setAttributeRep (const vector<TeAttributeRep>& attrRep)
	{ attrRep_ = attrRep; }

	//! Get attribute indexes that were generated
	vector<int>& attrOutIndexes()
	{ return attrOut_; }

	//! Set attribute indexes that were generated
	void attrOutIndexes (const vector<int>& indexes)
	{ attrOut_ = indexes; }

	//! Get the attribute or property value of the index-th attribute
	double getPropertyValue(const int& index)
	{ return  attrValue_[index]; }

	//! Gets the attribute value (as a string) of the i-th property
	bool getPropertyValue (string& val, int i= 0);	

	//! Set the attribute value of the indexAttr-th attribute
	void setPropertyValue(const int& indexAttr, const double& value);
	
	//! Get the attribute name of the index-th attribute
	string getPropertyName(const int& index)
	{ return  attrRep_[index].name_; }

	//! Get the property index of attribute called "name"
	int getPropertyIndex(const string& name); 
	
	//! Operator [], get attribute value 
	double operator[] (const int& index)
	{ return  attrValue_[index]; 	}

	//! Add a property that contain the attribute representation and the attribute value
	void addProperty(const TeProperty& prop, bool attrIn = true); 

	//! Add a property from a attribute representation and a attribute value
	void addProperty(const double& val, TeAttributeRep rep = TeAttributeRep(), bool attrIn = true);

	//! Return the attributes (representation and value) that were generated as a vector of properties
	TePropertyVector getOutPropertyVector(); 

	//! Return all attributes (representation and value) as a vector of properties
	TePropertyVector getPropertyVector(); 
};

//! A class that represents a set of spatial temporal events
class TeSTEventSet
{
protected:
	map<string, TeSTEvent>		eventSet_;  // a map from event identification to event
	TeChronon					chronon_;	// chronon 
	TeTheme*					theme_;		// theme associated to this event set
	TeBox						box_;		
	
	typedef map<string, TeSTEvent>::iterator	STElemIterator;
	
public:

	//! Constructor	
	TeSTEventSet(TeTheme* theme=0): 
	  theme_(theme)
	{ }

	//! Get theme pointer
	TeTheme* theme(); 

	//! Set theme pointer
	void theme(TeTheme* t);

	//! Return the number of events
	unsigned int size();

	//! Set the chronon
	void chronon(const TeChronon& chr) { chronon_ = chr; } 

	//! Set the box
	void box(const TeBox& b) { box_ = b; } 
	
	//! Gets the value of an attribute (as string) called "attr_name"
	bool getAttributeValue (const string&  object_id, const string& attr_name, 
		                    string& val, TeTimeInterval time = TeTimeInterval()); 
		
	//! Gets the value of the attr_index-th attribute (as string) 
	bool getAttributeValue (const string& object_id, int attr_index, string& val,
							TeTimeInterval time = TeTimeInterval());
	
	//! Gets the value of the attr_index-th attribute (as double) 
	bool getAttributeValue (const string& object_id, int attr_index, double& val, long time = -1);

	//! Get the attribute index of attribute called "attrName"
	int getAttributeIndex(const string& attrName);

	//! Retuns the number of spatial temporal instances in the set
	int numSTInstance()
	{ return eventSet_.size(); }

	//! Retuns the number of elements in the set
	int numElements() 
	{ return eventSet_.size(); }
	
	//! An iterator to a set of spatial temportal events
	/*! 
		An strucuture that allows the traversal STEvent set 
		(ST elements) in a similar way as the STL iterators. 
	*/
	class iterator 
	{
		public:

		//! Constructor
		iterator(TeSTEventSet* elemSet, STElemIterator curElem) :
			curElem_(curElem),
			elemSet_(elemSet)			
			{ }

		//! Prefix move forward operator
		iterator& operator++();
				
		//! Operator that return the pointed event
		TeSTEvent& operator*()
		{	return (curElem_->second); 	}

		//! Return a pointer to the event set
		TeSTEventSet* elemSet()
		{	return elemSet_; 	}

		//! Equal operator
		bool operator==(const iterator& rhs) const
		{ return (this->curElem_== rhs.curElem_); }

		//! Unequal operator
		bool operator!=(const iterator& rhs) const
		{ return (this->curElem_!= rhs.curElem_);	}

		protected:
			STElemIterator			curElem_;
			TeSTEventSet*			elemSet_;
	};
	
	friend class TeSTEventSet::iterator;

	//! Returns an iterator to the first element of the set
	iterator begin();     
	
	//! Returns an iterator to the one past last element of the set
	iterator end();       

	//! Insert a new event in the set
	void insertSTInstance(TeSTEvent& ev); 
};

#endif







