/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <stdio.h>
#include <unistd.h>
#include <pthread.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>

#include <libsysactivity/libsysactivity.h>

void print_memory_info(struct sa_memory* memory) {
#ifdef SA_MEMORY_TOTAL
		printf("total: %"PRIu64, memory->total);
#endif
#ifdef SA_MEMORY_FREE
		printf(", free: %"PRIu64, memory->free);
#endif
#ifdef SA_MEMORY_ACTIVE
		printf(", active: %"PRIu64, memory->active);
#endif
#ifdef SA_MEMORY_INACTIVE
		printf(", inactive: %"PRIu64, memory->inactive);
#endif
#ifdef SA_MEMORY_BUFFERS
		printf(", buffers: %"PRIu64, memory->buffers);
#endif
#ifdef SA_MEMORY_SWAP_TOTAL
		printf(", swap_total: %"PRIu64, memory->swap_total);
#endif
#ifdef SA_MEMORY_SWAP_FREE
		printf(", swap_free: %"PRIu64, memory->swap_free);
#endif
#ifdef SA_MEMORY_SWAP_CACHED
		printf(", swap_cached: %"PRIu64, memory->swap_cached);
#endif
#ifdef SA_MEMORY_DIRTY
		printf(", dirty: %"PRIu64, memory->dirty);
#endif
#ifdef SA_MEMORY_CACHED
		printf(", cached: %"PRIu64, memory->cached);
#endif
#ifdef SA_MEMORY_WIRED
		printf(", wired: %"PRIu64, memory->wired);
#endif
#ifdef SA_MEMORY_EXECUTABLE
		printf(", executable: %"PRIu64, memory->executable);
#endif
#ifdef SA_MEMORY_FILES
		printf(", files: %"PRIu64, memory->files);
#endif
#ifdef SA_MEMORY_LOCKED
		printf(", locked: %"PRIu64, memory->locked);
#endif
		printf("\n");
}

void print_swap_info(struct sa_swap* swap) {
#ifdef SA_SWAP_NAME
	printf("swap name: %s", swap->name);
#endif
#ifdef SA_SWAP_TYPE
	printf(swap->type == 1 ? ", type: partition" : ", type: file");
#endif
#ifdef SA_SWAP_TOTAL
	printf(", total: %"PRIu64, swap->total);
#endif
#ifdef SA_SWAP_FREE
	printf(", free: %"PRIu64, swap->free);
#endif
	printf("\n");
}

void test_memory_info(struct sa_memory* memory) {
	if (memory->free == 0) {
		printf("\nERROR: The free memory space is zero\n");
		exit(EXIT_FAILURE);
	}
	if (memory->total == 0) {
		printf("\nERROR: The total memory space is zero\n");
		exit(EXIT_FAILURE);
	}
	if (memory->free >= memory->total) {
		printf("\nERROR: The free memory space is larger or equal than the total space\n");
		exit(EXIT_FAILURE);
	}
#ifdef SA_MEMORY_ACTIVE
	if (memory->active < 1000) {
		printf("\nERROR: Active memory is lower than 1 kb\n");
		exit(EXIT_FAILURE);
	}
#endif
#ifdef SA_MEMORY_INACTIVE
	if (memory->inactive < 1000) {
		printf("\nERROR: Inactive memory is lower than 1 kb\n");
		exit(EXIT_FAILURE);
	}
#endif
}

void test_swap_info(struct sa_swap* swap) {
	if (swap->free > swap->total) {
		printf("\nERROR: The free swap space is larger than the total space\n");
		exit(EXIT_FAILURE);
	}
#ifdef SA_SWAP_NAME
	size_t len = strnlen(swap->name, SA_SWAP_NAME);
	if (len == 0) {
		printf("\nERROR: The length of the swap name is zero\n");
		exit(EXIT_FAILURE);
	}
	if (len == SA_SWAP_NAME) {
		printf("\nERROR: The length of the swap name is equals to the maximum name length\n");
		exit(EXIT_FAILURE);
	}
#endif
#ifdef SA_SWAP_TYPE
	if (swap->type != 1 && swap->type != 2) {
		printf("\nERROR: The value of the type of swap is: %d\n", swap->type);
		exit(EXIT_FAILURE);
	}
#endif
}

void* get_memory_info(void* arg) {
	int i;
	int ret;
	struct sa_memory memory;

#ifdef SA_OPEN_MEMORY
	ret = sa_open_memory();
	if (ret != 0) {
		printf("sa_open_memory(): %s\n", strerror(ret));
		exit(EXIT_FAILURE);
	}
#endif

	struct sa_swap swap;
	ret = sa_get_swap(0, &swap);
	if (ret != 0) {
		printf("sa_get_swap(): %s\n", strerror(ret));
		exit(EXIT_FAILURE);
	}
	print_swap_info(&swap);
	printf("\n");

	uint16_t number_swaps = 0;
	ret = sa_count_swaps(&number_swaps);
	if (ret != 0) {
		printf("sa_count_swaps(): %s\n", strerror(ret));
		exit(EXIT_FAILURE);
	}
	printf("number of swaps: %d\n", number_swaps);

	struct sa_swap* swaps = (struct sa_swap*) malloc(number_swaps * sizeof (struct sa_swap));
	uint16_t written = 0;
	ret = sa_get_swaps(swaps, number_swaps, &written);
	if (ret != 0) {
		printf("sa_get_swaps(): %s\n", strerror(ret));
		exit(EXIT_FAILURE);
	}
	for (i = 0; i < written; i++) {
		printf("swap number %d\n", i);
		print_swap_info(&swaps[i]);
		printf("\n");
		test_swap_info(&swaps[i]);
	}
	free(swaps);
	printf("\n");

	for (i = 0; i < 6; i++) {
		ret = sa_get_memory(&memory);
		if (ret != 0) {
			printf("sa_get_memory(): %s\n", strerror(ret));
			exit(EXIT_FAILURE);
		}
		print_memory_info(&memory);
		printf("\n");
		test_memory_info(&memory);
		sleep(1);
	}
#ifdef SA_CLOSE_MEMORY
	ret = sa_close_memory();
	if (ret != 0) {
		printf("sa_close_memory(): %s\n", strerror(ret));
		exit(EXIT_FAILURE);
	}
#endif
#ifdef SA_OPEN_MEMORY
	ret = sa_open_memory();
	if (ret != 0) {
		printf("sa_open_memory(): %s\n", strerror(ret));
		exit(EXIT_FAILURE);
	}
#endif
#ifdef SA_CLOSE_MEMORY
	ret = sa_close_memory();
	if (ret != 0) {
		printf("sa_close_memory(): %s\n", strerror(ret));
		exit(EXIT_FAILURE);
	}
#endif
	return NULL;
}

int main() {
	pthread_t thread1;
	struct timespec delay;
	delay.tv_sec = 0;
	delay.tv_nsec = 500000000;

	pthread_create(&thread1, NULL, get_memory_info, NULL);
	nanosleep(&delay, NULL);
	get_memory_info(NULL);

	return EXIT_SUCCESS;
}
