/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <stdio.h>
#include <string.h>
#include <inttypes.h>
#include <stdlib.h>
#include <errno.h>
#include <ctype.h>

#include "libsysactivity.h"
#include "utils.h"

static void get_value(char* pos, uint64_t* value);
static void parse_swap(struct sa_swap* dst, char* pos);

extern int kernel_version;
__thread FILE* file_meminfo;
__thread FILE* file_swaps;

int sa_open_memory() {
	file_meminfo = NULL;
	file_swaps = NULL;

	if (kernel_version < 11)
		return ENOTSUP;

	file_meminfo = fopen("/proc/meminfo", "r");
	if (file_meminfo == NULL)
		return EIO;
	file_swaps = fopen("/proc/swaps", "r");
	if (file_swaps == NULL)
		return EIO;
	return 0;
}

int sa_close_memory() {
	if (file_meminfo != NULL)
		fclose(file_meminfo);
	if (file_swaps != NULL)
		fclose(file_swaps);
	return 0;
}

int sa_get_memory(struct sa_memory* dst) {
	if (dst == NULL)
		return EINVAL;

	rewind(file_meminfo);
	if (fflush(file_meminfo) != 0)
		return EIO;

	char line_buffer[64];
	int aux = 0; // for better performance
	while (fgets(line_buffer, sizeof(line_buffer), file_meminfo) != NULL) {
		if (aux < 1 && strncmp(line_buffer, "MemTotal:", sizeof("MemTotal:") - 1) == 0)
			get_value(&line_buffer[sizeof("MemTotal:") - 1], &dst->total);
		else if (aux < 2 && strncmp(line_buffer, "MemFree:", sizeof("MemFree:") - 1) == 0)
			get_value(&line_buffer[sizeof("MemFree:") - 1], &dst->free);
		else if (aux < 3 && strncmp(line_buffer, "Buffers:", sizeof("Buffers:") - 1) == 0)
			get_value(&line_buffer[sizeof("Buffers:") - 1], &dst->buffers);
		else if (aux < 4 && strncmp(line_buffer, "Cached:", sizeof("Cached:") - 1) == 0)
			get_value(&line_buffer[sizeof("Cached:") - 1], &dst->cached);
		else if (aux < 5 && strncmp(line_buffer, "SwapCached:", sizeof("SwapCached:") - 1) == 0)
			get_value(&line_buffer[sizeof("SwapCached:") - 1], &dst->swap_cached);
		else if (aux < 6 && strncmp(line_buffer, "Active:", sizeof("Active:") - 1) == 0)
			get_value(&line_buffer[sizeof("Active:") - 1], &dst->active);
		else if (aux < 7 && strncmp(line_buffer, "Inactive:", sizeof("Inactive:") - 1) == 0)
			get_value(&line_buffer[sizeof("Inactive:") - 1], &dst->inactive);
		else if (aux < 8 && strncmp(line_buffer, "SwapTotal:", sizeof("SwapTotal:") - 1) == 0)
			get_value(&line_buffer[sizeof("SwapTotal:") - 1], &dst->swap_total);
		else if (aux < 9 && strncmp(line_buffer, "SwapFree:", sizeof("SwapFree:") - 1) == 0)
			get_value(&line_buffer[sizeof("SwapFree:") - 1], &dst->swap_free);
		else if (aux < 10 && strncmp(line_buffer, "Dirty:", sizeof("Dirty:") - 1) == 0) {
			get_value(&line_buffer[sizeof("Dirty:") - 1], &dst->dirty);
			break;
		} else
			continue;
		aux++;
	}

	return 0;
}

int sa_count_swaps(uint16_t* number) {
	if (number == NULL)
		return EINVAL;

	rewind(file_swaps);
	if (fflush(file_swaps) != 0)
		return EIO;

	char line_buffer[512];
	fgets(line_buffer, sizeof(line_buffer), file_swaps);

	*number = 0;
	while (fgets(line_buffer, sizeof(line_buffer), file_swaps) != NULL) {
		(*number)++;
	}
	return 0;
}

int sa_get_swap(uint16_t index, struct sa_swap* dst) {
	if (dst == NULL)
		return EINVAL;

	rewind(file_swaps);
	if (fflush(file_swaps) != 0)
		return EIO;

	char line_buffer[512];
	fgets(line_buffer, sizeof(line_buffer), file_swaps);

	int i;
	for (i = 0; fgets(line_buffer, sizeof(line_buffer), file_swaps) != NULL; i++) {
		if (i != index)
			continue;

		errno = 0;
		parse_swap(dst, line_buffer);
		if (errno != 0)
			return ENOSYS;

		return 0;
	}

	return ENODEV;
}

int sa_get_swaps(struct sa_swap* dst, uint16_t dst_size, uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	rewind(file_swaps);
	if (fflush(file_swaps) != 0)
		return EIO;

	char line_buffer[512];
	fgets(line_buffer, sizeof(line_buffer), file_swaps);

	*written = 0;
	int i;
	for (i = 0; i < dst_size; i++) {
		if (fgets(line_buffer, sizeof line_buffer, file_swaps) == NULL)
			return EIO;

		errno = 0;
		parse_swap(dst, line_buffer);
		if (errno != 0)
			return ENOSYS;

		(*written)++;
	}

	return 0;
}

static void get_value(char* pos, uint64_t* value) {
	while (isblank(*pos))
		pos++;

	*value = strtoull(pos, NULL, 10);
	*value *= 1024;
}

static void parse_swap(struct sa_swap* dst, char* pos) {
	int end = strchr(pos, ' ') - pos;
	if (sizeof(dst->name) - 1 < end)
		end = sizeof(dst->name) - 1;
	strlcpy(dst->name, pos, end);
	dst->name[end] = '\0';
	pos = skip_value(pos);
	dst->type = strncmp(pos, "file", 4) == 0 ? 2 : 1;
	pos = skip_value(pos);
	dst->total = strtoull(pos, NULL, 10) * 1024;
	pos = skip_value(pos);
	dst->free = dst->total - strtoull(pos, NULL, 10) * 1024;
}
