/*
    LIBSNSP - A C Library for the Simple Network Scanning Protocol
    Copyright (C) 2001 Michael R. Kllejan 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* $Id: report.c,v 1.3 2001/11/04 14:47:02 michael Exp $ */

#include <stdlib.h>
#include "report.h"
#include "const.h"

#define MAX_XMLDOCSTR_LEN 9000

snsp_report_t*
snsp_report_alloc () {
  snsp_report_t* report;

  if (!(report = (snsp_report_t*)malloc(sizeof(snsp_report_t))))
    return NULL;

  report->hosts = LINKED_LIST_ALLOC;

  return report;
};

void
snsp_report_free ( snsp_report_t **report ) {
  if ((!report) || (!*report))
    return;

  do {
    if (!((*report)->hosts))
      return;

    snsp_report_host_free ( &(*report)->hosts->data );

  } while ( LINKED_LIST_NEXT( &(*report)->hosts) );
  
  LINKED_LIST_FREE ( &(*report)->hosts );
  free ( *report ); 
  *report = NULL;
};

snsp_report_port_t*
snsp_report_port_alloc () {
  snsp_report_port_t* port;

  if ((port = (snsp_report_port_t*)
       malloc(sizeof(snsp_report_port_t))) == NULL)
    return NULL;

  port->portlist = LINKED_LIST_ALLOC;

  return port;
};

void
snsp_report_port_free ( snsp_report_port_t **port ) {

  if ((!port) || (!*port))
    return;

  do {
    if ((*port)->portlist)
      return;

    free ( (*port)->portlist->data );

  } while ( LINKED_LIST_NEXT ( &(*port)->portlist ) );

  LINKED_LIST_FREE ( &(*port)->portlist );
  free ( *port ); 
  *port = NULL;
};

snsp_report_host_t*
snsp_report_host_alloc () {
  snsp_report_host_t* host;

  if ((host = (snsp_report_host_t*)malloc(sizeof(snsp_report_host_t))) == NULL)
    return NULL;

  host->ports = LINKED_LIST_ALLOC;

  return host;
};

void
snsp_report_host_free ( snsp_report_host_t **host ) {

  if ((!host) || (!*host))
    return;

  free ((*host)->address);
  free ((*host)->scantype);

  do {
    if ((*host)->ports)
      return;

    snsp_report_port_free ( &(*host)->ports->data );

  } while ( LINKED_LIST_NEXT ( &(*host)->ports ) );

  LINKED_LIST_FREE ( &(*host)->ports );
  free ( *host ); 
  *host = NULL;
};

char*
snsp_report_render ( snsp_report_t *report ) {
  xmlDocPtr doc;
  xmlNodePtr node, subnode;
  xmlChar *xmldocstr = (xmlChar*)malloc(MAX_XMLDOCSTR_LEN);
  int size = MAX_XMLDOCSTR_LEN;
  char *curportrangestr = malloc(SNSP_PORTRANGE_STR_MAX+1);
  char *curportliststr = malloc(SNSP_PORTLIST_STR_MAX+1);
  
  doc = xmlNewDoc("1.0");

  doc->children = xmlNewDocNode ( doc, NULL, "SNSP-Report", NULL );
  xmlSetProp(doc->children, "version", "0.1" );

  do {
    if (!report->hosts)
      return NULL;

    node = xmlNewChild(doc->children, NULL, "Host", NULL );
    xmlSetProp(node, "address", report->hosts->data->address );
    xmlSetProp(node, "scantype", report->hosts->data->scantype);

    do {
      if (!(report->hosts->data->ports))
        return NULL;

      if (!(report->hosts->data->ports->data))
        break;

      curportliststr[0] = '\0';

      do {
        if (!report->hosts->data->ports->data->portlist ||
            !report->hosts->data->ports->data->portlist->data)
          break;
      
        if ((report->hosts->data->ports->data->portlist->data->bport) == 
            (report->hosts->data->ports->data->portlist->data->eport))
          sprintf(curportrangestr, "%i", 
            report->hosts->data->ports->data->portlist->data->bport );
        else
          sprintf(curportrangestr, "%i-%i", 
            report->hosts->data->ports->data->portlist->data->bport,
            report->hosts->data->ports->data->portlist->data->eport );

        if ( strlen(curportliststr) != 0 )
          strcat ( curportliststr, "," );
        strcat ( curportliststr, curportrangestr );

      } while ( LINKED_LIST_NEXT ( 
                &report->hosts->data->ports->data->portlist ));
      
      subnode = xmlNewChild(node, NULL, "Ports", curportliststr );
      xmlSetProp(subnode, "status", "open"); 
    } while ( LINKED_LIST_NEXT ( &report->hosts->data->ports ));
  } while ( LINKED_LIST_NEXT ( &report->hosts ) );

  xmlDocDumpMemory ( doc, &xmldocstr, &size );

  free(curportrangestr);
  free(curportliststr);

  return (char*)xmldocstr;
};

void*
snsp_report_shrink ( snsp_report_t *report ) {
  snsp_report_port_t *first = NULL;
  snsp_report_port_status_t status = snsp_report_port_status_open;
  snsp_portrange_t *curportrange = NULL;
        
  if ((!report) || (!report->hosts)) return NULL;

  do {
    if (!report->hosts->data) return NULL;

    do {
      if ((!report->hosts->data->ports) ||
          (!report->hosts->data->ports->data)) return NULL;

      if (report->hosts->data->ports->data->status == status) 
        if (!first)
          first = report->hosts->data->ports->data;
        else {
          while ( (curportrange = LINKED_LIST_POP ( 
                  &report->hosts->data->ports->data->portlist )) != NULL ) {
            LINKED_LIST_PUSH ( first->portlist, curportrange, sizeof(snsp_portrange_t));
          }
          LINKED_LIST_DEL ( &report->hosts->data->ports );
        };
    } while ( LINKED_LIST_NEXT ( &report->hosts->data->ports ));
  } while ( LINKED_LIST_NEXT ( &report->hosts ));
       
  return report;
};

#define XML_WALK_TO_ELEMENT(node,toname) \
  while ( xmlStrcmp ( node->name, toname ) ) { \
    node = node->next; \
    if ( node == NULL ) \
      return NULL; \
  };

#define XML_WALK_TO_CHILD(node) \
  node = node->xmlChildrenNode; \
  if (node == NULL) \
    return NULL;

#define XML_IGNORE_WS(node) \
  if (node) \
    while ( xmlIsBlankNode ( node ) ) { \
      node = node->next; \
    }

#define SNSP_REPORT_VERSION "0.1"

snsp_report_t*
snsp_report_parse (char *report_str)
{
  snsp_report_t* report = snsp_report_alloc(); 
  xmlDocPtr doc;
  xmlNodePtr cur1, cur2, cur3;
  xmlAttrPtr attr1, attr2;
  xmlChar *status_str;

  snsp_report_host_t *curhost = NULL;
  snsp_report_port_t *curport = NULL;

  doc = xmlParseMemory (report_str, strlen(report_str));
  cur1 = xmlDocGetRootElement (doc);

  if (!doc) return NULL;
 
  if (xmlStrcmp(cur1->name, (xmlChar*) "SNSP-Report"))
    return NULL;

  attr1 = cur1->properties;
  XML_WALK_TO_ELEMENT (attr1,"version");

  if (xmlStrcmp(attr1->children->content, (xmlChar*)
    SNSP_REPORT_VERSION))
    return NULL;

  cur2 = cur1;

  XML_WALK_TO_CHILD(cur2);
  XML_WALK_TO_ELEMENT (cur2,"Host");

  while (cur2 && !xmlStrcmp(cur2->name, ((xmlChar*) "Host" )))
  {
    if(cur2 == NULL)
      return report;

    attr1 = cur2->properties;
    XML_WALK_TO_ELEMENT (attr1, "address");

    attr2 = cur2->properties;
    XML_WALK_TO_ELEMENT (attr2, "scantype");

    curhost = snsp_report_host_alloc();  
    curhost->address = xmlStrdup(attr1->children->content);
    curhost->scantype = xmlStrdup(attr2->children->content);
    curhost->ports = LINKED_LIST_ALLOC;

    cur3 = cur2;
    XML_WALK_TO_CHILD(cur3);

    while (cur3 && (!xmlStrcmp(cur3->name, "Ports")))
    {
      curport = snsp_report_port_alloc();

      attr1 = cur3->properties;
      XML_WALK_TO_ELEMENT (attr1, "status");

      if (!(status_str = xmlStrdup(attr1->children->content)))
        return NULL;

      if (!strcmp(status_str, "open"))
        curport->status = snsp_report_port_status_open; 
      if (!strcmp(status_str, "closed"))
        curport->status = snsp_report_port_status_closed; 
      if (!strcmp(status_str, "notscanned"))
        curport->status = snsp_report_port_status_notscanned; 
      if (!strcmp(status_str, "filtered"))
        curport->status = snsp_report_port_status_filtered; 

      LINKED_LIST_PUSH(curhost->ports, curport,
        sizeof(snsp_report_port_t)); 

      free(curport);

      cur3 = cur3->next;
      XML_IGNORE_WS(cur3);
    }

    LINKED_LIST_PUSH(report->hosts, curhost,
      sizeof(snsp_report_host_t));
    free (curhost);

    cur2 = cur2->next;
    XML_IGNORE_WS(cur2);
  }

  return report;
}
