/* "$Id: fastimho.c,v 1.8 2001/09/19 19:52:59 stewa Exp $" */
/*

  FastIMHO - compiled Pike module for the IMHO webmail
  This module contains functions which are slowly performed in Pike,
  such as parsing.
  
  Copyright  
  Stefan Wallstrm <stewa@lysator.liu.se>
  & Bosse Lincoln <lincoln@lysator.liu.se> 1998-1999
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  See the file "LICENSE" for details.

*/

#include <global.h>
// PIKE_DEBUG uses functions which don't exist if Roxen is compiled 
// without PIKE_DEBUG. 
#undef PIKE_DEBUG
#include <interpret.h>
#include <svalue.h>
#include <array.h>
#include <object.h>
#include <pike_macros.h>
#include <threads.h>
#include <mapping.h>
#include <array.h>
#include <module_support.h>
#include <stralloc.h>


#define PT_MAPPING 0
#define PT_LIST 1
#define PT_QSTR 2
#define PT_TOKENS 3

/*
 *  array parseimap(string data, int parsetype, int tokens, array state)
 *
 */
static void f_parseimap( INT32 args )
{
  int len0,len = 0;
  char *in;
  int strpos = 0;
  struct svalue value;
  struct array *intstack;
  int startpos = 0, rp, wp;
  struct pike_string *ps, *key = 0;
  int slash = 0;
  int integer = 0;
  int parsetype = PT_MAPPING, parsetokens = 0;
  int leaf = 0;
  int done = 0;
  int literalremain = 0;
  struct array *return_state;
  struct string_builder sb;

  check_all_args("parseimap",args,BIT_STRING,BIT_INT,BIT_INT,BIT_ARRAY|BIT_INT,0);

  /* Get the arguments from the pike stack. */
  if (sp[-args].u.string->size_shift > 0)
    Pike_error("parseimap(): Only 8-bit strings allowed.\n");
  if (sp[3-args].type == T_ARRAY && sp[3-args].u.array->size != 6)
    Pike_error("parseimap(): State array must contain 6 elements.\n");
  //FIXME: check types in state array
  in = STR0(sp[-args].u.string);
  len0 = len = sp[-args].u.string->len;
  while (--len > 0 && in[len] != '\n');
  if (len > 0)
    len++;
  parsetype = sp[-args+1].u.integer;
  parsetokens = sp[-args+2].u.integer;
  if (return_state = sp[-args+3].u.array) {
    array_index_no_free(&value, return_state, 3);
    intstack = value.u.array;
    array_index_no_free(&value, return_state, 4);
    key = value.u.string;
    if (!key->len) {
      free_string(key);
      key = 0;
    }
    array_index_no_free(&value, return_state, 5);
    literalremain = value.u.integer;
    array_index_no_free(&value, return_state, 1);
  } else {
    intstack = allocate_array_no_init(0,10);
    if (parsetype == PT_TOKENS) {
      value.u.array = allocate_array_no_init(0,10);
      value.type = T_ARRAY;
      intstack = append_array(intstack, &value);
      free_svalue(&value);
    }
    value.type = T_DELETED;
  }
    
  while (strpos < len && !done) { 
    leaf = 0;
    if (literalremain) {
      struct pike_string *s;
      if (literalremain <= len) {
	s = add_and_free_shared_strings(value.u.string, 
					make_shared_binary_string(in, literalremain));
	value.u.string = s;
	value.type = T_STRING;
	strpos += literalremain;
	literalremain = 0;
	leaf = 1;
      } else {
	s = add_and_free_shared_strings(value.u.string, 
					make_shared_binary_string(in, len));
	value.u.string = s;
	value.type = T_STRING;
	strpos += len;
	literalremain -= len;
      }
    } else {
      
      // Remove blanks
      while(strpos < len && (in[strpos] == ' ' || in[strpos] == '\t' || 
			     in[strpos] == '\r' || in[strpos] == '\n'))
	strpos++;

      startpos = strpos;
      if (strpos < len) {
	
	switch(in[strpos]) {
	case '"': // Quoted string
	  strpos++;
	  init_string_builder(&sb, 0);
	  startpos++;
	  for (slash = rp = 0; startpos+rp < len && !(!slash && 
		 (in[startpos+rp] == '\"')); rp++) {
	    if (in[startpos+rp] != '\\' || slash) {
	      string_builder_putchar(&sb, in[startpos+rp]);
	      slash = 0;
	    } else
	      slash = 1;
	  }
	  if (startpos+rp < len)
	    strpos+=rp+1;
 	  value.u.string = finish_string_builder(&sb);
	  value.type = T_STRING;
	  if (parsetype == PT_QSTR)
	    done = 1;
	  else
	    leaf = 1;
	  break;
	  
	  
	case '(': // List or mapping depending on parsetype and depth
	  if (intstack->size == 0 && parsetype == PT_MAPPING) {
	    value.u.mapping = allocate_mapping(0);
	    value.type = T_MAPPING;
	  } else {
	    value.u.array = allocate_array_no_init(0,10);
	    value.type = T_ARRAY;
	  }
	  intstack = append_array(intstack,&value);
	  free_svalue(&value);
	  strpos++;
	  break;
	  
	case ')': // End of list or mapping
	  if (!intstack->size) {
	    error("parseimap(): Unbalanced paranthesis.\n");
	  }
	  array_index_no_free(&value, intstack, intstack->size-1);
	  intstack = array_remove(intstack, intstack->size-1);
	  strpos++;
	  leaf = 1;
	  break;	
	  
	case '{': // Literal, can be broken in different strings
	  strpos++;
	  integer = 0;
	  while (strpos < len && in[strpos] >= '0' && in[strpos] <= '9') {
	    integer = integer * 10 + ((int) in[strpos]-'0');
	    strpos++;
	  }
	  while (strpos < len && in[strpos] != '\n')
	    strpos++;
	  if (strpos < len)
	    strpos++;
	  literalremain = (integer>len-strpos?len-strpos:integer);
	  ps = make_shared_binary_string(in+strpos, literalremain);
	  strpos+=literalremain;
	  literalremain = integer-literalremain;
	  value.u.string = ps;
	  value.type = T_STRING;
	  if (!literalremain)
	    leaf = 1;
	  break;
	  
	default:
	  while(strpos < len && in[strpos] != ' ' &&
		in[strpos] != '\r' &&
		in[strpos] != '\n' &&
		in[strpos] != '\t' &&
		in[strpos] != ')')
	    strpos++;
	  if ((strpos-startpos == 3) && 
	      (strncmp("NIL", in+startpos, 3) == 0)) {
	    value.u.integer = 0;
	    value.type = T_INT;
	  } else {
	    ps = make_shared_binary_string(in+startpos, 
					   strpos-startpos);
	    value.u.string = ps;
	    value.type = T_STRING;
	  }
	  leaf = 1;
	}
      }
    }
    if (leaf) {
      if (intstack->size == 0)
	done = 1;
      else {
	if (intstack->item[intstack->size-1].type == T_ARRAY) {
	  intstack->item[intstack->size-1].u.array  = 
	    append_array(intstack->item[intstack->size-1].u.array,
			 &value);
	  free_svalue(&value);
	  value.type = T_DELETED;
	} else if (intstack->item[intstack->size-1].type == T_MAPPING) {
	  if (key) {
	    mapping_string_insert(intstack->item[intstack->size-1].u.mapping,
				  key, &value);
	    free_string(key);
	    free_svalue(&value);
	    value.type = T_DELETED;
	    key = 0;
	  } else {
	    if (value.type == T_STRING)
	      key = value.u.string;
	    else {
	      error("parseimap(): Mapping key is not a string (%d).\n", 
		    value.type);
	    }
	  }
	}
      }
    }
    if (parsetype == PT_TOKENS && 
	intstack->item->u.array->size >= parsetokens) {
      array_index_no_free(&value, intstack, 0);
      intstack = array_remove(intstack, intstack->size-1);

      done = 1;
    }
  }
  return_state = allocate_array(6);
  if (value.type == T_DELETED) { // == No data in in value
    value.u.integer = 0;
    value.type = T_INT;
  }
  array_set_index(return_state, 1, &value); // Value
  free_svalue(&value);
  value.u.integer = done;
  value.type = T_INT;
  array_set_index(return_state, 0, &value); // Done?
  free_svalue(&value);
  value.u.string = make_shared_binary_string(in+strpos, len0-strpos);
  value.type = T_STRING;
  array_set_index(return_state, 2, &value); // Remaining data
  free_svalue(&value);
  value.u.array = intstack;
  value.type = T_ARRAY;
  array_set_index(return_state, 3, &value); // Stack
  free_svalue(&value);
  if (key) {
    value.u.string = key;
    value.type = T_STRING;
  } else {
    value.u.string = make_shared_binary_string(0, 0);
    value.type = T_STRING;
  }
  array_set_index(return_state, 4, &value); // Key
  free_svalue(&value);
  value.u.integer = literalremain;
  value.type = T_INT;
  array_set_index(return_state, 5, &value); // Remaining octets in literal
  free_svalue(&value);

  pop_n_elems( args );
  value.u.array = return_state;
  value.type = T_ARRAY;
  push_svalue(&value); 
  free_svalue(&value);
}

void pike_module_init()
{
  add_function( "parseimap", f_parseimap, "function(string,int,int,array:array)", 0);
}

void pike_module_exit()
{
}




