/* Copyright (c) 2000 Nick Moffitt
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public Licence
 * as published by the Free Software Foundation; either version 2
 * of the Licence, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRENTY; without even the implied warrenty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public Licence in the COPYING file for more
 * details.
 *
 * You should have received a copy of the GNU Library General 
 * Public License along with the GNU C Library; see the file 
 * COPYING.LIB.  If not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
 */

/*================================================================
================================================================*/
#include <glib.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>

#include <printsys.h>

#include "config.h"

void get_printcap_info(LPS_System_t *sys,wchar_t *qname, wchar_t *oname, 
			int verbose);
void enum_printers(LPS_System_t *sys,const wchar_t **printerlist);
void get_queue_from_spooldir(LPS_System_t *sys, const char *spooldir, 
			     int verbose);
void version(void);
void usage(void);


/*----------------------------------------------------------------
  Send printcap information to standard output. If an option name
  (oname) is specified, just print that option, otherwise, print
  all options for the specified queue name (qname).
----------------------------------------------------------------*/
void get_printcap_info(LPS_System_t *sys,wchar_t *qname, wchar_t *oname, 
		       int verbose){
  LPS_Printer_t *target_printer = lps_get_printer(sys,qname,NULL);
  wchar_t *value_str = NULL;
  LPS_Pair_t *cur = NULL;

  /* See if the requested printer exists. */
  if (!target_printer) {
    fprintf(stderr, "Could not find requested printer: %S\n", qname);
    exit(1);
  }

  if (verbose)
    printf("QUEUE=\"%S\";", qname);

  if (oname) {
    /* Print just the requested option name. */
    value_str = lps_pr_lookup_field(target_printer, oname);
    if (value_str == NULL)
      printf("%S=\"\"\n", oname);
    else
      printf("%S=\"%S\"\n", oname, value_str);
  } else {
    /* Print all the option names. */
    for (cur = target_printer->fields; cur != NULL && cur->key != NULL; 
	 cur++) {
      printf("%S=\"%S\";", cur->key, cur->value);
    }
    printf("\n");
  }

}

/*----------------------------------------------------------------
----------------------------------------------------------------*/
void enum_printers(LPS_System_t *sys,const wchar_t **printerlist)
{
  int i = 0;
  LPS_Printer_t *my_printer;

  for (i = 0; printerlist[i] != NULL; i++) {
    my_printer = lps_get_printer(sys,printerlist[i],NULL);
    if (my_printer) {
      get_printcap_info(sys,my_printer->names[0], NULL, 1);
    } else {
      g_error("Could not retrieve printer named '%S'.\n", printerlist[i]);
      exit(1);
    }
  }
  return;
}

/*----------------------------------------------------------------
  Do a reverse lookup on spooldir, print queue name that it 
  is associated with.
----------------------------------------------------------------*/
void get_queue_from_spooldir(LPS_System_t *sys, const char *spooldir, 
			     int verbose __attribute__ ((unused)) )
  /* No idea what the verbose parameter is used for -- ben */
{
  int i = 0, n = 0;
  const wchar_t **plist = NULL;
  wchar_t *key = lps_promote("sd");
  char *curr_sd;
  LPS_Printer_t *curr_prn;

  plist = lps_get_printer_list(sys,NULL);
 
  for (i = 0; plist[i] != NULL; i++) {
    curr_prn = lps_get_printer(sys,plist[i],NULL);
    curr_sd = lps_demote(lps_pair_lookup(curr_prn->fields, key));
    n = strlen(curr_sd);

    /* kill any trailing '/' here (scripts don't have it) */
    /* hack off the crap that  keeps getting left on the end of 
       this */
    // printf("currsd=%s,len=%d\n",curr_sd,n);
    while (curr_sd[n-1] == '/')
      curr_sd[--n] = '\0';

    if (strcmp(spooldir, curr_sd) == 0)
    {
      printf("queue=\"%S\"\n", curr_prn->names[0]);
      free(curr_sd);
      return;
    }
  }
  /* give the empty queue if we don't find a match */
  printf("queue=\"\"\n");
  free(curr_sd);
}

/*----------------------------------------------------------------
  Print version information to statndard output.
----------------------------------------------------------------*/
void version(void){
  printf("\npcap version %s\n",VERSION);
}

/*----------------------------------------------------------------
  Print usage information to standard output.
----------------------------------------------------------------*/
void usage(void){
  version();

  printf("\nReads entries from /etc/printcap, and writes them to\n"
	 "standard output.\n\n" "Usage:\n\n"
	 "pcap -P queue_name[:option_name]\n" 
	 "pcap -S spooldir\n" "pcap -v\n" "pcap -?\n" "\n");
  return;
}

/*----------------------------------------------------------------
  Main routine.
----------------------------------------------------------------*/
int main(int argc, char *argv[])
{
  const wchar_t **plist = NULL;
  wchar_t *queue_name = NULL;
  wchar_t *option_name = NULL;
  wchar_t *cur = NULL;
  char *spooldir = NULL;
  int curarg;
  int verbose = 0;
  LPS_System_t *sys=lps_init(NULL);

  static struct option long_options[] = {
    {"printer", required_argument, NULL, 'P'},
    {"version", no_argument, NULL, 'V'},
    {"verbose", no_argument, NULL, 'v'},
    {"help", no_argument, NULL, '?'},
    {"spooldir", required_argument, NULL, 'S'},
    {0, 0, 0, 0}
  };

  /* Process command line arguments. */
  while ((curarg = getopt_long(argc, argv, "+?P:vS:", long_options, NULL)) 
	 != EOF) {
    switch (curarg) {
    case '?':
      usage();
      return 1;
      break;
    case 'V':
      version();
      printf("\n");
      return 1;
      break;
    case 'v':
      verbose = 1;
      break;
    case 'P':
      queue_name=lps_promote(optarg);
      if ((cur = wcschr(queue_name, ':'))) {
	if (queue_name == NULL) {
	  fprintf(stderr, "Unable to allocate memory for queue_name.\n");
	  exit(1);
	}
	queue_name[cur - queue_name] = 0;
	option_name = (wchar_t *)malloc(sizeof(wchar_t)*(wcslen(cur + 1)+1));
	if (option_name == NULL) {
	  fprintf(stderr, "Unable to allocate memory for option_name.\n");
	  exit(1);
	}
	wcscpy(option_name,cur+1);
      }
      break;
    case 'S':
      spooldir=optarg;
      break;
    }
  }

  /* Get all the printcap information. */

  if (queue_name)
    get_printcap_info(sys,queue_name, option_name, verbose);
  else if (spooldir) {
    int len;
    // printf("spooldir=%s\tlen=%d\n",spooldir,strlen(spooldir));
    while( spooldir[(len=strlen(spooldir)-1)]=='/')
      spooldir[len]=0;
    get_queue_from_spooldir(sys, spooldir, verbose);
  }
  else {
    plist = lps_get_printer_list(sys,NULL);
    enum_printers(sys,plist);
  }

  /* Clean up. */
  lps_end(sys);
  return 0;
}
