/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "libpolyxmass-chemgroup.h"




/* NEW'ING FUNCTIONS, DUPLICATING FUNCTIONS ...
 */
PxmChemgroup *
libpolyxmass_chemgroup_new (void)
{
  PxmChemgroup *chemgroup = g_malloc0 (sizeof (PxmChemgroup));
  
  chemgroup->pka = -1;
  
  chemgroup->cgrGPA = g_ptr_array_new ();

  return chemgroup;
}


PxmChemgroupRule *
libpolyxmass_chemgrouprule_new (void)
{
  PxmChemgroupRule *cgr = g_malloc0 (sizeof (PxmChemgroupRule));
  
  return cgr;
}


PxmProp *
libpolyxmass_chemgroup_prop_new (PxmChemgroup *chemgroup)
{
  PxmProp *prop = libpolyxmass_prop_new ();
  
  PxmChemgroup *chemgroup_new = NULL;

  if (chemgroup == NULL)
    chemgroup_new = libpolyxmass_chemgroup_new ();
  else
    chemgroup_new = chemgroup;
    
  libpolyxmass_prop_set_name (prop, "CHEMGROUP");
  
  prop->data = (gpointer) chemgroup_new;
  
  prop->custom_free = libpolyxmass_chemgroup_prop_free;
  
  return prop;
}




/* DATA MODIFYING FUNCTIONS
 */
gchar *
libpolyxmass_chemgroup_set_name (PxmChemgroup *chemgroup, gchar *name)
{
  g_assert (chemgroup != NULL);
  
  if (chemgroup->name != NULL)
    g_free (chemgroup->name);
  
  chemgroup->name = g_strdup (name);
  
  return chemgroup->name;
}
  

gchar *
libpolyxmass_chemgroup_set_userdata (PxmChemgroup *chemgroup, 
				     gchar *userdata)
{
  g_assert (chemgroup != NULL);
  
  if (chemgroup->userdata != NULL)
    g_free (chemgroup->userdata);
  
  chemgroup->userdata = g_strdup (userdata);
  
  return chemgroup->userdata;
}

  
gchar *
libpolyxmass_chemgrouprule_set_entity (PxmChemgroupRule *cgr, gchar *entity)
{
  g_assert (cgr != NULL);
  
  if (cgr->entity != NULL)
    g_free (cgr->entity);
  
  cgr->entity = g_strdup (entity);
  
  return cgr->entity;
}
  

gchar *
libpolyxmass_chemgrouprule_set_name (PxmChemgroupRule *cgr, gchar *name)
{
  g_assert (cgr != NULL);
  
  if (cgr->name != NULL)
    g_free (cgr->name);
  
  cgr->name = g_strdup (name);
  
  return cgr->name;
}
  


/* LOCATING FUNCTIONS
 */
PxmChemgroupRule *
libpolyxmass_chemgroup_get_chemgrouprule_ptr (PxmChemgroup *chemgroup,
					      gchar *entity,
					      gchar *name)
{
  gint iter = 0;
  
  PxmChemgroupRule *cgr = NULL;
  
  g_assert (chemgroup != NULL);
  g_assert (entity != NULL || name != NULL);
  
  
  for (iter = 0; iter < chemgroup->cgrGPA->len ; iter++)
    {
      cgr = g_ptr_array_index (chemgroup->cgrGPA, iter);
      g_assert (cgr != NULL);
      
      if (entity != NULL)
	{
	  if (0 == strcmp (entity, cgr->entity))
	    {
	      if (name != NULL)
		{
		  if (0 == strcmp (name, cgr->name))
		    return cgr;
		  else
		    continue;
		}
	      else
		return cgr;
	    }
	  else
	    continue;
	}
      
      if (name != NULL)
	{
	  if (0 == strcmp (name, cgr->name))
	    {
	      if (entity != NULL)
		{
		  if (0 == strcmp (entity, cgr->entity))
		    return cgr;
		  else
		    continue;
		}
	      else
		return cgr;
	    }
	  else
	    continue;
	}
    }
  
  return NULL;
}
  



  
/* Returns the number of PxmChemgroup instances that were found 
   containing PxmChemgroupRule objects of entity 'entity'.
*/
gint
libpolyxmass_chemgroup_ensure_single_containing_entity (PxmMonomer *monomer,
							gchar *entity)
{
  gint iter = 0;
  gint jter = 0;
  gint count = 0;
  gint found_chemgroup_index = -1;
  
  PxmProp *prop = NULL;
  
  PxmChemgroup *chemgroup = NULL;
  PxmChemgroupRule *cgr = NULL;



  /* We get the pointer to a monomer, and we are asked to ensure that
     in that monomer->propGPA array, there is at most one property of
     name "CHEMGROUP" that lists (in its PxmChemgroup data member)
     PxmChemgroupRule instances of entity 'entity'.
  */
  g_assert (entity != NULL);
  
  g_assert (monomer != NULL);
  if (monomer->propGPA->len <= 0)
    return 0;
  

  for (iter = 0; iter < monomer->propGPA->len; iter++)
    {
      prop = g_ptr_array_index (monomer->propGPA, iter);
      g_assert (prop != NULL);
      
      if (0 == strcmp ("CHEMGROUP", prop->name))
	chemgroup = (PxmChemgroup *) prop->data;
      else
	continue;
      
      /* OK, we are iterating in a chemgroup instance. Let's see if we
	 find a PxmChemgroupRule instance there. Since we might remove
	 instances, we iterate in the chemgroup->cgrGPA array in the
	 reverse order.
      */
      for (jter = chemgroup->cgrGPA->len - 1; jter >= 0 ; jter--)
	{
	  cgr = g_ptr_array_index (chemgroup->cgrGPA, jter);
	  g_assert (cgr != NULL);
	  
	  if (0 == strcmp (cgr->entity, entity))
	    {
	      /* The currently iterated PxmChemgroupRule instance 
		 is of the same entity as the one passed as parameter.
	      
		 This means that, if another chemgrouprule in another
		 chemgroup had been found already having that entity,
		 we should remove it.

		 Otherwise, we just acknowledge that we found the
		 first chemgroup sporting a chemgrouprule of the same
		 entity as the one passed as parameter. And we
		 continue. To next chemgroup.
	      */
	      if (found_chemgroup_index != -1)
		{
		  /*
		    One chemgrouprule of entity 'entity' (parameter)
		    had already been found in the chemgroup at index
		    'found_chemgroup_index' of monomer->monomerGPA, so
		    we have to remove the one we just came across and
		    free it.
		    
		  */
		  g_assert (TRUE == g_ptr_array_remove (chemgroup->cgrGPA,
							cgr));

		  debug_printf (("removing chemgrouprule of entity '%s' "
				 "at index '%d' of cgrGPA for chemgroup "
				 "at index %d",cgr->entity, jter, iter));
		  
		  libpolyxmass_chemgrouprule_free (cgr);

		  count++;
		  
		  /* If there are other chemgrouprule objects of the
		     same 'entity', we have to remove them also, so we
		     continue iterating in this cgrGPA array of
		     chemgrouprule instances.
		  */
		  continue;
		}
	      else
		{
		  /* We found the first chemgrouprule ever in this
		     monomer, and it happens to be in the chemgroup
		     object at index iter of the array
		     monomer->monomerGPA. So we store that index and
		     we break, so that we can deal with the other
		     chemgroup instances in that array in order to
		     remove doublons (duplicates).
		  */
		  found_chemgroup_index = iter;
		  
		  break;
		}
	    }
	  /* End of 
	     if (0 == strcmp (cgr->entity, entity))
	  */
	}
      /* End of
	 for (jter = chemgroup->cgrGPA->len - 1; jter >= 0 ; jter--)
      */
    }
  /* End of 
     for (iter = 0; iter < monomer->monomerGPA->len; iter++) 
  */
  
  return count;
}




/* RENDERING FUNCTIONS - FORMATTER FUNCTIONS
 */

PxmChemgroupRule *
libpolyxmass_chemgroup_render_xml_node_chemgrouprule (xmlDocPtr xml_doc,
						      xmlNodePtr xml_node,
						      gpointer data)
{
  /* The DTD for the "chemgroup" node says this:

  <!ELEMENT chemgrouprule (entity,name,outcome)>
  
  <!ELEMENT chemgroup ((pka,acidcharged)?,polrule*,chemgrouprule*)>

  Here is an example of a chemgroup elements:

      <chemgroup>
	<pka>9.6</pka>
	<acidcharged>TRUE</acidcharged>
	<polrule>left_trapped</polrule>
	<chemgrouprule>
	  <entity>LE_PLM_MODIF</entity>
	  <name>LE_Acetylation</name>
	  <outcome>LOST</outcome>
	</chemgrouprule>
      </chemgroup>
      <chemgroup>
	<pka>2.35</pka>
	<acidcharged>FALSE</acidcharged>
	<polrule>right_trapped</polrule>
      </chemgroup>


  The xml_node pointer points to the element <chemgrouprule>.
  */
  PxmChemgroupRule *cgr = NULL;
  
  gchar *help = NULL;

  /* Make sure we have parameters pointing bona fide to the right xml
     element.
   */
  g_assert (xml_node != NULL);
  g_assert (0 == xmlStrcmp (xml_node->name, 
			    (const xmlChar *) "chemgrouprule"));
  
  /* Now go to the first child of current node. It has to be a <entity>
     element.
   */
  xml_node = xml_node->children;
  
  /* From a rigorous XML parsing point of view, the blanks found in
     the XML document are considered to be nodes, and we have to
     detect these and take proper action: go next sibling (next blank)
     as long as blanks are encountered.
   */
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;

  g_assert (0 == xmlStrcmp ("entity",
			    (const xmlChar*) xml_node->name));
  
  help = xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
  g_assert (help != NULL);
  
  /* We have an entity character string, so it makes sense to allocate
     a PxmChemgroupRule object now.
  */
  cgr = libpolyxmass_chemgrouprule_new ();
  
  libpolyxmass_chemgrouprule_set_entity (cgr, help);
  
  g_free (help);
  
  /* We can now move on to the <name> element.
   */
  xml_node = xml_node->next;
  /* From a rigorous XML parsing point of view, the blanks found
     in the XML document are considered to be nodes, and we have
     to detect these and take proper action: go next sibling (next
     blank) as long as blanks are encountered.
  */
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  
  g_assert (0 == xmlStrcmp ("name",
			    (const xmlChar*) xml_node->name));
  
  help = xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
  g_assert (help != NULL);
  
  libpolyxmass_chemgrouprule_set_name (cgr, help);

  g_free (help);
  
  /* We can now move on to the <outcome> element.
   */
  xml_node = xml_node->next;
  /* From a rigorous XML parsing point of view, the blanks found
     in the XML document are considered to be nodes, and we have
     to detect these and take proper action: go next sibling (next
     blank) as long as blanks are encountered.
  */
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  

  g_assert (0 == xmlStrcmp ("outcome",
			    (const xmlChar*) xml_node->name));
  
  help = xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
  g_assert (help != NULL);
  
  if (0 == strcmp ("LOST", help))
    {
      g_free (help);
      
      cgr->outcome = PXM_CHEMGROUP_RULE_LOST;
    }
  else if (0 == strcmp ("PRESERVED", help))
    {
      g_free (help);
      
      cgr->outcome = PXM_CHEMGROUP_RULE_PRESERVED;
    }
  else
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     "%s@%d: failed getting value for PxmChemgroupRule "
	     "element <outcome>\n",
	     __FILE__, __LINE__);
      
      libpolyxmass_chemgrouprule_free (cgr);

      g_free (help);
      
      return NULL;
    }
  
  /* At this point, we have finished parsing the <chemgrouprule> element. 
     We just return the allocated PxmChemgroupRule element now.
  */
  return cgr;
}





PxmChemgroup *
libpolyxmass_chemgroup_render_xml_node_chemgroup (xmlDocPtr xml_doc,
				      xmlNodePtr xml_node,
				      gpointer data)
{
  /* The DTD for the "chemgroup" node says this:

  <!ELEMENT chemgroup ((pka,acidcharged)?,polrule*,chemgrouprule*)>
  <!ELEMENT chemgrouprule (entity,name,outcome)>


  Here is an example of such chemgroup elements:

      <chemgroup>
	<pka>9.6</pka>
	<acidcharged>TRUE</acidcharged>
	<polrule>left_trapped</polrule>
	<chemgrouprule>
	  <entity>LE_PLM_MODIF</entity>
	  <name>LE_Acetylation</name>
	  <outcome>LOST</outcome>
	</chemgrouprule>
      </chemgroup>
      <chemgroup>
	<pka>2.35</pka>
	<acidcharged>FALSE</acidcharged>
	<polrule>right_trapped</polrule>
      </chemgroup>

  The pointer we get here with xml_node points to <chemgroup>.
  */
  PxmChemgroup *chemgroup = NULL;
  PxmChemgroupRule *cgr = NULL;

  gchar *help = NULL;

  gdouble test = 0;
  
  gboolean pka_acidcharged_done = FALSE;
  gboolean polrule_done = FALSE;
  gboolean chemgrouprule_done = FALSE;
  
  /* Make sure we have parameters pointing bona fide to the right xml
     element.
   */
  g_assert (xml_node != NULL);
  g_assert (0 == xmlStrcmp (xml_node->name, 
			    (const xmlChar *) "chemgroup"));
  
  /* At this point we know we have a chemgroup so we can allocate a
     chemgroup object.
   */
  chemgroup = libpolyxmass_chemgroup_new ();
  
  /* Now go to the first child of current node (we cannot predict what
     it is going to be).
   */
  xml_node = xml_node->children;
  
  /* From a rigorous XML parsing point of view, the blanks found in
     the XML document are considered to be nodes, and we have to
     detect these and take proper action: go next sibling (next blank)
     as long as blanks are encountered.
   */
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  
  /* The first element is compulsory: <name>
   */
  g_assert (0 == xmlStrcmp (xml_node->name, 
			    (const xmlChar *) "name"));
  
  help = xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
  g_assert (help != NULL);
  
  libpolyxmass_chemgroup_set_name (chemgroup, help);
  
  g_free (help);
  
  xml_node = xml_node->next;
  
  /* From a rigorous XML parsing point of view, the blanks found in
     the XML document are considered to be nodes, and we have to
     detect these and take proper action: go next sibling (next blank)
     as long as blanks are encountered.
  */
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  

  /* Check if we have a <pka> element here.
   */
  if (0 == xmlStrcmp (xml_node->name, 
		      (const xmlChar *) "pka"))
    {
      help = xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
      g_assert (help != NULL);
      
      if (FALSE == libpolyxmass_globals_strtod (help, &test))
	{
	  g_free (help);
	  
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 "%s@%d: could not convert mass '%s' to gdouble\n",
		 __FILE__, __LINE__, 
		 xml_node->xmlChildrenNode->content);
	  
	  libpolyxmass_chemgroup_free (chemgroup);
	  
	  return NULL;
	}
  
      g_free (help);

      if (test < 0 || test > 14)
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 "%s@%d: value of pka is not valid : '%lf'\n",
		 __FILE__, __LINE__, 
		 test);
	  
	  libpolyxmass_chemgroup_free (chemgroup);
	  
	  return NULL;
	}
      else
	chemgroup->pka = test;

      
      /* At this point, because we just got a <pka> element, we know
	 we must have a <acidcharged> element.
       */
      xml_node = xml_node->next;
      
      /* From a rigorous XML parsing point of view, the blanks found in
	 the XML document are considered to be nodes, and we have to
	 detect these and take proper action: go next sibling (next blank)
	 as long as blanks are encountered.
      */
      while (TRUE == xmlIsBlankNode (xml_node))
	xml_node = xml_node->next;
  
      g_assert (0 == xmlStrcmp (xml_node->name, 
				(const xmlChar *) "acidcharged"));
      
      help = xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
      g_assert (help != NULL);

      if (0 == strcmp ("FALSE", help))
	chemgroup->acidcharged = FALSE;
      else if  (0 == strcmp ("TRUE", help))
	chemgroup->acidcharged = TRUE;
      else
	{
	  g_free (help);
	  
	  libpolyxmass_chemgroup_free (chemgroup);
	  
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 "%s@%d: could not parse the <acidcharged> element\n",
		 __FILE__, __LINE__);
	  
	  return NULL;
	}
      
      g_free (help);
      
      pka_acidcharged_done = TRUE;
      
      /* At this point we have finished parsing the (pka,acidcharged)
	 pair, just go down one sibling, but since the siblings below
	 are optional, we have to go through some kind of loop, since
	 there also can be a number of such elements (DTD says
	 polrule*,chemgrouprule*).
      */

      xml_node = xml_node->next;
      
      /* From a rigorous XML parsing point of view, the blanks found in
	 the XML document are considered to be nodes, and we have to
	 detect these and take proper action: go next sibling (next blank)
	 as long as blanks are encountered.
      */
      while (TRUE == xmlIsBlankNode (xml_node))
	xml_node = xml_node->next;
    }
  
  while (xml_node != NULL)
    {
      if (0 == xmlStrcmp (xml_node->name, (const xmlChar*) "polrule"))
	{
	  /* We are parsing a <polrule> element. Do the work right here.
	   */
	  help = 
	    xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
      
	  g_assert (help != NULL);
      
	  if (0 == strcmp ("left_trapped", help))
	    chemgroup->polrule |= PXM_CHEMGROUP_GROUP_LEFT_TRAPPED;
	  else if (0 == strcmp ("right_trapped", help))
	    chemgroup->polrule |= PXM_CHEMGROUP_GROUP_RIGHT_TRAPPED;
	  else if (0 == strcmp ("never_trapped", help))
	    chemgroup->polrule |= PXM_CHEMGROUP_GROUP_NEVER_TRAPPED;
	  else
	    {
	      g_free (help);
	  
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     "%s@%d: failed getting value for <polrule>\n",
		     __FILE__, __LINE__);
	  
	      libpolyxmass_chemgroup_free (chemgroup);
      
	      return NULL;
	    }
      
	  g_free (help);

	  polrule_done = TRUE;
      
	  /* Done parsing the <polrule> element, we can go to next one 
	     if it exists.
	  */
	  xml_node = xml_node->next;
	  /* From a rigorous XML parsing point of view, the blanks found
	     in the XML document are considered to be nodes, and we have
	     to detect these and take proper action: go next sibling (next
	     blank) as long as blanks are encountered.
	  */
	  while (TRUE == xmlIsBlankNode (xml_node))
	    xml_node = xml_node->next;
	}
      /* Finished parsing <polrule> nodes.
       */
      else
	/* The element at hand is not a <polrule> element, and thus
	   we ought to quit the current while loop.
	*/
	break;
    }
  
  
  /* Finally we have all the <chemgrouprule> elements, as many as they
     will... so we will loop into them.
  */
  while (xml_node != NULL)
    {
      /* Check that we have effectively a <chemgrouprule> element
	 here, we can make an assertion because as of now at this
	 point the only elements that we can have are <chemgrouprule>
	 elements.
      */
      g_assert (0 == xmlStrcmp (xml_node->name, 
				(const xmlChar*) "chemgrouprule"));
      
      cgr = libpolyxmass_chemgroup_render_xml_node_chemgrouprule (xml_doc,
						     xml_node,
						     NULL);
      if (cgr == NULL)
	{
	  libpolyxmass_chemgroup_free (chemgroup);
	  
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 "%s@%d: could not parse a <chemgrouprule> element\n",
		 __FILE__, __LINE__);
	  
	  return NULL;
	}

      /* At this point we have an allocated PxmChemgroupRule object, that we
	 can set to the array of such item in chemgroup.
      */
      g_ptr_array_add (chemgroup->cgrGPA, cgr);

      chemgrouprule_done = TRUE;
            
      /* We can iterate in the next xml node.
       */
      xml_node = xml_node->next;
      
      /* From a rigorous XML parsing point of view, the blanks found
	 in the XML document are considered to be nodes, and we have
	 to detect these and take proper action: go next sibling (next
	 blank) as long as blanks are encountered.
      */
      while (TRUE == xmlIsBlankNode (xml_node))
	xml_node = xml_node->next;
    }
  
  /* At this point we have finished parsing the <chemgroup>
     element. All we have to do is return it after having established
     that at least one element was filled with data from the xml data
     file.
  */
  if (pka_acidcharged_done == FALSE 
      && chemgrouprule_done == FALSE 
      && polrule_done == FALSE)
    {
      libpolyxmass_chemgroup_free (chemgroup);
      chemgroup = NULL;
    }
  
  return chemgroup;
}

  

/* FREE'ING FUNCTIONS
 */
gint 
libpolyxmass_chemgroup_free (PxmChemgroup *chemgroup)
{
  g_assert (chemgroup != NULL);
  
  if (chemgroup->userdata != NULL)
    g_free (chemgroup->userdata);
  
  g_assert (chemgroup->cgrGPA != NULL);
  
  libpolyxmass_chemgrouprule_GPA_free (chemgroup->cgrGPA);
  
  return 1;
}


gint 
libpolyxmass_chemgrouprule_free (PxmChemgroupRule *cgr)
{
  g_assert (cgr != NULL);
  
  if (cgr->entity != NULL)
    g_free (cgr->entity);
  
  if (cgr->name != NULL)
    g_free (cgr->name);
  
  g_free (cgr);
  
  return 1;
}


gint
libpolyxmass_chemgrouprule_GPA_free (GPtrArray *GPA)
{
  gint count = 0;
  
  PxmChemgroupRule *cgr = NULL;
  

  g_assert (GPA != NULL);
  
  while (GPA->len > 0)
    {
      cgr = g_ptr_array_remove_index (GPA, 0);
      g_assert (cgr != NULL);
      
      libpolyxmass_chemgrouprule_free (cgr);
      
      count++;
    }
  
  g_ptr_array_free (GPA, TRUE);
  
  return count;
}



gint 
libpolyxmass_chemgroup_prop_free (PxmProp *prop)
{
  /* This function is called when a prop named "CHEMGROUP" must be
     freed.
  */
  g_assert (prop != NULL);
  
  if (prop->name != NULL)
    g_free (prop->name);
  
  if (prop->data != NULL)
    libpolyxmass_chemgroup_free ((PxmChemgroup *) prop->data);
  
  g_free (prop);
  
  return 1;
}

  
gint 
libpolyxmass_chemgroup_prop_free_but_not_data (PxmProp *prop)
{
  /* This function is called when a prop named "CHEMGROUP" must be freed,
     but without freeing the data that were allocated in other places,
     and that are pointed to by the prop->data member.
  */
  g_assert (prop != NULL);
  
  if (prop->name != NULL)
    g_free (prop->name);
  
  g_free (prop);
  
  return 1;
}

  



