/*
 * Written by Jan Rekorajski <baggins@mimuw.edu.pl> 1999/08/08
 * 	based od pam_cracklib by Cristian Gafton <gafton@redhat.com>
 * Password generator code by Tom Van Vleck <thvv@multicians.org>
 */

#include <stdio.h>
#include <stdlib.h>
#include <crypt.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <stdarg.h>
#include <math.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include <syslog.h>

#ifndef DEFAULT_TRIGRAM_FILE
#define	DEFAULT_TRIGRAM_FILE	"/etc/security/trigram"
#endif

/* letter - look alike digit
 * 0     O
 * 1     I
 * 2     Z
 * 3     E
 * 4     A
 * 5     S
 * 6
 * 7     T
 * 8
 * 9
 */

#define PWGEN_PROMPT1	"Do you want (choose one letter only):\n\n"\
			"	Pronounceable passwords generated for you (g) ?\n"\
			"	A string of characters generated for you (c) ?\n"\
			"	To pick your password (p) ?\n\n"\
			"Enter choice here (q to quit): "

#define PWGEN_PROMPT2	"Generating %d random %s for %s.\n"\
			"	(Password generation may be a bit slow.)\n"\
			"	Hit <RETURN> or <ENTER> until you like the choice.\n"\
			"	When you have chosen the password you want, type it in.\n"
#define PWGEN_PROMPT3	"\nEnter new UNIX password: "
#define PWGEN_PROMPT4	"Retype new UNIX password: "
#define MISTYPED_PASS1	"Sorry, password do not match any of the proposed"
#define MISTYPED_PASS2	"Sorry, passwords do not match"

#define GENSTR01	"pronounceable passwords"
#define GENSTR02	"strings of characters"

#define GENTYPE_NONE	0
#define GENTYPE01	1
#define GENTYPE02	2

			
/*
 * here, we make a definition for the externally accessible function
 * in this file (this definition is required for static a module
 * but strongly encouraged generally) it is used to instruct the
 * modules include file to define the function prototypes.
 */

#define PAM_SM_PASSWORD

#include <security/pam_modules.h>
#include <security/_pam_macros.h>

#ifndef LINUX_PAM
#include <security/pam_appl.h>
#endif				/* LINUX_PAM */

/* argument parsing */
#define PAM_DEBUG_ARG       0x0001

struct tris_data {
	long sigma;		/* Total letters */
	int tris[26][26][26];	/* Trigraph frequencies */
};

struct gpassword {
	int length;
	char passwd[128];
};

static char *gen_pronouncable_passwd(int length, void *tr);
static char *gen_random_passwd(int length, void *null_data);

/* some syslogging */

static void _pam_log(int err, const char *format,...)
{
	va_list args;

	va_start(args, format);
	openlog("PAM-pwgen", LOG_CONS | LOG_PID, LOG_AUTH);
	vsyslog(err, format, args);
	va_end(args);
	closelog();
}

/* use this to free strings. ESPECIALLY password strings */
static char *_pam_delete(register char *xx)
{
	_pam_overwrite(xx);
	free(xx);
	return NULL;
}

/* Helper functions */

void _cleanup_data(pam_handle_t * pamh, void *data, int err)
{
	_pam_delete(data);
}

/* this is a front-end for module-application conversations */
static int converse(pam_handle_t * pamh, int ctrl, int nargs,
		    struct pam_message **message,
		    struct pam_response **response)
{
	int retval;
	struct pam_conv *conv;

	retval = pam_get_item(pamh, PAM_CONV, (const void **) &conv);

	if (retval == PAM_SUCCESS) {
		retval = conv->conv(nargs, (const struct pam_message **) message,
				    response, conv->appdata_ptr);
		if (retval != PAM_SUCCESS && (ctrl & PAM_DEBUG_ARG)) {
			_pam_log(LOG_DEBUG, "conversation failure [%s]",
				 pam_strerror(pamh, retval));
		}
	} else {
		_pam_log(LOG_ERR, "couldn't obtain coversation function [%s]",
			 pam_strerror(pamh, retval));
	}

	return retval;		/* propagate error status */
}

static int make_remark(pam_handle_t * pamh, unsigned int ctrl,
		       int type, const char *text)
{
	struct pam_message *pmsg[1], msg[1];
	struct pam_response *resp;
	int retval;

	pmsg[0] = &msg[0];
	msg[0].msg = text;
	msg[0].msg_style = type;
	resp = NULL;

	retval = converse(pamh, ctrl, 1, pmsg, &resp);
	if (retval == PAM_SUCCESS)
		_pam_drop_reply(resp, 1);

	return retval;
}

#define OLD_PASSWORDS_FILE	"/etc/security/opasswd"

static const char *check_old_password(const char *forwho, const char *newpass)
{
	static char buf[16384];
	char *s_luser, *s_uid, *s_npas, *s_pas;
	const char *msg = NULL;
	FILE *opwfile;

	opwfile = fopen(OLD_PASSWORDS_FILE, "r");
	if (opwfile == NULL)
		return NULL;

	while (fgets(buf, 16380, opwfile)) {
		if (!strncmp(buf, forwho, strlen(forwho))) {
			buf[strlen(buf) - 1] = '\0';
			s_luser = strtok(buf, ":,");
			s_uid = strtok(NULL, ":,");
			s_npas = strtok(NULL, ":,");
			s_pas = strtok(NULL, ":,");
			while (s_pas != NULL) {
				if (!strcmp(crypt(newpass, s_pas), s_pas)) {
					msg = "has been already used";
					break;
				}
				s_pas = strtok(NULL, ":,");
			}
			break;
		}
	}
	fclose(opwfile);

	return msg;
}

static int _pam_unix_approve_pass(pam_handle_t * pamh,
				  unsigned int ctrl,
				  const char *pass_old,
				  const char *pass_new)
{
	const char *msg = NULL;
	const char *user;
	int retval;

	if (pass_new == NULL || (pass_old && !strcmp(pass_old, pass_new))) {
		if (ctrl & PAM_DEBUG_ARG)
			_pam_log(LOG_DEBUG, "bad authentication token");
		make_remark(pamh, ctrl, PAM_ERROR_MSG,
			    pass_new == NULL ?
			  "No password supplied" : "Password unchanged");
		return PAM_AUTHTOK_ERR;
	}
	/*
	 * if one wanted to hardwire authentication token strength
	 * checking this would be the place
	 *
	msg = password_check(pass_old, pass_new);
	 */
	if (!msg) {
		retval = pam_get_item(pamh, PAM_USER, (const void **) &user);
		if (retval != PAM_SUCCESS) {
			if (ctrl & PAM_DEBUG_ARG) {
				_pam_log(LOG_ERR, "Can not get username");
				return PAM_AUTHTOK_ERR;
			}
		}
		msg = check_old_password(user, pass_new);
	}
	if (msg) {
		char remark[BUFSIZ];

		memset(remark, 0, sizeof(remark));
		sprintf(remark, "BAD PASSWORD: %s", msg);
		if (ctrl & PAM_DEBUG_ARG)
			_pam_log(LOG_NOTICE, "new passwd fails strength check: %s",
				 msg);
		make_remark(pamh, ctrl, PAM_ERROR_MSG, remark);
		return PAM_AUTHTOK_ERR;
	};
	return PAM_SUCCESS;

}

static int select_generator(pam_handle_t * pamh, unsigned int ctrl, int *retval)
{
	struct pam_message msg[1], *pmsg[1];
	struct pam_response *resp;
	int pwgen_func = GENTYPE_NONE;
	char prompt[BUFSIZ];
	int select_done;

	select_done = 0;

	do {
		memset(prompt, 0, sizeof(prompt));
		sprintf(prompt, PWGEN_PROMPT1);
		pmsg[0] = &msg[0];
		msg[0].msg_style = PAM_PROMPT_ECHO_ON;
		msg[0].msg = prompt;

		resp = NULL;
		*retval = converse(pamh, ctrl, 1, pmsg, &resp);
		if (resp != NULL) {
			/* interpret the response */
			if (*retval == PAM_SUCCESS) {	/* a good conversation */
				if (resp[0].resp == NULL || !strlen(resp[0].resp)) {
					_pam_log(LOG_NOTICE,
						 "could not recover authentication token 1");
					*retval = PAM_AUTHTOK_RECOVER_ERR;
				} else {
					switch (tolower(resp[0].resp[0])) {
						case 'g':
							pwgen_func = GENTYPE01;
							select_done = 1;
							break;
						case 'c':
							pwgen_func = GENTYPE02;
							select_done = 1;
							break;
						case 'p':
							pwgen_func = GENTYPE_NONE;
							*retval = PAM_IGNORE;
							select_done = 1;
							break;
						case 'q':
							pwgen_func = GENTYPE_NONE;
							*retval = PAM_ABORT;
							select_done = 1;
							break;
						default:
							break;
					}
				}
			}
			/*
			 * tidy up the conversation (resp_retcode) is ignored
			 */
			_pam_drop_reply(resp, 1);
		} else {
			*retval = (*retval == PAM_SUCCESS) ?
			    PAM_AUTHTOK_RECOVER_ERR : *retval;
		}
	} while (!select_done);

	if (*retval != PAM_SUCCESS) {
		if (ctrl & PAM_DEBUG_ARG)
			_pam_log(LOG_DEBUG, "unable to obtain a password");
	}

	return pwgen_func;
}

/* Main generator */
/* GPW - Generate pronounceable passwords
   This program uses statistics on the frequency of three-letter sequences
   in your dictionary to generate passwords.  The statistics are in trigram.h,
   generated there by the program loadtris.  Use different dictionaries
   and you'll get different statistics.

   This program can generate every word in the dictionary, and a lot of
   non-words.  It won't generate a bunch of other non-words, call them the
   unpronounceable ones, containing letter combinations found nowhere in
   the dictionary.  My rough estimate is that if there are 10^6 words,
   then there are about 10^9 pronounceables, out of a total population
   of 10^11 8-character strings.  I base this on running the program a lot
   and looking for real words in its output.. they are very rare, on the
   order of one in a thousand.

   This program uses "drand48()" to get random numbers, and "srand48()"
   to set the seed to the microsecond part of the system clock.  Works
   for AIX C++ compiler and runtime.  Might have to change this to port
   to another environment.

   The best way to use this program is to generate multiple words.  Then
   pick one you like and transform it with punctuation, capitalization,
   and other changes to come up with a new password.

   THVV 6/1/94 Coded

 */

static int read_trigrams(char *name, struct tris_data *tr)
{
	FILE *f;
	int c1, c2, c3;

	f = fopen(name, "rt");
	if (f == NULL)
		return PAM_SYSTEM_ERR;

	fscanf(f, "%ld\n", &tr->sigma);
	D(("%ld", tr->sigma));

	for (c1 = 0; c1 < 26; c1++) {
		for (c2 = 0; c2 < 26; c2++) {
			fscanf(f, "\n%*s");
			for (c3 = 0; c3 < 26; c3++) {
				tr->tris[c1][c2][c3] = 0;
				fscanf(f, "%d", &tr->tris[c1][c2][c3]);
				D(("%d", tr->tris[c1][c2][c3]));
			}
		}
	}

	fclose(f);
	return PAM_SUCCESS;
}

static char *gen_pronouncable_passwd(int length, void *tr)
{
	int c1, c2, c3;		/* array indices */
	long sumfreq;		/* total frequencies[c1][c2][*] */
	double pik;		/* raw random number in [0.1] from drand48() */
	long ranno;		/* random number in [0,sumfreq] */
	long sum;		/* running total of frequencies */
	int nchar;		/* number of chars in password so far */
	static char password[128];	/* buffer to develop a password */
	struct tris_data *tris = tr;

	/* Pick a random starting point. */
	/* (This cheats a little; the statistics for three-letter
	   combinations beginning a word are different from the stats
	   for the general population.  For example, this code happily
	   generates "mmitify" even though no word in my dictionary
	   begins with mmi. So what.) */

	pik = drand48();	/* random number [0,1] */
	sumfreq = tris->sigma;	/* sigma calculated by loadtris */
	ranno = pik * sumfreq;	/* Weight by sum of frequencies. */
	sum = 0;

	for (c1 = 0; c1 < 26; c1++) {
		for (c2 = 0; c2 < 26; c2++) {
			for (c3 = 0; c3 < 26; c3++) {
				sum += tris->tris[c1][c2][c3];
				if (sum > ranno) {	/* Pick first value */
					password[0] = 'a' + c1;
					password[1] = 'a' + c2;
					password[2] = 'a' + c3;
					c1 = c2 = c3 = 26;	/* Break all loops. */
				}
			}
		}
	}

	/* Do a random walk. */
	nchar = 3;		/* We have three chars so far. */
	while (nchar < length) {
		password[nchar] = '\0';
		password[nchar + 1] = '\0';
		c1 = tolower(password[nchar - 2]) - 'a';	/* Take the last 2 chars */
		c2 = tolower(password[nchar - 1]) - 'a';	/* .. and find the next one. */
		sumfreq = 0;
		for (c3 = 0; c3 < 26; c3++)
			sumfreq += tris->tris[c1][c2][c3];
		/* Note that sum < duos[c1][c2] because
		   duos counts all digraphs, not just those
		   in a trigraph. We want sum. */
		if (sumfreq == 0) {	/* If there is no possible extension.. */
			break;	/* Break while nchar loop & print what we have. */
		}
		/* Choose a continuation. */
		pik = drand48();
		ranno = pik * sumfreq;	/* Weight by sum of frequencies for row. */
		sum = 0;
		for (c3 = 0; c3 < 26; c3++) {
			sum += tris->tris[c1][c2][c3];
			if (sum > ranno) {
				password[nchar++] = 'a' + c3;
				c3 = 26;	/* Break the for c3 loop. */
			}
		}
	}
	return password;
}

static char *gen_random_passwd(int length, void *null_data)
{
	static const char ASCII[94] = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789`~!@#$%^&*()_+-=[]{};':\",./<>?\\|";
	static char password[128];	/* buffer to develop a password */
	int i;
	char c;

	for (i=0; i<length; i++) {
		c = ASCII[(int)(drand48()*94)];
		password[i] = c;
	}
	password[i] = '\0';

	return password;
}

PAM_EXTERN int pam_sm_chauthtok(pam_handle_t * pamh, int flags,
				int argc, const char **argv)
{
	unsigned int ctrl;
	const char *user;
	int retval;
	char tri_file[128];
	int retry_times = 0;
	int min_length = 10;
	int pw_count = 5;

	D(("called."));

	retry_times = 1;
	strcpy(tri_file, DEFAULT_TRIGRAM_FILE);

	/* step through arguments */
	for (ctrl = 0; argc-- > 0; ++argv) {
		char *ep = NULL;

		/* generic options */

		if (!strcmp(*argv, "debug"))
			ctrl |= PAM_DEBUG_ARG;
		else if (!strncmp(*argv, "retry=", 6)) {
			retry_times = strtol(*argv + 6, &ep, 10);
			if (!ep || (retry_times < 1))
				retry_times = 1;
		} else if (!strncmp(*argv, "minlen=", 7)) {
			min_length = strtol(*argv + 7, &ep, 10);
			if (!ep || (min_length < 5))
				min_length = 5;
			if (min_length > 64)
				min_length = 64;
		} else if (!strncmp(*argv, "count=", 6)) {
			pw_count = strtol(*argv + 6, &ep, 10);
			if (!ep || (pw_count < 5))
				pw_count = 5;
			if (pw_count > 100)
				pw_count = 100;
		} else if (!strncmp(*argv, "trifile=", 7)) {
			strncpy(tri_file, *argv + 7, 128);
		} else {
			_pam_log(LOG_ERR, "pam_parse: unknown option; %s", *argv);
		}
	}

	retval = pam_get_item(pamh, PAM_USER, (const void **)&user);
	if (retval != PAM_SUCCESS) {
		if (ctrl & PAM_DEBUG_ARG) {
			_pam_log(LOG_ERR,"Can not get username");
			return PAM_AUTHTOK_ERR;
		}
	}

	if (flags & PAM_PRELIM_CHECK) {
		/* Check for generator data file */
		struct stat st;

		D(("prelim check"));

		if (!stat(tri_file, &st) && st.st_size) {
			return PAM_SUCCESS;
		} else {
			if (ctrl & PAM_DEBUG_ARG)
				_pam_log(LOG_NOTICE, "trigram path '%s' is invalid",
					 tri_file);
			return PAM_ABORT;
		}

		/* Not reached */
		return PAM_SERVICE_ERR;

	} else if (flags & PAM_UPDATE_AUTHTOK) {
		char *oldtoken;
		const char *gen_prompt;
		char * (*genfn)(int, void *) = NULL;
		int genpw;
		struct tris_data *tr;
		struct gpassword *passwds;

		D(("do update"));
		retval = pam_get_item(pamh, PAM_OLDAUTHTOK,
				      (const void **) &oldtoken);
		if (retval != PAM_SUCCESS) {
			if (ctrl & PAM_DEBUG_ARG)
				_pam_log(LOG_ERR, "Can not get old passwd");
			oldtoken = NULL;
			retval = PAM_SUCCESS;
		}

		/*
		 * Planned modus operandi:
		 * 1) Present the generators.
		 * 2) Get a response.
		 * 3) Present passwords or return PAM_IGNORE.
		 * 4) Get a passwd.
		 * 5) If okay get it a second time.
		 *      else goto 3
		 * 6) Check to be the same with the first one.
		 * 7) set PAM_AUTHTOK and return
		 */

		srand48(time(NULL));	/* Set random seed. */
		
		genpw = select_generator(pamh, ctrl, &retval);

		switch (genpw) {
			case GENTYPE01:
				genfn = gen_pronouncable_passwd;
				gen_prompt = GENSTR01;
				tr = malloc(sizeof(struct tris_data));
				if (tr == NULL)
					return PAM_BUF_ERR;
				if (read_trigrams(tri_file, tr) < 0)
					return PAM_ABORT;
				break;
			case GENTYPE02:
				genfn = gen_random_passwd;
				gen_prompt = GENSTR02;
				tr = NULL;
				break;
			case GENTYPE_NONE:
				return retval;
			default:
				return PAM_ABORT;
		}

		passwds = malloc(sizeof(struct gpassword)*(pw_count+1));
		if (passwds == NULL)
			return PAM_BUF_ERR;

		do {
			char *token1, *token2;
			char *pw_buff;
			const char *item;
			struct pam_message msg[1], *pmsg[1];
			struct pam_response *resp;
			char prompt[BUFSIZ];
			int i, j;
			/*
			 * make sure nothing inappropriate gets returned
			 */
			token1 = token2 = NULL;
			for (i=0; i<pw_count; i++)
				_pam_overwrite(passwds[i].passwd);

			if (!retry_times) {
				D(("returning %s because maxtries reached",
				   pam_strerror(pamh, retval)));
				return retval;
			}

			memset(prompt, 0, sizeof(prompt));
			sprintf(prompt, PWGEN_PROMPT2, pw_count, gen_prompt, user);
			make_remark(pamh, ctrl, PAM_TEXT_INFO, prompt);

			memset(prompt, 0, sizeof(prompt));

			j = min_length + (rand() & 7);
			pw_buff = genfn(j, tr);
			passwds[0].length = j;
			strncpy(passwds[0].passwd, pw_buff, j);

			i = 0;
			do {
				j = min_length + (rand() & 7);
				pw_buff = genfn(j, tr);
				passwds[i].length = j;
				strncpy(passwds[i].passwd, pw_buff, j);
				strncat(prompt, pw_buff, j);
				strcat(prompt, "\n");
			} while (++i < pw_count);

			/* Prepare to ask the user for the first time */
			strcat(prompt, PWGEN_PROMPT3);
			pmsg[0] = &msg[0];
			msg[0].msg_style = PAM_PROMPT_ECHO_OFF;
			msg[0].msg = prompt;

			resp = NULL;
			retval = converse(pamh, ctrl, 1, pmsg, &resp);
			if (resp != NULL) {
				/* interpret the response */
				if (retval == PAM_SUCCESS) {	/* a good conversation */
					token1 = x_strdup(resp[0].resp);
					if (token1 == NULL) {
						_pam_log(LOG_NOTICE,
							 "could not recover authentication token 1");
						retval = PAM_AUTHTOK_RECOVER_ERR;
					}
				}
				/*
				 * tidy up the conversation (resp_retcode) is ignored
				 */
				_pam_drop_reply(resp, 1);
			} else {
				retval = (retval == PAM_SUCCESS) ?
				    PAM_AUTHTOK_RECOVER_ERR : retval;
			}

			if (strlen(token1) == 0) {
				retry_times++;
				continue;
			}

			if (retval != PAM_SUCCESS) {
				if (ctrl & PAM_DEBUG_ARG)
					_pam_log(LOG_DEBUG, "unable to obtain a password");
				continue;
			}
			D(("testing password, retval = %s", pam_strerror(pamh, retval)));
			/* now test this passwd */
			{
				if (strlen(token1) < min_length)
					retval = PAM_AUTHTOK_RECOVER_ERR;
				else {
					for (i=0; i<pw_count; i++) {
						if (strncmp(token1, passwds[i].passwd, passwds[i].length))
							break;
					}
					if (i == pw_count) {
						retval = PAM_AUTHTOK_RECOVER_ERR;
					} else {                                                            
						/* check it for strength too... */
						D(("for strength"));
						if (oldtoken) {
							retval = _pam_unix_approve_pass(pamh,ctrl,
									oldtoken,token1);
							if (retval != PAM_SUCCESS) {
								if (getuid() || (flags & PAM_CHANGE_EXPIRED_AUTHTOK))
									retval = PAM_AUTHTOK_ERR;
								else
									retval = PAM_SUCCESS;
							}
						}
					}
				}	
			}

			D(("after testing: retval = %s", pam_strerror(pamh, retval)));
			/* if cracklib/strength check said it is a bad passwd... */
			if ((retval != PAM_SUCCESS) && (retval != PAM_IGNORE)) {
				int temp_unused;

				make_remark(pamh, ctrl, PAM_ERROR_MSG, MISTYPED_PASS1);
				temp_unused = pam_set_item(pamh, PAM_AUTHTOK, NULL);
				token1 = _pam_delete(token1);
				continue;
			}
			/* Now we have a good passwd. Ask for it once again */

			bzero(prompt, sizeof(prompt));
			sprintf(prompt, PWGEN_PROMPT4);
			pmsg[0] = &msg[0];
			msg[0].msg_style = PAM_PROMPT_ECHO_OFF;
			msg[0].msg = prompt;

			resp = NULL;
			retval = converse(pamh, ctrl, 1, pmsg, &resp);
			if (resp != NULL) {
				/* interpret the response */
				if (retval == PAM_SUCCESS) {	/* a good conversation */
					token2 = x_strdup(resp[0].resp);
					if (token2 == NULL) {
						_pam_log(LOG_NOTICE,
							 "could not recover authentication token 2");
						retval = PAM_AUTHTOK_RECOVER_ERR;
					}
				}
				/*
				 * tidy up the conversation (resp_retcode) is ignored
				 */
				_pam_drop_reply(resp, 1);
			} else {
				retval = (retval == PAM_SUCCESS) ?
				    PAM_AUTHTOK_RECOVER_ERR : retval;
			}

			if (retval != PAM_SUCCESS) {
				if (ctrl & PAM_DEBUG_ARG)
					_pam_log(LOG_DEBUG
						 ,"unable to obtain the password a second time");
				continue;
			}
			/* Hopefully now token1 and token2 the same password ... */
			if (strcmp(token1, token2) != 0) {
				/* tell the user */
				make_remark(pamh, ctrl, PAM_ERROR_MSG, MISTYPED_PASS2);
				token1 = _pam_delete(token1);
				token2 = _pam_delete(token2);
				pam_set_item(pamh, PAM_AUTHTOK, NULL);
				if (ctrl & PAM_DEBUG_ARG)
					_pam_log(LOG_NOTICE, "Password mistyped");
				retval = PAM_AUTHTOK_RECOVER_ERR;
				continue;
			}
			/* Yes, the password was typed correct twice
			 * we store this password as an item
			 */

			retval = pam_set_item(pamh, PAM_AUTHTOK, token1);
			/* clean it up */
			token1 = _pam_delete(token1);
			token2 = _pam_delete(token2);
			if ((retval != PAM_SUCCESS) ||
					((retval = pam_get_item(pamh, PAM_AUTHTOK, (const void **) &item)) != PAM_SUCCESS)) {
				_pam_log(LOG_CRIT, "error manipulating password");
				continue;
			}
			for (i=0; i<pw_count; i++)
				_pam_overwrite(passwds[i].passwd);
			free(passwds);
			item = NULL;	/* break link to password */
			return PAM_SUCCESS;
		} while (retry_times--);
	} else {
		if (ctrl & PAM_DEBUG_ARG)
			_pam_log(LOG_NOTICE, "UNKNOWN flags setting %02X", flags);
		return PAM_SERVICE_ERR;
	}

	/* Not reached */
	return PAM_SERVICE_ERR;
}

#ifdef PAM_STATIC
/* static module data */
struct pam_module _pam_pwgen_modstruct =
{
	"pam_pwgen",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	pam_sm_chauthtok
};
#endif
