(**
   Implements windows.

  TODO
  * When unmaped, HandleEvents calls top.Hide, which then tries to draw in an
    allready unmaped window. Solution: object.Hide has to check window.maped.

  * Cleanup window-generation code, divide it into more methods

  * Redesign eventloop handling, on method for each event, more functionality
    in baseclase. Maybe some abstraction mechanism for certain events.

  * Support future Object.Refresh
**)

MODULE VO:Window;

(*
    Windows.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D   := VO:Base:Display,
       E   := VO:Base:Event,
       F   := VO:Base:Frame,
       O   := VO:Base:Object,
       U   := VO:Base:Util,

       P   := VO:Prefs:Base,

       DI  := VO:OS:Display,

       FG  := VO:FrameGroup,
       G   := VO:Object;

CONST
  (* various messages *)

  openMsg*      = 0;
  openModalMsg* = 1;
  closeMsg*     = 2;
  exitMsg*      = 3;

  (* window styles *)

  normal*    = 0;
  popup*     = 1;

  (* special shortcut modes *)

  none    * = 0;
  return  * = 1;
  escape  * = 2;
  default * = 3;

TYPE
  ShortcutEntry     = POINTER TO ShortcutEntryDesc;
  ShortcutEntryDesc = RECORD
  (**
    Store one shortcut entry.
  *)
                        next      : ShortcutEntry;
                        gadget    : G.Gadget;
                        qualifier : SET;
                        char      : CHAR;
                        id,mode   : LONGINT;
                      END;

  KeyEntry         = POINTER TO KeyEntryDesc;
  KeyEntryDesc     = RECORD
                       next,last : KeyEntry;
                       gadget    : G.Gadget;
                     END;

  KeyGroup         = POINTER TO KeyGroupDesc;
  KeyGroupDesc     = RECORD
                       next,last : KeyGroup;
                       firstEntry,
                       lastEntry : KeyEntry;
                     END;

  KeyHandler       = POINTER TO KeyHandlerDesc;
  KeyHandlerDesc   = RECORD
                       first,last : KeyGroup;
                       current    : G.Gadget;

                       sFirst,
                       sLast,
                       sCurrent : ShortcutEntry;
                     END;

  Prefs*     = POINTER TO PrefsDesc;
  PrefsDesc* = RECORD (P.PrefsDesc)
  (**
    In this class all preferences stuff of the button is stored.
  *)
                 frame* : LONGINT; (* the frame to use for the button *)
               END;

  Window*       = POINTER TO WindowDesc;
  WindowDesc*   = RECORD (DI.WindowDesc)
                    prefs       : Prefs;
                    top         : G.Object;
                    current     : G.Object;
                    reference   : G.Object;
                    keyHandler  : KeyHandler;
                    frame       : FG.FrameGroup;
                    style       : LONGINT;
                  END;

  OpenMsg*        = POINTER TO OpenMsgDesc;
  OpenMsgDesc*    = RECORD (O.MessageDesc)
                    END;

  OpenModalMsg*     = POINTER TO OpenModalMsgDesc;
  OpenModalMsgDesc* = RECORD (O.MessageDesc)
                      END;

  CloseMsg*       = POINTER TO CloseMsgDesc;
  CloseMsgDesc*   = RECORD (O.MessageDesc)
                    END;

  ExitMsg*        = POINTER TO ExitMsgDesc;
  ExitMsgDesc*    = RECORD (O.MessageDesc)
                    END;

  (* Some messagehandler *)

  Msg2Open*       = POINTER TO Msg2OpenDesc;
  Msg2OpenDesc*   = RECORD (O.HandlerDesc)
                    END;

  Msg2OpenModal*     = POINTER TO Msg2OpenModalDesc;
  Msg2OpenModalDesc* = RECORD (O.HandlerDesc)
                       END;

  Msg2Close*      = POINTER TO Msg2CloseDesc;
  Msg2CloseDesc*  = RECORD (O.HandlerDesc)
                    END;

  Msg2Exit*      = POINTER TO Msg2ExitDesc;
  Msg2ExitDesc*  = RECORD (O.HandlerDesc)
                   END;

VAR
  prefs* : Prefs;

  PROCEDURE (e : KeyEntry) Init;

  BEGIN
    e.next:=NIL;
    e.last:=NIL;
    e.gadget:=NIL;
  END Init;

  PROCEDURE (g : KeyGroup) Init;

  BEGIN
    g.next:=NIL;
    g.last:=NIL;
    g.firstEntry:=NIL;
    g.lastEntry:=NIL;
  END Init;

  PROCEDURE (g : KeyGroup) AddGadget(gadget : G.Gadget);

  VAR
    entry : KeyEntry;
    focus : F.Frame;

  BEGIN
    NEW(entry);
    entry.Init;
    entry.gadget:=gadget;

    gadget.SetFlags({G.mayFocus});

    IF G.stdFocus IN gadget.flags THEN
      NEW(focus);
      focus.Init;     (* We must of cause call Frame.Init of the frame *)
      (* We need a frame-image for a button *)
      focus.SetFrame(F.doubleFocus);
      gadget.SetFocusObject(focus);
    END;

    IF g.firstEntry=NIL THEN
      g.firstEntry:=entry;
    ELSE
      g.lastEntry.next:=entry;
      entry.last:=g.lastEntry;
    END;
    g.lastEntry:=entry;
  END AddGadget;

  PROCEDURE (k : KeyHandler) Init;

  BEGIN
    k.first:=NIL;
    k.last:=NIL;
    k.current:=NIL;

    k.sFirst:=NIL;
    k.sLast:=NIL;
    k.sCurrent:=NIL;
  END Init;

  PROCEDURE (k : KeyHandler) SetFocus(gadget : G.Gadget);

  BEGIN
    IF k.current#NIL THEN
      k.current.LostFocus;
    END;

    k.current:=gadget;
    k.current.CatchedFocus;
  END SetFocus;

  PROCEDURE (k : KeyHandler) GetCurrentFocusGadget():G.Gadget;

  BEGIN
    RETURN k.current;
  END GetCurrentFocusGadget;

  PROCEDURE (k : KeyHandler) GetCurrentFocusGroup():KeyGroup;

  VAR
    group : KeyGroup;
    entry : KeyEntry;

  BEGIN
    IF k.current=NIL THEN
      RETURN NIL;
    END;

    group:=k.first;
    WHILE group#NIL DO
      entry:=group.firstEntry;

      WHILE entry#NIL DO
        IF entry.gadget=k.current THEN
          RETURN group;
        END;
        entry:=entry.next;
      END;

      group:=group.next;
    END;

    RETURN NIL;
  END GetCurrentFocusGroup;

  PROCEDURE (k : KeyHandler) SetFocusFirst();

  VAR
    group : KeyGroup;
    entry : KeyEntry;

  BEGIN
    IF k.first=NIL THEN
      RETURN;
    END;

    group:=k.first;

    WHILE group#NIL DO
      entry:=group.firstEntry;
      WHILE entry#NIL DO
        IF entry.gadget.visible & ~entry.gadget.disabled THEN
          k.SetFocus(entry.gadget);
        END;
        entry:=entry.next;
      END;
      group:=group.next;
    END;
  END SetFocusFirst;

  PROCEDURE (k : KeyHandler) SetFocusNext();

  VAR
    gadget   : G.Gadget;
    entry    : KeyEntry;
    group,
    newGroup : KeyGroup;

  BEGIN
    IF k.first=NIL THEN
      RETURN;
    END;

    IF k.first=k.last THEN
      k.SetFocusFirst;
      RETURN;
    END;

    gadget:=k.GetCurrentFocusGadget();
    IF gadget=NIL THEN
      k.SetFocusFirst;
      RETURN;
    END;

    group:=k.GetCurrentFocusGroup();
    IF group=NIL THEN
      k.SetFocusFirst;
      RETURN;
    END;

    newGroup:=group.next;
    IF newGroup=NIL THEN
      newGroup:=k.first;
    END;

    WHILE newGroup#group DO
      entry:=newGroup.firstEntry;

      WHILE entry#NIL DO
        IF entry.gadget.visible & ~entry.gadget.disabled THEN
          k.SetFocus(entry.gadget);
          RETURN;
        END;
        entry:=entry.next;
      END;

      newGroup:=newGroup.next;
      IF newGroup=NIL THEN
        newGroup:=k.first;
      END;
    END;
  END SetFocusNext;

  PROCEDURE (k : KeyHandler) SetFocusLast();

  VAR
    gadget   : G.Gadget;
    entry    : KeyEntry;
    group,
    newGroup : KeyGroup;

  BEGIN
    IF k.first=NIL THEN
      RETURN;
    END;

    IF k.first=k.last THEN
      k.SetFocusFirst;
      RETURN;
    END;

    gadget:=k.GetCurrentFocusGadget();
    IF gadget=NIL THEN
      k.SetFocusFirst;
      RETURN;
    END;

    group:=k.GetCurrentFocusGroup();
    IF group=NIL THEN
      k.SetFocusFirst;
      RETURN;
    END;

    newGroup:=group.last;
    IF newGroup=NIL THEN
      newGroup:=k.last;
    END;

    WHILE newGroup#group DO
      entry:=newGroup.lastEntry;

      WHILE entry#NIL DO
        IF entry.gadget.visible & ~entry.gadget.disabled THEN
          k.SetFocus(entry.gadget);
          RETURN;
        END;
        entry:=entry.last;
      END;

      newGroup:=newGroup.last;
      IF newGroup=NIL THEN
        newGroup:=k.last;
      END;
    END;
  END SetFocusLast;

  PROCEDURE (k : KeyHandler) SetFocusNextInGroup();

  VAR
    group  : KeyGroup;
    entry  : KeyEntry;
    gadget : G.Gadget;

  BEGIN
    gadget:=k.GetCurrentFocusGadget();
    IF gadget=NIL THEN
      RETURN;
    END;

    group:=k.GetCurrentFocusGroup();
    IF group=NIL THEN
      RETURN;
    END;

    IF group.firstEntry=group.lastEntry THEN
      RETURN;
    END;

    entry:=group.firstEntry;
    WHILE (entry#NIL) & (entry.gadget#gadget) DO
      entry:=entry.next;
    END;

    IF entry#NIL THEN
      IF entry.next#NIL THEN
        k.SetFocus(entry.next.gadget);
      ELSE
        k.SetFocus(group.firstEntry.gadget);
      END;
    END;
  END SetFocusNextInGroup;

  PROCEDURE (k : KeyHandler) SetFocusLastInGroup();

  VAR
    group  : KeyGroup;
    entry  : KeyEntry;
    gadget : G.Gadget;

  BEGIN
    gadget:=k.GetCurrentFocusGadget();
    IF gadget=NIL THEN
      RETURN;
    END;

    group:=k.GetCurrentFocusGroup();
    IF group=NIL THEN
      RETURN;
    END;

    IF group.firstEntry=group.lastEntry THEN
      RETURN;
    END;

    entry:=group.firstEntry;
    WHILE (entry#NIL) & (entry.gadget#gadget) DO
      entry:=entry.next;
    END;

    IF entry#NIL THEN
      IF entry.last#NIL THEN
        k.SetFocus(entry.last.gadget);
      ELSE
        k.SetFocus(group.lastEntry.gadget);
      END;
    END;
  END SetFocusLastInGroup;

  PROCEDURE (k : KeyHandler) Activate;

  BEGIN
    IF k.current#NIL THEN
      k.current.CatchedFocus;
    ELSE
      k.SetFocusFirst;
    END;
  END Activate;

  PROCEDURE (k : KeyHandler) Deactivate;

  BEGIN
    IF k.current#NIL THEN
      k.current.LostFocus;
    END;
  END Deactivate;

  PROCEDURE (k : KeyHandler) AddFocusGadget(gadget : G.Gadget);

  VAR
    group : KeyGroup;

  BEGIN
    ASSERT(gadget.CanFocus()); (* The object must support foccusing *)

    NEW(group);
    group.Init;
    group.AddGadget(gadget);

    IF k.first=NIL THEN
      k.first:=group;
    ELSE
      group.last:=k.last;
      k.last.next:=group;
    END;
      k.last:=group;
  END AddFocusGadget;

  PROCEDURE (k : KeyHandler) AddFocusGadgetToGroup(gadget : G.Gadget);

  BEGIN
    ASSERT(gadget.CanFocus()); (* The object must support foccusing *)
    ASSERT(k.last#NIL);

    k.last.AddGadget(gadget);
  END AddFocusGadgetToGroup;

  PROCEDURE (k : KeyHandler) AddShortcutObject(gadget : G.Gadget; qualifier : SET;
                                               char : CHAR; id,mode : LONGINT);
  (**
    Adds a shortcut with the given qualifier and character for an object.
    Additional a specialmode can be given for the shortcut whith states
    additional situations the shortcut get evaluated. An object must
    be visible to get notified, so shortcuts can be shared as long as
    only one object at the time is visible.
  *)

  VAR
    entry : ShortcutEntry;

  BEGIN
    ASSERT(G.handleSC IN gadget.flags);

    NEW(entry);
    entry.next:=NIL;
    entry.qualifier:=qualifier;
    entry.char:=CAP(char);
    entry.gadget:=gadget;
    entry.id:=id;
    entry.mode:=mode;

    IF k.sLast#NIL THEN
      k.sLast.next:=entry;
    ELSE
      k.sFirst:=entry;
    END;
    k.sLast:=entry;
  END AddShortcutObject;

  PROCEDURE (sc : ShortcutEntry) Match(qualifier : SET; keysym : LONGINT; char  : CHAR):BOOLEAN;
  (**
    Checks if the given shortcut entry matches the given key.
  *)

  VAR
    found : BOOLEAN;

  BEGIN
    qualifier:=qualifier-E.shiftMask;

    found:=FALSE;
    IF (sc.qualifier=qualifier) & (sc.char=CAP(char)) THEN
      found:=TRUE;
    ELSIF ((sc.mode=return) & (keysym=E.return))
       OR ((sc.mode=escape) & (keysym=E.escape))
       OR ((sc.mode=default) & ((keysym=E.return) OR (keysym=E.escape))) THEN
      found:=TRUE;
    END;

    RETURN found & ((sc.gadget.visible) OR (G.scAlways IN sc.gadget.flags));
  END Match;

  PROCEDURE (k : KeyHandler) GetSCEntry(qualifier : SET; keysym : LONGINT; char  : CHAR):ShortcutEntry;
  (**
    Returns the shortcut matching or NIL.
  *)

  VAR
    sc    : ShortcutEntry;
    found : BOOLEAN;

  BEGIN
    found:=FALSE;
    sc:=k.sFirst;

    qualifier:=qualifier-E.shiftMask;
    WHILE sc#NIL DO
      IF (sc.qualifier=qualifier) & (sc.char=CAP(char)) THEN
        found:=TRUE;
      ELSIF ((sc.mode=return) & (keysym=E.return))
         OR ((sc.mode=escape) & (keysym=E.escape))
         OR ((sc.mode=default) & ((keysym=E.return) OR (keysym=E.escape))) THEN
        found:=TRUE;
      END;

      IF found & ((sc.gadget.visible) OR (G.scAlways IN sc.gadget.flags)) THEN
        RETURN sc;
      ELSE
        found:=FALSE;
        sc:=sc.next;
      END;
    END;
    RETURN NIL;
  END GetSCEntry;

  PROCEDURE (k : KeyHandler) CancelCurrent;

  BEGIN
    IF k.sCurrent#NIL THEN
      k.sCurrent.gadget.HandleShortcutEvent(k.sCurrent.id,G.canceled);
    END;
    k.sCurrent:=NIL;
  END CancelCurrent;

  PROCEDURE (k : KeyHandler) HandleEvent(event : E.KeyEvent; preObject : BOOLEAN):BOOLEAN;

  VAR
    keysym : LONGINT;
    sc     : ShortcutEntry;
    buffer : ARRAY 256 OF CHAR;

  BEGIN
    IF preObject THEN
      keysym:=event.GetKey();
      IF event.GetText(buffer)>0 THEN
        IF event.type=E.keyDown THEN

          sc:=k.GetSCEntry(event.qualifier,keysym,buffer[0]);

          IF sc#NIL THEN
            IF sc#k.sCurrent THEN
              k.CancelCurrent;
              k.sCurrent:=sc;
              k.sCurrent.gadget.HandleShortcutEvent(sc.id,G.pressed);
              RETURN TRUE;
            END;
          ELSE
            k.CancelCurrent;
          END;
        ELSIF k.sCurrent#NIL THEN
          IF ~k.sCurrent.Match(event.qualifier,keysym,buffer[0]) THEN
            k.CancelCurrent;
          ELSE
            k.sCurrent.gadget.HandleShortcutEvent(k.current.id,G.released);
            k.sCurrent:=NIL;
          END;
        END;
      ELSE
        k.CancelCurrent;
      END;
      RETURN FALSE;
    ELSE
      IF k.current#NIL THEN
        IF k.current.HandleKeyEvent(event) THEN
          RETURN TRUE;
        END;
      END;

      IF event.type=E.keyDown THEN
        keysym:=event.GetKey();
        CASE keysym OF
          E.tab:
          IF event.qualifier={} THEN
            k.SetFocusNext;
            RETURN TRUE;
          END;
        | E.leftTab:
          IF event.qualifier=E.shiftMask THEN
            k.SetFocusLast;
            RETURN TRUE;
          END;
        | E.left,E.up:
          IF event.qualifier={} THEN
            k.SetFocusLastInGroup;
            RETURN TRUE;
          END;
        | E.right,E.down:
          IF event.qualifier={} THEN
            k.SetFocusNextInGroup;
            RETURN TRUE;
          END;
        ELSE
        END;
      END;

      RETURN FALSE;
    END;
  END HandleEvent;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.frame:=F.double3DOut;
  END Init;

  PROCEDURE (w : Window) Init*;

  BEGIN
    w.Init^;

    w.prefs:=prefs;

    w.top:=NIL;
    w.current:=NIL;
    NEW(w.keyHandler);
    w.keyHandler.Init;
    w.frame:=NIL;
  END Init;

  PROCEDURE (w : Window) SetTop*(top : G.Object);
  (**
    Set the top object for this window.
  *)

  BEGIN
    w.top:=top;
  END SetTop;

  PROCEDURE (w : Window) SetReference*(reference : G.Object);

  BEGIN
    w.reference:=reference;
  END SetReference;

  PROCEDURE (w : Window) GetReference():G.Object;

  BEGIN
    RETURN w.reference;
  END GetReference;

  PROCEDURE (w : Window) SetStyle*(style : LONGINT);

  BEGIN
    w.style:=style;

    IF w.style=normal THEN
      w.Grab(FALSE);
      w.Borderless(FALSE);
    ELSIF w.style=popup THEN
      w.Grab(TRUE);
      w.Borderless(TRUE);
    END;
  END SetStyle;

  PROCEDURE (w : Window) AddFocusObject*(gadget : G.Gadget);
  (**
    Adds a new gadget to the current KeyChain.
  *)

  BEGIN
    w.keyHandler.AddFocusGadget(gadget);
  END AddFocusObject;

  PROCEDURE (w : Window) AddFocusObjectToGroup*(gadget : G.Gadget);
  (**
    Adds a new gadget to the current KeyChain.
  *)

  BEGIN
    w.keyHandler.AddFocusGadgetToGroup(gadget);
  END AddFocusObjectToGroup;

  PROCEDURE (w : Window) AddShortcutObject*(gadget : G.Gadget; qualifier : SET;
                                            char : CHAR; id,mode : LONGINT);
  BEGIN
    w.keyHandler.AddShortcutObject(gadget,qualifier,char,id,mode);
  END AddShortcutObject;

  PROCEDURE (w : Window) CursorIsIn():BOOLEAN;

  VAR
    rx,ry,
    wx,wy  : LONGINT;

  BEGIN
    w.GetMousePos(rx,ry,wx,wy);
    IF (wx>=0) & (wx<=w.width-1) & (wy>=0) & (wy<=w.height-1) THEN
      RETURN TRUE;
    ELSE
      RETURN FALSE;
    END;
  END CursorIsIn;

  PROCEDURE (w : Window) CalcSize;

  BEGIN
    NEW(w.frame);
    w.frame.Init;
    IF w.style=normal THEN
      w.frame.SetFrame(F.none); (* Windowmanager does decoration *)
    ELSE
      w.frame.SetFrame(w.prefs.frame);
    END;
    w.frame.SetFlags({G.horizontalFlex,G.verticalFlex});
    w.frame.SetSpace(FALSE);
    w.frame.SetObject(w.top);
    w.frame.SetWindow(w);

    w.frame.CalcSize;
    IF (w.style=popup) & (w.reference#NIL) THEN
      (* Is is nice, when reference and popup have the same width *)
      w.frame.Resize(w.reference.oWidth,-1);
    END;
    w.SetSize(w.frame.oWidth,w.frame.oHeight);

    IF G.horizontalFlex IN w.top.flags THEN
      w.SetMinSize(w.frame.oMinWidth,-1);
      w.SetMaxSize(w.frame.oMaxWidth,-1);
    ELSE
      w.SetMinSize(w.width,-1);
      w.SetMaxSize(w.width,-1);
    END;
    IF G.verticalFlex IN w.top.flags THEN
      w.SetMinSize(-1,w.frame.oMinHeight);
      w.SetMaxSize(-1,w.frame.oMaxHeight);
    ELSE
      w.SetMinSize(-1,w.height);
      w.SetMaxSize(-1,w.height);
    END;
  END CalcSize;

  PROCEDURE (w : Window) PreInit*;
  (**
     Creates a window on the display with top as top object
     and with title as title.
  *)

  VAR
    parent : D.Window;
    x,y    : LONGINT;

  BEGIN
    w.CalcSize;

    IF (w.style=popup) & (w.reference#NIL) THEN
      parent:=w.reference.GetWindow();
      parent.GetXY(x,y);
      INC(x,w.reference.oX);
      INC(y,w.reference.oY+w.reference.oHeight);

      x:=U.RoundRange(x,0,D.display.scrWidth-1-w.width);
      y:=U.RoundRange(y,0,D.display.scrHeight-1-w.height);

      w.SetPos(x,y);
      w.SetPosition(D.manualPos,D.manualPos);
    END;

    w.PreInit^;
  END PreInit;

  PROCEDURE (w : Window) ReinitWindow*;

  VAR
    maped : BOOLEAN;

  BEGIN
    maped:=w.IsMaped();
    IF maped THEN
      w.top.Hide;
    END;

    w.CalcSize;

    w.Resize(w.frame.oWidth,w.frame.oHeight);

    IF w.IsMaped() THEN
      w.frame.Move(0,0);
      w.frame.Draw(0,0,w.width,w.height);
      w.keyHandler.SetFocusFirst;
    ELSE
      w.frame.Resize(w.width,w.height);
    END;
  END ReinitWindow;

  (* -- handler for some common windowevents -- *)

  PROCEDURE (w : Window) ClosePressed*;

  VAR
    close : CloseMsg;

  BEGIN
    NEW(close);
    w.Send(close,closeMsg);
  END ClosePressed;

  PROCEDURE (w : Window) Unmaped*;

  BEGIN
    w.Unmaped^;
    w.frame.Hide;
  END Unmaped;

  PROCEDURE (w : Window) FocusIn*;

  BEGIN
    w.FocusIn^;
    w.keyHandler.Activate;
  END FocusIn;

  PROCEDURE (w : Window) FocusOut*;

  BEGIN
    w.FocusOut^;
    w.keyHandler.Deactivate;
  END FocusOut;

  PROCEDURE (w : Window) Redraw*(x,y,width,height : LONGINT);

  BEGIN
    w.frame.Move(0,0);
    w.frame.Draw(x,y,width,height);
  END Redraw;

  PROCEDURE (w : Window) Resized*(width,height : LONGINT);

  BEGIN
    w.frame.Resize(width,height);
    w.frame.Move(0,0);
    w.frame.Draw(0,0,w.width,w.height);
  END Resized;

  PROCEDURE (w : Window) GetPosObject(mode : LONGINT):G.Object;
  (**
    Returns the object under the cursor to supports the given mode.
  *)

  VAR
    rx,ry,
    wx,wy : LONGINT;

  BEGIN
    w.GetMousePos(rx,ry,wx,wy);
    RETURN w.frame.GetPosObject(wx,wy,mode);
  END GetPosObject;

  PROCEDURE (w : Window) GetDnDObject*(x,y : LONGINT; drag : BOOLEAN):G.Object;
  (**
    Returns the object that coveres the given point and that supports
    dragging of data.

    If drag is TRUE, when want to find a object that we can drag data from,
    else we want an object to drop data on.
  *)

  BEGIN
    RETURN w.frame.GetDnDObject(x,y,drag);
  END GetDnDObject;

  PROCEDURE (w : Window) ContextHelp*;
  (**
    Show the context help of an object under the cursor.
  *)

  VAR
    object : G.Object;
    help   : D.Window;

  BEGIN
    IF w.current=NIL THEN
      object:=w.GetPosObject(G.helpGadget);
      IF object#NIL THEN
        help:=object.GetHelpObject();
        IF help#NIL  THEN
          help.Open;
        END;
      END;
    END;
  END ContextHelp;

  PROCEDURE (w : Window) ContextMenu*():BOOLEAN;
  (**
    Open the context menu for an object under the cursor.
  *)

  VAR
    object : G.Object;
    menu   : D.Window;

  BEGIN
    object:=w.GetPosObject(G.menuGadget);
    IF object#NIL THEN
      menu:=object.GetMenuObject();
      IF menu#NIL THEN
        menu.Open;
        RETURN TRUE;
      END;
    END;
    RETURN FALSE;
  END ContextMenu;

  PROCEDURE (w : Window) FocusNext*;
  (**
    Move the keyboard focus to the next valid object.
  *)

  BEGIN
    w.keyHandler.SetFocusNext;
  END FocusNext;

  PROCEDURE (w : Window) SetFocus*(gadget : G.Gadget);
  (**
  *)

  BEGIN
    w.keyHandler.SetFocus(gadget);
  END SetFocus;

  PROCEDURE (w : Window) HandleEvent*(event : E.Event):BOOLEAN;

  BEGIN
    IF w.HandleEvent^(event) THEN
      RETURN TRUE;
    END;

    (*
      We close the window if we are popup and the user clicks
      outside the window.
    *)
    IF (w.style=popup) & (event IS E.ButtonEvent) & (event(E.ButtonEvent).type=E.mouseDown) THEN
      IF ~w.CursorIsIn() THEN
        IF w.IsInEventLoop() THEN
          w.Exit;
          w.Close;
        ELSE
          w.Close;
        END;
        RETURN TRUE;
      END;
    END;

    (*
      Before everything else we send key events to the keyhandler
    *)
    IF (event IS E.KeyEvent) THEN
      IF w.keyHandler.HandleEvent(event(E.KeyEvent),TRUE) THEN
        RETURN TRUE;
      END;
    END;

    IF (event IS E.MouseEvent) THEN
      IF w.current#NIL THEN
        IF w.current.HandleMouseEvent(event(E.MouseEvent),w.current) THEN
(*          Err.String("."); Err.Ln;*)
        END;

(*        IF w.current=NIL THEN
          Err.String("-"); Err.Ln; Err.Ln;
        END;*)
        (* Should we further delegate event? *)
      ELSE
        IF w.frame.HandleMouseEvent(event(E.MouseEvent),w.current) THEN
          IF w.current#NIL THEN
(*            Err.String("+"); Err.Ln;*)
            w.keyHandler.SetFocus(w.current(G.Gadget));
          END;
        END;
      END;
    ELSIF (event IS E.KeyEvent) THEN
      IF w.keyHandler.HandleEvent(event(E.KeyEvent),FALSE) THEN END;
    END;

    RETURN TRUE;
  END HandleEvent;

  PROCEDURE (w : Window) Receive*(message : O.Message);

  BEGIN
    WITH
      message: OpenMsg DO
        w.Open;
    | message: OpenModalMsg DO
        w.Open;
        w.EventLoop;
        w.Close;
    | message: CloseMsg DO
        w.Close;
    | message: ExitMsg DO
        w.Exit;
    ELSE
    END;
  END Receive;

  (*
    some predefined message-converter
  *)

  PROCEDURE (h : Msg2Open) Convert*(message : O.Message):O.Message;

  VAR
    new : OpenMsg;

  BEGIN
    NEW(new);
    RETURN new;
  END Convert;

  PROCEDURE (h : Msg2OpenModal) Convert*(message : O.Message):O.Message;

  VAR
    new : OpenModalMsg;

  BEGIN
    NEW(new);
    RETURN new;
  END Convert;

  PROCEDURE (h : Msg2Close) Convert*(message : O.Message):O.Message;

  VAR
    new : CloseMsg;

  BEGIN
    NEW(new);
    RETURN new;
  END Convert;

  PROCEDURE (h : Msg2Exit) Convert*(message : O.Message):O.Message;

  VAR
    new : ExitMsg;

  BEGIN
    NEW(new);
    RETURN new;
  END Convert;

  PROCEDURE CreateWindow*():Window;

  VAR
    window : Window;

  BEGIN
    NEW(window);
    window.Init;

    RETURN window;
  END CreateWindow;

BEGIN
  NEW(prefs);
  prefs.Init;

END VO:Window.