(**
  Baseclass for all displayable objects.
*)

MODULE VO:Object;

(*
    Basesclass for all graphical objects.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)


IMPORT A := VO:Base:Adjustment,
       D := VO:Base:Display,
       E := VO:Base:Event,
       F := VO:Base:Frame,
       O := VO:Base:Object,
       U := VO:Base:Util,
       Z := VO:Base:Size,

       P := VO:Prefs:Base;

CONST

  (* values for Object.flags *)

  horizontalFlex * =  0; (* Object can change its size in horizontal direction              *)
  verticalFlex   * =  1; (* Object can change its size in vertical direction                *)
  noHighlight    * =  2; (* Object shouldn't highlight when selected                        *)
  stdFocus       * =  3; (* Object uses the std way of displ. focus, ie. Object does it all *)
  hasFocus       * =  4; (* The objects currently has the focus                             *)
  canFocus       * =  5; (* We can handle keyboard focus events                             *)
  mayFocus       * =  6; (* It is possible that the object must draw its focus frame, it should therfor
                           interpet this flag to reserve some display space etc... *)
  showFocus      * =  7; (* We can't handle focus events, but should display one. Likely to happend,
                           when focus displaying has been delegated from super object      *)
  handleSC       * =  8; (* Object can handle shortcuts                                     *)
  scAlways       * =  9; (* Can handle sc even when not visible (menus f.e.)                *)
  canDisable     * = 10; (* Object has special handling for drawing disabled *)
  inited         * = 11; (* CalcSize has been called at least one *)

  (* values for minXXX and maxXXX *)

  minSize = 0;
  maxSize = MAX(LONGINT)-4;   (* Because of hardcoded focusborderwidth of 2 than might be added *)

  (* Constants for Object.GetPosObject *)

  menuGadget * = 0; (* Each Object can have its private context-sensitive menu       *)
  helpGadget * = 1; (* Each Object can have its private context-sensitive helpobject *)
  dragGadget * = 2; (* An object you can drag data from *)

  (* Constants for different kind of alignments an object may support *)

  alignLeft      * = 0;
  alignCenter    * = 1;
  alignRight     * = 2;
  alignBound     * = 3;
  alignTop       * = 4;
  alignBottom    * = 5;

  alignmentCount * = 6;

  (* shortcut events states *)

  pressed  * = 0;
  released * = 1;
  canceled * = 2;

TYPE
  AlignmentName = ARRAY 15 OF CHAR;

  Object*     = POINTER TO ObjectDesc;

  Background*     = POINTER TO BackgroundDesc;
  BackgroundDesc* = RECORD
  (**
    An object can have a background color or an background object that is responsible
    for drawing the given region in the background style. Background object could draw
    the background simply using the background color (though you can achive this b
    y simply setting the background color without using an background object),
    a regular pattern, a titled image, an image stretched to fit the whole object or
    something even more complicated.

    The apropiate method of filling the background will be handled the DrawBackground
    method. It will fill the given area in the backgrojund area or will delegate
    drawing to the background object - if available.

    source stores the orign of this Background. E.g., yo assign a background object
    to a button. The button will delegate the background to it label. However in some
    cases the background must not be rendered relative to the size of the label but
    still relative to the button.
  *)
                      source* : Object; (* The object, the background is background of.*)
                    END;


  Prefs*      = POINTER TO PrefsDesc;
  PrefsDesc*  = RECORD (P.PrefsDesc)
  (**
    Special Preferences object for graphical objects.
  *)

                  background* : Background;
                  frame*      : LONGINT;
                END;

  ObjectDesc* = RECORD (D.ObjectDesc)
  (**
    Special Preferences object for graphical objects.
  *)

                  next*,                             (* All objects a singlelinked *)
                  parent-              : Object;

                  (* preferences *)
                  prefs-               : Prefs;

                  (* Sizing stuff *)

                  x*,y*,                             (* The current bounds of the object *)
                  width*,height*,

                  minWidth*,minHeight*,              (* Its minimal and maximal bounds *)
                  maxWidth*,maxHeight* : LONGINT;

                  oX-,oY-,
                  oWidth-,oHeight-    : LONGINT;     (* The outer bounds of the object
                                                        Inner bounds + possible focus frame *)

                  oMinWidth*,
                  oMinHeight*,                       (* Its minimal and maximal bounds *)
                  oMaxWidth*,
                  oMaxHeight*          : LONGINT;

                  setWidth,
                  setHeight,
                  setMinWidth,
                  setMinHeight         : Z.SizeDesc;

                  (* generalFlags *)

                  flags-               : SET;        (* Some special flags *)

                  (* visibility stuff *)

                  background-          : LONGINT;
                  backgroundObject-    : Background;
                  visible-,                          (* The object is currently visible, i.e. not hidden *)
                  disabled-,
                  layouted             : BOOLEAN;    (* Every object can be disabled *)

                  (* objects *)
                  window               : D.Window;
                  menu                 : D.Window;   (* pointer the context sensitive menuobject *)
                  help                 : D.Window;   (* pointer the context sensitive helpobject *)

                  objectFrameType-     : LONGINT;
                  objectFrame-         : F.Frame;

                  focus-               : F.Frame;
                  label-               : Object;     (* An object can have a label *)
                END;

  Control*     = POINTER TO ControlDesc;
  ControlDesc* = RECORD (ObjectDesc)
  (**
    New baseclasses that distinguish between objects which
    control data and those which are used to control
    layout.
   *)
                  END;

  LayoutObject * = POINTER TO LayoutObjectDesc;
  LayoutObjectDesc * = RECORD (ObjectDesc)
                       END;

  Image*      = POINTER TO ImageDesc;  (* An image is a special kind of Object *)
  ImageDesc*  = RECORD (ObjectDesc)
  (**
    The baseclass for all images. Images do not have any event-handling.
  *)
                END;


  Gadget*     = POINTER TO GadgetDesc;
  GadgetDesc* = RECORD (ControlDesc)
  (**
    The baseclass for all gadgets. Gadgets have methods for event handling, unlike images for example.
  *)
                END;


  ScrollableGadget*     = POINTER TO ScrollableGadgetDesc;
  ScrollableGadgetDesc* = RECORD (GadgetDesc)
  (**
    The baseclass for all scrollable gadgets.
  *)
                            hAdjustment*,
                            vAdjustment* : A.Adjustment;
                          END;

  Group*      = POINTER TO GroupDesc;
  GroupDesc*  = RECORD (LayoutObjectDesc)
  (**
    An group is a special kind of Object. Although it is a collection of
    objects, it acts only as a layout instance. So it inherits from
    LayoutObject.
    It collects a number of object and presents them as one object to
    the rest of the world.
  *)
                  list*,               (* A linked list of all members *)
                  last*  : Object;
                  count* : LONGINT;    (* The number of members *)
                END;

 (* -- messages -- *)

  LabelClickedMsg*     = POINTER TO LabelClickedMsgDesc;
  LabelClickedMsgDesc* = RECORD (O.MessageDesc)
  (**
    The LabelClickedMsg maybe generated when someone clicks on your label.
  *)
                         END;


VAR
  alignments- : ARRAY alignmentCount OF AlignmentName;

  (* -------------------------------------------- *)

  PROCEDURE (o : Object) Init*;
  (**
    Every object needs an initialiser.

    If you provide an Init-method for an object you must first cann the
    Init-method of the baseclass.

    Note, that this does not hinder you ro provide function-like initializer, too.
    They just have to call this method.
  *)

  BEGIN
    o.Init^;
    o.x:=0;
    o.y:=0;

    o.oX:=0;
    o.oY:=0;

    o.width:=0;
    o.height:=0;

    o.oWidth:=0;
    o.oHeight:=0;

    o.next:=NIL;

    o.parent:=NIL;

    o.minWidth:=minSize;
    o.minHeight:=minSize;
    o.maxWidth:=maxSize;
    o.maxHeight:=maxSize;

    o.oMinWidth:=minSize;
    o.oMinHeight:=minSize;
    o.oMaxWidth:=maxSize;
    o.oMaxHeight:=maxSize;

    o.setWidth.Init;
    o.setHeight.Init;
    o.setMinWidth.Init;
    o.setMinHeight.Init;

    o.flags:={stdFocus};
    o.visible:=FALSE;
    o.disabled:=FALSE;

    o.layouted:=FALSE;

    o.focus:=NIL;
    o.label:=NIL;

    o.background:=D.backgroundColor;
    o.backgroundObject:=NIL;
    o.window:=NIL;
    o.menu:=NIL;
    o.help:=NIL;

    o.objectFrameType:=-1; (* THis is a trick! *)
    o.objectFrame:=NIL;
  END Init;

  PROCEDURE (o : Object) SetFlags*(flags : SET);
  (**
    Via this functions you can add the given flags to the
    flags already set.
  *)

  BEGIN
    o.flags:=o.flags+flags;
  END SetFlags;

  PROCEDURE (o : Object) RemoveFlags*(flags : SET);
  (**
    Via this functions you can add the given flags to the
    flags already set.
  *)

  BEGIN
    o.flags:=o.flags-flags;
  END RemoveFlags;

  PROCEDURE (o : Object) ClearFlags*;
  (**
    Via this functions you can clear all flags.

    NOTE
    This is a dangerous operations. There are many internal flags used by
    VO and it is likely that some things will not operate the way you
    want them to. Use removeFlags instead whenever possible.
  *)

  BEGIN
    o.flags:={};
  END ClearFlags;

  PROCEDURE (o : Object) SetWidth*(mode, value : LONGINT);
  (**
    Set the width of the object.
    "Mode" defines the interpretation of the given value.
  *)

  BEGIN
    o.setWidth.SetSize(mode,value);
  END SetWidth;

  PROCEDURE (o : Object) SetHeight*(mode, value : LONGINT);
  (**
    Set the height of the object.
    "Mode" defines the interpretation of the given value.
  *)

  BEGIN
    o.setHeight.SetSize(mode,value);
  END SetHeight;

  PROCEDURE (o : Object) SetMinWidth*(mode, value : LONGINT);
  (**
    Set the minWidth of the object.
    "Mode" defines the interpretation of the given value.
  *)

  BEGIN
    o.setMinWidth.SetSize(mode,value);
  END SetMinWidth;

  PROCEDURE (o : Object) SetMinHeight*(mode, value : LONGINT);
  (**
    Set the minHeight of the object.
    "Mode" defines the interpretation of the given value.
  *)

  BEGIN
    o.setMinHeight.SetSize(mode,value);
  END SetMinHeight;

  PROCEDURE (o : Object) SetBackground*(backgroundColor : D.Color);
  (**
    Set the background color for the object. Behaviour for complex/group
    objects is undefined.
  *)

  BEGIN
    o.background:=backgroundColor;
    o.backgroundObject:=NIL;
  END SetBackground;

  PROCEDURE (o : Object) SetBackgroundObject*(background : Background);
  (**
    Set the background object for the object. Behaviour for complex/group
    objects is undefined.
  *)

  BEGIN
    o.backgroundObject:=background;
  END SetBackgroundObject;

  PROCEDURE (o : Object) CopyBackground*(destination : Object);

  BEGIN
    ASSERT(destination#NIL);

    destination.backgroundObject:=o.backgroundObject;
    destination.background:=o.background;
  END CopyBackground;

  PROCEDURE (o : Object) SetLabelObject*(label : Object);
  (**
    Assign a label to the object. Some object may support some
    tricky operations with their label. A checkbox f.e. may delegate its
    focus frame to its label.
  *)

  BEGIN
    o.label:=label;
  END SetLabelObject;

  PROCEDURE (o : Object) SetWindow*(window : D.Window);

  BEGIN
    ASSERT((o.window=NIL) OR (o.window=window));

    o.window:=window;
  END SetWindow;

  PROCEDURE (o : Object) GetWindow*():D.Window;
  (**
    Returns the help window for this object.

    NOTE
    Special objects may overload this method to f.e. implement dynamic tooltips.
    They must then also overload GetPosObject.
  *)

  BEGIN
    IF o.window#NIL THEN
      RETURN o.window;
    ELSIF o.parent#NIL THEN
      o.window:=o.parent.GetWindow();
      RETURN o.window;
    ELSE
       RETURN NIL;
     END;
  END GetWindow;

  PROCEDURE (o : Object) SetParent*(parent : Object);
  (**
    Sets the parent object of this object.

    NOTE
    Normaly the objects tries to get its windows and thus is draw info object
    asking its parent. So be carefull to set a parent object for each object
    use instantiate. Most time this is automatically handled by the container
    object.

    Since top level object cannot have a parent, you must explicitely set
    the window calling SetWindow().
  *)

  BEGIN
(*    ASSERT((o.parent=NIL) OR (o.parent=parent));*)

    o.parent:=parent;
    o.window:=NIL;
  END SetParent;

  PROCEDURE (o : Object) GetDrawInfo*():D.DrawInfo;
  (**
    Returns the VODisplay.DrawInfo used for all drawing.
  *)

  VAR
    window : D.Window;

  BEGIN
    window:=o.GetWindow();

    RETURN window.GetDrawInfo();
  END GetDrawInfo;

  PROCEDURE (o : Object) SetObjectFrame*(frame : LONGINT);

  BEGIN
    o.objectFrameType:=frame;
  END SetObjectFrame;

  PROCEDURE (o : Object) SetHelpObject*(helpObject : D.Window);
  (**
    Assign an window that will displayed as tooltip. The baseclass will store
    the object within a private variable and will return it when GetHelpObject gets
    called.

    NOTE
    Special objects may overload this method this implement different behaviour.
    They must then also overload GetPosObject.
  *)

  BEGIN
    o.help:=helpObject;
  END SetHelpObject;

  PROCEDURE (o : Object) GetHelpObject*():D.Window;
  (**
    Returns the help window for this object.

    NOTE
    Special objects may overload this method to f.e. implement dynamic tooltips.
    They must then also overload GetPosObject.
  *)

  BEGIN
    RETURN o.help;
  END GetHelpObject;

  PROCEDURE (o : Object) SetMenuObject*(menuObject : D.Window);
  (**
    Assign a window that will displayed as menu. The baseclass will store
    the object within a private variable and will return it when GetMenuObject gets
    called.

    NOTE
    Special objects may overload this method this implement different behaviour.
    They must then also overload GetPosObject.
  *)

  BEGIN
    o.menu:=menuObject;
  END SetMenuObject;

  PROCEDURE (o : Object) GetMenuObject*():D.Window;
  (**
    Returns the menu window for this object.

    NOTE
    Special objects may overload this method to f.e. implement dynamic menues.
    They must then also overload GetPosObject.
  *)

  BEGIN
    RETURN o.menu;
  END GetMenuObject;

  PROCEDURE (o : Object) SetFocusObject*(focus : F.Frame);

  BEGIN
    o.focus:=focus;
  END SetFocusObject;

  PROCEDURE (o : Object) StdFocus*():BOOLEAN;
  (**
    Returns TRUE, when the object displays itself using the standard functionality
    of displaying focus frameing, e.g. stdFocus
    is in Object.flags.
  *)

  BEGIN
    RETURN stdFocus IN o.flags;
  END StdFocus;

  PROCEDURE (o : Object) HasFocus*():BOOLEAN;
  (**
    Returns TRUE, when the object has the focus, e.g. hasFocus
    is in Object.flags.
  *)

  BEGIN
    RETURN hasFocus IN o.flags;
  END HasFocus;

  PROCEDURE (o : Object) CanFocus*():BOOLEAN;
  (**
    Returns TRUE, when the object can handle keyboard focus events,a e.g. canFocus
    is in Object.flags.
  *)

  BEGIN
    RETURN canFocus IN o.flags;
  END CanFocus;

  PROCEDURE (o : Object) MayFocus*():BOOLEAN;
  (**
    Returns TRUE, when the object may show its focusframe,a e.g. mayFocus
    is in Object.flags.
  *)

  BEGIN
    RETURN mayFocus IN o.flags;
  END MayFocus;

  PROCEDURE (o : Object) ShowFocus*():BOOLEAN;
  (**
    Returns TRUE, when the object should show its focusframe,a e.g. showFocus
    is in Object.flags.
  *)

  BEGIN
    RETURN showFocus IN o.flags;
  END ShowFocus;

  PROCEDURE (o : Object) DisplayFocus*():BOOLEAN;
  (**
    Returns TRUE, when the object should display its focusframe,a e.g. showFocus
    or hasFocus is in Object.flags.
  *)

  BEGIN
    RETURN (hasFocus IN o.flags) OR (showFocus IN o.flags);
  END DisplayFocus;

  (* -------------------------------------------- *)

  PROCEDURE (b : Background) CalcSize*;
  (**
    Initilalize the object with the display.
  *)

  BEGIN
  END CalcSize;

  PROCEDURE (b : Background) Draw*(draw : D.DrawInfo; x,y,w,h : LONGINT);
  (**
    Draw the background. The baseclass will simply fill the given area
    with the defined object background color.
  *)

  BEGIN
    draw.PushForeground(D.backgroundColor);
    draw.FillRectangle(x,y,w,h);
    draw.PopForeground;
  END Draw;

  PROCEDURE (b : Background) Copy*():Background;
  (**
    Return a copy of the current background object.
  *)

  VAR
    copy : Background;

  BEGIN
    NEW(copy);
    copy^:=b^;

    RETURN copy;
  END Copy;

  PROCEDURE (b : Background) Free*;
  (**
    Free internal data.
  *)

  BEGIN
  END Free;

  (* -------------------------------------------- *)

  PROCEDURE (o : Object) SetPrefs*(p : Prefs);

  BEGIN
    o.prefs:=p; (* We set the prefs *)

    IF p.background#NIL THEN
      o.SetBackgroundObject(p.background.Copy());
      o.backgroundObject.source:=o;
    END;
  END SetPrefs;

  PROCEDURE (o : Object) CalcSize*;
  (**
    Tell the object it should calculate its size depending on its current settings

    This method normally gets calleed once before the first call of ObjectDraw.
    The window calls this method for its top object before it opens itself
    and draws its content. The top object call this method automatically for all
    its children.

    If this method get called for you object you normally have to simly
    propagate it to all your children and set the values for minWidth,
    minHeight, width and height. The setting of the max-values is optional,
    the defaultvalue for them is MAX(LONGINT).

    After this your *must* call this method of your baseclass.
  *)

  BEGIN
    IF (o.objectFrameType<0) THEN
      IF o.prefs#NIL THEN
        o.SetObjectFrame(o.prefs.frame);
      ELSE
        o.SetObjectFrame(F.none);
      END;
    END;

    IF o.setMinWidth.IsSizeSet() THEN
      o.minWidth:=U.RoundRange(o.setMinWidth.GetSize(),o.minWidth,o.maxWidth);
    END;

    IF o.setMinHeight.IsSizeSet() THEN
      o.minHeight:=U.RoundRange(o.setMinHeight.GetSize(),o.minHeight,o.maxHeight);
    END;

    o.oMinWidth:=o.minWidth;
    o.oMinHeight:=o.minHeight;

    o.oMaxWidth:=o.maxWidth;
    o.oMaxHeight:=o.maxHeight;

    o.width:=U.RoundRange(o.width,o.minWidth,o.maxWidth);
    o.height:=U.RoundRange(o.height,o.minHeight,o.maxHeight);


    IF o.setWidth.IsSizeSet() THEN
      o.width:=U.RoundRange(o.setWidth.GetSize(),o.minWidth,o.maxWidth);
    END;

    IF o.setHeight.IsSizeSet() THEN
      o.height:=U.RoundRange(o.setHeight.GetSize(),o.minHeight,o.maxHeight);
    END;

    o.oWidth:=o.width;
    o.oHeight:=o.height;

    IF o.objectFrame=NIL THEN
      NEW(o.objectFrame);
      o.objectFrame.Init;
    END;

    o.objectFrame.SetFrame(o.objectFrameType);

    INC(o.oWidth,o.objectFrame.leftBorder+o.objectFrame.rightBorder);
    INC(o.oHeight,o.objectFrame.topBorder+o.objectFrame.bottomBorder);

    INC(o.oMinWidth,o.objectFrame.leftBorder+o.objectFrame.rightBorder);
    INC(o.oMinHeight,o.objectFrame.topBorder+o.objectFrame.bottomBorder);
    INC(o.oMaxWidth,o.objectFrame.leftBorder+o.objectFrame.rightBorder);
    INC(o.oMaxHeight,o.objectFrame.topBorder+o.objectFrame.bottomBorder);

    IF o.MayFocus() & o.StdFocus() THEN
      INC(o.oWidth,o.focus.leftBorder+o.focus.rightBorder);
      INC(o.oHeight,o.focus.topBorder+o.focus.bottomBorder);

      INC(o.oMinWidth,o.focus.leftBorder+o.focus.rightBorder);
      INC(o.oMinHeight,o.focus.topBorder+o.focus.bottomBorder);
      INC(o.oMaxWidth,o.focus.leftBorder+o.focus.rightBorder);
      INC(o.oMaxHeight,o.focus.topBorder+o.focus.bottomBorder);
    END;

    IF ~(horizontalFlex IN o.flags) THEN
      o.oMinWidth:=o.oWidth;
      o.minWidth:=o.width;
      o.oMaxWidth:=o.oWidth;
      o.maxWidth:=o.width;
    END;
    IF ~(verticalFlex IN o.flags) THEN
      o.oMinHeight:=o.oHeight;
      o.minHeight:=o.height;
      o.oMaxHeight:=o.oHeight;
      o.maxHeight:=o.height;
    END;

    INCL(o.flags,inited);

    IF o.backgroundObject#NIL THEN
      o.backgroundObject.CalcSize;
    END;

    o.layouted:=FALSE;
  END CalcSize;

  PROCEDURE (o : Object) CanGrow*(horiz : BOOLEAN):BOOLEAN;
  (**
    Returns true, if the object can grow in the stated direction

    An object is resizable, if

    * It hase the xxxFlex-flag set for that direction

    * And if its current size is smaller than it maximal size

  *)

  BEGIN
    IF horiz THEN
      IF horizontalFlex IN o.flags THEN
        RETURN o.width<o.maxWidth;
      ELSE
        RETURN FALSE;
      END;
    ELSE
      IF verticalFlex IN o.flags THEN
        RETURN o.height<o.maxHeight;
      ELSE
        RETURN FALSE;
      END;
    END;
  END CanGrow;

  PROCEDURE (o : Object) CanShrink*(horiz : BOOLEAN):BOOLEAN;
  (**
    Same as Object.CanGrow only for shrinking
  *)

  BEGIN
    IF horiz THEN
      IF horizontalFlex IN o.flags THEN
        RETURN o.width>o.minWidth;
      ELSE
        RETURN FALSE;
      END;
    ELSE
      IF verticalFlex IN o.flags THEN
        RETURN o.height>o.minHeight;
      ELSE
        RETURN FALSE;
      END;
    END;
  END CanShrink;

  PROCEDURE (o : Object) CanResize*(grow, horiz : BOOLEAN):BOOLEAN;
  (**
    This calls Object.CanGrow or Object.Can.Shrink depending on its options
  *)

  BEGIN
    IF grow THEN
      RETURN o.CanGrow(horiz);
    ELSE
      RETURN o.CanShrink(horiz);
    END;
  END CanResize;

  PROCEDURE (o : Object) CanDrag*():BOOLEAN;
  (**
    Returns TRUE if you can drag data from the object.
  *)

  BEGIN
    RETURN FALSE;
  END CanDrag;

  PROCEDURE (o : Object) Resize*(width,height : LONGINT);
  (**
    This function tries to resize the object to the given measurement

    Note, that the object may not have the giving size after calling since
    it have the follow the settings for o.flags and it maximal size

    Normally the features of this implementation are enought.
    But you can overload it (At the moment none of the classes does this).

  *)

  VAR
    oldOWidth,oldOHeight : LONGINT;

  BEGIN
    oldOWidth:=o.oWidth;
    oldOHeight:=o.oHeight;

    IF o.MayFocus() & o.StdFocus() THEN
      IF (horizontalFlex IN o.flags) & (width>=0) THEN
        o.oWidth:=U.MaxLong(width,o.oMinWidth);
        o.width:=o.oWidth-o.focus.leftBorder-o.focus.rightBorder
                         -o.objectFrame.leftBorder-o.objectFrame.rightBorder;
      END;
      IF (verticalFlex IN o.flags) & (height>=0) THEN
        o.oHeight:=U.MaxLong(height,o.oMinHeight);
        o.height:=o.oHeight-o.focus.topBorder-o.focus.bottomBorder
                           -o.objectFrame.topBorder-o.objectFrame.bottomBorder;
      END;
    ELSE
      IF (horizontalFlex IN o.flags) & (width>=0) THEN
        o.oWidth:=U.MaxLong(width,o.oMinWidth);
        o.width:=o.oWidth-o.objectFrame.leftBorder-o.objectFrame.rightBorder;
      END;
      IF (verticalFlex IN o.flags) & (height>=0) THEN
        o.oHeight:=U.MaxLong(height,o.oMinHeight);
        o.height:=o.oHeight-o.objectFrame.topBorder-o.objectFrame.bottomBorder;
      END;
    END;

    IF (oldOWidth#o.oWidth) OR (oldOHeight#o.oHeight) THEN
      o.layouted:=FALSE;
    END;
  END Resize;

  PROCEDURE (o : Object) SetRelayout*;

  BEGIN
    o.layouted:=FALSE;
  END SetRelayout;

  PROCEDURE (o : Object) Layout*;

  BEGIN
    o.layouted:=TRUE;
  END Layout;

  PROCEDURE (o : Object) Draw*(x,y,w,h : LONGINT);
  (**
    Tells the object to draw itself at the given position and with the mode
    set in draw.mode.

    You can igonore modes you do not support and simply assume mode=normal.

    You *must* call the Draw-method of your superclass first!
  *)

  VAR
    draw : D.DrawInfo;
    a,b  : LONGINT;

  BEGIN
    o.visible:=TRUE;

    draw:=o.GetDrawInfo();

    a:=o.oX;
    b:=o.oY;

    IF o.MayFocus() & o.StdFocus() THEN
      IF o.DisplayFocus() & (o.focus#NIL) THEN
        o.focus.Draw(draw,a,b,o.oWidth,o.oHeight);
      END;
      INC(a,o.focus.leftBorder);
      INC(b,o.focus.topBorder);
    END;

    o.objectFrame.Draw(draw,
                       a,b,
                       o.width+o.objectFrame.leftBorder+o.objectFrame.rightBorder,
                       o.height+o.objectFrame.topBorder+o.objectFrame.bottomBorder);

    IF ~o.layouted THEN
      o.Layout;
    END;
  END Draw;

  PROCEDURE (o : Object) Move*(x,y : LONGINT);
  (**
  *)

  VAR
    oldOX,oldOY : LONGINT;

  BEGIN
    oldOX:=o.oX;
    oldOY:=o.oY;

    o.oX:=x;
    o.oY:=y;

    o.x:=x;
    o.y:=y;

    IF o.MayFocus() & o.StdFocus() THEN
      INC(o.x,o.focus.leftBorder);
      INC(o.y,o.focus.topBorder);
    END;

    INC(o.x,o.objectFrame.leftBorder);
    INC(o.y,o.objectFrame.topBorder);

    IF (oldOX#o.oX) OR (oldOY#o.oY) THEN
      o.layouted:=FALSE;
    END;
  END Move;

  PROCEDURE (o : Object) MoveResize*(x,y,width,height : LONGINT);

  BEGIN
    o.Move(x,y);
    o.Resize(width,height);
  END MoveResize;

  PROCEDURE (o : Object) Redraw*;
  (**
    Tells the object to redraw itself
    You normally do not need to overload this method.
  *)

  BEGIN
    IF o.visible THEN
      o.Draw(o.oX,o.oY,o.oWidth,o.oHeight);
    END;
  END Redraw;

  PROCEDURE (o : Object) Intersect*(x,y,w,h : LONGINT):BOOLEAN;
  (**
    Returns TRUE if the two areas intersect, else FALSE.
  *)

  BEGIN
    RETURN ~(
       (y+h-1<o.oY) (* above *)
    OR (y>o.oY+o.oHeight-1) (* under *)
    OR (x+w-1<o.oX) (* left *)
    OR (x>o.oX+o.oWidth-1)); (* right *)
  END Intersect;
(*
  PROCEDURE (o : Object) RestrictToBounds*(VAR x,y,w,h : LONGINT);

  BEGIN
    IF x<o.oX THEN
      DEC(w,o.oX-x);
      x:=o.oX;
    END;
    IF x>=o.oX+o.oWidth THEN
      x:=o.oX+o.oWidth-1;
      w:=0;
    END;

    IF y<o.oY THEN
      DEC(h,o.oY-y);
      y:=o.oY;
    END;
    IF y>=o.oY+o.oHeight THEN
      y:=o.oY+o.oHeight-1;
      h:=0;
    END;

    IF x+w>o.oX+o.oWidth-1 THEN
      w:=o.oWidth-(x-o.oX);
    END;

    IF y+h>o.oY+o.oHeight-1 THEN
      h:=o.oHeight-(y-o.oY);
    END;
  END RestrictToBounds;
*)

  PROCEDURE (o : Object) DrawDisabled*;
  (**
    This is just a convinience function. Call this if your are a gadget and want to
    show yourself disabled. It draws a pattern over the bound of the gadget.

    Not that this is just a proposal. You can of cause use other methods to show yourself
    disabled, but keep some style guide in mind.
  *)

  VAR
    draw : D.DrawInfo;

  BEGIN
    draw:=o.GetDrawInfo();

    draw.PushForeground(D.disabledColor);
    draw.PushPattern(D.disablePattern,D.disableWidth,D.disableHeight,D.fgPattern);
    draw.FillRectangle(o.oX,o.oY,o.oWidth,o.oHeight);
    draw.PopPattern;
    draw.PopForeground;
  END DrawDisabled;

  PROCEDURE (o : Object) DrawBackground*(x,y,w,h : LONGINT);
  (**
    This is just a convinience function. Draws the given rectangle in the
    background color of the object.
  *)

  VAR
    draw : D.DrawInfo;

  BEGIN
    draw:=o.GetDrawInfo();
    IF o.backgroundObject#NIL THEN
      o.backgroundObject.Draw(draw,x,y,w,h);
    ELSE
      draw.PushForeground(o.background);
      draw.FillRectangle(x,y,w,h);
      draw.PopForeground;
    END;
  END DrawBackground;

  PROCEDURE (o : Object) DrawHide*;
  (**
    This is just a convinience function. Call this if your are an object and want to
    hide yourself. It just draw a rectangle in the backgroundcolor of the bounds of the
    gadget.

    Not that this is just a proposal. You can of cause use other methods to hide yourself.
  *)

  VAR
    draw : D.DrawInfo;

  BEGIN
    draw:=o.GetDrawInfo();

    draw.PushForeground(D.backgroundColor);
    draw.FillRectangle(o.oX,o.oY,o.oWidth,o.oHeight);
    draw.PopForeground;
  END DrawHide;

  PROCEDURE (o : Object) Disable*(disable : BOOLEAN);
  (**
    Disable the object.
    The object should show it's disabled state via it's
    visual apperance.
  *)

  VAR
    window : D.Window;

  BEGIN
    IF o.disabled#disable THEN
      o.disabled:=disable;

      IF o.disabled THEN
        window:=o.GetWindow();
        IF window#NIL THEN
          window.FocusNext();
        END;
      END;

      o.Redraw;
    END;

  END Disable;

  PROCEDURE (o : Object) Hide*;
  (**
    Tell the object to hide itself.
    The space of the object should have the backgroundcolor after hiding.

    The method of the baseclass simply sets Object.visible to FALSE. If you
    implement your own object, you must overload this methods and hide your
    object (you may use Object.DrawHide for this). After that call
    Object.Hide of your baseclass. Since Object.Hide may have other purposes
    in the future, too.

    NOTE
    If you use other objects for your drawing you have to overload this
    method and call Hide for each of them to get Object.visible correct.
  *)

  VAR
    window : D.Window;

  BEGIN
    o.visible:=FALSE;

    o.objectFrame.Hide;

    (* Tell the windw, that the focus object hides *)
    IF o.HasFocus() THEN
      window:=o.GetWindow();
      window.FocusNext();
    END;
  END Hide;

  PROCEDURE (o : Object) DrawFocus*;
  (**
    This method gets called when the window thinks, that the
    object should show somekind of keyboardfocus. This will
    happen, when the object gets the focus, or when the
    focus has to be refreshed because of a window refresh.

    NOTE
    You can use the supplied focusFrame but you need not to do.
    The baseclass does nothing.
  *)

  BEGIN
    IF o.focus#NIL THEN
      o.focus.Draw(o.GetDrawInfo(),o.oX,o.oY,o.oWidth,o.oHeight);
    END;
  END DrawFocus;

  PROCEDURE (o : Object) HideFocus*;
  (**
    This method gets called when the window thinks, that the
    object should hide the keyboardfocus. This happens when
    the pbject loses the focus.
  *)

  BEGIN
    IF o.focus#NIL THEN
      o.focus.Hide;
    END;
  END HideFocus;

  PROCEDURE (o : Object) LeaveFocus*;
  (**
    Ask the window to change the focus to the next entry.
    This is usefull f.e. for stringgadgets. If you enter return, the
    gadget should get deactivated and the focus should change to the
    next element.

    NOTE
    The object must have the focus, otherwise nothing will happen.
  *)

  VAR
    window : D.Window;

  BEGIN
    IF o.HasFocus() THEN
      window:=o.GetWindow();
      window.FocusNext;
    END;
  END LeaveFocus;

  PROCEDURE (o : Object) PointIsIn*(x,y : LONGINT):BOOLEAN;
  (**
    Ask the object, if the given point it in its bounds.
    We also check if the pointer is within the bounds of the
    label of the object, if the object has one.
  *)

  BEGIN
    IF ~o.visible THEN
      RETURN FALSE;
    END;

    RETURN (x>=o.oX) & (y>=o.oY) & (x<=o.oX+o.oWidth-1) & (y<=o.oY+o.oHeight-1);
  END PointIsIn;

  PROCEDURE (o : Object) MouseIsIn*():BOOLEAN;
  (**
    Tells if the mouse pointer is currently ober the object.
    We also check if the mouse pointer is within the bounds of the
    label of the object, if the object has one.
  *)

  VAR
    rx,ry,
    wx,wy  : LONGINT;
    window : D.Window;

  BEGIN
    IF ~o.visible OR o.disabled THEN
      RETURN FALSE;
    ELSE
      window:=o.GetWindow();
      window.GetMousePos(rx,ry,wx,wy);
      RETURN o.PointIsIn(wx,wy);
    END;
  END MouseIsIn;

  PROCEDURE (g : Object) HandleMouseEvent*(event : E.MouseEvent;
                                           VAR grab : Object):BOOLEAN;
  (**
    Handles Mouse events.

    Returns true, if the object has handled the event.

    If the objects want to grab mouse events (directly receive mouseevents),
    it must store itself in grab. If it wants to loose the grab, it must
    set grab to NIL.
  *)

  BEGIN
    RETURN FALSE; (* Return 'TRUE' if you want to loose focus *)
  END HandleMouseEvent;

  PROCEDURE (o : Object) GetPosObject*(x,y : LONGINT; type : LONGINT):Object;
  (**
    Get a pointer to the object, which is in the given bounds and has
    an menu or help-object attached. If there is no object, return NIL.

    Return also true, if the object is currently not visible.
  *)

  BEGIN
    IF o.visible THEN
      IF o.PointIsIn(x,y) THEN
        CASE type OF
          menuGadget:
            IF o.menu#NIL THEN
              RETURN o;
            END;
        | helpGadget:
            IF o.help#NIL THEN
              RETURN o;
            END;
        | dragGadget:
            IF o.CanDrag() THEN
              RETURN o;
            END;
        END;
      END;
    END;
    RETURN NIL;
  END GetPosObject;

  PROCEDURE (o : Object) GetDnDObject*(x,y : LONGINT; drag : BOOLEAN):Object;
  (**
    Returns the object that coveres the given point and that supports
    drag and drop of data.

    If drag is TRUE, when want to find a object that we can drag data from,
    else we want an object to drop data on.
  *)

  BEGIN
    RETURN NIL;
  END GetDnDObject;

  PROCEDURE (o : Object) Free*;
  (**
    Before freeing the object call this function.

    This method is not supported by the current objects.
  *)
  BEGIN
  END Free;

  (* -------------------------------------------- *)

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.background:=NIL;
    p.frame:=F.none;
  END Init;

  PROCEDURE (p : Prefs) Free*;

  BEGIN
    IF p.background#NIL THEN
      p.background.Free;
      p.background:=NIL;
    END;

    p.Free^;
  END Free;


  (* -------------------------------------------- *)

  PROCEDURE (o : Control) SetModel * (m : O.Model);
  (**
    This method connects a model to the control.
   *)

  BEGIN
  END SetModel;

  PROCEDURE (o : Control) ModelAccepted * (m : O.Model):BOOLEAN;
  (**
    As SetModel is too general to guarantee that the model
    is accepted, we need to check the result. This is done by
    ModelAccepted. It returns true if the actual model is the
    same as the argument. This has to be implemented by real
    views.
   *)

  BEGIN
    RETURN FALSE
  END ModelAccepted;

  (* ----------------------------------- *)

  PROCEDURE (g : Gadget) CatchedFocus*;
  (**
    Called, when you got the keyboard focus.
  *)

  BEGIN
    INCL(g.flags,hasFocus);
    g.DrawFocus;
  END CatchedFocus;

  PROCEDURE (g : Gadget) LostFocus*;
  (**
    Call, when the keyboard focus has been taken away from you.
  *)

  BEGIN
    EXCL(g.flags,hasFocus);
    g.HideFocus;
  END LostFocus;

  PROCEDURE (g : Gadget) HandleKeyEvent*(event : E.KeyEvent):BOOLEAN;
  (**
    This gets called, when you have the current keyboard focus and
    the users presses keys. You can expect to get only keyboard events.

    RESULT
    Return TRUE if you have handled the event, else FALSE.

    NOTE
    If you activtly have grabed the focus using Object.GetFocus and
    Object.handleEvent you will get the keyboardevents there. This
    function gets only called when you don't grabed the focus.
  *)

  BEGIN
    RETURN FALSE;
  END HandleKeyEvent;

  PROCEDURE (g : Gadget) HandleShortcutEvent*(id,state : LONGINT);
  (**
    Via this method you get informed about shortcut events you've
    registered via the keyboard handler of the window.

    PARAMETER
    id - the id you've registered your shortcut with.
    state - the state of the shortcut event. This may either be
    pressed, released or canceled.
  *)

  BEGIN
  END HandleShortcutEvent;

  (* ----------------------------------- *)

  PROCEDURE (g : Group) Init*;
  (**
    A group object is derived from Object. Its purpure is, to  collect a number
    of object and represent them to the ouer space as one object.
    It has some more methods and attributs.
  *)

  BEGIN
    g.Init^;

    g.count:=0;
    g.list:=NIL;
  END Init;

  PROCEDURE (g : Group) Disable*(disabled : BOOLEAN);
  (**
    Disables all elements of the group object.
    Note, that inherited group objects may interpret beeing disabled
    different. VOTab f.e. disables itself.
  *)

  VAR
    help : Object;

  BEGIN
    help:=g.list;
    WHILE help#NIL DO
      help.Disable(disabled);
      help:=help.next;
    END;
  END Disable;

  PROCEDURE (g : Group) Add*(object : Object);
  (**
    Add a new Object to the group
    Removing objects is currently not supported

    Note that some group-objects have a more special functions for adding
    members. However Add should always be supported
  *)

  BEGIN
    IF g.list=NIL THEN
      g.list:=object;
    ELSE
      g.last.next:=object;
    END;
    g.last:=object;
    object.SetParent(g);
    INC(g.count);
  END Add;

  PROCEDURE (g : Group) HandleMouseEvent*(event : E.MouseEvent;
                                          VAR grab : Object):BOOLEAN;
  (**
    The defaulthandler ask all members of the Group for the focus
  *)

  VAR
    object : Object;

  BEGIN
    object:=g.list;
    WHILE object#NIL DO
      IF object.HandleMouseEvent(event,grab) THEN
        RETURN TRUE;
      END;
      object:=object.next;
    END;
    RETURN FALSE;
  END HandleMouseEvent;

  PROCEDURE (g : Group) GetPosObject*(x,y : LONGINT; type : LONGINT):Object;
  (**
    The defaulthandler ask all members of the Group

    You have to overload this method if you do not use
    Group.Add and Group.list.
  *)

  VAR
    object,
    return  : Object;

  BEGIN
    object:=g.list;
    WHILE object#NIL DO
      return:=object.GetPosObject(x,y,type);
      IF return#NIL THEN
        RETURN return;
      END;
      object:=object.next;
    END;
    RETURN g.GetPosObject^(x,y,type);
  END GetPosObject;

  PROCEDURE (g : Group) GetDnDObject*(x,y : LONGINT; drag : BOOLEAN):Object;
  (**
    Returns the object that coveres the given point and that supports
    drag and drop of data.

    If drag is TRUE, when want to find a object that we can drag data from,
    else we want an object to drop data on.
  *)

  VAR
    object,
    return  : Object;

  BEGIN
    object:=g.list;
    WHILE object#NIL DO
      return:=object.GetDnDObject(x,y,drag);
      IF return#NIL THEN
        RETURN return;
      END;
      object:=object.next;
    END;
    RETURN g.GetDnDObject^(x,y,drag);
  END GetDnDObject;

  PROCEDURE (g : Group) Draw*(x,y,w,h : LONGINT);

  VAR
    object : Object;
    draw   : D.DrawInfo;

  BEGIN
    g.Draw^(x,y,w,h);

    IF ~g.Intersect(x,y,w,h) THEN
      RETURN;
    END;

    draw:=g.GetDrawInfo();

    draw.InstallClip(x,y,w,h);

    object:=g.list;
    WHILE object#NIL DO
      draw.SubRegion(object.oX,object.oY,object.oWidth,object.oHeight);
      object:=object.next;
    END;

    g.DrawBackground(g.x,g.y,g.width,g.height);

    draw.FreeLastClip;

    object:=g.list;
    WHILE object#NIL DO
      object.Draw(x,y,w,h);
      object:=object.next;
    END;
  END Draw;

  PROCEDURE (g : Group) Hide*;

  VAR
    help : Object;

  BEGIN
    IF g.visible THEN
      help:=g.list;
      WHILE help#NIL DO
        help.Hide;
        help:=help.next;
      END;
      g.DrawHide;
      g.Hide^;
    END;
  END Hide;

  PROCEDURE  GetAlignment*(name : ARRAY OF CHAR):LONGINT;
  (**
    Returns the alignment value corresponding to the the given string value.
    Returns -1, if the string contains no valid alignment name.
  *)

  VAR
    x : LONGINT;

  BEGIN
    FOR x:=0 TO alignmentCount-1 DO
      IF alignments[x]=name THEN
        RETURN x;
      END;
    END;
    RETURN -1;
  END GetAlignment;

BEGIN
  alignments[alignLeft]:="alignLeft";
  alignments[alignCenter]:="alignCenter";
  alignments[alignRight]:="alignRight";
  alignments[alignBound]:="alignBound";
  alignments[alignTop]:="alignTop";
  alignments[alignBottom]:="alignBottom";
END VO:Object.