(**

   Implements popup menues.

  TODO
  * Add support for check menu and radio menu

  * Cleanup code and share more code with VOWindow
**)

MODULE VO:Menu;

(*
    Implements menues.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)


IMPORT D   := VO:Base:Display,
       E   := VO:Base:Event,
       F   := VO:Base:Frame,
       O   := VO:Base:Object,
       U   := VO:Base:Util,

       DI  := VO:OS:Display,

       G   := VO:Object,
       T   := VO:Text,
       V   := VO:VecImage,
       W   := VO:Window;

CONST
  selectedMsg * = 0; (* a menuitem has been selected *)
  closeMsg    * = 1; (* a toplevel menu has been closed *)

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
                 menuFrame*,            (* the frame to use for the menu *)
                 stripFrame* : LONGINT; (* the frame to use for the menustrip *)
               END;

  (* Menustrip stuff *)

  MenuStrip*     = POINTER TO MenuStripDesc;

  (* menu stuff *)

  Menu*         = POINTER TO MenuDesc;

  MenuEntry     = POINTER TO MenuEntryDesc;
  MenuEntryDesc = RECORD (G.GadgetDesc)
                    w1,w2  : LONGINT;
                  END;

  (* Menustrip stuff *)

  MenuStripDesc* = RECORD (G.GroupDesc)
                   END;

  (* menu stuff *)

  MenuItem      = POINTER TO MenuItemDesc;
  MenuItemDesc  = RECORD (MenuEntryDesc)
                    name,
                    shortcut   : G.Object;
                    qualifier  : SET;
                    char       : CHAR;
                  END;
  PullDownMenu  = POINTER TO PullDownMenuDesc;
  PullDownMenuDesc = RECORD (MenuEntryDesc);
                    name     : G.Image;
                    subMenu  : Menu;
                    subOpen  : BOOLEAN;
                  END;

  MenuTitle     = POINTER TO MenuTitleDesc;
  MenuTitleDesc = RECORD (MenuEntryDesc)
                    name     : G.Object;
                  END;

  Separator     = POINTER TO SeparatorDesc;
  SeparatorDesc = RECORD (MenuEntryDesc)
                  END;

  SubMenu       = POINTER TO SubMenuDesc;
  SubMenuDesc   = RECORD(MenuEntryDesc)
                    name    : G.Object;
                    arrow   : V.VecImage;
                    subMenu : Menu;
                    subOpen : BOOLEAN;
                  END;

  MenuDesc*     = RECORD (DI.WindowDesc)
                    prefs            : Prefs;
                    backgroundObject : G.Background;
                    parentMenu       : SubMenu;

                    child            : Menu;

                    frame            : F.Frame;   (* frames menu *)

                    selected         : MenuEntry; (* the currently selected entry *)

                    count            : LONGINT;   (* number of menuentries *)
                    list,last        : MenuEntry; (* menuentry-list *)

                    (* Pulldown stuff *)
                    reference        : G.Object;
                    strip            : MenuStrip;

                    popup            : BOOLEAN;   (* Top menu of a popup menu *)
                    pullDown         : BOOLEAN;   (* This menu is the top of a pullDown *)
                  END;

  SelectedMsg*     = POINTER TO SelectedMsgDesc;
  SelectedMsgDesc* = RECORD (O.MessageDesc)
                       id* : LONGINT;
                     END;

  CloseMsg*     = POINTER TO CloseMsgDesc;
  CloseMsgDesc* = RECORD (O.MessageDesc)
                  END;

VAR
  prefs* : Prefs;

  (**
    Initalisation of preferences.
  **)

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.menuFrame:=F.double3DOut;
    p.stripFrame:=F.double3DOut;
  END Init;

  (* ---------- Menu stuff ------------------*)

  PROCEDURE InitMenu*():Menu;

  VAR
    m : Menu;

  BEGIN
    NEW(m);
    m.Init;

    RETURN m;
  END InitMenu;

  PROCEDURE (m : Menu) Init*;

  BEGIN
    m.Init^;

    m.prefs:=prefs;   (* We set the prefs *)

    IF prefs.background#NIL THEN
      m.backgroundObject:=prefs.background.Copy();
      m.backgroundObject.source:=NIL;
    END;

    m.Borderless(TRUE);

    m.count:=0;
    m.list:=NIL;
    m.last:=NIL;
    m.selected:=NIL;
    m.parentMenu:=NIL;
    m.child:=NIL;
    m.reference:=NIL;
    m.strip:=NIL;
    m.popup:=TRUE;
    m.pullDown:=FALSE;

    NEW(m.frame);
    m.frame.Init;
  END Init;

  (**
    This makes the menu a pulldown relative to the given object.

    NOTE
    The pulldown flag gets cleared after closing the menu, you must explizitely
    set it before calling Menu.Open.
  **)

  PROCEDURE (m : Menu) SetPullDown(strip : MenuStrip; reference : G.Object);
(* horizontal offset!!!*)
  BEGIN
    m.pullDown:=TRUE;
    m.reference:=reference;
    m.strip:=strip;
  END SetPullDown;


  (* ---------- Menustrip stuff ------------------*)

  PROCEDURE (m : MenuStrip) Init*;

  BEGIN
    m.Init^;

    m.SetFlags({G.horizontalFlex});

    m.SetPrefs(prefs);

    m.SetObjectFrame(m.prefs(Prefs).stripFrame);
  END Init;

  PROCEDURE (m : MenuStrip) AddPullDownMenu*(label : G.Image; subMenu : Menu);

  VAR menu : PullDownMenu;

  BEGIN
    NEW(menu);
    menu.Init;
    menu.name:=label;
    menu.name.SetParent(menu);
    menu.subMenu:=subMenu;

    m.Add(menu);
  END AddPullDownMenu;

  PROCEDURE (m : MenuStrip) AddPullDownMenuText*(title   : ARRAY OF CHAR;
                                                 subMenu : Menu);

  VAR text  : T.Text;

  BEGIN
    text:=T.MakeCenterText(title);
    text.SetFlags({G.horizontalFlex,G.verticalFlex});
    m.AddPullDownMenu(text,subMenu);
  END AddPullDownMenuText;

  PROCEDURE (m : MenuStrip) CalcSize*;

  VAR
    object : G.Object;

  BEGIN
    m.width:=0;
    m.height:=0;

    object := m.list;
    WHILE object#NIL DO
      object.CalcSize;
      object.Resize(object.oWidth + 2 * D.display.spaceWidth, - 1);
      m.height:=U.MaxLong(m.height,object.oHeight);
      INC(m.width,object.oWidth);

      object:=object.next;
    END;

    INC(m.height,D.display.spaceHeight);
    m.CalcSize^;
  END CalcSize;

  PROCEDURE (m : MenuStrip) Layout*;

  VAR
    pos    : LONGINT;
    object : G.Object;

  BEGIN
    pos:=m.x;
    object:=m.list;
    WHILE object#NIL DO
      object.Resize(-1,m.height);
      object.Move(pos,m.y + (m.height-object.oHeight) DIV 2);
      INC(pos,object.oWidth);

      object:=object.next;
    END;

    m.Layout^;
  END Layout;

  (* ---------- PullDownMenu stuff ------------------*)

  PROCEDURE (m : PullDownMenu) Init*;

  BEGIN
    m.Init^;

    m.SetPrefs(prefs);

    m.SetFlags({G.horizontalFlex,G.verticalFlex});
    m.name:=NIL;
    m.subMenu:=NIL;
    m.subOpen:=FALSE;
  END Init;

  PROCEDURE (m : PullDownMenu) CalcSize*;

  BEGIN
    m.name.CalcSize;
    m.width:=m.name.width;
    m.height:=m.name.height;

    m.CalcSize^;
  END CalcSize;

  PROCEDURE (m : PullDownMenu) HandleMouseEvent*(event : E.MouseEvent;
                                                 VAR grab : G.Object):BOOLEAN;

  VAR
    strip : MenuStrip;

  BEGIN
    IF ~m.visible OR m.disabled THEN
      RETURN FALSE;
    END;

    WITH event : E.ButtonEvent DO
      IF (event.type=E.mouseDown) & m.PointIsIn(event.x,event.y)
      & ((event.button=E.button1) OR (event.button=E.button3)) THEN
        IF m.parent IS MenuStrip THEN
          strip:=m.parent(MenuStrip);
          m.subMenu.SetPullDown(strip,m);
          m.subMenu.Open();

          RETURN TRUE;
        END;
      END;
    | event : E.MotionEvent DO
      IF ~m.subMenu.IsOpen() THEN
        IF m.PointIsIn(event.x,event.y) THEN
          IF ({E.button1}=event.qualifier) OR ({E.button3}=event.qualifier) THEN
            IF m.parent IS MenuStrip THEN
              strip:=m.parent(MenuStrip);
              m.subMenu.SetPullDown(strip,m);
              m.subMenu.Open();

              RETURN TRUE;
            END;
          END;
        END;
      END;
    ELSE
    END;

    RETURN FALSE;
  END HandleMouseEvent;

  PROCEDURE (m : PullDownMenu) Draw*(x,y,w,h : LONGINT);

  BEGIN
    m.Draw^(x,y,w,h);

    m.DrawBackground(m.x,m.y,m.width,m.height);

    m.name.Move(m.x,m.y);
    m.name.Draw(x,y,w,h);
  END Draw;

  PROCEDURE (m : PullDownMenu) Resize*(width,height: LONGINT);

  BEGIN
    m.name.Resize( width, height );
    m.width:=m.name.width;
    m.height:=m.name.height;
    m.CalcSize^;
  END Resize;

  PROCEDURE (m : PullDownMenu) Hide*;

  BEGIN
    IF m.visible THEN
      m.name.Hide;
    END;
    m.Hide^;
  END Hide;





  (* ---------- Menu stuff ------------------*)

  PROCEDURE (e : MenuEntry) ResizeMenu(w1,w2, height : LONGINT);

  BEGIN
    e.w1:=w1;
    e.w2:=w2;
(*    e.Resize(w1+w2,height);*)
    e.width:=w1+w2;
    e.height:=height;
  END ResizeMenu;

  PROCEDURE (e : MenuItem) Init*;

  BEGIN
    e.Init^;

    e.SetFlags({G.handleSC,G.scAlways});

    e.name:=NIL;
    e.shortcut:=NIL;
  END Init;

  (* MenuItem *)

  PROCEDURE (i : MenuItem) Draw*(x,y,w,h : LONGINT);

  VAR
    draw   : D.DrawInfo;

  BEGIN
    i.Draw^(x,y,w,h);

    draw:=i.GetDrawInfo();

    IF D.selected IN draw.mode THEN
      draw.PushForeground(D.fillColor);
      draw.FillRectangle(i.x,i.y,i.width,i.height);
      draw.PopForeground;
    ELSE
      i.DrawBackground(i.x,i.y,i.width,i.height);
    END;


    i.name.Move(i.x+D.display.spaceWidth DIV 2,
                i.y+(i.height-i.name.height) DIV 2);
    i.name.Draw(x,y,w,h);
    IF i.shortcut#NIL THEN
      i.shortcut.Move(i.x+i.w1+2*D.display.spaceWidth,
                      i.y+(i.height-i.shortcut.height) DIV 2);
      i.shortcut.Draw(x,y,w,h);
    END;
  END Draw;

  PROCEDURE (i : MenuItem) HandleShortcutEvent*(id,state : LONGINT);

  VAR
    menu     : Menu;
    help     : D.Window;
    selected : SelectedMsg;

  BEGIN
    IF state=G.pressed THEN
      (* Nothing to do *)
    ELSE
      IF state=G.released THEN
        help:=i.GetMenuObject();
        menu:=help(Menu);
        NEW(selected);
        selected.id:=i.id;
        menu.Send(selected,selectedMsg);
      ELSE
        (* Nothing to do *)
      END;
    END;
  END HandleShortcutEvent;

  (* MenuTitle *)

  PROCEDURE (i : MenuTitle) Draw*(x,y,w,h : LONGINT);

  VAR
    draw : D.DrawInfo;

  BEGIN
    i.Draw^(x,y,w,h);

    draw:=i.GetDrawInfo();

    i.DrawBackground(i.x,i.y,i.width,i.height);

    draw.mode:={};
    i.name.Move(i.x+(i.width-i.name.width) DIV 2,i.y);
    i.name.Draw(x,y,w,h);
    draw.PushForeground(D.shadowColor);
    draw.DrawLine(i.x,i.y+i.name.height,i.x+i.width-1,i.y+i.name.height);
    draw.PopForeground;
    draw.PushForeground(D.shineColor);
    draw.DrawLine(i.x,i.y+i.name.height+1,i.x+i.width-1,i.y+i.name.height+1);
    draw.PopForeground;
  END Draw;

  PROCEDURE (i : MenuTitle) ResizeMenu(w1,w2, height : LONGINT);

  BEGIN
    i.ResizeMenu^(w1,w2,height);
    i.name.Resize(w1+w2,height-3);
  END ResizeMenu;

  (* Separator *)

  PROCEDURE (s : Separator) Draw*(x,y,w,h : LONGINT);

  VAR
    draw : D.DrawInfo;

  BEGIN
    s.Draw^(x,y,w,h);

    draw:=s.GetDrawInfo();

    s.DrawBackground(s.x,s.y,s.width,s.height);

    draw.PushForeground(D.shadowColor);
    draw.DrawLine(s.x,s.y+D.display.spaceHeight DIV 2,
                  s.x+s.width-1,s.y+D.display.spaceHeight DIV 2);
    draw.PopForeground;
    draw.PushForeground(D.shineColor);
    draw.DrawLine(s.x,s.y+1+D.display.spaceHeight DIV 2,
                  s.x+s.width-1,s.y+1+D.display.spaceHeight DIV 2);
    draw.PopForeground;
  END Draw;

  (* SubMenuI *)

  PROCEDURE (s : SubMenu) Draw*(x,y,w,h : LONGINT);

  VAR
    help : D.Window;
    draw : D.DrawInfo;

  BEGIN
    s.Draw^(x,y,w,h);

    draw:=s.GetDrawInfo();

    s.DrawBackground(s.x,s.y,s.width,s.height);

    IF (D.selected IN draw.mode) & ~s.subOpen THEN
      s.subMenu.Open;
      help:=s.GetMenuObject();
      help(Menu).child:=s.subMenu;
      s.subOpen:=TRUE;
    ELSIF s.subOpen THEN
      s.subMenu.Close;
    END;
    IF s.arrow#NIL THEN
      s.arrow.Move(s.x+s.w1+s.w2-s.arrow.width-D.display.spaceWidth DIV 2,
                 s.y+D.display.spaceHeight DIV 2);
      s.arrow.Draw(x,y,w,h);
    END;
    draw.mode:={};
    s.name.Move(s.x+D.display.spaceWidth DIV 2,s.y+D.display.spaceHeight DIV 2);
    s.name.Draw(x,y,w,h);
  END Draw;


  PROCEDURE (m : SubMenu) CalcSize*;

  BEGIN

    m.name.CalcSize;
    m.w1:=m.name.width+D.display.spaceWidth;
    m.arrow:=NIL;
    m.w2:=2*D.display.spaceWidth;
    m.width:=m.w1+m.w2;
    m.height:=m.name.height(*+display.spaceHeight*);
    m.minWidth:=m.width;
    m.minHeight:=m.height;
    m.CalcSize^;
  END CalcSize;

  (**
     Initializes an empty menu.
  **)

  PROCEDURE (m : Menu) Add(entry : MenuEntry);

  BEGIN
    IF m.list=NIL THEN
      m.list:=entry;
    ELSE
      m.last.next:=entry;
    END;
    m.last:=entry;
    entry.SetWindow(m);
    entry.SetMenuObject(m);
    entry.CalcSize;

    INC(m.count);
  END Add;

  PROCEDURE (m : Menu) AddItem*(name, shortcut : G.Object;
                                qualifier : SET; char : CHAR; id : LONGINT);

  VAR
    item : MenuItem;

  BEGIN
    NEW(item);
    item.Init;

    IF name#NIL THEN
      name.SetParent(item);
    END;
    IF shortcut#NIL THEN
      shortcut.SetParent(item);
    END;

    item.name:=name;
    item.shortcut:=shortcut;
    item.qualifier:=qualifier;
    item.char:=char;
    item.SetId(id);

    item.name.SetFlags({G.horizontalFlex,G.verticalFlex});
    item.name.CalcSize;
    item.w1:=item.name.width+D.display.spaceWidth DIV 2;
    item.height:=item.name.height+D.display.spaceHeight;

    IF item.shortcut#NIL THEN
      item.shortcut.SetFlags({G.horizontalFlex,G.verticalFlex});
      item.shortcut.CalcSize;
      item.w2:=item.shortcut.width+2*D.display.spaceWidth+D.display.spaceWidth DIV 2;
      item.height:=U.MaxLong(item.height,item.name.height+D.display.spaceHeight);
    ELSE
      item.w2:=D.display.spaceWidth DIV 2;
    END;

    m.Add(item);
  END AddItem;

  PROCEDURE (m : Menu) AddTextItem*(label : ARRAY OF CHAR; id : LONGINT);

  VAR
    labelText : T.Text;
    help      : U.Text;

  BEGIN
    NEW(labelText);
    labelText.Init;
    labelText.SetFlags({G.horizontalFlex,G.verticalFlex});
    labelText.SetDefault(T.leftAlligned,{},D.normalFont);
    help:=U.EscapeString(label);
    labelText.SetText(help^);

    m.AddItem(labelText,NIL,{},0X,id);
  END AddTextItem;

  PROCEDURE (m : Menu) AddTextItemSC*(label, shortcut : ARRAY OF CHAR;
                                      id : LONGINT; qualifier : SET; char : CHAR;
                                      window : W.Window);

  VAR
    labelText,
    scText    : T.Text;
    help      : U.Text;

  BEGIN
    NEW(labelText);
    labelText.Init;
    labelText.SetFlags({G.horizontalFlex,G.verticalFlex});
    labelText.SetDefault(T.leftAlligned,{},D.normalFont);
    help:=U.EscapeString(label);
    labelText.SetText(help^);

    IF shortcut#"" THEN
      NEW(scText);
      scText.Init;
      scText.SetFlags({G.horizontalFlex,G.verticalFlex});
      scText.SetDefault(T.leftAlligned,{},D.normalFont);
      help:=U.EscapeString(shortcut);
      scText.SetText(help^);
    ELSE
      scText:=NIL;
    END;

    m.AddItem(labelText,scText,qualifier,char,id);
    IF window#NIL THEN
      window.AddShortcutObject(m.last,qualifier,char,id,W.none); (* m.last = hack! *)
    END;
  END AddTextItemSC;

  PROCEDURE (m : Menu) AddSeparator*;

  VAR
    s : Separator;

  BEGIN
    NEW(s);
    s.w1:=2;
    s.height:=2+D.display.spaceHeight;

    m.Add(s);
  END AddSeparator;

  PROCEDURE (m : Menu) AddTitle*(name : G.Object);

  VAR
    item : MenuTitle;

  BEGIN
    name.SetParent(item);

    NEW(item);
    item.name:=name;
    item.name.SetFlags({G.horizontalFlex,G.verticalFlex});
    item.name.CalcSize;
    item.w1:=item.name.width+D.display.spaceWidth;
    item.height:=item.name.height+D.display.spaceHeight;
    INC(item.height,3);
    m.Add(item);
  END AddTitle;

  PROCEDURE (m : Menu) AddTextTitle*(label : ARRAY OF CHAR);

  VAR
    labelText : T.Text;
    help      : U.Text;

  BEGIN
    NEW(labelText);
    labelText.Init;
    labelText.SetFlags({G.horizontalFlex,G.verticalFlex});
    labelText.SetDefault(T.centered,{},D.normalFont);
    help:=U.EscapeString(label);
    labelText.SetText(help^);

    m.AddTitle(labelText);
  END AddTextTitle;

  PROCEDURE (m : Menu) AddSubMenu*(name : G.Object; menu : Menu);

  VAR
    sub : SubMenu;

  BEGIN
    name.SetParent(sub);

    m.popup:=FALSE;
    NEW(sub);
    sub.subMenu:=menu;
    sub.subMenu.parentMenu:=sub;
    sub.name:=name;
    sub.name.SetFlags({G.horizontalFlex,G.verticalFlex});
    sub.name.CalcSize;
    sub.w1:=sub.name.width+D.display.spaceWidth;
    NEW(sub.arrow);
    sub.arrow.Init;
    sub.arrow.SetFlags({G.horizontalFlex,G.verticalFlex});
    sub.arrow.Set(V.arrowRight);
    sub.arrow.CalcSize;
    sub.w2:=2*D.display.spaceWidth+sub.arrow.width;
    sub.height:=U.MaxLong(sub.name.height,sub.arrow.height)+D.display.spaceHeight;

    m.Add(sub);

  END AddSubMenu;

  PROCEDURE (m : Menu) AddTextSubMenu*(label : ARRAY OF CHAR; menu : Menu);

  VAR
    labelText : T.Text;

  BEGIN
    labelText:=T.MakeCenterText(label);
    m.AddSubMenu(labelText,menu);
  END AddTextSubMenu;

  PROCEDURE (m : Menu) CalcSize;

  VAR
    entry   : MenuEntry;
    w1,w2,h : LONGINT;

  BEGIN
    m.frame.SetFrame(m.prefs.menuFrame);

    w1:=0;
    w2:=0;
    h:=0;

    entry:=m.list;
    WHILE entry#NIL DO
      INC(h,entry.height);
      w1:=U.MaxLong(w1,entry.w1);
      w2:=U.MaxLong(w2,entry.w2);
      IF entry.next#NIL THEN
        entry:=entry.next(MenuEntry);
      ELSE
        entry:=NIL;
      END;
    END;

    entry:=m.list;
    WHILE entry#NIL DO
      entry.ResizeMenu(w1,w2,entry.height);
      IF entry.next#NIL THEN
        entry:=entry.next(MenuEntry);
      ELSE
        entry:=NIL;
      END;
    END;

    m.SetSize(m.frame.leftBorder+w1+w2+m.frame.rightBorder,
              m.frame.topBorder+h+m.frame.bottomBorder);
  END CalcSize;

  PROCEDURE (m : Menu) Send*(message : O.Message; type : LONGINT);

  VAR
    menu : Menu;
    help : D.Window;

  BEGIN
    menu:=m;
    WHILE (menu.parentMenu#NIL) & (menu.parentMenu.GetMenuObject()#NIL) DO
      help:=menu.parentMenu.GetMenuObject();
      menu:=help(Menu);
    END;

    IF menu#m THEN
      menu.Send(message,type);
    ELSE
      menu.Send^(message,type);
    END;
  END Send;

  PROCEDURE (m : Menu) CursorIsIn():BOOLEAN;

  VAR
    rx,ry,
    wx,wy  : LONGINT;

  BEGIN
    m.GetMousePos(rx,ry,wx,wy);
    IF (rx>=m.x) & (rx<=m.x+m.width-1) & (ry>=m.y) & (ry<=m.y+m.height-1) THEN
      RETURN TRUE;
    ELSE
      RETURN FALSE;
    END;
  END CursorIsIn;

  PROCEDURE (m : Menu) GetSelected():MenuEntry;

  VAR
    entry  : MenuEntry;

    rx,ry,
    wx,wy  : LONGINT;

  BEGIN
    m.GetMousePos(rx,ry,wx,wy);
    entry:=m.list;
    WHILE entry#NIL DO
      IF (rx>=m.x) & (rx<=m.x+m.width-1) & (wy>=entry.y) & (wy<=entry.y+entry.height-1) THEN
        RETURN entry;
      END;
      IF entry.next#NIL THEN
        entry:=entry.next(MenuEntry);
      ELSE
        entry:=NIL;
      END;
    END;
    RETURN entry;
  END GetSelected;

  PROCEDURE (m : Menu) PreInit*;

  VAR
    rx,ry,
    cx,cy,
    x,y    : LONGINT;
    help   : D.Window;

  BEGIN
    m.CalcSize;

    IF m.selected=NIL THEN
      m.selected:=m.list;
    END;
    IF m.pullDown & (m.reference#NIL) THEN
      help:=m.reference.GetWindow();
      help.GetXY(x,y);
      INC(x,m.reference.x);
      INC(y,m.reference.y+m.reference.height);
      m.Grab(TRUE);
    ELSE
      IF m.parentMenu#NIL THEN (* submenu *)
        help:=m.parentMenu.GetMenuObject();
        x:=help.x+help.width;
        y:=help.y+m.parentMenu.y-help(Menu).list.y;
      ELSE (* main-menu *)
        D.display.currentWin.GetMousePos(rx,ry,cx,cy);
        IF m.popup THEN
          x:=rx+D.display.spaceWidth;
          y:=ry+D.display.spaceHeight;
        ELSE
          x:=rx-m.selected.x-m.selected.width DIV 2;
          y:=ry-m.selected.y-m.selected.height DIV 2;
        END;
        IF ~m.pullDown THEN
          m.Grab(TRUE);
        END;
      END;
    END;

    x:=U.RoundRange(x,0,D.display.scrWidth-1-m.width);
    y:=U.RoundRange(y,0,D.display.scrHeight-1-m.height);

    m.SetPos(x,y);
    m.SetPosition(D.manualPos,D.manualPos);

    m.PreInit^;
  END PreInit;

  PROCEDURE (m : Menu) Close*;

  VAR
    close : CloseMsg;
    help  : D.Window;

  BEGIN
    m.Grab(FALSE);

    IF m.child#NIL THEN
      m.child.Close;
      m.child:=NIL;
    END;
    IF m.parentMenu#NIL THEN
      m.parentMenu.subOpen:=FALSE;
      help:=m.parentMenu.GetMenuObject();
      help(Menu).child:=NIL;
    ELSE
      NEW(close);
      m.Send(close,closeMsg);
    END;

    m.pullDown:=FALSE;
    m.reference:=NIL;

    m.Close^;
  END Close;

  PROCEDURE ( m : Menu) CloseAll;

  VAR
    current : Menu;
    help    : D.Window;

  BEGIN
    m.pullDown:=FALSE;
    m.reference:=NIL;
    current:=m;
    WHILE current#NIL DO
      current.Close;
      IF current.parentMenu#NIL THEN
        help:=current.parentMenu.GetMenuObject();
        current:=help(Menu);
      ELSE
        current:=NIL;
      END;
    END;
  END CloseAll;

  PROCEDURE (m : Menu) Redraw*(x,y,width,height : LONGINT);

  VAR
    entry : MenuEntry;
    xPos,
    yPos  : LONGINT;
    draw  : D.DrawInfo;

  BEGIN
    draw:=m.GetDrawInfo();

    m.frame.Draw(draw,0,0,m.width,m.height);

    xPos:=m.frame.topBorder;
    yPos:=m.frame.leftBorder;

    m.selected:=m.GetSelected();
    entry:=m.list;
    WHILE entry#NIL DO

      IF entry=m.selected THEN
        draw.mode:={D.selected};
      END;

      entry.Move(xPos,yPos);
      entry.Draw(x,y,width,height);
      draw.mode:={};

      INC(yPos,entry.height);

      IF entry.next#NIL THEN
        entry:=entry.next(MenuEntry);
      ELSE
        entry:=NIL;
      END;
    END;
  END Redraw;

  PROCEDURE (m : Menu) Hidden*;

  BEGIN
    m.CloseAll;
  END Hidden;

  (* ---------------------------------- *)

  PROCEDURE (m : Menu) HandleEvent*(event : E.Event):BOOLEAN;

  VAR
    newSelected : MenuEntry;
    selected    : SelectedMsg;
    menu        : Menu;
    rx,ry,wx,wy : LONGINT;
    object      : G.Object;
    window      : D.Window;
    draw        : D.DrawInfo;

  BEGIN
    IF m.HandleEvent^(event) THEN
      RETURN TRUE;
    END;

    WITH event : E.ButtonEvent DO
      selected:=NIL;
      IF (event.type=E.mouseUp)
      &  ((event.button=E.button3) OR (event.button=E.button1)) THEN
        menu:=m;
        WHILE (menu#NIL) & ~(menu.GetSelected()#NIL) & (menu.selected#NIL) DO
          IF menu.selected IS SubMenu THEN
            menu:=menu.selected(SubMenu).subMenu;
          ELSE
            menu:=NIL;
          END;
        END;

        IF menu#NIL THEN
          menu.selected:=menu.GetSelected();
          IF (menu.selected#NIL) & (menu.selected IS MenuItem) THEN
            NEW(selected);
            selected.id:=menu.selected(MenuItem).id;
          END;
        END;
        IF m.pullDown THEN (* The strip must get its mouse up *)
          D.display.PutBackEvent(event,m.strip.GetWindow());
        END;
        m.CloseAll;
        IF selected#NIL THEN
          m.Send(selected,selectedMsg);
        END;
      END;

    | event : E.MotionEvent DO
      menu:=m;
      WHILE (menu#NIL) & (menu.selected#NIL) & ~menu.CursorIsIn() DO
        IF (menu.selected#NIL) & (menu.selected IS SubMenu) THEN
          menu:=menu.selected(SubMenu).subMenu;
        ELSE
          menu:=NIL;
        END;
      END;

      IF menu#NIL THEN

        IF menu.CursorIsIn() THEN
          newSelected:=menu.GetSelected();
          IF newSelected#menu.selected THEN
            IF menu.selected#NIL THEN
              menu.selected.Redraw;
            END;
            IF newSelected#NIL THEN
              draw:=menu.GetDrawInfo();
              draw.mode:={D.selected};
              newSelected.Redraw;
              draw.mode:={};
            END;
            menu.selected:=newSelected;
          END;
        ELSIF m.pullDown THEN
          window:=m.reference.GetWindow();
          window.GetMousePos(rx,ry,wx,wy);
          IF m.strip.MouseIsIn() THEN
            object:=m.strip.list;
            WHILE object#NIL DO
              IF object.MouseIsIn () & (m.reference#object) THEN
                (*D.display.PutBackEvent(event,m.strip.draw.vWindow);*)
                m.CloseAll ();
              END;
              object:=object.next;
            END;
          END;
        END;

      ELSE

        menu:=m;
        WHILE (menu#NIL) & (menu.selected#NIL) & (menu.selected IS SubMenu) DO
          menu:=menu.selected(SubMenu).subMenu;
        END;
        IF (menu#NIL) & (menu.selected#NIL) THEN
          menu.selected.Redraw;
          menu.selected:=NIL;
        END;

        IF m.pullDown THEN
          window:=m.reference.GetWindow();
          window.GetMousePos(rx,ry,wx,wy);
          IF m.strip.MouseIsIn() THEN
            object:=m.strip.list;
            WHILE object#NIL DO
              IF object.MouseIsIn () & (m.reference#object) THEN
                m.CloseAll ();
              END;
              object:=object.next;
            END;
          END;
        END;
      END;

    ELSE
    END;
    RETURN TRUE;
  END HandleEvent;

BEGIN
  NEW(prefs);
  prefs.Init;
END VO:Menu.