(**
   A groupclass that places nicely formated textlabels (or any other
   object) to the left of a vertical list of objects.
**)

MODULE VO:Label;

(*
    A layouting class that gives you labels.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D   := VO:Base:Display,
       E   := VO:Base:Event,
       U   := VO:Base:Util,
       Z   := VO:Base:Size,

       G   := VO:Object,
       T   := VO:Text;

CONST
  left*  = 0;
  right* = 1;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button row is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
                 hSpace*,
                 vSpace*      : Z.SizeDesc;
                 labelPos*    : LONGINT;
                 labelAlign*  : LONGINT;
                 objectAlign* : LONGINT;
               END;


  Text       = POINTER TO TextDesc;
  TextDesc   = RECORD
                 next     : Text;
                 object,
                 text     : G.Object;
               END;

  Label*     = POINTER TO LabelDesc;
  LabelDesc* = RECORD(G.ObjectDesc)
                 count      : LONGINT;    (* The number of members *)
                 lSize,
                 rSize      : LONGINT;
                 labelList,
                 lLast      : Text;
                 labelFlex  : BOOLEAN;
               END;

VAR
  prefs* : Prefs;


  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.hSpace.Init;
    p.vSpace.Init;
    p.hSpace.SetSize(Z.unit,1);
    p.vSpace.SetSize(Z.unit,1);

    p.labelPos:=left;
    p.labelAlign:=right;
    p.objectAlign:=left;
  END Init;

  PROCEDURE (l : Label) Init*;

  BEGIN
    l.Init^;

    l.SetPrefs(prefs);

    l.labelList:=NIL;
    l.labelFlex:=FALSE;
  END Init;

  (**
    Add a label entry to label group. A label entry consists of two objects. One
    object is the label for the other.

  **)

  PROCEDURE (l : Label) AddLabel*(text, object : G.Object);

  VAR
    label : Text;

  BEGIN
    text.SetParent(l);
    object.SetParent(l);

    NEW(label);
    label.object:=object;
    label.text:=text;

    IF l.labelList=NIL THEN
      l.labelList:=label;
    ELSE
      l.lLast.next:=label;
    END;
    l.lLast:=label;

    INC(l.count);

    object.SetLabelObject(text);
  END AddLabel;

  (**
    Add a textual label entry. The label object will generate a text object out of
    the given string and will then assign it as label object to the other object.
  **)

  PROCEDURE (l : Label) AddTextLabel*(string : ARRAY OF CHAR; object : G.Object);

  VAR
    text : T.Text;
    help : U.Text;

  BEGIN
    NEW(text);
    text.Init;
    help:=U.EscapeString(string);
    text.SetText(help^);
    l.AddLabel(text,object);
  END AddTextLabel;

  (**
    Tell, if the label object should stretch the label or the object when beeing resized
    horizontally.
  **)

  PROCEDURE ( l : Label) Set*(labelFlex : BOOLEAN);

  BEGIN
    l.labelFlex:=labelFlex;
  END Set;

  PROCEDURE (l : Label) CalcSize*;

  VAR
    label     : Text;
    rMinWidth : LONGINT;

  BEGIN
    l.height:=0;
    l.minHeight:=0;
    l.lSize:=0;
    l.rSize:=0;
    rMinWidth:=0;

    label:=l.labelList;
    WHILE (label#NIL) DO

      label.object.CalcSize;
      label.text.CalcSize;

      l.lSize:=U.MaxLong(label.text.width,l.lSize);

      IF label.object.oHeight<label.text.oHeight THEN
        INC(l.height,label.text.oHeight);
        INC(l.minHeight,label.text.oMinHeight);
      ELSE
        INC(l.height,label.object.oHeight);
        INC(l.minHeight,label.object.oMinHeight);
      END;

      l.rSize:=U.MaxLong(l.rSize,label.object.oWidth);
      rMinWidth:=U.MaxLong(rMinWidth,label.object.oMinWidth);

      label:=label.next;
    END;

    IF l.count>1 THEN
      INC(l.height,(l.count-1)*l.prefs(Prefs).vSpace.GetSize());
      INC(l.minHeight,(l.count-1)*l.prefs(Prefs).vSpace.GetSize());
    END;

    l.width:=l.lSize+l.prefs(Prefs).hSpace.GetSize()+l.rSize;
    l.minWidth:=l.lSize+l.prefs(Prefs).hSpace.GetSize()+rMinWidth;

    l.CalcSize^;

    IF l.labelFlex THEN
      l.lSize:=l.width-l.rSize-l.prefs(Prefs).hSpace.GetSize();
    ELSE
      l.rSize:=l.width-l.lSize-l.prefs(Prefs).hSpace.GetSize();
    END;
  END CalcSize;

  PROCEDURE (l : Label) Resize*(width,height : LONGINT);

  BEGIN
    l.Resize^(width,height);

    IF l.labelFlex THEN
      l.lSize:=l.width-l.rSize-l.prefs(Prefs).hSpace.GetSize();
    ELSE
      l.rSize:=l.width-l.lSize-l.prefs(Prefs).hSpace.GetSize();
    END;
  END Resize;

  PROCEDURE (l : Label) Layout*;

  VAR
    label     : Text;
    pos,
    curHeight,
    old,
    count     : LONGINT;

  BEGIN
    curHeight:=0;
    label:=l.labelList;
    WHILE (label#NIL) DO
      IF label.object.oHeight<label.text.oHeight THEN
        label.object.Resize(-1,label.text.oHeight);
        INC(curHeight,label.text.oHeight);
      ELSIF label.object.oHeight>label.text.oHeight THEN
        label.text.Resize(-1,label.object.oHeight);
        INC(curHeight,label.object.oHeight);
      ELSE
        INC(curHeight,label.object.oHeight);
      END;
      label.text.Resize(l.lSize,-1);
      label.object.Resize(l.rSize,-1);
      label:=label.next;
    END;

    IF l.count>1 THEN
      INC(curHeight,(l.count-1)*l.prefs(Prefs).vSpace.GetSize());
    END;

    IF curHeight<l.height THEN
      LOOP
        count:=0;
        label:=l.labelList;
        WHILE label#NIL DO
          IF label.object.CanResize(l.height>curHeight,FALSE) THEN
            INC(count);
          END;
          label:=label.next;
        END;

        IF count=0 THEN
          EXIT;
        END;

        label:=l.labelList;
        WHILE label#NIL DO
          IF label.object.CanResize(l.height>curHeight,FALSE) THEN
            old:=label.object.oHeight;
            label.object.Resize(-1,label.object.oHeight+U.UpDiv(l.height-curHeight,count));
            INC(curHeight,label.object.oHeight-old);
            DEC(count);
          END;
          label:=label.next;
        END;

        IF curHeight=l.height THEN
          EXIT;
        END;
      END;
    END;

    (* Draw all objects *)

    label:=l.labelList;
    pos:=l.y;
    WHILE (label#NIL) DO

      IF l.prefs(Prefs).labelPos=right THEN
        IF l.prefs(Prefs).objectAlign=left THEN
          label.object.Move(l.x,pos);
        ELSE
          label.object.Move(l.x+l.rSize-label.object.oWidth,pos);
        END;
        IF l.prefs(Prefs).labelAlign=right THEN
          label.text.Move(l.x+l.rSize+l.prefs(Prefs).hSpace.GetSize()+l.lSize-label.text.oWidth,
                          pos+label.object.oHeight-label.text.oHeight);
        ELSE
          label.text.Move(l.x+l.rSize+l.prefs(Prefs).hSpace.GetSize(),
                          pos+label.object.oHeight-label.text.oHeight);
        END;
      ELSE
        IF l.prefs(Prefs).objectAlign=left THEN
          label.object.Move(l.x+l.lSize+l.prefs(Prefs).hSpace.GetSize(),pos);
        ELSE
          label.object.Move(l.x+l.lSize+l.prefs(Prefs).hSpace.GetSize()+l.rSize-label.object.oWidth,pos);
        END;
        IF l.prefs(Prefs).labelAlign=right THEN
          label.text.Move(l.x+l.lSize-label.text.oWidth,
                          pos+label.object.oHeight-label.text.oHeight);
        ELSE
          label.text.Move(l.x,pos+label.object.oHeight-label.text.oHeight);
        END;
      END;

      IF label.object.oHeight>label.text.height THEN
        INC(pos,label.object.oHeight);
      ELSE
        INC(pos,label.text.height);
      END;

      INC(pos,l.prefs(Prefs).vSpace.GetSize());

      label:=label.next;
    END;

    l.Layout^;
  END Layout;

  PROCEDURE (l : Label) Draw*(x,y,w,h : LONGINT);

  VAR
    object : G.Object;
    label  : Text;
    draw   : D.DrawInfo;

  BEGIN
    l.Draw^(x,y,w,h);

    IF ~l.Intersect(x,y,w,h) THEN
      RETURN;
    END;

    draw:=l.GetDrawInfo();

    draw.InstallClip(x,y,w,h);

    label:=l.labelList;
    WHILE label#NIL DO
      object:=label.text;
      IF object#NIL THEN
        draw.SubRegion(object.oX,object.oY,object.oWidth,object.oHeight);
      END;
      object:=label.object;
      IF object#NIL THEN
        draw.SubRegion(object.oX,object.oY,object.oWidth,object.oHeight);
      END;
      label:=label.next;
    END;

    l.DrawBackground(l.x,l.y,l.width,l.height);

    draw.FreeLastClip;

    label:=l.labelList;
    WHILE label#NIL DO
      object:=label.text;
      IF object#NIL THEN
        object.Draw(x,y,w,h);
      END;
      object:=label.object;
      IF object#NIL THEN
        object.Draw(x,y,w,h);
      END;
      label:=label.next;
    END;
  END Draw;

  PROCEDURE (l : Label) Hide*;

  VAR
    label : Text;

  BEGIN
    IF l.visible THEN
      label:=l.labelList;
      WHILE label#NIL DO
        label.object.Hide;
        label.text.Hide;
        label:=label.next;
      END;
      l.DrawHide;
      l.Hide^;
    END;
  END Hide;

  PROCEDURE (l : Label) HandleMouseEvent*(event : E.MouseEvent;
                                          VAR grab : G.Object):BOOLEAN;


  VAR
    label : Text;

  BEGIN
    IF l.visible THEN
      label:=l.labelList;
      WHILE label#NIL DO
        IF label.object.HandleMouseEvent(event,grab) THEN
          RETURN TRUE;
        END;

        IF label.text.HandleMouseEvent(event,grab) THEN
          RETURN TRUE;
        END;

        label:=label.next;
      END;
    END;

    RETURN FALSE;
  END HandleMouseEvent;

  PROCEDURE (l : Label) GetPosObject*(x,y : LONGINT; type : LONGINT):G.Object;

  VAR
    return : G.Object;
    label  : Text;

  BEGIN
    IF l.visible THEN
      label:=l.labelList;
      WHILE label#NIL DO
        return:=label.object.GetPosObject(x,y,type);
        IF return#NIL THEN
          RETURN return;
        END;

        return:=label.text.GetPosObject(x,y,type);
        IF return#NIL THEN
          RETURN return;
        END;

        label:=label.next;
      END;
    END;

    RETURN l.GetPosObject^(x,y,type);
  END GetPosObject;

  PROCEDURE (l : Label) GetDnDObject*(x,y : LONGINT; drag : BOOLEAN):G.Object;

  VAR
    return : G.Object;
    label  : Text;

  BEGIN
    IF l.visible THEN
      label:=l.labelList;
      WHILE label#NIL DO
        return:=label.object.GetDnDObject(x,y,drag);
        IF return#NIL THEN
          RETURN return;
        END;

        return:=label.text.GetDnDObject(x,y,drag);
        IF return#NIL THEN
          RETURN return;
        END;

        label:=label.next;
      END;
    END;

    RETURN NIL;
  END GetDnDObject;

  PROCEDURE CreateLabel*():Label;

  VAR
    label : Label;

  BEGIN
    NEW(label);
    label.Init;

    RETURN label;
  END CreateLabel;

BEGIN
  NEW(prefs);
  prefs.Init;
END VO:Label.