(**
   This module implements a button.
   The button label  can be a text or any other graphical object.

   This module is well documented to show how to implement a simple
   gadget. It also demonstrates, how the VO-engine works.
*)

MODULE VO:Button;

(*
    Implements a button.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D   := VO:Base:Display,
       E   := VO:Base:Event,
       F   := VO:Base:Frame,
       O   := VO:Base:Object,
       U   := VO:Base:Util,
       Z   := VO:Base:Size,

       G   := VO:Object,
       T   := VO:Text,
       W   := VO:Window,

       str := Strings;


CONST
  (* Types *)

  normal  * = 0;
  small   * = 1;
  image   * = 2;
  toolBar * = 3;

  (* actions *)

  pressedMsg * = 0; (* The constant for the PressedMsg.
                       A PressedMsg will be generated everytime
                       our button gets pressed.
                     *)

  repeatTimeOut = 75; (* Time between button repeat *)

TYPE
  Prefs*     = POINTER TO PrefsDesc;
  PrefsDesc* = RECORD (G.PrefsDesc)
  (**
    In this class all preferences stuff of the button is stored.
  *)
                 hSpace*,
                 vSpace*      : Z.SizeDesc;
                 sFrame*,
                 iFrame*,
                 tFrame*      : LONGINT; (* the frame to use for the button *)
                 highlight*   : BOOLEAN;
                 gridDisable* : BOOLEAN;
               END;


  Button*     = POINTER TO ButtonDesc;
  ButtonDesc* = RECORD (G.GadgetDesc)
  (**
    THe ButtonClass.
    Since a button needs some eventhandling, so we inherit
    from VOGUIObject.Gadget.
  *)
                  image      : G.Object; (* and an image *)
                  timeOut    : D.TimeOut;(* timeout info for pulse-mode *)
                  font       : LONGINT;  (* Special font for the button *)
                  type       : LONGINT;  (* normal, Small or image *)
                  state,               (* The state of the button. TRUE when currently selected *)
                  active,
                  pulse      : BOOLEAN;  (* send permanent pressed Msg on ButtonDown and none on ButtonUp *)
                  shortCut   : CHAR;
                  scMode     : LONGINT;
                  scAssigned : BOOLEAN;
                END;


  (* messages *)

  PressedMsg*     = POINTER TO PressedMsgDesc;
  PressedMsgDesc* = RECORD (O.MessageDesc)
  (**
    The PressedMsg generated everytime the button get clicked.
  *)
                    END;


VAR
  prefs* : Prefs;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.hSpace.Init;
    p.vSpace.Init;
    p.hSpace.SetSize(Z.softUnitP,50);
    p.vSpace.SetSize(Z.softUnitP,50);

    p.frame:=F.double3DOut;
    p.sFrame:=F.single3DOut;
    p.iFrame:=F.double3DOut;
    p.tFrame:=F.double3DTool;
    p.highlight:=TRUE;
    p.gridDisable:=TRUE;
  END Init;

  PROCEDURE (b : Button) Init*;
  (**
    The Init-method of our button.
  *)

  BEGIN
    b.Init^;          (* We *must* call Init of our baseclass first *)

    b.SetBackground(D.buttonBackgroundColor);

    b.SetPrefs(prefs);

    b.timeOut:=NIL;   (* initialize the timeout info to NIL *)

    b.SetFlags({G.canFocus,G.handleSC}); (* We can handle keyboard focus *)
                                         (* We can handle shortcuts *)
    b.type:=normal;   (* Normally we use normal button frames *)
    b.pulse:=FALSE;   (* normaly one pressedMsg on mousebutton up *)
    b.font:=D.normalFont; (* Just the global default font fo the button label *)

    (* There is no default image *)
    b.image:=NIL;

    (* The button is by default unpressed *)
    b.state:=FALSE;
    b.active:=FALSE;

    b.shortCut:=0X;   (* No shortcut key *)
    b.scMode:=W.none; (* No special key for keyboard shortcut *)
    b.scAssigned:=TRUE;
  END Init;

  PROCEDURE (b : Button) SetFont*(font : LONGINT);
  (**
    Set a new font to be used by the button gadget.
  *)

  BEGIN
    b.font:=font;
  END SetFont;

  PROCEDURE (b : Button) SetText*(string : ARRAY OF CHAR);
  (**
    Call this method, if you want the given text to be displayed in
    the button.

    This creates simply an VOText.Text instance for Button.image.
  *)

  VAR
    text   : T.Text;
    help   : U.Text;

  BEGIN
    NEW(text); (* Allocate a VOText.Text object for displaying text *)
    text.Init; (* We must call Text.Init asfor all objects *)
    text.SetParent(b);
    help:=U.EscapeString(string);   (* We escape the string because it could contain escape-sequences.
                                         This is because Oberon-2 does not specify an escape-character for strings *)

    text.SetFlags({G.horizontalFlex,G.verticalFlex}); (* Our text should be resizeable in all directions *)
    text.SetDefault(T.centered,{},b.font); (* We want the text of tex textimage centered in its bounds *)
    text.SetText(help^);            (* Set the escaped text to the Text-object *)
    b.CopyBackground(text);         (* Copy our background to the text *)
    b.image:=text;                  (* Use it as our image *)
  END SetText;

  PROCEDURE (b : Button) SetLabelText*(string : ARRAY OF CHAR);
  (**
    Call this method, if you want the given text to be displayed in
    the button and want the button to interpret the given string
    regarding shortcuts.

    This creates simply an VOText.Text instance for Button.image.
  *)

  VAR
    text     : T.Text;
    help     : U.Text;
    sc,x,
    length,
    offset   : INTEGER;

  BEGIN
    NEW(text); (* Allocate a VOText.Text object for displaying text *)
    text.Init; (* We must call Text.Init asfor all objects *)
    text.SetParent(b);

    offset:=0;
    sc:=-1;

    length:=str.Length(string);
    x:=0;
    WHILE x<length DO
      CASE string[x] OF
        "\":
          CASE string[x+1] OF
            "_",
            "*",
            "#",
            "^": str.Delete(string,x,1);
                 DEC(length);
          ELSE
          END;
      | "_" : sc:=x;
              b.shortCut:=string[x+1];
      | "*" : b.scMode:=W.return;
      | "^" : b.scMode:=W.escape;
      | "#" : b.scMode:=W.default;
      ELSE
      END;
      INC(x);
    END;

    IF sc>=0 THEN
      INC(offset,5); (* "\eu" "\en" - underscore *)
    END;

    IF b.scMode#W.none THEN
      INC(offset,2); (* \eX - ( *|^|# ) *)
    END;

    IF (sc>=0) OR (b.scMode#W.none) THEN
      NEW(help,length+1+offset);
      COPY(string,help^);

      IF sc>=0 THEN
        str.Delete(help^,sc,1);
        str.Insert("\en",sc+1,help^);
        str.Insert("\eu",sc,help^);
        INC(length,5);
      END;

      IF b.scMode#W.none THEN
        str.Delete(help^,length-1,1);
        CASE b.scMode OF
          W.return:  str.Insert("\eR",length-1,help^);
        | W.escape:  str.Insert("\eE",length-1,help^);
        | W.default: str.Insert("\eD",length-1,help^);
        END;
      END;

      IF (b.shortCut#0X) OR (b.scMode#W.none) THEN
        b.scAssigned:=FALSE;
      END;

      help:=U.EscapeString(help^);
    ELSE
      help:=U.EscapeString(string);   (* We escape the string because it could contain escape-sequences.
                                         This is because Oberon-2 does not specify an escape-character for strings *)
    END;

    text.SetFlags({G.horizontalFlex,G.verticalFlex}); (* Our text should be resizeable in all directions *)
    text.SetDefault(T.centered,{},b.font); (* We want the text of tex textimage centered in its bounds *)
    text.SetText(help^);            (* Set the escaped text to the Text-object *)
    b.CopyBackground(text);         (* Copy our background to the text *)
    b.image:=text;                  (* Use it as our image *)
  END SetLabelText;

  PROCEDURE (b : Button) SetImage*(image : G.Object);
  (**
    Use this method if you do not want text displayed in the button.
    We use n external initialized image. Not that for Button there is no difference
    between text and any oher image after this point.
  *)

  BEGIN
    b.image:=image;
    b.image.SetParent(b);
    b.CopyBackground(b.image);
  END SetImage;

  PROCEDURE (b : Button) SetType*(type : LONGINT);
  (**
    We can define special types of buttons. Currently supported are normal, small and image.
  *)

  BEGIN
    b.type:=type;
  END SetType;

  PROCEDURE (b : Button) SetPulse*(pulse : BOOLEAN);
  (**
    Is pulsemode is true, the button send permanent pressedMsg on mouse button down
    and none on the final button up.

    This is usefull for buttons in a scroller or similar.
  *)

  BEGIN
    b.pulse:=pulse;
  END SetPulse;

  PROCEDURE (b : Button) GetDnDObject*(x,y : LONGINT; drag : BOOLEAN):G.Object;
  (**
    Returns the object that coveres the given point and that supports
    drag and drop of data.

    If drag is TRUE, when want to find a object that we can drag data from,
    else we want an object to drop data on.
  *)

  BEGIN
    IF b.visible & b.PointIsIn(x,y) & (b.image#NIL) THEN
      RETURN b.image.GetDnDObject(x,y,drag);
    ELSE
      RETURN NIL;
    END;
  END GetDnDObject;

  PROCEDURE (b : Button) CalcSize*;
  (**
    This method gets called by the parent object before the first call to Button.Draw.

    We have to calculate the bounds of our button.
  *)

  BEGIN

    IF b.image#NIL THEN
      (*
        Tell the image not to highlight, if the prefs say it shouldn't.
      *)
      IF ~b.prefs(Prefs).highlight THEN
        b.image.SetFlags({G.noHighlight});
      ELSE
        b.image.RemoveFlags({G.noHighlight});
      END;
    END;

    (*
      We check, if the image can show some kind of frame. If so, we do not display
      the frame ourself, but delegate it to the image.
    *)

    IF (b.image#NIL) & ~b.image.StdFocus() & b.MayFocus() THEN
      b.RemoveFlags({G.stdFocus});
      b.image.SetFlags({G.mayFocus});
    END;

    (* Let the frame calculate its size *)
    CASE b.type OF
      normal:
        b.SetObjectFrame(b.prefs(Prefs).frame);
     | small:
        b.SetObjectFrame(b.prefs(Prefs).sFrame);
     | image:
        b.SetObjectFrame(b.prefs(Prefs).iFrame);
     | toolBar:
        b.SetObjectFrame(b.prefs(Prefs).tFrame);
    END;
    (*
      Our size is the size of the frame plus a little space we want to have
      to have between the frame and the image.
    *)
    b.width:=b.prefs(Prefs).hSpace.GetSize()*2;
    b.height:=b.prefs(Prefs).vSpace.GetSize()*2;

    (* Our minimal size is equal to the normal size *)
    b.minWidth:=b.width;
    b.minHeight:=b.height;

    IF b.image#NIL THEN
      (*
        Now we let the image calculate its bounds and simply add its size
        to the size of the button.
      *)
      b.image.CalcSize;
      INC(b.width,b.image.width);
      INC(b.height,b.image.height);
      INC(b.minWidth,b.image.minWidth);
      INC(b.minHeight,b.image.minHeight);
    END;

    (* We *must* call CalcSize of our superclass! *)
    b.CalcSize^;
  END CalcSize;

  PROCEDURE (b : Button) HandleMouseEvent*(event : E.MouseEvent;
                                           VAR grab : G.Object):BOOLEAN;
  (**
    This method gets called when the window gets an event and looks for
    someone that processes it.

    If GetFocus return an object, that objets HandleEvent-method
    get called untill it gives away the focus.
  *)

  VAR
    pressed : PressedMsg; (* We want to create a pressedMsg
                             when we found out we got pressed *)
    redraw  : BOOLEAN;

  BEGIN
    (* It makes no sense to get the focus if we are currently not visible *)
    IF ~b.visible OR b.disabled THEN
      RETURN FALSE;
    END;

    (*
      When the left mousebutton gets pressed without any qualifier
      in the bounds of our button...
    *)

    WITH event : E.ButtonEvent DO
      IF (event.type=E.mouseDown) & b.PointIsIn(event.x,event.y)
       & (event.button=E.button1) THEN
        IF ~b.state THEN
          (* We change our state to pressed and redisplay ourself *)
          b.state:=TRUE;
          b.Redraw;

          IF b.pulse THEN
            NEW(pressed);
            b.Send(pressed,pressedMsg);
            b.timeOut:=D.display.AddTimeOut(0,repeatTimeOut,b);
          END;

          (*
            Since we want the focus for waiting for buttonup we return
            a pointer to ourself.
          *)
          grab:=b;
          RETURN TRUE;
        ELSE
          (* We change our state to pressed and redisplay ourself *)
          b.state:=TRUE;
          b.Redraw;

          IF b.pulse THEN
            NEW(pressed);
            b.Send(pressed,pressedMsg);
            b.timeOut:=D.display.AddTimeOut(0,repeatTimeOut,b);
          END;
        END;
      ELSIF (event.type=E.mouseUp) & (event.button=E.button1) THEN
        (* We get unselected again and must redisplay ourself *)
        b.state:=FALSE;
        b.Redraw;

        (*
          Clean up and remove possibly remaining timer event.
        *)
        IF b.timeOut#NIL THEN
          D.display.RemoveTimeOut(b.timeOut);
          b.timeOut:=NIL;
        END;

        grab:=NIL;

        (*
          If the users released the left mousebutton over our bounds we really
          got selected.
        *)
        IF b.PointIsIn(event.x,event.y) & ~b.pulse THEN
          (*
            We create a PressedMsg and send it away.
            Button.Send (inherited from Object) does the managing
            of the possible attached handlers for use.
          *)
          (* Action: Button pressed *)
          NEW(pressed);
          b.Send(pressed,pressedMsg);
        END;

        IF ~b.PointIsIn(event.x,event.y) THEN
          IF b.active THEN
            b.active:=FALSE;
            b.Redraw;
          END;
        END;
      END;
    | event : E.MotionEvent DO

      redraw:=FALSE;
      IF (b.PointIsIn(event.x,event.y)) THEN
        IF grab=b THEN
          IF ~b.state THEN
            b.state:=TRUE;
            redraw:=TRUE;
          END;
        END;

        IF ~b.active THEN
          b.active:=TRUE;
          redraw:=TRUE;
        END;
      ELSE
        IF grab=b THEN
          IF b.state THEN
            b.state:=FALSE;
            redraw:=TRUE;
          END;
        END;

        IF b.active THEN
          b.active:=FALSE;
          redraw:=TRUE;
        END;
      END;

      IF redraw THEN
        b.Redraw;
      END;

    ELSE
    END;

    RETURN FALSE;
  END HandleMouseEvent;

  PROCEDURE (b : Button) HandleKeyEvent*(event : E.KeyEvent):BOOLEAN;
  (**
    This method gets called when we have the keyboard focus and
    a key is pressed. We analyse the key. If it is the space-key,
    we've been selected.

    TODO
    Go into the selected state, when we get a KeyPress-event for
    the space key. Get unselected again, if we receive a KeyRelease-
    event and send a PressedMsg. Get unselected without sending
    a PressedMsg when Object.LostFocus got called.
  *)

  VAR
    keysym  : LONGINT;

    pressed : PressedMsg; (* We want to create a pressedMsg
                             when we found out we got pressed *)

  BEGIN
    IF event.type=E.keyDown THEN
      keysym:=event.GetKey();
      IF keysym=E.space THEN
        b.state:=TRUE;
        b.Redraw;
        (* TODO: Add some delay here *)
        b.state:=FALSE;
        b.Redraw;
        NEW(pressed);
        b.Send(pressed,pressedMsg);
        RETURN TRUE;
      END;
    END;
    RETURN FALSE;
  END HandleKeyEvent;

  PROCEDURE (b : Button) HandleShortcutEvent*(id,state : LONGINT);

  VAR
    pressed : PressedMsg; (* We want to create a pressedMsg
                             when we found out we got pressed *)

  BEGIN
    IF state=G.pressed THEN
      b.state:=TRUE;
      b.Redraw;
    ELSE
      b.state:=FALSE;
      b.Redraw;
      IF state=G.released THEN
        NEW(pressed);
        b.Send(pressed,pressedMsg);
      END;
    END;
  END HandleShortcutEvent;

  PROCEDURE (b : Button) Layout*;

  BEGIN
   (*
      We tell the image to resize themself to
      our current bounds. Our bounds could have changed
      because Resize may have been called by some layout-objects
      between Button.CalcSize and Button.Draw.
    *)

    IF b.image#NIL THEN
      b.image.Resize(b.width,b.height);
      b.image.Move(b.x+(b.width-b.image.oWidth) DIV 2,
                   b.y+(b.height-b.image.oHeight) DIV 2);
    END;

    b.Layout^;
  END Layout;

  PROCEDURE (b : Button) Draw*(x,y,w,h : LONGINT);
  (**
    Gets called if the engine whants us to draw ourself.
  *)

  VAR
    draw   : D.DrawInfo;
    window : D.Window;


  BEGIN
    IF ~b.Intersect(x,y,w,h) THEN
      RETURN;
    END;

    draw:=b.GetDrawInfo();

    (*
      Set the correct draw mode before calling the baseclass,
      since the baseclass draw the object frame.
    *)
    IF b.active & ~b.state THEN
      draw.mode:={D.activated};
    ELSIF b.state THEN
      draw.mode:={D.selected};
    ELSE
      draw.mode:={};
    END;

    b.Draw^(x,y,w,h); (* We must call Draw of our superclass *)

    (* Change the drawmode to normal, since we share draw with our parent object. *)
    draw.mode:={};

    IF b.image#NIL THEN
      (*
        We fill the entier region with the background color
      *)

      draw.InstallClip(x,y,w,h);
      draw.SubRegion(b.image.oX,b.image.oY,b.image.oWidth,b.image.oHeight);
      b.DrawBackground(b.x,b.y,b.width,b.height);
      draw.FreeLastClip;

      (*
        If we are selected, we set the drawmode to VODisplay.selected.

        It may be, that some objects draw themselfs different when they are
        selected. Others may ignore it.

        Check if disabling should be delegated to image and check
        if the image has special care for diabling, too.
      *)

      IF b.disabled & ~b.prefs(Prefs).gridDisable & (G.canDisable IN b.image.flags) THEN
        draw.mode:={D.disabled};
      ELSIF b.state THEN
        draw.mode:={D.selected};
      ELSE
        draw.mode:={};
      END;

      (*
        Draw the image centered in the bounds of the button

        Note, that we assume that the frame is symetric. This is true
        for buttonFrame (wich we use), but for groupFrame with label this is
        *not* the case!
      *)

      b.image.Draw(x,y,w,h);

      draw.mode:={};
    ELSE
      b.DrawBackground(b.x,b.y,b.width,b.height);
    END;

    IF b.disabled & (b.prefs(Prefs).gridDisable OR ~(G.canDisable IN b.image.flags)) THEN
      b.DrawDisabled;
    END;

    IF ~b.scAssigned THEN
      window:=b.GetWindow();
      WITH window : W.Window DO
        IF (b.shortCut#0X) OR (b.scMode#W.none) THEN
          window.AddShortcutObject(b,{},b.shortCut,0,b.scMode);
          b.scAssigned:=TRUE;
        END;
      END;
    END;
  END Draw;

  PROCEDURE (b : Button) DrawFocus*;
  (**
    Draw the keyboard focus.
  *)

  VAR
    draw : D.DrawInfo;

  BEGIN
    (* If our image can draw a keyboard focus, delegate it *)
    IF (b.image#NIL) & ~b.image.StdFocus() THEN
      draw:=b.GetDrawInfo();

      IF b.state THEN
        draw.mode:={D.selected};
      ELSE
        draw.mode:={};
      END;
      b.image.DrawFocus;
      draw.mode:={};
    ELSE
      (* Delegate drawing to the baseclass *)
      b.DrawFocus^;
    END;
  END DrawFocus;

  PROCEDURE (b : Button) HideFocus*;
  (**
    Hide the keyboard focus.
  *)

  VAR
    draw : D.DrawInfo;

  BEGIN
    (* If our image can draw a keyboard focus, delegate it *)
    IF (b.image#NIL) & ~b.image.StdFocus() THEN

      draw:=b.GetDrawInfo();
      IF b.state THEN
        draw.mode:={D.selected};
      ELSE
        draw.mode:={};
      END;
      b.image.HideFocus;
      draw.mode:={};
    ELSE
      (* Delegate drawing to the baseclass *)
      b.HideFocus^;
    END;
  END HideFocus;

  PROCEDURE (b : Button) Hide*;
  (**
    Hides the button. The buttons hides itself by hiding its label and its frame. Note, that you must
    set Object.visible yourself when you overload Object.Hide.
  *)

  BEGIN
    IF b.visible THEN
      IF b.image#NIL THEN
        (* Hide the image *)
        b.image.Hide;
      END;
      (* hide the frame *)
      b.DrawHide;
      b.Hide^;
    END;
  END Hide;

  PROCEDURE (b : Button) Receive*(message : O.Message);

  VAR
    pressed : PressedMsg; (* We want to create a pressedMsg
                             when we found out we got pressed *)

  BEGIN
    WITH
      message : D.TimeOutMsg DO
        IF b.state THEN
          (*
            Time to send a new pressed message.
          *)
          NEW(pressed);
          b.Send(pressed,pressedMsg);
          b.timeOut:=D.display.AddTimeOut(0,repeatTimeOut,b);
        ELSE
          b.timeOut:=NIL;
        END;
    ELSE
      b.Receive^(message);
    END;
  END Receive;

  PROCEDURE CreateButton*():Button;

  VAR
    button : Button;

  BEGIN
    NEW(button);
    button.Init;

    RETURN button;
  END CreateButton;

BEGIN
  NEW(prefs);
  prefs.Init;

END VO:Button.