/* GNU Objective C Runtime Thread Interface
   Copyright (C) 1996, 1997 Free Software Foundation, Inc.
   Contributed by Galen C. Hunt (gchunt@cs.rochester.edu)
   Conditions added by Mircea Oancea (mircea@first.elcom.pub.ro)
      
This file is part of GNU CC.

GNU CC is free software; you can redistribute it and/or modify it under the
terms of the GNU General Public License as published by the Free Software
Foundation; either version 2, or (at your option) any later version.

GNU CC is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
GNU CC; see the file COPYING.  If not, write to the Free Software
Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

/* As a special exception, if you link this library with files compiled with
   GCC to produce an executable, this does not cause the resulting executable
   to be covered by the GNU General Public License. This exception does not
   however invalidate any other reasons why the executable file might be
   covered by the GNU General Public License.  */

#include <objc/thr.h>
#include "runtime.h"

#include <stdlib.h>
#include <thread.h>
#include <synch.h>
#include <errno.h>

#if OBJC_WITH_GC
#  include <gc.h>
#endif
#if OBJC_WITH_SGC
#  include <sgc.h>
#endif

/* Key structure for maintaining thread specific storage */
static thread_key_t __objc_thread_data_key = 0;

/* Backend initialization functions */

/* Initialize the threads subsystem. */
int __objc_init_thread_system(void) {
  /* Initialize the thread storage key */

  printf("Launching Objective-C threading backend: %s ..\n", __FILE__);
  fflush(stdout);
  
  return (thr_keycreate(&__objc_thread_data_key, NULL) == 0) ? 0 : -1;
}

/* Close the threads subsystem. */
int __objc_close_thread_system(void) {
  return 0;
}

/* Backend thread functions */

/* Create a new thread of execution. */
objc_thread_t __objc_thread_detach(void (*func)(void *arg), void *arg) {
  thread_t new_thread_id = 0;

  if (
#if OBJC_WITH_SGC
      SGC_thr_create
#elif OBJC_WITH_GC
      GC_thr_create
#else
      thr_create
#endif
      (NULL, 0, (void *)func, arg,
       THR_DETACHED | THR_NEW_LWP,
       &new_thread_id) == 0) {

    if (sizeof(objc_thread_t) != sizeof(new_thread_id))
      abort();

    return (objc_thread_t)new_thread_id;
  }
  else
    return NULL;
}

/* Set the current thread's priority. */
int __objc_thread_set_priority(int priority) {
  int sys_priority = 0;

  switch (priority) {
    case OBJC_THREAD_INTERACTIVE_PRIORITY:
      sys_priority = 300;
      break;
    default:
    case OBJC_THREAD_BACKGROUND_PRIORITY:
      sys_priority = 200;
      break;
    case OBJC_THREAD_LOW_PRIORITY:
      sys_priority = 1000;
      break;
  }

  /* Change priority */
  return (thr_setprio(thr_self(), sys_priority) == 0) ? 0 : -1;
}

/* Return the current thread's priority. */
int __objc_thread_get_priority(void) {
  int sys_priority;
                                                   
  if (thr_getprio(thr_self(), &sys_priority) == 0){
    if (sys_priority >= 250)
      return OBJC_THREAD_INTERACTIVE_PRIORITY;
    else if (sys_priority >= 150)
      return OBJC_THREAD_BACKGROUND_PRIORITY;
    return OBJC_THREAD_LOW_PRIORITY;
  }

  /* Couldn't get priority. */
  return -1;
}

/* Yield our process time to another thread. */
void __objc_thread_yield(void) {
  thr_yield();
}

/* Terminate the current thread. */
int __objc_thread_exit(void) {
  /* exit the thread */
  thr_exit(&__objc_thread_exit_status);

  /* Failed if we reached here */
  return -1;
}

/* Returns an integer value which uniquely describes a thread. */
objc_thread_t __objc_thread_id(void) {
  if (sizeof(objc_thread_t) != sizeof(thread_t))
    abort();
  
  return (objc_thread_t)thr_self();
}

/* Sets the thread's local storage pointer. */
int __objc_thread_set_data(void *value) {
  return (thr_setspecific(__objc_thread_data_key, value) == 0) ? 0 : -1;
}

/* Returns the thread's local storage pointer. */
void *__objc_thread_get_data(void) {
  void *value = NULL;

  if (thr_getspecific(__objc_thread_data_key, &value) == 0)
    return value;

  return NULL;
}

/* Backend mutex functions */

int __objc_mutex_allocate(objc_mutex_t mutex) {
  mutex->backend = calloc(1, sizeof(mutex_t));
  if (mutex->backend == NULL) {
    fprintf(stderr, "%s: could not allocate memory for Solaris mutex !\n"
            __PRETTY_FUNCTION__);
    return -1;
  }
  
  if (mutex_init(mutex->backend, USYNC_THREAD, 0) != 0) {
    free(mutex->backend); mutex->backend = NULL;
    return -1;
  }
  return 0;
}

int __objc_mutex_deallocate(objc_mutex_t mutex) {
  if (mutex->backend) {
    mutex_destroy(mutex->backend);
    free(mutex->backend); mutex->backend = NULL;
    return 0;
  }
  else
    return -1;
}

int __objc_mutex_lock(objc_mutex_t mutex) {
  return (mutex_lock(mutex->backend) != 0) ? -1 : 0;
}

int __objc_mutex_trylock(objc_mutex_t mutex) {
  return (mutex_trylock(mutex->backend) != 0) ? -1 : 0;
}

int __objc_mutex_unlock(objc_mutex_t mutex) {
  return (mutex_unlock(mutex->backend) != 0) ? -1 : 0;
}

/* Backend condition mutex functions */

int __objc_condition_allocate(objc_condition_t condition) {
  condition->backend = calloc(1, sizeof(cond_t));
  if (condition->backend == NULL) {
    fprintf(stderr, "%s: could not allocate memory for Solaris mutex !\n"
            __PRETTY_FUNCTION__);
    return -1;
  }

  if (cond_init(condition->backend, USYNC_THREAD, 0) != 0) {
    free(condition->backend); condition->backend = NULL;
    return -1;
  }
  return 0;
}

int __objc_condition_deallocate(objc_condition_t condition) {
  if (condition->backend) {
    cond_destroy(condition->backend);
    free(condition->backend); condition->backend = NULL;
    return 0;
  }
  else
    return -1;
}

int __objc_condition_wait(objc_condition_t condition, objc_mutex_t mutex) {
  return cond_wait(condition->backend, mutex->backend);
}

/* Wake up all threads waiting on this condition. */
int __objc_condition_broadcast(objc_condition_t condition) {
  return cond_broadcast(condition->backend);
}

/* Wake up one thread waiting on this condition. */
int __objc_condition_signal(objc_condition_t condition) {
  return cond_signal(condition->backend);
}
