#include <string.h>
#include <stdio.h>
#include <errno.h>
#include "libnjb.h"
#include "defs.h"
#include "protocol.h"
#include "njb_error.h"

static struct njb_error_stack_struct {
	int idx;
	int count;
	char msg[MAX_ERRORS+1][MAX_ERRLEN];
} estack;

int njb_error= 0;

int _error_overflow (void);
const char *njb_status_string (unsigned char code);

int _error_overflow (void)
{
	if ( estack.count >= MAX_ERRORS ) {
		strcpy(estack.msg[MAX_ERRORS], "Error stack overflow");
		estack.count= MAX_ERRORS+1;
		return 1;
	}

	return 0;
}

void njb_error_add3 (const char *sub, const char *prefix, const char
	*suffix, int code)
{
	char *ep= estack.msg[estack.count];

	if ( _error_overflow() ) return;

	snprintf(ep, MAX_ERRLEN, "%s: %s: %s %s", sub, prefix,
		njb_error_string(code), suffix);

	njb_error= code;
	estack.count++;
}

void njb_error_add2 (const char *sub, const char *prefix, int code)
{
	char *ep= estack.msg[estack.count];

	if ( _error_overflow() ) return;

	snprintf(ep, MAX_ERRLEN, "%s: %s: %s", sub, prefix,
		njb_error_string(code));

	njb_error= code;
	estack.count++;
}

void njb_error_add (const char *sub, int code)
{
	char *ep= estack.msg[estack.count];

	if ( _error_overflow() ) return;

	snprintf(ep, MAX_ERRLEN, "%s: %s", sub, njb_error_string(code));

	njb_error= code;
	estack.count++;
}

void njb_error_status (const char *sub, unsigned char code)
{
	char *ep= estack.msg[estack.count];

	if ( _error_overflow() ) return;

	snprintf(ep, MAX_ERRLEN, "%s: %s", sub, njb_status_string(code));

	njb_error= EO_BADSTATUS;
	estack.count++;
}

void njb_error_clear (void)
{
	estack.count= 0;
	estack.idx= 0;
}

void njb_error_reset_geterror (void)
{
	estack.idx= 0;
}

const char *njb_error_geterror (void)
{
	const char *cp;
	if ( estack.idx == estack.count ) return NULL;

	cp= estack.msg[estack.idx];
	estack.idx++;
	return cp;
}


const char *njb_error_string (int code)
{
	switch (code)
	{
	case -1:
		return strerror(errno);
	case 0:
		return "";
	case EO_USBCTL:
		return "I/O failure on USB control pipe";
	case EO_USBBLK:
		return "I/O failure on USB data pipe";
	case EO_RDSHORT:
		return "short read on USB data pipe";
	case EO_NOMEM:
		return "out of memory";
	case EO_BADDATA:
		return "invalid data";
	case EO_EOM:
		return "end of data";
	case EO_BADSTATUS:
		return "bad status from Jukebox";
	case EO_BADNJBID:
		return "Jukebox ID has changed";
	case EO_BADCOUNT:
		return "library count mismatch";
	case EO_WRSHORT:
		return "short write on USB data pipe";
	case EO_NULLTMP:
		return "temporary transfer dir not defined";
	case EO_TOOBIG:
		return "block size too big";
	case EO_CANTMOVE:
		return "can't move file to destination";
	case EO_TIMEOUT:
		return "operation timed out";
	case EO_ABORTED:
		return "operation aborted";
	case EO_EOF:
		return "received EOF";
	case EO_DEVICE:
		return "can't open device for read/write";
	case EO_INIT:
		return "can't initialize device";
	case EO_TMPFILE:
		return "can't create temporary file";
	case EO_XFERDENIED:
		return "transfer request denied";
	case EO_WRFILE:	
		return "error writing output file";
	case EO_XFERERROR:
		return "bad transfer completion status";
	case EO_SRCFILE:
		return "can't read source file";
	case EO_INVALID:
		return "invalid arguments";
	case EO_AGAIN:
		return "resource temporarily unavailable";
	default:
		return "(undefined error)";
	}
}

void njb_error_dump (FILE *fp)
{
	const char *sp;

	njb_error_reset_geterror();
	while ( (sp= njb_error_geterror()) != NULL ) {
		fprintf(fp, "%s\n", sp);
	}
}

const char *njb_status_string (unsigned char code)
{
        /* This is not very thread safe... */
        static char buffer[100];

	switch (code)
	{
	case NJB_MSG_OKAY:
		return "no error";

	case NJB_ERR_FAILED:
		return "operation failed";
	case NJB_ERR_DEVICE_BUSY:
		return "device busy";
	case NJB_ERR_STORAGE_FULL:
		return "storage full";
	case NJB_ERR_HD_GENERAL_ERROR:
		return "general hard drive failure";
	case NJB_ERR_SETTIME_REJECTED:
		return "set time rejected";

	case NJB_ERR_TRACK_NOT_FOUND:
		return "track not found";
	case NJB_ERR_TRACK_EXISTS:
		return "track exists";
	case NJB_ERR_TITLE_MISSING:
		return "title missing";
	case NJB_ERR_CODEC_MISSING:
		return "CODEC missing";
	case NJB_ERR_SIZE_MISSING:
		return "size missing";
	case NJB_ERR_IO_OPERATION_ABORTED:
		return "I/O operation aborted";
	case NJB_ERR_READ_WRITE_ERROR:
		return "read/write error";
	case NJB_ERR_NOT_OPENED:
		return "not opened";
	case NJB_ERR_UPLOAD_DENIED:
		return "upload denied";

	case NJB_ERR_PLAYLIST_NOT_FOUND:
		return "playlist not found";
	case NJB_ERR_PLAYLIST_EXISTS:
		return "playlist exists";
	case NJB_ERR_PLAYLIST_ITEM_NOT_FOUND:
		return "playlist item not found";
	case NJB_ERR_PLAYLIST_ITEM_EXISTS:
		return "playlist item exists";

	case NJB_MSG_QUEUED_AUDIO_STARTED:
		return "queued audio started";
	case NJB_MSG_PLAYER_FINISHED:
		return "player finished";
	case NJB_ERR_VOLUME_CONTROL_UNAVAILABLE:
		return "volume control unavailable";
	case NJB_ERR_QUEUE_ALREADY_EMPTY:
		 return "queue already empty";

	case NJB_ERR_DATA_NOT_FOUND:
		return "data not found";
	case NJB_ERR_DATA_NOT_OPENED:
		return "data not opened";

	case NJB_ERR_UNDEFINED_ERR:
		return "undefined error";

	case NJB_ERR_NOT_IMPLEMENTED:
		return "function not implemented";
	}

	sprintf(buffer, "unknown error %02X", code);
	return buffer;
}
