/* DiffieHellman.java -- Diffie-Hellman key exchange.
   Copyright (C) 2003  Casey Marshall <rsdio@metastatic.org>

This file is a part of Jessie.

Jessie is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

Jessie is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with Jessie; if not, write to the

   Free Software Foundation, Inc.,
   59 Temple Place, Suite 330,
   Boston, MA  02111-1307
   USA

Linking this library statically or dynamically with other modules is
making a combined work based on this library.  Thus, the terms and
conditions of the GNU General Public License cover the whole
combination.

As a special exception, the copyright holders of this library give you
permission to link this library with independent modules to produce an
executable, regardless of the license terms of these independent
modules, and to copy and distribute the resulting executable under terms
of your choice, provided that you also meet, for each linked independent
module, the terms and conditions of the license of that module.  An
independent module is a module which is not derived from or based on
this library.  If you modify this library, you may extend this exception
to your version of the library, but you are not obligated to do so.  If
you do not wish to do so, delete this exception statement from your
version.  */


package org.metastatic.jessie.provider;

import java.math.BigInteger;
import gnu.crypto.key.dh.GnuDHPrivateKey;

/**
 * <p>Simple implementation of two-party Diffie-Hellman key agreement.</p>
 *
 * <p>The primes used in this class are from the following documents:</p>
 *
 * <ul>
 * <li>D. Harkins and D. Carrel, "The Internet Key Exchange (IKE)", <a
 * href="http://www.ietf.org/rfc/rfc2409.txt">RFC 2409</a>.</li>
 * <li>T. Kivinen and M. Kojo, "More Modular
 * Exponential (MODP) Diffie-Hellman groups for Internet Key Exchange
 * (IKE)", <a href="http://www.ietf.org/rfc/rfc3526.txt">RFC
 * 3526</a>.</li>
 * </li>
 *
 * <p>The generator for all these primes is 2.</p>
 */
final class DiffieHellman
{

  // Class method.
  // -------------------------------------------------------------------------

  /**
   * Get the system's Diffie-Hellman parameters, in which <i>g</i> is 2
   * and <i>p</i> is determined by the property
   * <code>"jessie.keypool.dh.group"</code>. The default value for <i>p</i>
   * is 18, corresponding to {@link #GROUP_18}.
   */
  static GnuDHPrivateKey getParams()
  {
    BigInteger p = DiffieHellman.GROUP_18;
    String group = Util.getSecurityProperty("jessie.key.dh.group");
    if (group != null)
      {
        group = group.trim();
        if (group.equals("1"))
          p = DiffieHellman.GROUP_1;
        else if (group.equals("2"))
          p = DiffieHellman.GROUP_2;
        else if (group.equals("5"))
          p = DiffieHellman.GROUP_5;
        else if (group.equals("14"))
          p = DiffieHellman.GROUP_14;
        else if (group.equals("15"))
          p = DiffieHellman.GROUP_15;
        else if (group.equals("16"))
          p = DiffieHellman.GROUP_16;
        else if (group.equals("17"))
          p = DiffieHellman.GROUP_17;
        else if (group.equals("18"))
          p = DiffieHellman.GROUP_18;
      }
    return new GnuDHPrivateKey(null, p, DH_G, null);
  }

  // Constants.
  // -------------------------------------------------------------------------

  /**
   * The generator for all Diffie Hellman groups below.
   */
  static final BigInteger DH_G = BigInteger.valueOf(2L);

  /**
   * p = 2^768 - 2 ^704 - 1 + 2^64 * { [2^638 pi] + 149686 }
   */
  static final BigInteger GROUP_1 = new BigInteger("00" +
    "FFFFFFFFFFFFFFFFC90FDAA22168C234C4C6628B80DC1CD1" +
    "29024E088A67CC74020BBEA63B139B22514A08798E3404DD" +
    "EF9519B3CD3A431B302B0A6DF25F14374FE1356D6D51C245" +
    "E485B576625E7EC6F44C42E9A63A3620FFFFFFFFFFFFFFFF", 16);

  /**
   * p = 2^1024 - 2^960 - 1 + 2^64 * { [2^894 pi] + 129093 }
   */
  static final BigInteger GROUP_2 = new BigInteger("00" +
    "FFFFFFFFFFFFFFFFC90FDAA22168C234C4C6628B80DC1CD1" +
    "29024E088A67CC74020BBEA63B139B22514A08798E3404DD" +
    "EF9519B3CD3A431B302B0A6DF25F14374FE1356D6D51C245" +
    "E485B576625E7EC6F44C42E9A637ED6B0BFF5CB6F406B7ED" +
    "EE386BFB5A899FA5AE9F24117C4B1FE649286651ECE65381" +
    "FFFFFFFFFFFFFFFF", 16);

  /**
   * This prime p = 2^1536 - 2^1472 - 1 + 2^64 * { [2^1406 pi] + 741804 }.
   */
  static final BigInteger GROUP_5 = new BigInteger("00" +
    "FFFFFFFFFFFFFFFFC90FDAA22168C234C4C6628B80DC1CD1" +
    "29024E088A67CC74020BBEA63B139B22514A08798E3404DD" +
    "EF9519B3CD3A431B302B0A6DF25F14374FE1356D6D51C245" +
    "E485B576625E7EC6F44C42E9A637ED6B0BFF5CB6F406B7ED" +
    "EE386BFB5A899FA5AE9F24117C4B1FE649286651ECE45B3D" +
    "C2007CB8A163BF0598DA48361C55D39A69163FA8FD24CF5F" +
    "83655D23DCA3AD961C62F356208552BB9ED529077096966D" +
    "670C354E4ABC9804F1746C08CA237327FFFFFFFFFFFFFFFF", 16);

  /**
   * p = 2^2048 - 2^1984 - 1 + 2^64 * { [2^1918 pi] + 124476 }.
   */
  static final BigInteger GROUP_14 = new BigInteger("00" +
    "FFFFFFFFFFFFFFFFC90FDAA22168C234C4C6628B80DC1CD1" +
    "29024E088A67CC74020BBEA63B139B22514A08798E3404DD" +
    "EF9519B3CD3A431B302B0A6DF25F14374FE1356D6D51C245" +
    "E485B576625E7EC6F44C42E9A637ED6B0BFF5CB6F406B7ED" +
    "EE386BFB5A899FA5AE9F24117C4B1FE649286651ECE45B3D" +
    "C2007CB8A163BF0598DA48361C55D39A69163FA8FD24CF5F" +
    "83655D23DCA3AD961C62F356208552BB9ED529077096966D" +
    "670C354E4ABC9804F1746C08CA18217C32905E462E36CE3B" +
    "E39E772C180E86039B2783A2EC07A28FB5C55DF06F4C52C9" +
    "DE2BCBF6955817183995497CEA956AE515D2261898FA0510" +
    "15728E5A8AACAA68FFFFFFFFFFFFFFFF", 16);

  /**
   * p = 2^3072 - 2^3008 - 1 + 2^64 * { [2^2942 pi] + 1690314 }.
   */
  static final BigInteger GROUP_15 = new BigInteger("00" +
    "FFFFFFFFFFFFFFFFC90FDAA22168C234C4C6628B80DC1CD1" +
    "29024E088A67CC74020BBEA63B139B22514A08798E3404DD" +
    "EF9519B3CD3A431B302B0A6DF25F14374FE1356D6D51C245" +
    "E485B576625E7EC6F44C42E9A637ED6B0BFF5CB6F406B7ED" +
    "EE386BFB5A899FA5AE9F24117C4B1FE649286651ECE45B3D" +
    "C2007CB8A163BF0598DA48361C55D39A69163FA8FD24CF5F" +
    "83655D23DCA3AD961C62F356208552BB9ED529077096966D" +
    "670C354E4ABC9804F1746C08CA18217C32905E462E36CE3B" +
    "E39E772C180E86039B2783A2EC07A28FB5C55DF06F4C52C9" +
    "DE2BCBF6955817183995497CEA956AE515D2261898FA0510" +
    "15728E5A8AAAC42DAD33170D04507A33A85521ABDF1CBA64" +
    "ECFB850458DBEF0A8AEA71575D060C7DB3970F85A6E1E4C7" +
    "ABF5AE8CDB0933D71E8C94E04A25619DCEE3D2261AD2EE6B" +
    "F12FFA06D98A0864D87602733EC86A64521F2B18177B200C" +
    "BBE117577A615D6C770988C0BAD946E208E24FA074E5AB31" +
    "43DB5BFCE0FD108E4B82D120A93AD2CAFFFFFFFFFFFFFFFF", 16);

  /**
   * p = 2^4096 - 2^4032 - 1 + 2^64 * { [2^3966 pi] + 240904 }.
   */
  static final BigInteger GROUP_16 = new BigInteger("00" +
    "FFFFFFFFFFFFFFFFC90FDAA22168C234C4C6628B80DC1CD1" +
    "29024E088A67CC74020BBEA63B139B22514A08798E3404DD" +
    "EF9519B3CD3A431B302B0A6DF25F14374FE1356D6D51C245" +
    "E485B576625E7EC6F44C42E9A637ED6B0BFF5CB6F406B7ED" +
    "EE386BFB5A899FA5AE9F24117C4B1FE649286651ECE45B3D" +
    "C2007CB8A163BF0598DA48361C55D39A69163FA8FD24CF5F" +
    "83655D23DCA3AD961C62F356208552BB9ED529077096966D" +
    "670C354E4ABC9804F1746C08CA18217C32905E462E36CE3B" +
    "E39E772C180E86039B2783A2EC07A28FB5C55DF06F4C52C9" +
    "DE2BCBF6955817183995497CEA956AE515D2261898FA0510" +
    "15728E5A8AAAC42DAD33170D04507A33A85521ABDF1CBA64" +
    "ECFB850458DBEF0A8AEA71575D060C7DB3970F85A6E1E4C7" +
    "ABF5AE8CDB0933D71E8C94E04A25619DCEE3D2261AD2EE6B" +
    "F12FFA06D98A0864D87602733EC86A64521F2B18177B200C" +
    "BBE117577A615D6C770988C0BAD946E208E24FA074E5AB31" +
    "43DB5BFCE0FD108E4B82D120A92108011A723C12A787E6D7" +
    "88719A10BDBA5B2699C327186AF4E23C1A946834B6150BDA" +
    "2583E9CA2AD44CE8DBBBC2DB04DE8EF92E8EFC141FBECAA6" +
    "287C59474E6BC05D99B2964FA090C3A2233BA186515BE7ED" +
    "1F612970CEE2D7AFB81BDD762170481CD0069127D5B05AA9" +
    "93B4EA988D8FDDC186FFB7DC90A6C08F4DF435C934063199" +
    "FFFFFFFFFFFFFFFF", 16);

  static final BigInteger GROUP_17 = new BigInteger("00" +
    "FFFFFFFFFFFFFFFFC90FDAA22168C234C4C6628B80DC1CD129024E08" +
    "8A67CC74020BBEA63B139B22514A08798E3404DDEF9519B3CD3A431B" +
    "302B0A6DF25F14374FE1356D6D51C245E485B576625E7EC6F44C42E9" +
    "A637ED6B0BFF5CB6F406B7EDEE386BFB5A899FA5AE9F24117C4B1FE6" +
    "49286651ECE45B3DC2007CB8A163BF0598DA48361C55D39A69163FA8" +
    "FD24CF5F83655D23DCA3AD961C62F356208552BB9ED529077096966D" +
    "670C354E4ABC9804F1746C08CA18217C32905E462E36CE3BE39E772C" +
    "180E86039B2783A2EC07A28FB5C55DF06F4C52C9DE2BCBF695581718" +
    "3995497CEA956AE515D2261898FA051015728E5A8AAAC42DAD33170D" +
    "04507A33A85521ABDF1CBA64ECFB850458DBEF0A8AEA71575D060C7D" +
    "B3970F85A6E1E4C7ABF5AE8CDB0933D71E8C94E04A25619DCEE3D226" +
    "1AD2EE6BF12FFA06D98A0864D87602733EC86A64521F2B18177B200C" +
    "BBE117577A615D6C770988C0BAD946E208E24FA074E5AB3143DB5BFC" +
    "E0FD108E4B82D120A92108011A723C12A787E6D788719A10BDBA5B26" +
    "99C327186AF4E23C1A946834B6150BDA2583E9CA2AD44CE8DBBBC2DB" +
    "04DE8EF92E8EFC141FBECAA6287C59474E6BC05D99B2964FA090C3A2" +
    "233BA186515BE7ED1F612970CEE2D7AFB81BDD762170481CD0069127" +
    "D5B05AA993B4EA988D8FDDC186FFB7DC90A6C08F4DF435C934028492" +
    "36C3FAB4D27C7026C1D4DCB2602646DEC9751E763DBA37BDF8FF9406" +
    "AD9E530EE5DB382F413001AEB06A53ED9027D831179727B0865A8918" +
    "DA3EDBEBCF9B14ED44CE6CBACED4BB1BDB7F1447E6CC254B33205151" +
    "2BD7AF426FB8F401378CD2BF5983CA01C64B92ECF032EA15D1721D03" +
    "F482D7CE6E74FEF6D55E702F46980C82B5A84031900B1C9E59E7C97F" +
    "BEC7E8F323A97A7E36CC88BE0F1D45B7FF585AC54BD407B22B4154AA" +
    "CC8F6D7EBF48E1D814CC5ED20F8037E0A79715EEF29BE32806A1D58B" +
    "B7C5DA76F550AA3D8A1FBFF0EB19CCB1A313D55CDA56C9EC2EF29632" +
    "387FE8D76E3C0468043E8F663F4860EE12BF2D5B0B7474D6E694F91E" +
    "6DCC4024FFFFFFFFFFFFFFFF", 16);

  /**
   * p = 2^8192 - 2^8128 - 1 + 2^64 * { [2^8062 pi] + 4743158 }.
   *
   * <p>This value, while quite large, is estimated to provide the equivalent
   * cryptographic strength of a symmetric key between 190 and 320 bits.
   */
  static final BigInteger GROUP_18 = new BigInteger("00" +
    "FFFFFFFFFFFFFFFFC90FDAA22168C234C4C6628B80DC1CD1" +
    "29024E088A67CC74020BBEA63B139B22514A08798E3404DD" +
    "EF9519B3CD3A431B302B0A6DF25F14374FE1356D6D51C245" +
    "E485B576625E7EC6F44C42E9A637ED6B0BFF5CB6F406B7ED" +
    "EE386BFB5A899FA5AE9F24117C4B1FE649286651ECE45B3D" +
    "C2007CB8A163BF0598DA48361C55D39A69163FA8FD24CF5F" +
    "83655D23DCA3AD961C62F356208552BB9ED529077096966D" +
    "670C354E4ABC9804F1746C08CA18217C32905E462E36CE3B" +
    "E39E772C180E86039B2783A2EC07A28FB5C55DF06F4C52C9" +
    "DE2BCBF6955817183995497CEA956AE515D2261898FA0510" +
    "15728E5A8AAAC42DAD33170D04507A33A85521ABDF1CBA64" +
    "ECFB850458DBEF0A8AEA71575D060C7DB3970F85A6E1E4C7" +
    "ABF5AE8CDB0933D71E8C94E04A25619DCEE3D2261AD2EE6B" +
    "F12FFA06D98A0864D87602733EC86A64521F2B18177B200C" +
    "BBE117577A615D6C770988C0BAD946E208E24FA074E5AB31" +
    "43DB5BFCE0FD108E4B82D120A92108011A723C12A787E6D7" +
    "88719A10BDBA5B2699C327186AF4E23C1A946834B6150BDA" +
    "2583E9CA2AD44CE8DBBBC2DB04DE8EF92E8EFC141FBECAA6" +
    "287C59474E6BC05D99B2964FA090C3A2233BA186515BE7ED" +
    "1F612970CEE2D7AFB81BDD762170481CD0069127D5B05AA9" +
    "93B4EA988D8FDDC186FFB7DC90A6C08F4DF435C934028492" +
    "36C3FAB4D27C7026C1D4DCB2602646DEC9751E763DBA37BD" +
    "F8FF9406AD9E530EE5DB382F413001AEB06A53ED9027D831" +
    "179727B0865A8918DA3EDBEBCF9B14ED44CE6CBACED4BB1B" +
    "DB7F1447E6CC254B332051512BD7AF426FB8F401378CD2BF" +
    "5983CA01C64B92ECF032EA15D1721D03F482D7CE6E74FEF6" +
    "D55E702F46980C82B5A84031900B1C9E59E7C97FBEC7E8F3" +
    "23A97A7E36CC88BE0F1D45B7FF585AC54BD407B22B4154AA" +
    "CC8F6D7EBF48E1D814CC5ED20F8037E0A79715EEF29BE328" +
    "06A1D58BB7C5DA76F550AA3D8A1FBFF0EB19CCB1A313D55C" +
    "DA56C9EC2EF29632387FE8D76E3C0468043E8F663F4860EE" +
    "12BF2D5B0B7474D6E694F91E6DBE115974A3926F12FEE5E4" +
    "38777CB6A932DF8CD8BEC4D073B931BA3BC832B68D9DD300" +
    "741FA7BF8AFC47ED2576F6936BA424663AAB639C5AE4F568" +
    "3423B4742BF1C978238F16CBE39D652DE3FDB8BEFC848AD9" +
    "22222E04A4037C0713EB57A81A23F0C73473FC646CEA306B" +
    "4BCBC8862F8385DDFA9D4B7FA2C087E879683303ED5BDD3A" +
    "062B3CF5B3A278A66D2A13F83F44F82DDF310EE074AB6A36" +
    "4597E899A0255DC164F31CC50846851DF9AB48195DED7EA1" +
    "B1D510BD7EE74D73FAF36BC31ECFA268359046F4EB879F92" +
    "4009438B481C6CD7889A002ED5EE382BC9190DA6FC026E47" +
    "9558E4475677E9AA9E3050E2765694DFC81F56E880B96E71" +
    "60C980DD98EDD3DFFFFFFFFFFFFFFFFF", 16);

}
