/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  scale.cc - GtkScale, GtkHScale and GtkVScale C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "scale.h"
#include "private/scale_p.h"
#include "adjustment.h"
#include <math.h>
#include <stdlib.h>

using namespace Inti;

/*  Gtk::Scale
 */
 
Gtk::Scale::Scale(GtkScale *scale, bool reference) 
: Range((GtkRange*)scale, reference)
{
}

Gtk::Scale::Scale()
: Range((GtkRange*)ScaleClass::create())
{
}

Gtk::Scale::~Scale()
{
}

GtkScaleClass*
Gtk::Scale::gtk_scale_class() const 
{ 
	return get_class<GtkScaleClass>(); 
}
	
Gtk::Scale::operator GtkScale* () const 
{
	return this ? gtk_scale() : 0; 
}

int
Gtk::Scale::get_digits() const
{
	return gtk_scale_get_digits(gtk_scale());
}

bool
Gtk::Scale::get_draw_value() const
{
	return gtk_scale_get_draw_value(gtk_scale());
}

Gtk::PositionType 
Gtk::Scale::get_value_pos() const
{
	return (PositionType)gtk_scale_get_value_pos(gtk_scale());
}	

void 
Gtk::Scale::set_digits(int digits)
{
	gtk_scale_set_digits(gtk_scale(), digits);
}

void 
Gtk::Scale::set_draw_value(bool draw_value)
{
	gtk_scale_set_draw_value(gtk_scale(), draw_value);
}

void 
Gtk::Scale::set_value_pos(PositionType pos)
{
	gtk_scale_set_value_pos(gtk_scale(), (GtkPositionType)pos);
}

String
Gtk::Scale::format_value(double value)
{
	char *fmt = _gtk_scale_format_value(gtk_scale(), value);
	String tmp_fmt(fmt);
	g_free(fmt);
	return tmp_fmt;
}

/*  Gtk::ScaleClass
 */

void
Gtk::ScaleClass::init(GtkScaleClass *g_class)
{
	RangeClass::init((GtkRangeClass*)g_class);
	g_class->draw_value = &draw_value_proxy;
	g_class->format_value = &format_value_proxy;
}

GType
Gtk::ScaleClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GTK_TYPE_SCALE, (GClassInitFunc)init);
	}
	return type;
}

void*
Gtk::ScaleClass::create()
{
	return g_object_new(get_type(), 0);
}

void
Gtk::ScaleClass::draw_value_proxy(GtkScale *scale)
{
	Scale *tmp_scale = G::Object::pointer<Scale>(scale);
	if (tmp_scale)
		tmp_scale->do_draw_value();
	else
	{
		GtkScaleClass *g_class = G::TypeInstance::class_peek_parent<GtkScaleClass>(GTK_SCALE_GET_CLASS(scale));
		if (g_class->draw_value)
			g_class->draw_value(scale);
	}
}

gchar*
Gtk::ScaleClass::format_value_proxy(GtkScale *scale, gdouble value)
{
	Scale *tmp_scale = G::Object::pointer<Scale>(scale);
	gchar *fmt = 0;
	if (tmp_scale)
		fmt = tmp_scale->on_format_value(value);
	else
	{
		GtkScaleClass *g_class = G::TypeInstance::class_peek_parent<GtkScaleClass>(GTK_SCALE_GET_CLASS(scale));
		if (g_class->format_value)
			fmt = g_class->format_value(scale, value);
	}
	return fmt;
}

/*  Overridable methods
 */

void
Gtk::Scale::do_draw_value()
{
	GtkScaleClass *g_class = class_peek_parent<GtkScaleClass>(gtk_scale_class());
	if (g_class->draw_value)
		g_class->draw_value(gtk_scale());
}

/*  Signal handlers
 */

char*
Gtk::Scale::on_format_value(double value)
{
	GtkScaleClass *g_class = class_peek_parent<GtkScaleClass>(gtk_scale_class());
	char *fmt = 0;
	if (g_class->format_value)
		fmt = g_class->format_value(gtk_scale(), value);
	return fmt;
}

/*  Gtk::Scale properties
 */

const Gtk::Scale::DigitsPropertyType Gtk::Scale::digits_property("digits");

const Gtk::Scale::DrawValuePropertyType Gtk::Scale::draw_value_property("draw_value");

const Gtk::Scale::ValuePosPropertyType Gtk::Scale::value_pos_property("value_pos");

/*  Signals
 */

const Gtk::Scale::FormatValueSignalType Gtk::Scale::format_value_signal("format_value");

namespace {

void construct_scale_with_range(GtkScale *scale, gdouble min, gdouble max, gdouble step)
{
	g_return_if_fail(min < max);
	g_return_if_fail(step != 0.0);

	GtkObject *adjustment = gtk_adjustment_new (min, min, max, step, 10 * step, 0);
	g_object_set(scale, "adjustment", adjustment, 0);

	gint digits;
	if (fabs(step) >= 1.0 || step == 0.0)
    	digits = 0;
  	else
	{
	    digits = abs((gint)floor(log10(fabs(step))));
		if (digits > 5)
			digits = 5;
	}
	gtk_scale_set_digits(scale, digits);
}

} // namespace

/*  Gtk::HScale
 */
 
Gtk::HScale::HScale(GtkHScale *hscale, bool reference) 
: Scale((GtkScale*)hscale, reference)
{
}

Gtk::HScale::HScale()
: Scale((GtkScale*)HScaleClass::create())
{
}

Gtk::HScale::HScale(Adjustment& adjustment)
: Scale((GtkScale*)HScaleClass::create())
{
	set("adjustment", adjustment.gtk_adjustment(), 0);
}

Gtk::HScale::HScale(double min, double max, double step)
: Scale((GtkScale*)HScaleClass::create())
{
	construct_scale_with_range(gtk_scale(), min, max, step);
}

Gtk::HScale::~HScale()
{
}

GtkHScaleClass* 
Gtk::HScale::gtk_hscale_class() const 
{ 
	return get_class<GtkHScaleClass>(); 
}
	
Gtk::HScale::operator GtkHScale* () const 
{ 
	return this ? gtk_hscale() : 0; 
}
	
/*  Gtk::HScaleClass
 */

void
Gtk::HScaleClass::init(GtkHScaleClass *g_class)
{
	ScaleClass::init((GtkScaleClass*)g_class);
}

GType
Gtk::HScaleClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GTK_TYPE_HSCALE, (GClassInitFunc)init);
	}
	return type;
}

void*
Gtk::HScaleClass::create()
{
	return g_object_new(get_type(), 0);
}

/*  Gtk::VScale
 */

Gtk::VScale::VScale(GtkVScale *vscale, bool reference)
: Scale((GtkScale*)vscale, reference)
{
}

Gtk::VScale::VScale()
: Scale((GtkScale*)VScaleClass::create())
{
}

Gtk::VScale::VScale(Adjustment& adjustment)
: Scale((GtkScale*)VScaleClass::create())
{
	set("adjustment", adjustment.gtk_adjustment(), 0);
}

Gtk::VScale::VScale(double min, double max, double step)
: Scale((GtkScale*)VScaleClass::create())
{
	construct_scale_with_range(gtk_scale(), min, max, step);
}

Gtk::VScale::~VScale()
{
}

GtkVScaleClass* 
Gtk::VScale::gtk_vscale_class() const 
{ 
	return get_class<GtkVScaleClass>(); 
}
	
Gtk::VScale::operator GtkVScale* () const 
{ 
	return this ? gtk_vscale() : 0; 
}
	
/*  Gtk::VScaleClass
 */

void
Gtk::VScaleClass::init(GtkVScaleClass *g_class)
{
	ScaleClass::init((GtkScaleClass*)g_class);
}

GType
Gtk::VScaleClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GTK_TYPE_VSCALE, (GClassInitFunc)init);
	}
	return type;
}

void*
Gtk::VScaleClass::create()
{
	return g_object_new(get_type(), 0);
}

