/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  fileselection.cc - GtkFileSelection C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "fileselection.h"
#include "private/fileselection_p.h"
#include "entry.h"
#include "label.h"
#include "button.h"
#include "../glib/error.h"

using namespace Inti;

/*  Gtk::FileSelection
 */

Gtk::FileSelection::FileSelection(GtkFileSelection *file_selection, bool reference)
: Dialog((GtkDialog*)file_selection, reference)
{
}

Gtk::FileSelection::FileSelection() 
: Dialog((GtkDialog*)FileSelectionClass::create())
{
}

Gtk::FileSelection::FileSelection(const String& title) 
: Dialog((GtkDialog*)FileSelectionClass::create())
{
	set_title(title);
}

Gtk::FileSelection::~FileSelection()
{
}

GtkFileSelectionClass* 
Gtk::FileSelection::gtk_file_selection_class() const 
{ 
	return get_class<GtkFileSelectionClass>(); 
}
	
Gtk::FileSelection::operator GtkFileSelection* () const 
{ 
	return this ? gtk_file_selection() : 0; 
}
	
Gtk::Entry* 
Gtk::FileSelection::selection_entry() const
{
	return G::Object::wrap<Entry>((GtkEntry*)gtk_file_selection()->selection_entry);
}

Gtk::Label*
Gtk::FileSelection::selection_text() const
{
	return G::Object::wrap<Label>((GtkLabel*)gtk_file_selection()->selection_text);
}

Gtk::Button* 
Gtk::FileSelection::ok_button() const
{
	return G::Object::wrap<Button>((GtkButton*)gtk_file_selection()->ok_button);
}
	
Gtk::Button* 
Gtk::FileSelection::cancel_button() const
{
	return G::Object::wrap<Button>((GtkButton*)gtk_file_selection()->cancel_button);
}
	
Gtk::Button*
Gtk::FileSelection::help_button() const
{
	return G::Object::wrap<Button>((GtkButton*)gtk_file_selection()->help_button);
}

String 
Gtk::FileSelection::get_filename(G::Error *error) const
{
	String s;
	s.from_filename(gtk_file_selection_get_filename(gtk_file_selection()), error);
	return s;
}

bool
Gtk::FileSelection::get_selections(std::vector<String>& selections, G::Error *error) const
{
	g_return_val_if_fail(selections.empty(), false);
	char **tmp_selections = gtk_file_selection_get_selections(gtk_file_selection());

	int i = 0;
	while (tmp_selections && tmp_selections[i])
	{
		String s = String::from_filename(tmp_selections[i], error);
		if (error && error->get())
			break;
		selections.push_back(s);
		++i;
	}

	g_strfreev(tmp_selections);
	return !selections.empty();
}

bool
Gtk::FileSelection::get_select_multiple() const
{
	return gtk_file_selection_get_select_multiple(gtk_file_selection());
}

void 
Gtk::FileSelection::set_filename(const String& filename, G::Error *error)
{
	std::string s = filename.to_filename(error);
	gtk_file_selection_set_filename(gtk_file_selection(), s.c_str());
}

void 
Gtk::FileSelection::complete(const String& pattern)
{
	gtk_file_selection_complete(gtk_file_selection(), pattern.c_str());
}

void 
Gtk::FileSelection::show_fileop_buttons()
{
	gtk_file_selection_show_fileop_buttons(gtk_file_selection());
}

void
Gtk::FileSelection::hide_fileop_buttons()
{
	gtk_file_selection_hide_fileop_buttons(gtk_file_selection());
}

void
Gtk::FileSelection::set_select_multiple(bool select_multiple)
{
	gtk_file_selection_set_select_multiple(gtk_file_selection(), select_multiple);
}

/*  Gtk::FileSelectionClass
 */

void
Gtk::FileSelectionClass::init(GtkFileSelectionClass *g_class)
{
	DialogClass::init((GtkDialogClass*)g_class);
}

GType
Gtk::FileSelectionClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GTK_TYPE_FILE_SELECTION, (GClassInitFunc)init);
	}
	return type;
}

void*
Gtk::FileSelectionClass::create()
{
	return g_object_new(get_type(), 0);
}

/*  Properties
 */

const Gtk::FileSelection::FilenamePropertyType Gtk::FileSelection::filename_property("filename");

const Gtk::FileSelection::ShowFileopsPropertyType Gtk::FileSelection::show_fileops_property("show_fileops");

const Gtk::FileSelection::SelectMultiplePropertyType Gtk::FileSelection::select_multiple_property("select_multiple");



