/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
//! @file inti/gtk/arrow.h
//! @brief A GtkArrow C++ wrapper interface.
//!
//! Provides Arrow, a widget that can be used to draw simple arrows that need to point
//! in one of the four cardinal directions (up, down, left, or right).

#ifndef INTI_GTK_ARROW_H
#define INTI_GTK_ARROW_H

#ifndef INTI_GTK_MISC_H
#include <inti/gtk/misc.h>
#endif

#ifndef __GTK_ARROW_H__
#include <gtk/gtkarrow.h>
#endif

namespace Inti {

namespace Gtk {

class ArrowClass;

//! @class Arrow arrow.h inti/gtk/arrow.h
//! @brief A GtkArrow C++ wrapper class.
//!
//! Arrow should be used to draw simple arrows that need to point in one of the four
//! cardinal directions (up, down, left, or right). The style of the arrow can be one
//! of shadow in, shadow out, etched in, or etched out.
//!
//! Arrow will fill any space alloted to it, but since it is inherited from Misc, it
//! can be padded and/or aligned, to fill exactly the space the programmer desires.
//! The direction or style of an arrow can be changed after creation by using set().

class Arrow : public Misc
{
	friend class G::Object;
	friend class ArrowClass;

	Arrow(const Arrow&);
	Arrow& operator=(const Arrow&);
	
protected:
//! @name Constructors
//! @{
	
	explicit Arrow(GtkArrow *arrow, bool reference = false);
	//!< Construct a new Arrow from an existing GtkArrow.
	//!< @param arrow A pointer to a GtkArrow.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>arrow</EM> can be a newly created GtkArrow or an existing
	//!< GtkArrow. (see G::Object::Object).

//! @}
//  Properties

	typedef G::Property<ArrowType, int> ArrowTypePropertyType;
	typedef G::PropertyProxy<G::Object, ArrowTypePropertyType> ArrowTypePropertyProxy;
	static const ArrowTypePropertyType arrow_type_property;

	typedef G::Property<ShadowType, int> ShadowTypePropertyType;
	typedef G::PropertyProxy<G::Object, ShadowTypePropertyType> ShadowTypePropertyProxy;
	static const ShadowTypePropertyType shadow_type_property;

public:
//! @name Constructors
//! @{

	Arrow();
	//!< Construct an Arrow with the arrow type ARROW_RIGHT and the shadow type SHADOW_OUT.

	Arrow(ArrowType arrow_type, ShadowType shadow_type);
	//!< Construct an Arrow with the specified direction and style.
	//!< @param arrow_type The direction of the arrow, a valid ArrowType.
	//!< @param shadow_type The style of the arrow, a valid ShadowType.

	virtual ~Arrow();
	//!< Destructor.

//! @}
//! @name Accessors
//! @{

	GtkArrow* gtk_arrow() const { return (GtkArrow*)instance; }
	//!< Get a pointer to the GtkArrow structure.

	GtkArrowClass* gtk_arrow_class() const;
	//!< Get a pointer to the GtkArrowClass structure.

	operator GtkArrow* () const;
	//!< Conversion operator; safely converts an Arrow to a GtkArrow pointer.

//! @}
//! @name Methods
//! @{

	void set(ArrowType arrow_type, ShadowType shadow_type);
	//!< Sets the direction and style of the Arrow.
	//!< @param arrow_type The direction of the arrow, a valid ArrowType.
	//!< @param shadow_type The style of the arrow, a valid ShadowType.

//! @}
//! @name Property Proxies
//! @{

	const ArrowTypePropertyProxy prop_arrow_type()
	{
		return ArrowTypePropertyProxy(this, &arrow_type_property);
	}
	//!< The direction the arrow should point (ArrowType : Read / Write).


	const ShadowTypePropertyProxy prop_shadow_type()
	{
		return ShadowTypePropertyProxy(this, &shadow_type_property);
	}
	//!< Appearance of the shadow surrounding the arrow (ShadowType : Read / Write).

//! @}
};

} // namespace Gtk

} // namespace Inti

#endif // INTI_GTK_ARROW_H

