/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *  Copyright (C) 2000 Red Hat, Inc.
 *
 *  window.cc - GdkWindow C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "window.h"
#include "private/window_p.h"
#include "bitmap.h"
#include "color.h"
#include "cursor.h"
#include "display.h"
#include "pixmap.h"
#include "region.h"
#include "screen.h"
#include "visual.h"
#include "../gdk-pixbuf/pixbuf.h"
#include <gdk/gdkx.h>

using namespace Inti;

/*  Gdk::get_default_root_window
 */

Gdk::Window*
Gdk::get_default_root_window()
{
	return G::Object::wrap<Window>(gdk_get_default_root_window());
}

/*  Gdk::Geometry
 */

Gdk::Geometry::Geometry()
: mask_(0)
{
	memset(&geometry_, 0, sizeof(GdkGeometry));
}

int
Gdk::Geometry::min_width() const
{
	return geometry_.min_width;
}

int 
Gdk::Geometry::min_height() const
{
	return geometry_.min_height;
}

int 
Gdk::Geometry::max_width() const
{
	return geometry_.max_width; 
}

int 
Gdk::Geometry::max_height() const
{
	return geometry_.max_height; 
}
	
int 
Gdk::Geometry::base_width() const
{
	return geometry_.base_width;
}

int 
Gdk::Geometry::base_height() const
{
	return geometry_.base_height;
}

int 
Gdk::Geometry::width_increment() const
{
	return geometry_.width_inc; 
}

int 
Gdk::Geometry::height_increment() const
{ 
	return geometry_.height_inc;
}

double 
Gdk::Geometry::min_aspect_ratio() const
{
	return geometry_.min_aspect; 
}
      
double 
Gdk::Geometry::max_aspect_ratio() const
{
	return geometry_.max_aspect;
}

bool
Gdk::Geometry::min_size_set() const
{
	return (mask_ & GDK_HINT_MIN_SIZE) != 0;
}

bool
Gdk::Geometry::max_size_set() const
{
	return (mask_ & GDK_HINT_MAX_SIZE) != 0;
}

bool
Gdk::Geometry::base_size_set() const
{
	return (mask_ & GDK_HINT_BASE_SIZE) != 0;
}

bool
Gdk::Geometry::aspect_range_set() const
{
        return (mask_ & GDK_HINT_ASPECT) != 0;
}

bool
Gdk::Geometry::resize_increments_set() const
{
        return (mask_ & GDK_HINT_RESIZE_INC) != 0;
}

void
Gdk::Geometry::set_min_size(int width, int height)
{
	geometry_.min_width = width;
	geometry_.min_height = height;
	mask_ |= GDK_HINT_MIN_SIZE;
}

void
Gdk::Geometry::unset_min_size()
{
	mask_ &= ~GDK_HINT_MIN_SIZE;
}

void
Gdk::Geometry::set_max_size(int width, int height)
{
	geometry_.max_width = width;
	geometry_.max_height = height;
	mask_ |= GDK_HINT_MAX_SIZE;
}

void
Gdk::Geometry::unset_max_size()
{
	mask_ &= ~GDK_HINT_MAX_SIZE;
}

void
Gdk::Geometry::set_base_size(int width, int height)
{
	geometry_.base_width = width;
	geometry_.base_height = height;
	mask_ |= GDK_HINT_BASE_SIZE;
}

void
Gdk::Geometry::unset_base_size()
{
	mask_ &= ~GDK_HINT_BASE_SIZE;
}

void
Gdk::Geometry::set_resize_increments(int width_inc, int height_inc)
{
	geometry_.width_inc = width_inc;
	geometry_.height_inc = height_inc;
	mask_ |= GDK_HINT_RESIZE_INC;
}

void
Gdk::Geometry::unset_resize_increments()
{
	mask_ &= ~GDK_HINT_RESIZE_INC;
}

void
Gdk::Geometry::set_aspect_range(double min_aspect_ratio, double max_aspect_ratio)
{
	geometry_.min_aspect = min_aspect_ratio;
	geometry_.max_aspect = max_aspect_ratio;
	mask_ |= GDK_HINT_ASPECT;
}

void
Gdk::Geometry::unset_aspect_range()
{
	mask_ &= ~GDK_HINT_ASPECT;
}

void
Gdk::Geometry::constrain_size(unsigned int flags, int width, int height, int *new_width, int *new_height)
{
	gdk_window_constrain_size(&geometry_, mask_, width, height, new_width, new_height);
}

/*  Gdk::WindowAttr
 */

Gdk::WindowAttr::WindowAttr(int width, int height, WindowType type, EventMaskField event_mask)
: mask_(0)
{
	set_size(width, height);
	attr_.window_type = (GdkWindowType)type;
	attr_.wclass = GDK_INPUT_OUTPUT;
	attr_.event_mask = (int)event_mask;
}

Gdk::WindowAttr::WindowAttr(int x, int y, int width, int height, WindowType type, EventMaskField event_mask)
: mask_(0)
{
	set_allocation(x, y, width, height);
	attr_.wclass = GDK_INPUT_OUTPUT;
	attr_.window_type = (GdkWindowType)type;
	attr_.event_mask = (int)event_mask;
}

Gdk::WindowAttr::WindowAttr(const Rectangle& allocation, WindowType type, EventMaskField event_mask)
: mask_(0)
{
	set_allocation(allocation);
	attr_.wclass = GDK_INPUT_OUTPUT;
	attr_.window_type = (GdkWindowType)type;
	attr_.event_mask = (int)event_mask;
}

void
Gdk::WindowAttr::set_title(const String& title)
{
	attr_.title = const_cast<char*>(title.c_str());
	mask_ |= GDK_WA_TITLE;
}

void
Gdk::WindowAttr::set_type(WindowType type)
{
	attr_.window_type = (GdkWindowType)type;
}

void
Gdk::WindowAttr::set_events(EventMaskField event_mask)
{
	attr_.event_mask = (int)event_mask;
}

void
Gdk::WindowAttr::set_position(int x, int y)
{
	attr_.x = x;
	attr_.y = y;
	mask_ |= (GDK_WA_X | GDK_WA_Y);
}

void
Gdk::WindowAttr::set_size(int width, int height)
{
	attr_.width = width;
	attr_.height = height;
}

void
Gdk::WindowAttr::set_allocation(int x, int y, int width, int height)
{
	attr_.x = x;
	attr_.y = y;
	attr_.width = width;
	attr_.height = height;
	mask_ |= (GDK_WA_X | GDK_WA_Y);
}

void
Gdk::WindowAttr::set_allocation(const Rectangle& allocation)
{
	attr_.x = allocation.x();
	attr_.y = allocation.y();
	attr_.width = allocation.width();
	attr_.height = allocation.height();
	mask_ |= (GDK_WA_X | GDK_WA_Y);
}

void
Gdk::WindowAttr::set_cursor(const Cursor& cursor)
{
	attr_.cursor = cursor.gdk_cursor();
	mask_ |= GDK_WA_CURSOR;
}

void
Gdk::WindowAttr::set_visual(const Visual& visual)
{
	attr_.visual = visual.gdk_visual();
	mask_ |= GDK_WA_VISUAL;
}

void
Gdk::WindowAttr::set_colormap(const Colormap& colormap)
{
	attr_.colormap = colormap.gdk_colormap();
	mask_ |= GDK_WA_COLORMAP;
}

void
Gdk::WindowAttr::set_input_only()
{
	attr_.wclass = GDK_INPUT_ONLY;
}

void
Gdk::WindowAttr::set_override_redirect()
{
	attr_.override_redirect = TRUE;
	mask_ |= GDK_WA_NOREDIR;
}

/*  Gdk::Window
 */

Gdk::Window::Window(GdkWindow *window, bool reference)
: Drawable((GdkDrawable*)window, reference)
{
}

Gdk::Window::Window(const WindowAttr& attributes)
: Drawable((GdkDrawable*)gdk_window_new(0, attributes.gdk_window_attr(), attributes.mask()))
{
}

Gdk::Window::Window(const Window& parent, const WindowAttr& attributes)
: Drawable((GdkDrawable*)gdk_window_new(parent.gdk_window(), attributes.gdk_window_attr(), attributes.mask()))
{
}
	
Gdk::Window::~Window() 
{
}
	
GdkWindowObject*
Gdk::Window::gdk_window_object() const 
{ 
	return GDK_WINDOW_OBJECT(gdk_window()); 
}
	
GdkWindowObjectClass*
Gdk::Window::gdk_window_object_class() const
{
	return get_class<GdkWindowObjectClass>(); 
}

Pointer<Gdk::Window> 
Gdk::Window::create(GdkNativeWindow anid, const Display *display)
{
	GdkDisplay *tmp_display = display ? display->gdk_display() : gdk_display_get_default();
	return G::Object::wrap<Window>(gdk_window_foreign_new_for_display(tmp_display, anid));
}

Pointer<Gdk::Window>
Gdk::Window::lookup(GdkNativeWindow anid, const Display *display)
{
	GdkDisplay *tmp_display = display ? display->gdk_display() : gdk_display_get_default();
	return G::Object::wrap<Window>((GdkWindow*)gdk_xid_table_lookup_for_display(tmp_display, anid));
}

Gdk::Window*
Gdk::Window::at_pointer(int *win_x, int *win_y, const Display *display)
{
	GdkDisplay *tmp_display = display ? display->gdk_display() : gdk_display_get_default();
	return G::Object::wrap<Window>(gdk_display_get_window_at_pointer(tmp_display, win_x, win_y));
}

Gdk::Window*
Gdk::Window::at_pointer(Point& point, const Display *display)
{
	int x, y;
	Gdk::Window *window = at_pointer(&x, &y, display);
	point.set(x, y);
	return window;
}

Gdk::WindowType
Gdk::Window::get_window_type() const
{
	return (WindowType)gdk_window_get_window_type(gdk_window());
}

bool
Gdk::Window::is_visible() const
{
	return gdk_window_is_visible(gdk_window());
}

bool
Gdk::Window::is_viewable() const
{
	return gdk_window_is_viewable(gdk_window());
}

Gdk::WindowState
Gdk::Window::get_state() const
{
	return (WindowState)gdk_window_get_state(gdk_window());
}

void
Gdk::Window::get_user_data(void *&data) const
{
	gdk_window_get_user_data(gdk_window(), &data);
}

void
Gdk::Window::get_geometry(int *x, int *y, int *width, int *height, int *depth) const
{
	gdk_window_get_geometry(gdk_window(), x, y, width, height, depth);
}

void
Gdk::Window::get_geometry(Rectangle& rectangle, int *depth) const
{
	GdkRectangle tmp;
	gdk_window_get_geometry(gdk_window(), &tmp.x, &tmp.y, &tmp.width, &tmp.height, depth);
	rectangle.set(tmp.x, tmp.y, tmp.width, tmp.height);
}

void
Gdk::Window::get_position(int *x, int *y) const
{
	gdk_window_get_position(gdk_window(), x, y);
}

Gdk::Point
Gdk::Window::get_position() const
{
	GdkPoint point;
	gdk_window_get_position(gdk_window(), &point.x, &point.y);
	return point;
}

bool
Gdk::Window::get_origin(int *x, int *y) const
{
	return gdk_window_get_origin(gdk_window(), x, y);
}

Gdk::Point
Gdk::Window::get_origin() const
{
	GdkPoint point;
	gdk_window_get_origin(gdk_window(), &point.x, &point.y);
	return point;
}

void
Gdk::Window::get_root_origin(int *x, int *y) const
{
	gdk_window_get_root_origin(gdk_window(), x, y);
}

Gdk::Point
Gdk::Window::get_root_origin() const
{
	GdkPoint point;
	gdk_window_get_root_origin(gdk_window(), &point.x, &point.y);
	return point;
}

Gdk::Window*
Gdk::Window::get_pointer(int *x, int *y, ModifierTypeField *mask) const
{
	GdkWindow *window = gdk_window_get_pointer(gdk_window(), x, y, (GdkModifierType*)mask);
	return window ? G::Object::wrap<Window>(window) : 0;
}

Gdk::Window*
Gdk::Window::get_parent() const
{
	GdkWindow *window = gdk_window_get_parent(gdk_window());
	return window ? G::Object::wrap<Window>(window) : 0;
}

Gdk::Window*
Gdk::Window::get_toplevel() const
{
	return G::Object::wrap<Window>(gdk_window_get_toplevel(gdk_window()));
}

bool
Gdk::Window::get_children(std::vector<Window*>& child_list) const
{
	g_return_val_if_fail(child_list.empty(), false);
	GList *first = gdk_window_get_children(gdk_window());
	GList *next = first;
	
	while (next != 0)
	{
		child_list.push_back(G::Object::wrap<Gdk::Window>((GdkWindow*)next->data));
		next = g_list_next(next);
	}

	g_list_free(first);
	return !child_list.empty();
}

Gdk::EventMaskField
Gdk::Window::get_events() const
{
	return (EventMaskField)gdk_window_get_events(gdk_window());
}

bool
Gdk::Window::get_decorations(WMDecorationField *decorations) const
{
	return gdk_window_get_decorations(gdk_window(), (GdkWMDecoration*)decorations);
}

bool
Gdk::Window::get_toplevels(std::vector<Window*>& toplevels)
{
	Screen *screen = Screen::get_default();
	return screen->get_toplevels(toplevels);
}

Pointer<Gdk::Region>
Gdk::Window::get_update_area() const
{
	return new Region(gdk_window_get_update_area(gdk_window()));
}

Gdk::Rectangle
Gdk::Window::get_frame_extents() const
{
	GdkRectangle rectangle;
	gdk_window_get_frame_extents(gdk_window(), &rectangle);
	return rectangle;
}

void 
Gdk::Window::get_frame_extents(Rectangle& rectangle) const
{
	gdk_window_get_frame_extents(gdk_window(), rectangle.gdk_rectangle());
}

void
Gdk::Window::dispose()
{
	gdk_window_destroy(gdk_window());
}

void
Gdk::Window::show()
{
	gdk_window_show(gdk_window());
}

void 
Gdk::Window::hide()
{
	gdk_window_hide(gdk_window());
}

void 
Gdk::Window::withdraw()
{
	gdk_window_withdraw(gdk_window());
}

void 
Gdk::Window::show_unraised()
{
	gdk_window_show_unraised(gdk_window());
}

void 
Gdk::Window::move(int x, int y)
{
	gdk_window_move(gdk_window(), x, y);
}

void 
Gdk::Window::move(const Point& point)
{
	gdk_window_move(gdk_window(), point.x(), point.y());
}

void 
Gdk::Window::resize(int width, int height)
{
	gdk_window_resize(gdk_window(), width, height);
}

void 
Gdk::Window::move_resize(int x, int y, int width, int height)
{
	gdk_window_move_resize(gdk_window(), x, y, width, height);
}

void 
Gdk::Window::move_resize(const Rectangle& rectangle)
{
	gdk_window_move_resize(gdk_window(), rectangle.x(), rectangle.y(), rectangle.width(), rectangle.height());
}

void 
Gdk::Window::reparent(Window& new_parent, int x, int y)
{
	gdk_window_reparent(gdk_window(), new_parent.gdk_window(), x, y);
}
	
void 
Gdk::Window::clear()
{
	gdk_window_clear(gdk_window());
}

void 
Gdk::Window::clear_area(int x, int y, int width, int height, bool generate_expose)
{
	if (!generate_expose)
		gdk_window_clear_area(gdk_window(), x, y, width, height);
	else
		gdk_window_clear_area_e(gdk_window(), x, y, width, height);
}

void 
Gdk::Window::clear_area(const Rectangle& rectangle, bool generate_expose)
{
	if (!generate_expose)
		gdk_window_clear_area(gdk_window(), rectangle.x(), rectangle.y(), rectangle.width(), rectangle.height());
	else
		gdk_window_clear_area_e(gdk_window(), rectangle.x(), rectangle.y(), rectangle.width(), rectangle.height());
}

void
Gdk::Window::raise()
{
	gdk_window_raise(gdk_window());
}

void 
Gdk::Window::lower()
{
	gdk_window_lower(gdk_window());
}

void 
Gdk::Window::focus(unsigned int timestamp)
{
	gdk_window_focus(gdk_window(), timestamp);
}

void
Gdk::Window::set_user_data(void *user_data)
{
	gdk_window_set_user_data(gdk_window(), user_data);
}

void
Gdk::Window::set_override_redirect(bool override_redirect)
{
	gdk_window_set_override_redirect(gdk_window(), override_redirect);
}

namespace { // filter_slot_callback

GdkFilterReturn filter_slot_callback(GdkXEvent *xevent, GdkEvent *event, gpointer data)
{
	Gdk::Window::FilterSlot *slot = static_cast<Gdk::Window::FilterSlot*>(data);
	return (GdkFilterReturn)slot->call(xevent, event);
}

} // filter_slot_callback

void
Gdk::Window::add_filter(Window *window, const FilterSlot *filter)
{
	gdk_window_add_filter(*window, &filter_slot_callback, (void*)filter);
}

void
Gdk::Window::remove_filter(Window *window, const FilterSlot *filter)
{
	gdk_window_remove_filter(*window, &filter_slot_callback, (void*)filter);
}

void
Gdk::Window::scroll(int dx, int dy)
{
	gdk_window_scroll(gdk_window(), dx, dy);
}

void
Gdk::Window::shape_combine_mask(Bitmap *shape_mask, int x, int y)
{
	gdk_window_shape_combine_mask(gdk_window(), shape_mask->gdk_bitmap(), x, y);
}

void
Gdk::Window::shape_combine_region(Region *shape_region, int offset_x, int offset_y)
{
	gdk_window_shape_combine_region(gdk_window(), *shape_region, offset_x, offset_y);
}

void
Gdk::Window::set_child_shapes()
{
	gdk_window_set_child_shapes(gdk_window());
}

void
Gdk::Window::merge_child_shapes()
{
	gdk_window_merge_child_shapes(gdk_window());
}

bool
Gdk::Window::set_static_gravities(bool use_static)
{
	return gdk_window_set_static_gravities(gdk_window(), use_static);
}

void
Gdk::Window::set_type_hint(WindowTypeHint hint)
{
	gdk_window_set_type_hint(gdk_window(), (GdkWindowTypeHint)hint);
}

void
Gdk::Window::set_modal_hint(bool modal)
{
	gdk_window_set_modal_hint(gdk_window(), modal);
}

void 
Gdk::Window::set_skip_taskbar_hint(bool skips_taskbar)
{
	gdk_window_set_skip_taskbar_hint(gdk_window(), skips_taskbar);
}

void 
Gdk::Window::set_skip_pager_hint(bool skips_pager)
{
	gdk_window_set_skip_pager_hint(gdk_window(), skips_pager);
}

void
Gdk::Window::set_geometry_hints(const Geometry *geometry)
{
	GdkGeometry *tmp_geometry = geometry ? geometry->gdk_geometry() : 0;
	unsigned int hints_mask = geometry ? geometry->gdk_window_hints() :0;
	gdk_window_set_geometry_hints(gdk_window(), tmp_geometry, (GdkWindowHints)hints_mask);
}

void
Gdk::Window::begin_paint(const Rectangle& rectangle)
{
	gdk_window_begin_paint_rect(gdk_window(), rectangle.gdk_rectangle());
}

void
Gdk::Window::begin_paint(const Region& region)
{
	gdk_window_begin_paint_region(gdk_window(), region.gdk_region());
}

void
Gdk::Window::end_paint()
{
	gdk_window_end_paint(gdk_window());
}

void
Gdk::Window::set_title(const String& title)
{
	gdk_window_set_title(gdk_window(), title.c_str());
}

void
Gdk::Window::set_role(const String& role)
{
	gdk_window_set_role(gdk_window(), role.c_str());
}

void
Gdk::Window::set_transient_for(Window& parent)
{
	gdk_window_set_transient_for(gdk_window(), parent);
}

void
Gdk::Window::set_background(const Color& color)
{
	gdk_window_set_background(gdk_window(), color.gdk_color());
}

void
Gdk::Window::set_back_pixmap(Pixmap *pixmap, bool parent_relative)
{
	gdk_window_set_back_pixmap(gdk_window(), *pixmap, parent_relative);
}

void
Gdk::Window::set_cursor(const Cursor& cursor)
{
	gdk_window_set_cursor(gdk_window(), cursor.gdk_cursor());
}

void
Gdk::Window::set_events(EventMaskField event_mask)
{
	gdk_window_set_events(gdk_window(), (GdkEventMask)event_mask);
}

void
Gdk::Window::set_icon_list(const std::vector<Pixbuf*>& pixbufs)
{
	g_return_if_fail(!pixbufs.empty());
	GList *tmp_pixbufs = 0;
	int count = pixbufs.size();

	int i = 0;
	while (i < count)
	{
		tmp_pixbufs = g_list_append(tmp_pixbufs, pixbufs[i]->gdk_pixbuf());
		++i;
	}

	gdk_window_set_icon_list(gdk_window(), tmp_pixbufs);
	g_list_free(tmp_pixbufs);
}

void
Gdk::Window::set_icon(Pixmap *pixmap, Bitmap *mask, Window *icon_window)
{
	gdk_window_set_icon(gdk_window(), icon_window ? icon_window->gdk_window() : 0, pixmap->gdk_pixmap(), mask->gdk_bitmap());
}

void
Gdk::Window::set_icon_name(const String& name)
{
	gdk_window_set_icon_name(gdk_window(), name.c_str());
}

void
Gdk::Window::set_group(Window& leader)
{
	gdk_window_set_group(gdk_window(), leader.gdk_window());
}

void
Gdk::Window::set_decorations(WMDecorationField decorations)
{
	gdk_window_set_decorations(gdk_window(), (GdkWMDecoration)decorations);
}

void
Gdk::Window::set_functions(WMFunctionField functions)
{
	gdk_window_set_functions(gdk_window(), (GdkWMFunction)functions);
}

void
Gdk::Window::iconify()
{
	gdk_window_iconify(gdk_window());
}

void
Gdk::Window::deiconify()
{
	gdk_window_deiconify(gdk_window());
}

void
Gdk::Window::stick()
{
	gdk_window_stick(gdk_window());
}

void
Gdk::Window::unstick()
{
	gdk_window_unstick(gdk_window());
}

void
Gdk::Window::maximize()
{
	gdk_window_maximize(gdk_window());
}

void
Gdk::Window::unmaximize()
{
	gdk_window_unmaximize(gdk_window());
}

void 
Gdk::Window::fullscreen()
{
	gdk_window_fullscreen(gdk_window());
}

void 
Gdk::Window::unfullscreen()
{
	gdk_window_unfullscreen(gdk_window());
}

void
Gdk::Window::register_dnd()
{
	gdk_window_register_dnd(gdk_window());
}

void
Gdk::Window::begin_resize_drag(WindowEdge edge, int button, int root_x, int root_y, unsigned int timestamp)
{
	gdk_window_begin_resize_drag(gdk_window(), (GdkWindowEdge)edge, button, root_x, root_y, timestamp);
}

void
Gdk::Window::begin_move_drag(int button, int root_x, int root_y, unsigned int timestamp)
{
	gdk_window_begin_move_drag(gdk_window(), button, root_x, root_y, timestamp);
}

void 
Gdk::Window::invalidate(const Rectangle& rectangle, bool invalidate_children)
{
	gdk_window_invalidate_rect(gdk_window(), rectangle.gdk_rectangle(), invalidate_children);
}

void
Gdk::Window::invalidate(const Region& region, bool invalidate_children)
{
	gdk_window_invalidate_region(gdk_window(), region.gdk_region(), invalidate_children);
}

void
Gdk::Window::freeze_updates()
{
	gdk_window_freeze_updates(gdk_window());
}

void 
Gdk::Window::thaw_updates()
{
	gdk_window_thaw_updates(gdk_window());
}

void 
Gdk::Window::process_all_updates()
{
	gdk_window_process_all_updates();
}

void 
Gdk::Window::process_updates(bool update_children)
{
	gdk_window_process_updates(gdk_window(), update_children);
}

void 
Gdk::Window::set_debug_updates(bool setting)
{
	gdk_window_set_debug_updates(setting);
}

Gdk::GrabStatus
Gdk::Window::keyboard_grab(bool owner_events, unsigned int time)
{
	return (GrabStatus)gdk_keyboard_grab(gdk_window(), owner_events, time);
}

Gdk::GrabStatus
Gdk::Window::pointer_grab(EventMaskField event_mask, Window *confine_to, Cursor *cursor,
                          bool owner_events, unsigned int time)
{
	return (GrabStatus)gdk_pointer_grab(gdk_window(), owner_events, (GdkEventMask)event_mask,
	                                         *confine_to, *cursor, time);
}

void
Gdk::Window::set_extension_events(int mask, ExtensionMode mode)
{
	gdk_input_set_extension_events(gdk_window(), mask, (GdkExtensionMode)mode);
}

/*  Gdk::WindowClass
 */

void
Gdk::WindowClass::init(GdkWindowClass *g_class)
{
	DrawableClass::init((GdkDrawableClass*)g_class);
}

GType
Gdk::WindowClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GDK_TYPE_WINDOW, (GClassInitFunc)init);
	}
	return type;
}

void*
Gdk::WindowClass::create()
{
	return g_object_new(get_type(), 0);
}

