/** Copyright  2003 by Jean-Hugues de Raigniac <jhraigniac@workingfrog.org>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.workingfrog.i18n.swing;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;

import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JRootPane;

import org.workingfrog.i18n.util.LocaleEvent;
import org.workingfrog.i18n.util.LocaleListener;
import org.workingfrog.i18n.util.Translator;

/**
 * i18n version of javax.swing.JRootPane
 *
 * @author Jean-Hugues de Raigniac
 */
public class I18NJRootPane extends JRootPane implements LocaleListener {

    /**
     * Override javax.swing.JRootPane.JRootPane ().
     */
    public I18NJRootPane () {
        super();
        setGlassPane(createGlassPane());
        setLayeredPane(new I18NJLayeredPane());
        setContentPane(createContentPane());
        setLayout(createRootLayout());
        setDoubleBuffered(true);
        updateUI();
    }

    /** Called by the constructor methods to create the default glassPane.
      * By default this method creates a new JComponent with visibility
      * set to false.
      * @return Component
      */
    protected Component createGlassPane () {
        JComponent c = new I18NJPanel(); 
        c.setName(this.getName() + ".glassPane");
        c.setVisible(false);
        ((JPanel) c).setOpaque(false);
        return c;
    }  

    protected Container createContentPane() {
        JComponent c = new I18NJPanel();
        c.setName(this.getName() + ".contentPane");
        c.setLayout(new BorderLayout() {
            /* This BorderLayout subclass maps a null constraint to CENTER.
             * Although the reference BorderLayout also does this, some VMs
             * throw an IllegalArgumentException.
             */
            public void addLayoutComponent(Component comp, Object constraints) {
                if (constraints == null) {
                    constraints = BorderLayout.CENTER;
                }
                super.addLayoutComponent(comp, constraints);
            }
        });
        return c;
    }

    /**
     * Propagate the LocaleEvent to the components of this JRootPane.
     *
     * @param event contains the new Locale
     */
    public void localeChanged (LocaleEvent event) {

        Component[] components = getComponents();

        for (int i = 0; i < components.length; i++) {

            if (Translator.checkObject(components[i], this)) {
                ((LocaleListener) components[i]).localeChanged(event);
            }
        }
    }
}
