/* libgtrans_mysql_3_23
 * Copyright (C) 1999-2000 the Free Software Foundation
 * Authors: Jos Miguel Ronquillo
 *          Matias Mutchinick
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */



/**
 * This is a plugin for GNOME Transcript.
 * GNOME Transcript is a SQL Database Client that uses a plugins system 
 * to access diferent database servers.
 * The plugins are handled by libgtrans_ifase a library to designed to 
 * homogeneize access to different sql database servers.
 * This plugins defines the basic functions needed by libgtrans_ifase
 * and by GNOME Transcript to work with MYSQL servers, this plugin 
 * was written for and tested only in MYSQL-3.23.6-alpha-log and might
 * not work in any other version of MYSQL, how ever it would take little
 * time and effort to port this to other version of MYSQL.
 * In order for the plugin to work you will need the shared versions of
 * MYSQL libraries, you could find them if available at http://www.mysql.com
 *
 * This plugin uses the data types and definitions from the libgtrans_ifase.
 * These are...
 * GTransIFaseConn   : Homogenized connection to database servers.
 * GTransIFaseResult : Homogenized query result.
 *
 * As this library, will be loaded as a plugin, there are two functions (symbols)
 * that are mandatory:
 * Name    : Returns the name of the plugin.
 * Connect : Establish a GTransIFaseConn to the database server.
 *
 * All the other functions needed will be passed pointed at by members
 * in the GTransIFaseConn. These are those functions.
 *  
 * * The function that executes queries, must return a valid
 *   GTransIFaseResult. 
 * conn->Exec = Exec;
 * 
 * * The funcion tha finishes the connection.
 * conn->Disconnect = Disconnect;
 *
 * * The function that checks the status of the connection.
 * conn->Status = Status;
 *
 * * The funcion that returns the connections las error message.
 * conn->ErrorMsg = ErrorMsg;
 *
 * * The funcion that returns a list of the tables. 
 *   (The result mus be a GTransIFaseResult)
 * conn->ListTables = ListTables;
 *
 * * The funcion that returns a list of databases in the same server.
 *   (The result mus be a GTransIFaseResult)
 * conn->ListDb = ListDb;
 *
 * * The function that returns the names of the field type attributes.
 * conn->TableDesignerAttrNames   = TableDesignerAttrNames;
 *
 * * The function that returns the values of the field type attributes.
 * conn->TableDesignerAttrVals    = TableDesignerAttrVals;
 *
 * * The function that returns the names of the field types.
 * conn->TableDesignerFieldTypes  = TableDesignerFieldTypes;
 *
 * * The function that returns the names of the field type masks.
 * conn->TableDesignerTypeMasks   = TableDesignerTypeMasks;
 *
 * * The function that creates the table in the database.
 * conn->TableDesignerCreateTable = TableDesignerCreateTable;
 */



#include <mysql/mysql.h>
#include <glib.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <gtrans_ifase/gtrans_ifase.h>
#include "reserved.h"


#define  _(_str_)  _str_ 
#define N_(_str_)  _str_



/*
 * Name
 *
 * This function returns the name of the plugin.
 */
gchar *Name (void) {
	return "MySQL_3_23";
}



/**
 * The purpose of GNOME Transcript if to provide a maximum of
 * interoperability beetwen the user and the database server,
 * for this reason instead of writting a custom table designer
 * that will work in ok most database servers but leave specific
 * server capabilities, we have decided to include functions in 
 * the plugins to define it and provide all the capabilities that 
 * the actual server provides.
 *
 * This is the way that the plugin functions for the table designer 
 * work:
 *
 * 1.- You must provide functions for defining the table designer.
 * 
 *     1.1  A function that returns a glib doubly linked list (GList *)
 *          with the names of the available field types.
 *          The first item in the list must be an empty string.
 *
 *     1.2  A function that returns a glib doubly linked list (GList *)
 *          with the names of all the available field attributes 
 *          for example (Length, Indices, Precision).
 *
 *     1.3  A function that returns a glib doubly linked list (GList *)
 *          with integer numbers that will work as bit masks used
 *          by the table designer to decide which field type is in title
 *          to which field attributes. This list should have the same
 *          number of elements that the field type list and must start
 *          with a 0 (zero) that corresponds to the empty string in the
 *          type list.
 *
 *     1.4  A function that returns a glib doubly linked list (GList *)
 *          of glib doubly linked lists with the values available to
 *          each field attribute. This is, first you create separate
 *          GList's one for each field attribute, and each list must
 *          be a list of gchar's of the valid attribute values, this
 *          way the table designer will display a combo with te valid 
 *          options, a link with NULL data, means that this attribute 
 *          is a type-in and anything can be typed in.
 *          Then all this linked lists will be linked in another list.
 *          (You must provide a function this last list).
 *
 * 
 * 2.- Then you must provide function for creating and executing the query
 *     that will create the table defined in the table designer.
 *     The table designer takes the data you entered an creates a GList of
 *     arrays of character strings
 *     Each link of the list reperesents the definition of one field.
 *     And each string of the array, the value entered for a particular 
 *     attribute.
 */




/**
 * TableDesignerFieldTypes
 *
 * Return a (GList *) of strings of the valid field types.    
 * This is one of the functions pointed at by a member of the conn.
 */
static GList *
TableDesignerFieldTypes()
{
	GList   *list = NULL;
	gint     i;
	gchar   *types[] = { ""           , 
			     "TINYINT"    ,
			     "SMALLINT"   ,
			     "MEDIUMINT"  ,
			     "INT"        ,
			     "BIGINT"     ,
			     "FLOAT"      ,
			     "DOUBLE"     ,
			     "DECIMAL"    ,
			     "TIMESTAMP"  ,
			     "DATE"       ,
			     "TIME"       ,
			     "DATETIME"   ,
			     "YEAR"       , 
			     "CHAR"       , 
			     "BINARY"     ,
			     "VARCHAR"    ,
			     "VARBINARY"  ,
			     "TINYTEXT"   ,
			     "TINIBLOB"   ,
			     "TEXT"       ,
			     "BLOB"       ,
			     "MEDIUMTEXT" ,
			     "MEDIUMBLOB" , 
			     "LONGTEXT"   ,
			     "LONGBLOB"   ,
//			     "ENUM"       ,
//			     "SET"        ,
			     NULL        };
	gchar *tmp;
	
	i=0;
	while ( (tmp = types[i++]) != NULL )
		list = g_list_append (list, g_strdup (tmp));
	
	return list;
}




/**
 * TableDesignerTypeMasks
 *
 * Return a (GList *) of integers that are the bit masks
 * for the field arttributes.
 * e.g. 
 * The binary representation of 125 is 1111101, you will see
 * later that this plugin defines 7 attributes for the table designer,
 * a 1 means that a field type is in title to certain attribute.
 * 125 (1111101) as a bit mask, means that this particular field type,
 * the one that this mask belongs to is in title to the 
 * 1,3,4,5,6 and 7 attributes and the 2 attribute is forbidden.
 * This is one of the functions pointed at by a member of the conn.
 */
static GList *
TableDesignerTypeMasks()
{
	GList   *list = NULL;
	gint     i;
	gint     mask[] = { 0, 
			    125,
			    125,
			    125,
			    125,
			    125,
			    115,
			    115,
			    115,
			    113,
			    112,
			    112,
			    112,
			    112,
			    113,
			    113,
			    113,
			    113,
			    80,
			    80,
			    80,
			    80,
			    80,
			    80,
			    80,
			    80,
//			    12,
//			    12,
			    -1 };
	gint tmp;
	
	i=0;
	while ( (tmp = mask[i++]) != -1 )
		list = g_list_append (list,(gpointer) tmp);
	
	return list;
}



/**
 * TableDesignerNullity
 *
 * Return a (GList *) of strings that are the options for the
 * field's NULL attribute.
 */
static GList *
TableDesignerNullity()
{
	GList   *list;
	
	list = g_list_append (NULL, g_strdup(""));
	list = g_list_append (list, g_strdup("NOT NULL"));
	
	return list;
}




/**
 * TableDesignerIndexes
 *
 * Return a (GList *) of strings that are the options for the
 * field's KEY (INDEX) attribute.
 */
static GList *
TableDesignerIndexes()
{
	GList   *list;
	
	list = g_list_append (NULL, g_strdup(""));
	list = g_list_append (list, g_strdup("INDEX"));
	list = g_list_append (list, g_strdup("UNIQUE KEY"));
	list = g_list_append (list, g_strdup("PRIMARY KEY"));
	
	return list;
}



/**
 * TableDesignerSign
 *
 * Return a (GList *) of strings that are the options for the
 * field's UNSIGNED attribute.
 */
static GList *
TableDesignerSign()
{
	GList   *list;
	
	list = g_list_append (NULL, g_strdup(""));
	list = g_list_append (list, g_strdup("UNSIGNED"));
	
	return list;
}



/**
 * TableDesignerAutoIncrement
 *
 * Return a (GList *) of strings that are the options for the
 * field's Auto-Increment attribute.
 */
static GList *
TableDesignerAutoIncrement()
{
	GList   *list;
	
	list = g_list_append (NULL, g_strdup(""));
	list = g_list_append (list, g_strdup("AUTO_INCREMENT"));
	
	return list;
}




/**
 * TableDesignerAttrVals
 *
 * Return a (GList *) of (GList *) that are the options for the
 * field's Attributes.
 * This is one of the functions pointed at by a member of the conn.
 */
static GList *
TableDesignerAttrVals()
{
	GList  *list;
	
	list = g_list_append(NULL,NULL);
	list = g_list_append(list,NULL);
	list = g_list_append(list,TableDesignerSign());
	list = g_list_append(list,TableDesignerAutoIncrement());
	list = g_list_append(list,TableDesignerNullity());
	list = g_list_append(list,NULL);
	list = g_list_append(list,TableDesignerIndexes());
	
	return list;
}




/**
 * TableDesignerAttrNames
 *
 * Return a (GList *) of strings that are the names for the
 * field's attributes.
 * This is one of the functions pointed at by a member of the conn.
 */
static GList *
TableDesignerAttrNames()
{
	GList  *list;
	
	list = g_list_append(NULL,g_strdup("Length"));
	list = g_list_append(list,g_strdup("Precision"));
	list = g_list_append(list,g_strdup("Unsigned"));
	list = g_list_append(list,g_strdup("Auto-Increment"));
	list = g_list_append(list,g_strdup("Null"));
	list = g_list_append(list,g_strdup("Default"));
	list = g_list_append(list,g_strdup("Key"));
	
	return list;
}





/**
 * Here start the functions that will take the table designer's data
 * turn it into a query and then execute it.
 * First we will do some (or a lott) of error checking.
 *
 * Here I parse a field definition at a time, and return an error
 * message if an invalid vlue has been entered in the table designer,
 * Then I can check more global things, like indexes. 
 */ 

/**
 * These are error types.
 */
enum {
	EVERYT_OK            = 0,
	
	/* Field definition Errors */
	MISSING_FIELD_NAME      = 1,
	INVALID_FIELD_RESERVED  = 2,
	INVALID_FIELD_SYNTAX    = 3,
	MISSING_TYPE            = 4,
	MISSING_LEN             = 5,
	INVALID_LEN_RANGE       = 6,
	INVALID_LEN_SYNTAX      = 7,
	MISSING_PREC            = 8,
	INVALID_PREC_RANGE      = 9,
	INVALID_PREC_SYNTAX     = 10,
	INVALID_DEFA_SYNTAX     = 11,
	MISSING_NOT_NULL        = 12,
	MISSING_PRIMARY_KEY     = 13,
	
	/* Table definition errors */
	NO_FIELDS_DEFINED       = 14,
	MISSING_TABLE_NAME      = 15,
	INVALID_TABLE_RESERVED  = 16,
	INVALID_TABLE_SYNTAX    = 17,
	MULTIPLE_PRIMARY_KEY    = 18,
	MULTIPLE_AUTO_INCREMENT = 19
};




/*
 * TableDesignerErrorMsg
 * Here we keep the error messages that must be
 * syncronized with the error types enum.
 */
static gchar *
TableDesignerErrorMsg[] = {
	"",
	
	/* Field definition error messages */
	N_("Missing field name"),
	N_("Field name is a reserved word"),
	N_("Field name invalid syntax"),
	N_("Missing field type"),
	N_("Field length needed"),
	N_("Invalid field length\nValid range 1 - 255"),
	N_("Invalid field length\nMust be an integer number"),
	N_("Field precision needed"),
	N_("Invalid field precision\nValid range 0 - 30"),
	N_("Invalid field precision\nMust be an integer number"),
	N_("Invalid default value\nSyntax error"),
	N_("A PRIMARY KEY must be declared NUT NULL"),
	N_("An AUTO_INCREMENT field must\nbe declared as a PRIMARY KEY"),
	
	/* Table definition error messages */
	N_("Error : No fields defined"),
	N_("Error : Missing table name"),
	N_("Error : Table name is a reserved word"),
	N_("Error : Table name invalid syntax"),
	N_("Error : There can be only one PRIMARY KEY"),
	N_("Error : There can be only one AUTO_INCREMENT")
};




/**
 * TableDesignerParseInt
 * @str : String representation of an integer number
 *
 * Returns 0 if the string passes the syntax test or 1
 * if it does not.
 * This again is not a function you must provide.
 */
static gint
TableDesignerParseInt(gchar *str)
{
	gint i = 0;
	
	while (isspace(str[i]))	
		i++;
	if (str[i] == '-' || str[i] == '+' )
		i++;
	while (isdigit(str[i]))
		i++;
	
	if (str[i] != 0)
		return 1;
	
	return 0;
}





/**
 * TableDesignerValidateLength
 * @str : A string reperesenting the fields length.
 *
 * If the string does not reperesents an integer,
 * or the integer is out of range, this returns a
 * error code, otherwise returns 0.
 */
static gint
TableDesignerValidateLength(gchar *str)
{
	gint val;
	
	if (TableDesignerParseInt(str))
		return INVALID_LEN_SYNTAX;
	
	val = atoi(str);
	if ( val > 255 || val < 1  )
		return INVALID_LEN_RANGE;
	
	return 0;
}



/**
 * TableDesignerValidatePrecision
 * @str : A string
 * 
 * Pretty much as the above, but the value range is different.
 */
static gint
TableDesignerValidatePrecision(gchar *str)
{
	gint val;
	
	if (TableDesignerParseInt(str))
		return INVALID_PREC_SYNTAX;
	
	val = atoi(str);
	if (val > 30 || val < 0)
		return INVALID_PREC_RANGE;
	
	return 0;
}



/**
 * TableDesignerValidateString
 * @str  : A character string.
 *
 * This function checks that the string wont cause any problems
 * inside the query. 
 * Return 0 if the string is ok and 1 if there is an
 * error.
 */
static gint
TableDesignerValidateString(gchar *str)
{
	gint  i = 0;
	
	while (str[i] != 0){
		
		if (str[i] == '\"')
			return 1;
		
		if (str[i] == '\\'){
			i++;
			if (str[i] == 0)
				return 1;
		}
		i++;
	}
	return 0;
}



/**
 * TableDesignerValidateWord
 * @word : A string
 *
 * Check if the string is a reserved mysql word, if it is return 1,
 * otherwise return 0.
 */
gint static
TableDesignerValidateWord(gchar *word)
{
	gchar *tmp;
	gint   i = 0;
	
	while ((tmp = reserved[i++]))
		if (!g_strcasecmp(tmp,word))
			return 1;
	
	return 0;
}



/**
 * TableDesignerValidateWSyntax
 * @word : A string
 *
 * Check if the string is a valid table-column name, 
 * if it is return 0 otherwise return 1.
 */
gint static
TableDesignerValidateWSyntax(gchar *word)
{
	gchar  c;
	gint   i = 1;
	
	if (!isalpha(word[0]) && word[0] != '_')
		return 1;
	
	while ((c = word[i++]))		
		if (!isalnum(c) && c != '_')
			return 1;
	
	return 0;
}





/**
 * TableDesignerValidateField
 * @row  : A gchar pointer to pointers that define a field 
 *         in the table designer
 *
 * Returns an error code if the field definition is wrong
 * and 0 if it is ok.
 * This function checks a lott of stuff, that the required arguments
 * are there, that the args are OK, etc...
 */
static gint
TableDesignerValidateField(gchar **field)
{
	gchar *type, *len, *prec, *name, *null, *indx, *sign, *incr, *defa;
	gint   error;
	
	name = field[0];
	type = field[1];
	len  = field[2];
	prec = field[3];
	sign = field[4];
	incr = field[5];
	null = field[6];
	defa = field[7];
	indx = field[8]; 
	

	/* Name is mandatory */
	if(!name)
		return MISSING_FIELD_NAME;
	
	/* Check for a valid field name */
	if(TableDesignerValidateWSyntax(name))
		return INVALID_FIELD_SYNTAX;
	
	/* Check that field name is not a reserved word */
	if(TableDesignerValidateWord(name))
		return INVALID_FIELD_RESERVED;
	
	
	/* If type is missing we have an error */
	if (!type)
		return MISSING_TYPE;
	
	/* BINARY, VARCHAR and VARBINARY demand field length */
	if ((!strcmp(type,"VARCHAR") || !strcmp(type,"VARBINARY") ||
	     !strcmp(type,"BINARY")) && !len)
		return MISSING_LEN;
	
	/* Length attribute must be within range */
	if (len)
		if ((error = TableDesignerValidateLength(len)))
			return error;
	
	/* If DOUBLE length is defined, precision is mandatory */
	if ( !strcmp(type,"DOUBLE") && len && !prec)
		return MISSING_PREC;
	
	/* If precision is defined, length is mandatory */
	if (prec && !len)
		return MISSING_LEN;
	
	/* Precision attribute must be within range */
	if (prec)
		if ((error = TableDesignerValidatePrecision(prec)))
			return error;
	
	/* Default value syntax check */
	if (defa)
		if ((error = TableDesignerValidateString(defa)))
			return INVALID_DEFA_SYNTAX;
	
	/* All PRIMARY KEY's must be NOT NULL */
	if (indx)
		if ( !strcmp(indx,"PRIMARY KEY") && !null )
			return MISSING_NOT_NULL;

	/* All AUTO_INCREMENTS must be PRIMARY KEYS */
	if (incr){
		if (indx){
			if (strcmp(indx,"PRIMARY KEY"))
				return MISSING_PRIMARY_KEY;
		} else
			return MISSING_PRIMARY_KEY;
	}
	
	/* Everithing OK */
	return EVERYT_OK;
}




/**
 * TableDesignerValidateFields
 * @tdef  : A GList of field definitions.
 *
 * Checks one by one the field definitions and return NULL
 * if everiithing is of, returns an error message other wise.
 * This is not a required function if you settle for the server's
 * error message, the error message is mandatory.
 */
static gchar *
TableDesignerValidateFields(GList *tdef)
{
	GList  *tmp;
	gchar **field;
	gint    error = 0 , i = 0;
		
	
	for (tmp = tdef ; tmp != NULL ; tmp = tmp->next , i++){
		
		if ( (field = (gchar **)tmp->data) == NULL)
			continue;
			
		error = TableDesignerValidateField(field);
		
		if(error){
			gchar *msg;
			
			msg = g_strdup_printf(_("Error in field definition %i\n%s"),
					      i,TableDesignerErrorMsg[error]);
			
			return msg;
		}
	}
	return NULL;
}




/**
 * TableDesignerValidateFieldNumber
 * @tdef : The GList of field definitions
 *
 * Check tha there is only one field defined as primary key.
 */
static gint
TableDesignerValidateFieldNumber(GList *tdef)
{
	GList  *tmp;
	gint    num = 0;
	
	for (tmp = tdef ; tmp != NULL ; tmp = tmp->next)
		if (tmp->data)
			num++;
	
	if (num == 0)
		return 1;
	
	return 0;
}




/**
 * TableDesignerValidatePrimaryKey
 * @tdef : The GList of field definitions
 *
 * Check tha there is only one field defined as primary key.
 */
static gint
TableDesignerValidatePrimaryKey(GList *tdef)
{
	GList  *tmp;
	gchar **field, *indx;
	gint    keys = 0;
	
	for (tmp = tdef ; tmp != NULL ; tmp = tmp->next){
		
		if ( (field = (gchar **)tmp->data) ){
			
			indx = field[8];
			if (indx)
				if (!strcmp(indx,"PRIMARY KEY")){
					if (keys == 1)
						return 1;
					keys++;
				}
		}
	}
	return 0;
}



/**
 * TableDesignerValidateAutoIncrement
 * @tdef : The GList of field definitions
 *
 * Check tha there is only one field defined as AUTO_INCREMENT.
 */
static gint
TableDesignerValidateAutoIncrement(GList *tdef)
{
	GList  *tmp;
	gchar **field;
	gint    incrs = 0;
	
	for (tmp = tdef ; tmp != NULL ; tmp = tmp->next){
		
		if ( (field = (gchar **)tmp->data) )
			if (field[5]){
				if (incrs == 1)
					return 1;
				incrs++;
			}
	}
	return 0;
}




/**
 * TableDesignerValidateTable
 * @name   : The name for the table
 * @tdef : The GList of field definitions
 *
 * Validate the table definition, syntax, indexes, values, etc...
 */
static gchar *
TableDesignerValidateTable(gchar *tname,
			   GList *tdef)
{
       	/* Table Name */
	if(strlen(tname) == 0)
		return g_strdup(TableDesignerErrorMsg[MISSING_TABLE_NAME]);
	
	/* Must be at least one field defined */
	if(TableDesignerValidateFieldNumber(tdef))
		return g_strdup(TableDesignerErrorMsg[NO_FIELDS_DEFINED]);
	
	/* No more than one primary key */
	if(TableDesignerValidatePrimaryKey(tdef))
		return g_strdup(TableDesignerErrorMsg[MULTIPLE_PRIMARY_KEY]);
	
	/* No more than one AUTO_INCREMENT */
	if(TableDesignerValidateAutoIncrement(tdef))
		return g_strdup(TableDesignerErrorMsg[MULTIPLE_AUTO_INCREMENT]);
	
	/* Is table_name valid for a table name? */
	if(TableDesignerValidateWSyntax(tname))
		return g_strdup(TableDesignerErrorMsg[INVALID_TABLE_SYNTAX]);
	
	/* Is table name a reserved word? */
	if(TableDesignerValidateWord(tname))
		return g_strdup(TableDesignerErrorMsg[INVALID_TABLE_RESERVED]);
	
	return TableDesignerValidateFields(tdef);
}





/**
 * Stuff for the table designer query generation and execution
 */

/**
 * TableDesignerCalcFieldSize
 * @field : An arry of strings (field definition).
 *
 * Return the number of characters that this field definition
 * would take inside a query string.
 */
static gint
TableDesignerCalcFieldSize(gchar **field)
{
	gchar *name, *type, *len, *prec, *sign, *incr, *null, *defa, *indx;
	gint   size = 4;
	
	if(!field)
		return 0;
	
	name = field[0];
	type = field[1];
	len  = field[2];
	prec = field[3];
	sign = field[4];
	incr = field[5];
	null = field[6];
	defa = field[7];
	indx = field[8];
	
	size += strlen(name) + strlen(type);
	
	if(len){
		size += strlen(len) + 2;
		
		if(prec)
			size += strlen(prec) + 1;
	}
	
	if(sign)
		size += strlen(sign) + 1;
	
	if(incr)
		size += strlen(incr) + 1;
	
	if(null)
		size += strlen(null) + 1;
	
	if(defa)
		size += strlen(defa) + 11;
	
	if(indx)
		size += strlen(indx) + 6 + strlen(name);
	
	return size;
}



/**
 * TableDesignerCalcQuerySize
 * @name  : Tha table name
 * @rows  : A GList of arry of strings (field definitions).
 *
 * Return the number of characters needed to create
 * a query string for the table defined in the table designer.
 */
static gint
TableDesignerCalcQuerySize(gchar *name,
			   GList *tdef)
{
	GList *tmp;
	gint   size = 20;
	
	size += strlen(name);
	
	for (tmp = tdef ; tmp != NULL ; tmp = tmp->next)
		size += TableDesignerCalcFieldSize((gchar **)tmp->data);
	
	return size;
}




/**
 * TableDesignerCreateQuery
 * @tname  : Table name
 * @tdef : A GList of arry of strings (field definitions).
 *
 * Create the query needed to create the table designed in @gtd.
 */
static gchar *
TableDesignerCreateQuery(gchar    *tname,
			 GList    *tdef)
{
	GList   *tmp;
	gchar  **field;
	gchar   *qstr, *fname, *ftype, *flen, *fprec, 
		*fsign, *fincr, *fnull, *fdefa, *findx;
	gint     size, i = 0;
	
	
	size = TableDesignerCalcQuerySize(tname,tdef);
	qstr = g_new0(gchar,size);
	
	strcpy(qstr,"CREATE TABLE ");
	strcat(qstr,tname);
	strcat(qstr," ( ");

	for ( tmp = tdef ; tmp != NULL ; tmp = tmp->next){
		
		/* NULL field, no data, get outta here! */
		if ((field = (gchar **)tmp->data) == NULL)
			continue;
		
		fname = field[0];
		ftype = field[1];
		flen  = field[2];
		fprec = field[3];
		fsign = field[4];
		fincr = field[5];
		fnull = field[6];
		fdefa = field[7];
		findx = field[8];
		
		/* Only add a coma if a field has already been 
		 * parsed into the qstr */
		if (i != 0)
			strcat(qstr,", ");
		
		strcat(qstr,fname);
		strcat(qstr," ");
		strcat(qstr,ftype);
		
		if (flen){
			strcat(qstr,"(");
			strcat(qstr,flen);
			
			if (fprec){
				strcat(qstr,",");
				strcat(qstr,fprec);
			}
			strcat(qstr,")");
		}
		
		strcat(qstr," ");
		
		if (fsign){
			strcat(qstr,fsign);
			strcat(qstr," ");
		}
		
		if (fincr){
			strcat(qstr,fincr);
			strcat(qstr," ");
		}
		
		if (fnull){
			strcat(qstr,fnull);
			strcat(qstr," ");
		}
		
		if (fdefa){
			strcat(qstr,"DEFAULT \"");
			strcat(qstr,fdefa);
			strcat(qstr,"\" ");
		}
		
		if (findx){
			strcat(qstr,", ");
			strcat(qstr,findx);
			strcat(qstr,"(");
			strcat(qstr,fname);
			strcat(qstr,") ");
		}
		
		i++; /* Field Count */
	}
	
	strcat(qstr,")");
	
	return qstr;
}





/**
 * TableDesignerFreeTableDefs
 * @tdef   : The definition of the table as giver by the 
 *           GTransTableDesigner
 *
 * Free all the memory allocated by the table definition.
 */
static void
TableDesignerFreeTableDefs(GList *tdef)
{
	GList  *tmp;
	gchar **field;
	gint    i;

	for (tmp = tdef ; tmp != NULL ; tmp = tmp->next){

		if ((field = tmp->data) == NULL)
			continue;
		
		/* For all the arrtibutes (5) + field name + field size = 7*/
		for ( i = 0 ; i < 7 ; i++)
		       	if (field[i])
				g_free(field[i]);
		g_free(field);
	}
	g_list_free(tdef);
}




/**
 * TableDesignerCreateTable
 * @conn    : The GTransIFaseConn connection to a database backend
 * @tname   : The name for the table
 * @tdef    : The linked list string arrays (field definitions).
 *
 * Check for errors in the field definition, if everything
 * it ok the table is created and NULL is returned, 
 * otherwise the error message is returned.
 * This funcion is mandatory for the plugin definition.
 * This is one of the functions pointed at by a member of the conn.
 */
static gchar *
TableDesignerCreateTable(GTransIFaseConn  *conn,
			 gchar            *tname,
			 GList            *tdef)
{
	gchar     *errmsg;
	gint       status;
	gchar     *qstr;
	
        errmsg = TableDesignerValidateTable(tname,tdef);
	if (errmsg){
		return errmsg;
	}
	
	qstr = TableDesignerCreateQuery(tname,tdef);

	status = mysql_real_query ((MYSQL *)conn->conn,qstr,
				   (strlen (qstr)+1)*sizeof (gchar));
        
	g_free (qstr);
	TableDesignerFreeTableDefs(tdef);
	
	if (status)
                return mysql_error((MYSQL*)conn->conn);
	
	
	return NULL;
}





/************************************************************************
 * Here is the start of the low level functions, these funcions take    *
 * care of the interaction with the server, this funcions use the       *
 * homogenized types from libgtrans_ifase                               *
 ************************************************************************/

/************************************************************************
 * Functions for creating GTransIFaseResult's                           *
 ************************************************************************/


/**
 * Exec:
 * @conn  : GTransIFaseConn to a MySQL server
 * @qstr  : SQL query string.
 *
 * Execute a SQL query in the DB server.
 * Return : A GTransIFaseResult.
 */
static GTransIFaseResult *Exec (GTransIFaseConn *conn,
				gchar           *qstr)
{
	GTransIFaseResult  *gt_res;
	MYSQL_RES          *my_res; 
	gchar             **tmp;
        gint                j, err;
	
	
	/* Allocate memory for the result */
	gt_res = gtrans_ifase_result_alloc();
	
       	err = mysql_real_query ((MYSQL *)conn->conn,qstr,
				(strlen (qstr)+1)*sizeof (gchar));
        
	/* If errored */
	if (err){
		gt_res->status = GTRANS_IFASE_RESULT_ERROR;
		gt_res->errormsg = g_strdup(mysql_error((MYSQL *)conn->conn));
		return gt_res;
        }
	
	my_res = mysql_store_result ((MYSQL *)conn->conn);
	
	/* If command returned no tupples */
	if (!my_res)
                return gt_res;
		
	
	/* If returned tupples */
	gt_res->type = GTRANS_IFASE_RESULT_TUPPLES;
        gt_res->n_rows = mysql_num_rows (my_res);
        gt_res->n_cols = mysql_num_fields (my_res);
        
	/* Copy the field names */
        gt_res->fields = g_new0(gchar *,gt_res->n_cols + 1);
        for (j = 0 ; j < gt_res->n_cols ; j++)
                gt_res->fields[j] = g_strdup(mysql_fetch_field(my_res)->name);
        gt_res->fields[j] = NULL;
	
        
	/* Copy the result tupples */
        while ((tmp = mysql_fetch_row (my_res)) != NULL){
                
                gchar **row; 
                
                row = g_new0(gchar *,gt_res->n_cols);
                
                for (j = 0 ; j < gt_res->n_cols ; j++)
                        row[j] = g_strdup (tmp[j]);
                
                gt_res->rows = g_list_append(gt_res->rows,row);
        }
	gt_res->rows = g_list_append(gt_res->rows,NULL);      
	
	/* Free the MYSQL result */
	mysql_free_result (my_res);
	
        return gt_res;
}



/**
 * ListTables:
 * @conn  : GTransIFaseConn to a MySQL server
 *
 * Return : A GTransIFaseResult with the available tables
 * in the database.
 */
static GTransIFaseResult *ListTables (GTransIFaseConn *conn)
{
	g_return_val_if_fail(conn != NULL,NULL);
	
	return Exec (conn, "SHOW tables");
}



/**
 * ListDb:
 * @conn  : GTransIFaseConn to a MySQL server
 *
 * Return : A GTransIFaseResult with the availble databases
 * in the server.
 */
static GTransIFaseResult *ListDb (GTransIFaseConn *conn)
{
	g_return_val_if_fail(conn != NULL,NULL);
	
	return Exec (conn, "SHOW databases");
}





/*******************************************************************
 * Connection Stuff                                                *
 *******************************************************************/

/* Disconnection function */

/**
 * Disconnect:
 * @conn  : GTransIFaseConn to a MySQL server
 *
 * Finich the connection to the server.
 * Also free the GTransIFaseConn.
 */
static void Disconnect (GTransIFaseConn *conn)
{
	g_return_if_fail (conn != NULL);

	mysql_close ((MYSQL *)conn->conn);

	if (conn->host)
		g_free (conn->host);
	if (conn->db)
		g_free (conn->db);
	if (conn->user)
		g_free (conn->user);
	if (conn->pwd)
		g_free (conn->pwd);
	if (conn->port)
		g_free (conn->port);
	
	g_free(conn);
}



/**
 * Status:
 * @conn  : GTransIFaseConn to a MySQL server
 *
 * Return 1 if there has been an error with the connection.
 * 0 otherwise.
 */
static gint Status (GTransIFaseConn *conn)
{
	g_return_val_if_fail (conn != NULL, 1);
	
	if ( conn->conn != NULL )
		if( !mysql_ping ((MYSQL *)conn->conn) )
			return 0;
        
        return 1;
}




/**
 * ErrorMsg:
 * @conn : GTransIFaseConn to a MySQL server
 *
 * Return the error string of the last error that 
 * happened with the connection.
 */
static gchar *ErrorMsg (GTransIFaseConn *conn)
{
	if (conn == NULL)
		return g_strdup ("Fatal Error: NULL Connection");
	
	if (conn->conn == NULL)
		return g_strdup ("Fatal Error: Connection Impossible");
	
	return mysql_error ((MYSQL *)conn->conn);
}




/**
 * Connect :
 * @host : server where the database backend is running
 * @port : port where the database backend is running
 * @db   : name of the database
 * @user : user name to establish a connection
 * @pwd  : password for the user
 *
 * Establish a connection to a MySQL database server.
 * Returns : A GTransIFaseResult.
 */
GTransIFaseConn *Connect (gchar   *host,
			  gchar   *port,
			  gchar   *db,
			  gchar   *user,
			  gchar   *pwd)
{
	GTransIFaseConn *conn;
	
	conn = gtrans_ifase_conn_alloc();

	if (host)
		conn->host = g_strdup (host);				
	if (port)
		conn->port = g_strdup (port);
	if (db)
		conn->db = g_strdup (db);
	if (user)
		conn->user = g_strdup (user);
	if (pwd)
		conn->pwd = g_strdup (pwd); 

	(MYSQL *)conn->conn = mysql_init (NULL);
	(MYSQL *)conn->conn = mysql_real_connect ((MYSQL *)conn->conn, 
						  host, 
						  user, 
						  pwd,
						  db, 
						  atoi (port), 
						  NULL, 
						  0);
	
	conn->Connect = NULL;
	conn->Exec = Exec;
	conn->Disconnect = Disconnect;
	conn->Status = Status;
	conn->ErrorMsg = ErrorMsg;
	conn->ListTables = ListTables;
	conn->ListDb = ListDb;
	
	/* Table Designer */
	conn->TableDesignerAttrNames   = TableDesignerAttrNames;
	conn->TableDesignerAttrVals    = TableDesignerAttrVals;
	conn->TableDesignerFieldTypes  = TableDesignerFieldTypes;
	conn->TableDesignerTypeMasks   = TableDesignerTypeMasks;
	conn->TableDesignerCreateTable = TableDesignerCreateTable;
	
	return conn;
}
