/*******************************************************************************************************************************************
 cmenu.h
 
 CClass						CClass
 |--CSerialized					|--CSerialized
    |--CMetaModule				   |--CMetaModule
       |--CObject				      |--CObjectListener
          |--CComponent			      	      	 |
	     |--CControl			      	 |
	        |--CWidget			      	 |--CWidgetListener
		   |--CContainer			    |
		      |--CMenu			 	    |
		         |--CMenuBar		 	    |--CMenuBarListener
		         |--CMenuItem		 	    |--CMenuItemListener
		         |  |--CMenuItemSeparator	    |
		         |  |--CMenuItemCheck	  	    |
		         |  |  |--CMenuItemRadio	    |
			 |  |--CMenuItemImage		    |
		         |--CMenuPopup		 	    |--CMenuPopupListener

 CMenu is an abstract base class that introduces the menu classes family. 
 CMenuBar handles GtkMenuBar and its purpose is to contain some instances of CMenuItem. 
 CMenuItem handles GtkMenuItem. It is a menu item itself and a container for sub instances of CMenuItem types.
 CMenuPopup encapsulates a contextual menu behaviour. All of the sub gtkol widgets of a specified widget that has a CMenuPopup child
 component may be a popup menu sender. 
 For CMenuItemRadio groups definition, it is the CMenuItemSeparator that acts as a logical separator for radio groups.
 To add an image to a CMenuItemImage menu item, append a CImage instance child or use menu item the stock id constructor.
*******************************************************************************************************************************************/

#ifndef __CMENU_H__
#define __CMENU_H__

// for accels access...
#include "cform.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// what's defined in this gtkol section
//-----------------------------------------------------------------------------------------------------------------------------------------

class CMenu;
class CMenuBar;
class CMenuItem;
class CMenuItemSeparator;
class CMenuItemCheck;
class CMenuItemRadio;
class CMenuItemImage;
class CMenuPopup;

class CMenuBarListener;
class CMenuItemListener;
class CMenuPopupListener;

/*******************************************************************************************************************************************
 hotkeys definition
*******************************************************************************************************************************************/
struct THotKey
{
	SInt16	Key;
	Bool	Control;
	Bool	Shift;
	Bool	Alt;

	THotKey	(const SInt16 inKey=0x0000, const Bool inControl=false, const Bool inShift=false, const Bool inAlt=false);
};

// common hotkeys standards...
const static THotKey _HKNone_	(0x0000, false, false, false);	// nothing...
const static THotKey _HKSave_	(GDK_s,  true,  false, false);	// CTRL + s
const static THotKey _HKOpen_	(GDK_o,  true,  false, false);	// CTRL + o
const static THotKey _HKPrint_	(GDK_p,  true,  false, false);	// CTRL + p
const static THotKey _HKExit_	(GDK_F4, false, false, true );	// ALT  + f4
const static THotKey _HKClose_	(GDK_F4, true,  false, false);	// CTRL + f4
const static THotKey _HKQuit_	(GDK_q,  true,  false, false);	// CTRL + q
const static THotKey _HKCopy_	(GDK_c,  true,  false, false);	// CTRL + c
const static THotKey _HKCut_	(GDK_x,  true,  false, false);	// CTRL + x
const static THotKey _HKPaste_	(GDK_v,  true,  false, false);	// CTRL + v
const static THotKey _HKUndo_	(GDK_z,  true,  false, false);	// CTRL + z
const static THotKey _HKRedo_	(GDK_y,  true,  false, false);	// CTRL + y
const static THotKey _HKFind_	(GDK_f,  true,  false, false);	// CTRL + f
const static THotKey _HKSelect_	(GDK_a,  true,  false, false);	// CTRL + a
const static THotKey _HKNew_	(GDK_n,  true,  false, false);	// CTRL + n

/*******************************************************************************************************************************************
 CMenu abstraction
*******************************************************************************************************************************************/

//-----------------------------------------------------------------------------------------------------------------------------------------
// CMenu class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CMenu : public CContainer
{
	// instanciation section
	public :

		CMenu				(CContainer *inOwner=NULL, const CWidgetListener *inListener=NULL);
		virtual ~CMenu			() =0;

	// generic widget request handling
	protected :

		// handled widget initialization
		virtual void			PerformWidgetInitialize		();

	// CComponent redefinition
	public :

		// expected owner type : __metaclasses(CContainer); expected children type : __metaclasses(CMenuItem)
		virtual CMetaClasses		OwnerMustBe			() const;
		virtual CMetaClasses		ChildMustBe			() const;

	// CContainer redefinition
	public :

		// get the container children number handling left : no limitation on menu i.e. -1
		virtual SInt16			GetGtkChildrenNumberLeft	() const;

		// metaclass association
		SECTION_GENERIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_GENERIC_METACLASS ('menu', CMenu, CContainer);

/*******************************************************************************************************************************************
 CMenuBar
*******************************************************************************************************************************************/

//-----------------------------------------------------------------------------------------------------------------------------------------
// CMenuBarListener class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CMenuBarListener : public CWidgetListener
{
	// instanciation section
	public :

		CMenuBarListener		();
		virtual ~CMenuBarListener	() =0;

	// specific listening
	public :

		// called when the selection menu loop ends
		virtual void			OnSelectionDone			(CObject *inSender)		{ }

		// metaclass association
		SECTION_GENERIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_GENERIC_METACLASS ('_mnb', CMenuBarListener, CWidgetListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// CMenuBar class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CMenuBar : public CMenu
{
	// instanciation section
	public :

		CMenuBar			(CContainer *inOwner=NULL, const CMenuBarListener *inListener=NULL);
		virtual ~CMenuBar		();

	// specific widget requests handling
	protected :

		// widget instanciation and initialization
		virtual GtkWidget *		PerformWidgetInstanciate	();
		virtual void			PerformWidgetInitialize		();

	// specific container requests handling
	protected :

		// child widget addon or removal from the menu bar
		virtual void			PerformContainerAdd		(CWidget *inChild);
		virtual void			PerformContainerRemove		(CWidget *inChild);

	// CObject redefinition
	protected :

		// listener affectation, must be derived from CMenuBarListener
		virtual const CMetaClass *	ListenerMustBe			() const;

	// protected section
	protected :

		// static menu bar listener addon
		static void			OnSelectionDone			(GtkMenuShell *, gpointer);

		// metaclass assocation
		SECTION_DYNAMIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_DYNAMIC_METACLASS ('mnbr', CMenuBar, CMenu);

/*******************************************************************************************************************************************
 CMenuItem
*******************************************************************************************************************************************/

//-----------------------------------------------------------------------------------------------------------------------------------------
// cmenuitem xml serialization constants
//-----------------------------------------------------------------------------------------------------------------------------------------
// <cmenuitem caption="string" [key="int" combinaison="string"]>
//    ...
// </cmenuitem>
//-----------------------------------------------------------------------------------------------------------------------------------------
static CString XML_MENUITEM_ELEMENT			("cmenuitem");
static CString XML_MENUITEM_ATTR_CAPTION		("caption");
static CString XML_MENUITEM_ATTR_HOTKEY_KEY		("key");
static CString XML_MENUITEM_ATTR_HOTKEY_COMBINAISON	("combinaison");

//-----------------------------------------------------------------------------------------------------------------------------------------
// CMenuItemListener class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CMenuItemListener : public CWidgetListener
{
	// instanciation section
	public :

		CMenuItemListener		();
		virtual ~CMenuItemListener	() =0;

	// specific listening functions
	public :

		// called when inSender menu item is selected or unselected
		virtual void			OnSelect			(CObject *inSender, const bool inSelected)		{ }

		// called when inSender menu item is activated / clicked
		virtual void			OnClick				(CObject *inSender)					{ }

		// metaclass association
		SECTION_GENERIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_GENERIC_METACLASS ('_ntm', CMenuItemListener, CWidgetListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// CMenuItem class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CMenuItem : public CMenu
{
	// instanciation section
	public :

		CMenuItem			(CMenu *inOwner=NULL, const CString & =CString(), const THotKey & =_HKNone_, 
						 const CMenuItemListener *inListener=NULL);
		virtual ~CMenuItem		();

	// specific widget requests handling
	protected :

		// widget instanciation and initialization
		virtual GtkWidget *		PerformWidgetInstanciate	();
		virtual void			PerformWidgetInitialize		();

	// specific container requests handling
	protected :

		// child widget addon or removal from the menu bar
		virtual void			PerformContainerAdd		(CWidget *inChild);
		virtual void			PerformContainerRemove		(CWidget *inChild);

	// CObject redefinition
	protected :

		// listener affectation, should be derived from CMenuItemListener
		virtual const CMetaClass *	ListenerMustBe			() const;

	// CComponent redefinition
	public :
	
		// expected owner type __metaclasses(CMenu), children type __metaclasses(CMenuItem) - see CMenu -
		virtual CMetaClasses		OwnerMustBe			() const;

	// CControl/CWidget redefinition
	public :

		// there are no gui bounds handling, let the menu item manage its coordinates and dimensions, just keep a local
		// copy of the actual ones
		virtual void			SetBounds			(const TBounds &inBounds);

		// get the menu item pixbuf representation
		virtual CPixbuf * 		GetControlPixbuf 		() const;

	// specific menu item functions
	public :

		// caption affectation and access (underscore char for "_mnemonic" specification)
		virtual void			SetCaption			(const CString &inCaption);
		CString				GetCaption			() const;

		// hotkey affectation / access (remove the old one if any)
		virtual void			SetHotKey			(const THotKey &inHotKey);
		THotKey				GetHotKey			() const;

		// OnClick event sender
		virtual void			Click				();

	// CSerialized redefinition
	public :

		// menu item xml serialization
		virtual void			Serialize			(CXMLElementNode *&ioXMLElementNode, const int inMode)
										 THROWABLE;

	// menu protected attributes
	protected :

		// menu item caption, menu item hotkey
		CString				m_Caption;
		THotKey				m_HotKey;

		// menu item potential sub menus
		GtkWidget *			m_SubMenu;

		// static associated listeners for dynamic redirection to the listener instance if any
		static void			OnSelect			(GtkItem     *, gpointer);
		static void			OnDeselect			(GtkItem     *, gpointer);
		static void			OnActivate			(GtkMenuItem *, gpointer);

		// metaclass association
		SECTION_DYNAMIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_DYNAMIC_METACLASS ('mntm', CMenuItem, CMenu);

/*******************************************************************************************************************************************
 CMenuItemSparator
*******************************************************************************************************************************************/

//-----------------------------------------------------------------------------------------------------------------------------------------
// CMenuItemSeparator class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CMenuItemSeparator : protected CMenuItem
{
	// instanciation section
	public :

		CMenuItemSeparator		(CMenu *inOwner=NULL);
		virtual ~CMenuItemSeparator	();

	// specific widget requests handling
	protected :

		// gtk widget instanciation
		virtual GtkWidget *		PerformWidgetInstanciate	();

	// CSerialized redefinition
	public :

		// jump directly to the CMenu definition, avoiding registering caption and others written by the CMenuItem definition
		virtual void			Serialize			(CXMLElementNode *&ioXMLElementNode, const int inMode)
										 THROWABLE;

		// metaclass association
		SECTION_DYNAMIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_DYNAMIC_METACLASS ('mnts', CMenuItemSeparator, CMenuItem);

/*******************************************************************************************************************************************
 CMenuItemCheck
*******************************************************************************************************************************************/

//-----------------------------------------------------------------------------------------------------------------------------------------
// cmenuitemcheck xml serialization constants
//-----------------------------------------------------------------------------------------------------------------------------------------
// <cmenuitemcheck checked="bool">
//    ...
// </cmenuitemcheck>
//-----------------------------------------------------------------------------------------------------------------------------------------
static CString XML_MENUITEMCHECK_ELEMENT	("cmenuitemcheck");
static CString XML_MENUITEMCHECK_ATTR_CHECKED	("checked");

//-----------------------------------------------------------------------------------------------------------------------------------------
// CMenuItemCheck class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CMenuItemCheck : public CMenuItem
{
	// instanciation section
	public :

		CMenuItemCheck			(CMenu *inOwner=NULL, const CString & =CString(), const THotKey & =_HKNone_, 
						 const CMenuItemListener *inListener=NULL);
		virtual ~CMenuItemCheck		();

	// specific widget requests handling
	protected :

		// gtk widget instanciation
		virtual GtkWidget *		PerformWidgetInstanciate	();

	// specific menu item check functions
	public :
		
		// menu item check access (do not generate OnClick event, use Click to do so)
		virtual void			Check				(const bool inCheck=true);
		bool				IsChecked			() const;

	// CSerialized redefinition
	public :

		// menu item check xml serialization
		virtual void			Serialize			(CXMLElementNode *&ioXMLElementNode, const int inMode)
										 THROWABLE;
		// metaclass association
		SECTION_DYNAMIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_DYNAMIC_METACLASS ('mntk', CMenuItemCheck, CMenuItem);

/*******************************************************************************************************************************************
 CMenuItemRadio
*******************************************************************************************************************************************/

//-----------------------------------------------------------------------------------------------------------------------------------------
// CMenuItemRadio class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CMenuItemRadio : public CMenuItemCheck
{
	// instanciation section
	public :

		CMenuItemRadio			(CMenu *inOwner=NULL, const CString & =CString(), const THotKey & =_HKNone_, 
						 const CMenuItemListener *inListener=NULL);
		virtual ~CMenuItemRadio		();

	// specific widget requests handling
	protected :

		// gtk widget instanciation
		virtual GtkWidget *		PerformWidgetInstanciate	();

	// CMenuItemCheck redefinition
	public :
		
		// menu item radio check (do not generate OnClick event, use Click to do so)
		virtual void			Check				(const bool inCheck=true);

		// metaclass association
		SECTION_DYNAMIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_DYNAMIC_METACLASS ('mntr', CMenuItemRadio, CMenuItemCheck);

/*******************************************************************************************************************************************
 CMenuItemImage
*******************************************************************************************************************************************/

//-----------------------------------------------------------------------------------------------------------------------------------------
// cmenuitemimage xml serialization constants
//-----------------------------------------------------------------------------------------------------------------------------------------
// <cmenuitemimage [stock-id="string"]>
//    ...
// </cmenuitemimage>
//-----------------------------------------------------------------------------------------------------------------------------------------
static CString XML_MENUITEMIMAGE_ELEMENT	("cmenuitemimage");
static CString XML_MENUITEMIMAGE_ATTR_STOCK_ID 	("stock-id");

//-----------------------------------------------------------------------------------------------------------------------------------------
// CMenuItemImage class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CMenuItemImage : public CMenuItem
{
	// instanciation section
	public :

		CMenuItemImage			(CMenu *inOwner=NULL, const CString & =CString(), const THotKey & =_HKNone_, 
						 const CMenuItemListener * =NULL);
		CMenuItemImage			(CMenu *inOwner, const char *inStockId, const CMenuItemListener *inListener=NULL);
		virtual ~CMenuItemImage		();

	// specific widget requests handling
	protected :

		// gtk widget instanciation
		virtual GtkWidget *		PerformWidgetInstanciate	();

	// specific container requests handling
	protected :

		// child widget addon or removal from the menu bar
		virtual void			PerformContainerAdd		(CWidget *inChild);
		virtual void			PerformContainerRemove		(CWidget *inChild);

	// ccomponent refinition
	public :

		// expected children types : __metaclass(CMenuItem) or __metaclass(CImage), expected owner __metaclass(CMenu)
		virtual CMetaClasses		ChildMustBe			() const;

	// cserialized redefinition
	public :

		// menu item image serialization handling
		virtual void			Serialize			(CXMLElementNode *&, const int) THROWABLE;

	// protected section
	protected :

		// menu item stock id if any
		CString				m_StockId;

		// metaclass association
		SECTION_DYNAMIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_DYNAMIC_METACLASS ('mnti', CMenuItemImage, CMenuItem);

/*******************************************************************************************************************************************
 CMenuPopup
*******************************************************************************************************************************************/

//-----------------------------------------------------------------------------------------------------------------------------------------
// CMenuPopupListener class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CMenuPopupListener : public CWidgetListener
{
	// instanciation section
	public :
	
		CMenuPopupListener		();
		virtual ~CMenuPopupListener	() =0;

	// specific popup menu listener interface
	public :

		// called when the inSender popup menu is about to be displayed
		virtual void			OnQueryPopup			(CObject *inSender, Bool &ioDoPopup)		{ }

		// called when the inSender popup menu is displayed
		virtual void			OnPopup				(CObject *inSender)				{ }

		// called when the selection menu loop ends
		virtual void			OnSelectionDone			(CObject *inSender)				{ }

		// metaclass association
		SECTION_GENERIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_GENERIC_METACLASS ('_pom', CMenuPopupListener, CWidgetListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// CMenuPopup class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CMenuPopup : public CMenu
{
	// instancation section
	public :

		CMenuPopup			(CContainer *inOwner=NULL, const CMenuPopupListener *inListener=NULL);
		virtual ~CMenuPopup		();
	
	// protected gtk widget requests handling
	protected :

		// gtk widget instanciation and initialization
		virtual GtkWidget *		PerformWidgetInstanciate	();
		virtual void			PerformWidgetInitialize		();

	// protected gtk container requests handling
	protected :

		// container child addon or removal
		virtual void			PerformContainerAdd		(CWidget *inChild);
		virtual void			PerformContainerRemove		(CWidget *inChild);

	// CObject redefinition
	protected :

		// listener affectation, must be derived from CMenuPopupListener
		virtual const CMetaClass *	ListenerMustBe			() const;

	// CControl redefinition
	public :

		// there are no gui bounds handling, let the menu item manage its coordinates and dimensions, just keep a local
		// copy of the actual ones
		virtual void			SetBounds			(const TBounds &inBounds);

	// popup menu specifics
	public :

		// get the popup menu sender that is what is the control that the popup menu is executed on
		CControl *			GetPopupSender			() const;

		// get the popup menu sender relative call point i.e. the mouse coordinates relative to the CWidget sender or to the 
		// CWidget owner of the CControl sender if sender is not a windowed widget
		TPoint				GetPopupRelativePoint		() const;

		// do popup
		void				Popup				();

	// protected attributes 
	protected :
		
		// popup menu sender, popup sender relative mouse point
		CControl *			m_Sender;
		TPoint				m_RelativePoint;

		// friend class declaration (the CWidget level manages the popup requests and affects the m_Sender and m_RelativePoint
		// attributes as needed before sending notification to the associated instance's listener if any)
		friend class			CWidget;

		// menu popup specific listener redirector
		static void			OnSelectionDone			(GtkMenuShell *, gpointer);

		// metaclass association
		SECTION_DYNAMIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_DYNAMIC_METACLASS ('popm', CMenuPopup, CMenu);

#endif



