/* 

                          Firewall Builder

                 Copyright (C) 2000 Vadim Kurland,  Vadim Zaliva

  Author:  Vadim Kurland     vadim@vk.crocodile.org
           Vadim Zaliva      lord@crocodile.org

  $Id: BackgroundOp.cc,v 1.4 2001/12/19 12:53:59 lord Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <fwbuilder/libfwbuilder-config.h>

#include <stdlib.h>
#include <stdio.h>
#include <pthread.h>
#include <unistd.h>
#include <sys/types.h>
#include <errno.h>
#include <signal.h>

#include <glib.h>

#include <fwbuilder/BackgroundOp.hh>

using namespace std;
using namespace libfwbuilder;

class SyncLogger:public Logger
{
    private:

    ostream  *strm;
    SyncFlag *lock;

    public:

    SyncLogger(ostream *s, SyncFlag *l)
    {
        strm = s;
        lock = l;
    }

    virtual ~SyncLogger() {}
    
    virtual Logger& operator<< (char c)
    {
        char s[2];
        s[0]=c;
        s[1]='\0';
        write(string(s));
        return *this;
    }
    
    virtual Logger& operator<< (char  *str)
    {
        write(string(str));
        return *this;
    }
    
    virtual Logger& operator<< (int    i)
    {
        char b[32];
        sprintf(b,"%d",i);
        write(string(b));
        return *this;
    }

    virtual Logger& operator<< (long l)
    {
        char b[32];
        sprintf(b,"%ld",l);
        write(string(b));
        return *this;
    }

    virtual Logger& operator<< (const string &str)
    {
        write(str);
        return *this;
    }

    protected:

    virtual void write(const string &s)
    {
        if(strm)
        {
            if(lock) 
            {
                lock->lock();
                if(!lock->peek())
                    *strm << s;
                lock->unlock();
            } else
            {
                *strm << s;
            }
        }
    }
};

Logger &libfwbuilder::start(Logger &l)
{
    l.line_lock.lock();
    return l;
}

Logger &libfwbuilder::end(Logger &l)
{
    l.line_lock.unlock();
    return l;
}

Logger& NullLogger::operator<< (char c)            { return *this;}
Logger& NullLogger::operator<< (char  *str)        { return *this;}
Logger& NullLogger::operator<< (const string &str) { return *this;}
Logger& NullLogger::operator<< (int    i  )        { return *this;}
Logger& NullLogger::operator<< (long   l  )        { return *this;}

StreamLogger::StreamLogger(ostream *s)
{
    stream = s;
}

Logger& StreamLogger::operator<< (char c)            { *stream << c   ; return *this;}
Logger& StreamLogger::operator<< (char  *str)        { *stream << str ; return *this;}
Logger& StreamLogger::operator<< (const string &str) { *stream << str ; return *this;}
Logger& StreamLogger::operator<< (int    i  )        { *stream << i   ; return *this;}
Logger& StreamLogger::operator<< (long   l  )        { *stream << l   ; return *this;}


BackgroundOp::BackgroundOp():stop_program(false),running(false)
{
    error = NULL;
    pool  = NULL;
}

void BackgroundOp::detach(Pool<BackgroundOp> *p)
{
    pool_mutex.lock();
    pool=p;
    pool_mutex.unlock();
}

bool  BackgroundOp::get_stop_program_flag()
{
    return stop_program;
}

void  BackgroundOp::set_stop_program_flag()
{
    stop_program=true;
}

void  BackgroundOp::clear_stop_program_flag()
{
    stop_program=false;
}

bool  BackgroundOp::get_running_flag()
{
    return running;
}

void  BackgroundOp::set_running_flag()
{
    running=true;
}

void  BackgroundOp::clear_running_flag()
{
    running=false;
}

void BackgroundOp::stop_operation()
{
    error = new FWException("Interrupted by user");
    set_stop_program_flag();
}

void BackgroundOp::operation_completed()
{
    clear_running_flag();
}

pthread_t BackgroundOp::start_operation(ostream *l) throw(FWException)
{
    // clear error - this instance of BackgroundOp may be reused, so
    // error may still carry old error value (vk)
    if(error) 
    {
        delete error;
        error = NULL;
    }

    clear_stop_program_flag();
    set_running_flag();

    //Gtk::Main::idle.connect(SigC::slot(this,&BackgroundOp::monitor_operation));
    
    void **void_pair = new void*[2];
    void_pair[0] = this;
    void_pair[1] = new SyncLogger(l, &stop_program);
    pthread_t tid;
    pthread_create(&tid, NULL, background_thread, void_pair);
    pthread_detach(tid);
    return tid;
}

gint BackgroundOp::monitor_operation()
{
    if(get_running_flag()==false) 
    {
        // sends signal, argument is 0 if ok and -1 if error
        if(get_stop_program_flag()==false)
            completed((error)?-1:0); 

        pool_mutex.lock();
        if(pool)
            pool->release(this);
        pool_mutex.unlock();
        
        return 0;
    } else
    {
        return 1;
    }
}

void *libfwbuilder::background_thread(void *args)
{
    void **void_pair=(void**)args;

    BackgroundOp *q = (BackgroundOp*)void_pair[0];
    Logger       *l = (Logger *)     void_pair[1];
    
    try 
    {
	q->run_impl(l);
    } catch (FWException &ex) 
    {
        if(q->get_stop_program_flag()==false)
	    *l << "Exception: " << ex.toString() << '\n';
	q->error=new FWException(ex);
    }

    q->operation_completed();
    delete void_pair;

    return(NULL);
}






