/*
  CoreLinux++ 
  Copyright (C) 2000 CoreLinux Consortium
  
   The CoreLinux++ Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The CoreLinux++ Library Library is distributed in the hope that it will 
   be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  
*/   

#if   !defined(__COMMON_HPP)
#include <corelinux/Common.hpp>
#endif

#if   !defined(__CORELINUXASSOCIATIVEITERATOR_HPP)
#include <corelinux/CoreLinuxAssociativeIterator.hpp>
#endif

#if   !defined(__SUBJECTOBSERVER_HPP)
#include <SubjectObserver.hpp>
#endif

#if   !defined(__EVENTS_HPP)
#include <Events.hpp>
#endif

using namespace corelinux;

// Default constructor

SubjectObserver::SubjectObserver( void )
   :
   Subject(),
   Observer(),
   theObserverMap()
{
   theObserverMap.clear();
}

// Copy constructor

SubjectObserver::SubjectObserver( SubjectObserverCref anotherSO ) throw ( Assertion )
   :
   Subject(anotherSO),
   Observer(anotherSO),
   theObserverMap()
{
   NEVER_GET_HERE;
}


// Virtual destructor

SubjectObserver::~SubjectObserver( void )
{
   theObserverMap.clear();
}

// Assignment operator

SubjectObserverRef SubjectObserver::operator=( SubjectObserverCref )
   throw ( Assertion )
{
   NEVER_GET_HERE;
   return ( *this );
}

// Equality operator

bool  SubjectObserver::operator=( SubjectObserverCref anotherSO ) const
{
   return ( this == &anotherSO );
}
            
// Add an observer

void  SubjectObserver::addObserver
   ( 
      ObserverPtr aObserver, 
      Event<corelinux::Identifier> *forEvent
   )
   throw ( NullPointerException ) 
{
   if( aObserver == NULLPTR || forEvent == NULLPTR )
   {
      throw NullPointerException( LOCATION );
   }
   else
   {
      //
      // Check if we have the observer already, if not we add it
      //

      ListEventPtr         anEvent( dynamic_cast<ListEventPtr>(forEvent) );
      DwordIdentifier      aDI( *DwordIdentifierPtr(*anEvent) );
      ObserverMapIterator  lowerB = theObserverMap.lower_bound( aDI ) ;
      ObserverMapIterator  upperB = theObserverMap.upper_bound( aDI ) ;
      bool                 notFound( true );

      while( lowerB != upperB && notFound == true )
      {
         if( (*lowerB).second == aObserver )
         {
            notFound = false;
         }
         else
         {
            ++lowerB;
         }
      }

      if( notFound == true )
      {
         theObserverMap.insert( ObserverMap::value_type(aDI,aObserver) );
      }
      else
      {
         ;  // do nothing
      }
   }
}

// Take an observer off the subject notification list

void  SubjectObserver::removeObserver( ObserverPtr aObserver ) 
   throw ( NullPointerException ) 
{
   if( aObserver == NULLPTR )
   {
      throw NullPointerException( LOCATION );
   }
   else
   {
      ObserverMapIterator  begin( theObserverMap.begin() );
      ObserverMapIterator  end( theObserverMap.end() );
      while( begin != end )
      {
         if( (*begin).second == aObserver )
         {
            ObserverMapIterator  tmpItr( begin );
            ++tmpItr;
            theObserverMap.erase(begin);
            begin = tmpItr;
         }
         else
         {
            ++begin;
         }
      }
   }
}

void  SubjectObserver::removeObserver
   ( 
      ObserverPtr aObserver, 
      Event<corelinux::Identifier> *forEvent
   )
   throw ( NullPointerException ) 
{
   if( aObserver == NULLPTR || forEvent == NULLPTR )
   {
      throw NullPointerException( LOCATION );
   }
   else
   {
      //
      // Check if we have the observer already, if not we add it
      //

      ListEventPtr         anEvent( dynamic_cast<ListEventPtr>(forEvent) );
      DwordIdentifier      aDI( *DwordIdentifierPtr(*anEvent) );
      ObserverMapIterator  lowerB( theObserverMap.lower_bound( aDI ) );
      ObserverMapIterator  upperB( theObserverMap.upper_bound( aDI ) );

      while( lowerB != upperB )
      {
         if( (*lowerB).second == aObserver )
         {
            ObserverMapIterator  tmpItr( lowerB );
            ++tmpItr;
            theObserverMap.erase(lowerB);
            lowerB = tmpItr;
         }
         else
         {
            ++lowerB;
         }
      }
   }
}

// Iterator Factory for all observers

Iterator<ObserverPtr> *SubjectObserver::createIterator( void )
{
   Iterator<ObserverPtr>  *pList(NULLPTR);

   pList = new CoreLinuxAssociativeIterator<ObserverMapIterator,DwordIdentifier,ObserverPtr>
      ( 
         theObserverMap.begin(),
         theObserverMap.end()
      );

   ENSURE( pList != NULLPTR );
   return pList;
}

// Iterator Factory for all observers interest in this event

Iterator<ObserverPtr> *SubjectObserver::createIterator
   ( 
      Event<corelinux::Identifier>  *anId
   )
   throw ( NullPointerException ) 
{
   Iterator<ObserverPtr>   *pList(NULLPTR);

   if( anId == NULLPTR )
   {
      throw NullPointerException(LOCATION);
   }
   else
   {
      ListEventPtr            anEvent( dynamic_cast<ListEventPtr>(anId) );

      DwordIdentifier         aDI( *DwordIdentifierPtr(*anEvent) );

      pList = new CoreLinuxAssociativeIterator<ObserverMapIterator,DwordIdentifier,ObserverPtr>
         ( 
            theObserverMap.lower_bound(aDI),
            theObserverMap.upper_bound(aDI)
         );

      if( pList == NULLPTR )
      {
         throw NullPointerException(LOCATION);
      }
      else
      {
         ;  // do nothing
      }
   }
   return pList;
}

// Destroy iterator

void SubjectObserver::destroyIterator( Iterator<ObserverPtr> *aIterator )
   throw ( NullPointerException ) 
{
   if( aIterator == NULLPTR )
   {
      throw NullPointerException(LOCATION);
   }
   else
   {
      delete aIterator;
   }
}

/*
   Common rcs information do not modify
   $Author: frankc $
   $Revision: 1.1 $
   $Date: 2000/05/08 22:29:04 $
   $Locker:  $
*/


