
#
# Macro:      AP_PROG_APXS
# Author:     Heiko Hund <heiko@ist.eigentlich.net>
# Version:    $Revision: 48 $ $Date: 2008-04-04 06:52:51 +0200 (Fri, 04 Apr 2008) $
# Parameters: none
# Requires:   none
# Output:     @APXS@ - path to apxs
#
# The AP_PROG_APXS macro looks for the apache extension tool (apxs or apxs2)
# in $PATH. It's also possible to give autoconf the right path to apxs via
# `APXS' environment variable or `--with-apxs' switch on the command line.
# If no apxs can be found an error is displayed and configure stops.
#
AC_DEFUN([AP_PROG_APXS],
[
	AC_ARG_VAR([APXS], [path to apache extension tool (apxs)])
	AC_ARG_WITH(
		[apxs],
		[AS_HELP_STRING([--with-apxs=FILE], [path to apache extension tool (apxs)])],
		[
			AC_MSG_CHECKING([for apxs])
			if test ! -x "$withval"
			then
				AC_MSG_RESULT([no ($withval is not executable)])
			else
				AC_MSG_RESULT([$withval])
				AC_SUBST([APXS], [$withval])
			fi
		],
		[AC_PATH_PROGS([APXS], [apxs apxs2])]
	)

	if test "x$APXS" = "x"
	then
		AC_MSG_ERROR([The apache extension tool (apxs) was not found in your PATH. You
                  might want to specify the correct path using `--with-apxs'.])
	fi
])



#
# Macro:      AP_LIB_APR
# Author:     Heiko Hund <heiko@ist.eigentlich.net>
# Version:    $Revision: 48 $ $Date: 2008-04-04 06:52:51 +0200 (Fri, 04 Apr 2008) $
# Parameters: none
# Requires:   AP_PROG_APXS
# Output:     @APR_CFLAGS@    - C compiler flags needed by APR
#             @APR_CPPFLAGS@  - C preprocessor flags needed by APR
#             @APR_LDFLAGS@   - linker flags needed by APR
#             @APR_LIBS@      - libraries that need to be linked for APR
# Defines:    HAVE_LIBAPR     - set 1 if libapr is installed and working
#             HAVE_LIBAPRUTIL - set 1 if libaprutil is installed and working
#
# The AP_LIB_APR macro sets some common variables needed if you use libapr or
# libaprutil in your module. It also checks that the libraries can be linked
# without an error.
#
AC_DEFUN([AP_LIB_APR],
[
	AC_REQUIRE([AP_PROG_APXS])

	ap_apr_config="`$APXS -q APR_CONFIG 2>/dev/null`"
	if test "x$ap_apr_config" != "x"
	then
		ap_apr_config_name="`basename $ap_apr_config`"
		AC_MSG_CHECKING([for $ap_apr_config_name])
		if test ! -x "$ap_apr_config"
		then
			AC_MSG_RESULT([no])
			AC_MSG_ERROR([$ap_apu_config_name is either not installed or not executable.
                   Please check your httpd installation.])
		fi
		AC_MSG_RESULT([$ap_apr_config])

		ap_apr_cflags="`$ap_apr_config --cflags`"
		ap_apr_cppflags="`$ap_apr_config --includes`"
		ap_apr_ldflags="`$ap_apr_config --ldflags`"
		ap_apr_libs="`$ap_apr_config --libs --apr-la-file`"
		ap_apr_libname="`$ap_apr_config --link-ld | sed -e 's/.*-l\([[^[:space:]]]*\).*/\1/'`"

		_ldflags="$LDFLAGS"
		LDFLAGS="$LDFLAGS `$ap_apr_config --link-ld`"
		AC_CHECK_LIB([$ap_apr_libname], [apr_initialize], [ap_apr_err=no],
		             [ap_apr_err=yes], [`$ap_apr_config --libs`])
		LDFLAGS="$_ldflags"
		if test "x$ap_apr_err" = "xyes"
		then
			AC_MSG_FAILURE([linking of `lib$ap_apr_libname' failed])
		fi

		AC_DEFINE([HAVE_LIBAPR], [1], [Define to 1 if you have the `apr' library.])
	fi

	ap_apu_config="`$APXS -q APU_CONFIG 2>/dev/null`"
	if test "x$ap_apu_config" != "x"
	then
		ap_apu_config_name="`basename $ap_apu_config`"
		AC_MSG_CHECKING([for $ap_apu_config_name])
		if test ! -x "$ap_apu_config"
		then
			AC_MSG_RESULT([no])
			AC_MSG_ERROR([$ap_apu_config_name is either not installed or not executable.
                   Please check your httpd installation.])
		fi
		AC_MSG_RESULT([$ap_apu_config])

		ap_apu_cppflags="`$ap_apu_config --includes`"
		ap_apu_ldflags="`$ap_apu_config --ldflags`"
		ap_apu_libs="`$ap_apu_config --libs --apu-la-file`"
		ap_apu_libname="`$ap_apu_config --link-ld | sed -e 's/.*-l\([[^[:space:]]]*\).*/\1/'`"

		_ldflags="$LDFLAGS"
		LDFLAGS="$LDFLAGS `$ap_apu_config --link-ld`"
		AC_CHECK_LIB([$ap_apu_libname], [apr_dynamic_fn_register], [ap_apu_err=no],
		             [ap_apu_err=yes], [`$ap_apu_config --libs`])
		LDFLAGS="$_ldflags"
		if test "x$ap_apu_err" = "xyes"
		then
			AC_MSG_FAILURE([linking of `lib$ap_apu_libname' failed])
		fi

		AC_DEFINE([HAVE_LIBAPRUTIL], [1], [Define to 1 if you have the `aprutil' library.])
	fi

	AC_SUBST([APR_CFLAGS], ["$ap_apr_cflags"])
	AC_SUBST([APR_CPPFLAGS], ["$ap_apr_cppflags $ap_apu_cppflags"])
	AC_SUBST([APR_LDFLAGS], ["$ap_apr_ldflags $ap_apu_ldflags"])
	AC_SUBST([APR_LIBS], ["$ap_apr_libs $ap_apu_libs"])
])



#
# Macro:      AP_HTTPD_VERSION
# Author:     Heiko Hund <heiko@ist.eigentlich.net>
# Version:    $Revision: 48 $ $Date: 2008-04-04 06:52:51 +0200 (Fri, 04 Apr 2008) $
# Parameters: [MATCH_1 [MATCH_2 [MATCH_n]]]
# Requires:   AP_PROG_APXS
# Output:     @HTTPD_VERSION@ - Apache HTTPD version
#
# The AP_HTTPD_VERSION macro uses apxs to determine the path of the apache
# httpd binary and invokes it with the `-v' command line switch. It extracts
# the version string from the output and compares it to any given parameter.
# You can give zero or more `MATCH' parameters to the macro to make sure
# sure HTTPD has a certain required version. Parameters must be separated by
# whitespace.
#
# The format for the parameters is: [COMPARATOR]VERSION
# The `VERSION' part can be any Apache HTTPD version (sub)string to compare to.
# The optional COMPARATOR' part can be one of:
#    =  equal to (default if not given)
#    >  greater than
#    <  less than
#   !=  not equal to
#   >=  greater or equal
#   <=  less or equal
#
# Thus, the strings `>1' and `>=2.0.0' both match HTTPD version 2.x.x and higher.
# All parameters must match the HTTPD version (logical AND) or a error message
# is displayed and execution of configure is halted.
#
AC_DEFUN([AP_HTTPD_VERSION],
[
	AC_REQUIRE([AP_PROG_APXS])
	ap_httpd="`$APXS -q SBINDIR`/`$APXS -q TARGET`"
	ap_httpd_version="`$ap_httpd -v | grep 'version' | sed -e 's!.*/\([[[:digit:].]]*\).*$!\1!'`"

	if test "x$1" != "x"
	then
		AC_MSG_CHECKING([for apache httpd version])
		AC_MSG_RESULT([$ap_httpd_version])
		for ap_param in m4_foreach_w([var], m4_quote($*), ['var' ])
		do
			ap_param_op="`echo $ap_param | sed -e 's/[[^!<>=]]*\([[!<>=]]*\).*/\1/'`"
			ap_param_ver="`echo $ap_param | sed -e 's/[[^[:digit:]]]*\([[[:digit:].]]*[[[:digit:]]]\).*/\1/'`"
			ap_param_ver_size="$((`echo $ap_param_ver | tr '.' ' ' | wc -w`))"

			ap_i=1
			while test $ap_i -le $ap_param_ver_size
			do
				ap_tmp="`echo $ap_httpd_version | cut -d. -f$ap_i`"
				ap_cmp_ver1="`printf "%s%6s" "$ap_cmp_ver1" ${ap_tmp:-0} | tr ' ' 0`"
				ap_tmp="`echo $ap_param_ver | cut -d. -f$ap_i`"
				ap_cmp_ver2="`printf "%s%6s" "$ap_cmp_ver2" $ap_tmp | tr ' ' 0`"
				ap_i="$(($ap_i+1))"
			done

			if ! expr "$ap_cmp_ver1" "${ap_param_op:-=}" "$ap_cmp_ver2" >/dev/null
			then
				ap_mod_works="No"
				break
			fi

			unset ap_cmp_ver1
			unset ap_cmp_ver2
		done
		if test "x$ap_works" = "xNo"
		then
			AC_MSG_ERROR([apache httpd version $* is required])
		fi
	fi

	AC_SUBST([HTTPD_VERSION])
])



#
# Macro:      AP_HTTPD_MODULE
# Author:     Heiko Hund <heiko@ist.eigentlich.net>
# Version:    $Revision: 48 $ $Date: 2008-04-04 06:52:51 +0200 (Fri, 04 Apr 2008) $
# Parameters: MATCH_1 [MATCH_2 [MATCH_n]]
# Requires:   AP_HTTPD_VERSION
# Output:     @AP_MOD_DIR@      - Apache HTTPD modules directory
#             @AP_MOD_CFLAGS@   - CFLAGS for module compilation
#             @AP_MOD_CPPFLAGS@ - CPPFLAGS for module preprocessing
#             @AP_MOD_LDFLAGS@  - LDFLAGS for linking modules
#
# The AP_HTTPD_MODULE macro uses apxs to gather information needed to build
# dynamically loadable modules for Apache HTTPD. It takes one or more `MATCH'
# parameters that ensure the module is built for the Apache version the module
# was made for. If multiple parameters are given, they must be separated by
# whitespace.
#
# The format for the parameters is: [COMPARATOR]VERSION
# The `VERSION' part can be any Apache HTTPD version (sub)string to compare to.
# The optional COMPARATOR' part can be one of:
#    =  equal to (default if not given)
#    >  greater than
#    <  less than
#   !=  not equal to
#   >=  greater or equal
#   <=  less or equal
#
# Thus, the strings `>1' and `>=2.0.0' both match HTTPD version 2.x.x and higher.
# All parameters must match the HTTPD version (logical AND) or a error message
# is displayed and execution of configure is halted.
#
AC_DEFUN([AP_HTTPD_MODULE],
[
	AC_REQUIRE([AP_HTTPD_VERSION])
	AC_REQUIRE([AP_LIB_APR])

	AC_MSG_CHECKING([if apache httpd supports loadable modules])
	if ! $ap_httpd -l | grep mod_so >/dev/null
	then
		AC_MSG_RESULT([no])
		AC_MSG_ERROR([Make sure to compile mod_so into your apache httpd binary.])
	fi
	AC_MSG_RESULT([yes])

	AC_MSG_CHECKING([if module works with apache httpd version $ap_httpd_version])
	for ap_param in m4_foreach_w([var], m4_quote($*), ['var' ])
	do
		ap_param_op="`echo $ap_param | sed -e 's/[[^!<>=]]*\([[!<>=]]*\).*/\1/'`"
		ap_param_ver="`echo $ap_param | sed -e 's/[[^[:digit:]]]*\([[[:digit:].]]*[[[:digit:]]]\).*/\1/'`"
		ap_param_ver_size="$((`echo $ap_param_ver | tr '.' ' ' | wc -w`))"

		ap_i=1
		while test $ap_i -le $ap_param_ver_size
		do
			ap_tmp="`echo $ap_httpd_version | cut -d. -f$ap_i`"
			ap_cmp_ver1="`printf "%s%6s" "$ap_cmp_ver1" ${ap_tmp:-0} | tr ' ' 0`"
			ap_tmp="`echo $ap_param_ver | cut -d. -f$ap_i`"
			ap_cmp_ver2="`printf "%s%6s" "$ap_cmp_ver2" $ap_tmp | tr ' ' 0`"
			ap_i="$(($ap_i+1))"
		done

		if ! expr "$ap_cmp_ver1" "${ap_param_op:-=}" "$ap_cmp_ver2" >/dev/null
		then
			ap_mod_works="No"
			break
		fi

		ap_cmp_ver1=
		ap_cmp_ver2=
	done
	if test "x$ap_mod_works" = "xNo"
	then
		AC_MSG_RESULT([no])
		AC_MSG_ERROR([module only works with apache httpd version $*])
	else
		AC_MSG_RESULT([yes ($*)])
	fi

	ap_cflags="`$APXS -q CFLAGS` `$APXS -q CFLAGS_SHLIB` `$APXS -q SHLTCFLAGS 2>/dev/null` `$APXS -q EXTRA_CFLAGS 2>/dev/null`"
	ap_cppflags="`$APXS -q NOTEST_CPPFLAGS 2>/dev/null` `$APXS -q EXTRA_CPPFLAGS 2>/dev/null` -I`$APXS -q INCLUDEDIR`"
	ap_ldflags="-avoid-version `$APXS -q LDFLAGS_SHLIB` `$APXS -q LIBS_SHLIB`"

	AC_SUBST([AP_MOD_CFLAGS], ["$ap_cflags"])
	AC_SUBST([AP_MOD_CPPFLAGS], ["$ap_cppflags $APR_CPPFLAGS"])
	AC_SUBST([AP_MOD_LDFLAGS], ["$ap_ldflags"])
	AC_SUBST([AP_MOD_DIR], ["`$APXS -q LIBEXECDIR`"])
])
