/* ======================================================================
 * Copyright (c) 1998-1999 The Johns Hopkins University.
 * All rights reserved.
 * The following code was written by Theo Schlossnagle for use in the
 * Backhand project at The Center for Networking and Distributed Systems
 * at The Johns Hopkins University.
 * Please refer to the LICENSE file before using this software.
 * ======================================================================
*/

/* NOTE: If we compile with pthreads, it is NOT necessary to compile
   with -D_REENTRANT unless gettimeofday is not reentrant already.
   That is the ONLY system call we make and we use NO shared
   variables... So Lets not take any hits by defining _REENTRANT */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/wait.h>

#ifdef PTHREADS
#include <pthread.h>
#endif

#define BB_ITERS 6666
#define BEGIN gettimeofday(&start, NULL)
#define END gettimeofday(&end, NULL)
#define CPS(a) (((float)(a))/timediff(start, end))

static float timediff(struct timeval a, struct timeval b) {
  long sec, usec;
  float elapsed;
  sec = b.tv_sec-a.tv_sec;
  usec = b.tv_usec-a.tv_usec;
  elapsed = (float)sec + (usec/1000000.0);
  return elapsed;
}

static void *time_consumption(void *st) {
  float z[101] = {0.0};
  float *elapsed;
  struct timeval *start=st, end;
  int i, j;
  for(i=0;i<BB_ITERS;i++)
    for(j=0;j<101;j++) {
      z[j] = (!z[j])?j: (z[(i-1)%101]+1.0)*(z[(i+1)%101]+1.0);
      if(z[j]>100000000.0 || z[j]<0) z[j]=3.0;
    }
  elapsed = (float *)malloc(sizeof(float));
  END;
  *elapsed = timediff(*start, end);
#ifdef PTHREADS
  pthread_exit(elapsed);
#else
  return elapsed;
#endif
}

int backhand_bench(void) {
  int pa=-1;
#define NUMTHREADS 12 /* This will exploit capabilities for
			 machines with up to 4 processors */
  struct timeval start;
#ifdef PTHREADS
  pthread_t threads[NUMTHREADS]; 
  int i;
#else /* PNOTHREADS */
  pid_t threads[NUMTHREADS] = {0}; /* Not really thread, less code changes */
  int i,status;
#endif
  float elapsed=0.0, *partialtime;
  BEGIN;
#ifdef PTHREADS
  for(i=0;i<NUMTHREADS;i++) {
    pthread_create(&threads[i], NULL, time_consumption, &start);
  }
  for(i=0;i<NUMTHREADS;i++) {
    pthread_join(threads[i], (void **)&partialtime);
    /* This is in case we need it (only works well with Solaris Threads
       though.. because we can wait on first exiting thread) */
    elapsed+=*partialtime;
    free(partialtime);
  }
#else /* PNOTHREADS */
  for(i=0;i<NUMTHREADS;i++) {
    if((threads[i] = fork())==0) {
      /* Child... */
      time_consumption(&start);
      exit(0);
    }
  }
  for(i=0;i<NUMTHREADS;i++) {
    if(threads[i]>0) {
      waitpid(threads[i], &status, 0);
    }
  }
#endif /* PTHREADS */
  /* Currently we DON'T use the above partialtimes (from THREADS)
     this is intentional, if we find it useful to incomporate it into
     our Arriba, then it will be simple */
  partialtime = (float *)time_consumption(&start);
  elapsed = *partialtime;
  free(partialtime);
  pa = (BB_ITERS*101)/elapsed;
  return pa;
}

#ifdef ARRIBA_TEST
int main() {
  printf("mod_backhand benchmark: %d\n", backhand_bench());
}
#endif
